/* This program tests the rate at which timer interrupts can be generated to a process */

#ifdef SVR3						/* System V */
#    include <sys/types.h>
#    include <sys/times.h>
#    include <sys/param.h>
#    define AccountInfo tms
#    define AccountData(storage) times( &storage )
#    define Time( user ) user.tms_utime * ( 1000000 / HZ )  /* time in usec */
#else							/* BSD */
#    include <sys/time.h>
#    include <sys/resource.h>
#    define AccountInfo rusage
#    define AccountData(storage) getrusage( RUSAGE_SELF, &storage )
#    define Time( user ) user.ru_utime.tv_sec * 1000000 + user.ru_utime.tv_usec  /* time in usec */
#endif

#include <signal.h>
#include <sys/time.h>

struct itimerval value;

int itimer_real;

int alarm() {
    itimer_real += 1;
    signal( SIGALRM, alarm );				/* System V requires handler be reset */
    setitimer( ITIMER_REAL, &value, 0 );
}

int itimer_virtual;

int vtalarm() {
    itimer_virtual += 1;
    signal( SIGVTALRM, vtalarm );			/* System V requires handler be reset */
    setitimer( ITIMER_VIRTUAL, &value, 0 );
}

int itimer_prof;

int prof() {
    itimer_prof += 1;
    signal( SIGPROF, prof );				/* System V requires handler be reset */
    setitimer( ITIMER_PROF, &value, 0 );
}

#define NoOfTestTimes 6
struct timeval TestTime[NoOfTestTimes] = { {0, 10}, {0, 100}, {0, 1000}, {0, 10000}, {0, 100000}, {1, 0} };

main() {
    int test, delay;
    int StartTime, EndTime;
    struct AccountInfo usage;
    
    for ( test = 0; test < NoOfTestTimes; test += 1 ) {
	itimer_real = 0;
	itimer_virtual = 0;
	itimer_prof = 0;
	
	value.it_value.tv_sec = TestTime[test].tv_sec;
	value.it_value.tv_usec = TestTime[test].tv_usec;
	value.it_interval.tv_sec = 0;
	value.it_interval.tv_usec = 0;
	
	signal( SIGALRM, alarm );
	signal( SIGVTALRM, vtalarm );
	signal( SIGPROF, prof );
    
	setitimer( ITIMER_REAL, &value, 0 );
	setitimer( ITIMER_VIRTUAL, &value, 0 );
	setitimer( ITIMER_PROF, &value, 0 );
	
	AccountData( usage );
	StartTime = Time( usage );
	
	for ( delay = 0; delay <= 5000000; delay += 1 ) {}
	
	AccountData( usage );
	EndTime = Time( usage );
	
	printf( "Interrupts every %7d usecs, ", TestTime[test].tv_sec * 1000000 + TestTime[test].tv_usec );
	printf( "Total time for delay %9d usecs, ", EndTime - StartTime );
	printf( "No. of Interrupts: real %5d, virtual %5d, prof %5d\n", itimer_real, itimer_virtual, itimer_prof );
    }
} /* main */
