/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.4.3, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uStream.c -- Stream I/O routines.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:17:47 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Tue Jan  4 19:10:30 1994
 * Update Count     : 464
 */

/* change the following to a 1 if you want the debugging code included */

/* #define __DEBUG_IO__ */

#define __U_KERNEL__

#include <uUnix.h>
#include <uSystem.h>
#include <uStream.h>

#include <sys/file.h>
#include <stdarg.h>

#if defined( __ibm__ )
#define U_FNDELAY FNONBLOCK				/* POSIX non-blocking */
#define U_EWOULDBLK EAGAIN
#elif defined( __sgi__ )
#define U_FNDELAY FNONBLK				/* POSIX non-blocking */
#define U_EWOULDBLK EAGAIN
#elif defined( __hp__ )
#define U_FNDELAY O_NONBLOCK				/* POSIX non-blocking */
#define U_EWOULDBLK EAGAIN
#elif defined( __sun__ ) && defined( __svr4__ )
#define U_FNDELAY O_NONBLOCK				/* POSIX non-blocking */
#define U_EWOULDBLK EAGAIN
#else
#define U_FNDELAY FNDELAY
#define U_EWOULDBLK EWOULDBLOCK
#endif

/*
 * This type is used to indicate whether and when a stream should be set into
 * non-blocking mode.
 *
 * U_FNDELAY_NEVER:  it never blocks (e.g. file access) so don't bother with FNDELAY
 * U_FNDELAY_DEMAND: it blocks, but setting it to non-blocking mode at open is too
 *                      dangerous beause an interrupt can leave the file in non-blocking
 *                      mode after the application terminates (e.g. stdin).
 *			To mitigate the problem, set it to non-blocking mode on each syscall.
 * U_FNDELAY_ALWAYS: it blocks.  Set it to non-blocking mode when the file is opened.
 *			It is reset to the original mode when the file is closed.
 */

typedef enum { U_FNDELAY_NEVER, U_FNDELAY_DEMAND, U_FNDELAY_ALWAYS } uBlockVal;

typedef struct uStreamD {
    uSemaphore mutex;
    FILE *file;
    uCluster cluster;
    uBlockVal block;
} uStreamD;

uStream uStdin  = NULL;
uStream uStdout = NULL;
uStream uStderr = NULL;

void uInstallStandardStream( void ) {
    uStdin = uFinstall( stdin );
    uStdout = uFinstall( stdout );
    uStderr = uFinstall( stderr );
} /* uInstallStandardStream */

void uDestallStandardStream( void ) {
    uFdestall( uStdin );
    uFdestall( uStdout );
    uFdestall( uStderr );
} /* uInstallStandardStream */

inline void uStreamEnter( uStream stream ) {
    uMigrate( stream->cluster );    
    uP( &(stream->mutex) );
} /* uStreamEnter */

inline void uStreamLeave( uStream stream, uCluster prevClus ) {
    uV( &(stream->mutex) );
    uMigrate( prevClus );
} /* uStreamLeave */

inline static uStream uCreateStream() {
    uStream stream;

    stream = uMalloc( sizeof( uStreamD ) );		/* allocate a stream descriptor */
    stream->mutex = U_SEMAPHORE( 1 );			/* initialize the semaphore */
    stream->cluster = uCreateCluster( 1, 0 );		/* create cluster with one processor and no time slicing */

    return stream;
} /* uCreateStream */

inline static void uDestroyStream( uStream stream ) {
    uDestroyCluster( stream->cluster );
    uFree( stream );
} /* uDestroyStream */

inline static uBlockVal uDoesStreamBlock( uStream stream ) {
    /*
     * Determine the type of the file, so that we know whether to block on access or not.
     */

    struct stat buf;
    int fd = fileno( stream->file );

    if ( fstat( fd, &buf ) < 0 ) {			/* can't stat the file...  set the FNDELAY flag on demand */
	return U_FNDELAY_DEMAND;
    } /* if */
    	
    /*
     * examine the "type" bits
     * 
     * File access may block if file type is:
     * 	S_IFIFO  : named pipe
     * 	S_IFCHR  : character special file (tty/pty)
     * 	S_IFSOCK : socket
     */

    if ( ( buf.st_mode & S_IFCHR ) == S_IFCHR ) {	/* if a tty, always on demand (only has bearing on input) */
	return U_FNDELAY_DEMAND;
#ifdef S_IFIFO
    } else if ( ( buf.st_mode & S_IFIFO ) == S_IFIFO ) {
	return U_FNDELAY_ALWAYS;
#endif
    } else if ( ( buf.st_mode & S_IFSOCK ) == S_IFSOCK ) {
	return U_FNDELAY_ALWAYS;
    } else {
	return U_FNDELAY_NEVER;
    } /* if */

} /* uDoesStreamBlock */

inline static void uSetBlockFlag( uStream stream ) {

    int flags;
    
    flags = fcntl( fileno( stream->file ), F_GETFL );
    if ( flags == -1 ) {
	stream->block = U_FNDELAY_NEVER;
	return;
    } /* if */

    flags = fcntl( fileno( stream->file ), F_SETFL, flags | U_FNDELAY );
    if ( flags == -1 ) {
	stream->block = U_FNDELAY_NEVER;
	return;
    } /* if */

} /* uSetBlockFlag */

inline static void uClearBlockFlag( uStream stream ) {

    int flags;
    
    flags = fcntl( fileno( stream->file ), F_GETFL );
    if ( flags == -1 ) {
	stream->block = U_FNDELAY_NEVER;
	return;
    } /* if */

    flags = fcntl( fileno( stream->file ), F_SETFL, flags & ~ U_FNDELAY );
    if ( flags == -1 ) {
	stream->block = U_FNDELAY_NEVER;
	return;
    } /* if */

} /* uClearBlockFlag */

inline static void uSetStreamAccess( uStream stream ) {
    /*
     * if the stream is a blocking stream, set the fcntl bits to allow nonblocking access
     */

    if ( stream->block == U_FNDELAY_ALWAYS ) uSetBlockFlag( stream );
} /* uSetStreamAccess */

inline static void uClearStreamAccess( uStream stream ) {
    /*
     * if the stream is a blocking stream, set the fcntl bits to what they originally were
     */

    if ( stream->block == U_FNDELAY_ALWAYS ) uClearBlockFlag( stream );
} /* uClearStreamAccess */

static int uCheckStream( int delay ) {

    int found;
    struct timeval timeout = { delay, 0 };
    uCluster cluster = uThisCluster();
    uProcessor processor = uThisProcessor();
    fd_set rfds = cluster->rfds;			/* make local copy of fd sets because select operation destroys */
    fd_set wfds = cluster->wfds;			/* the sets and the io operation pending information is lost */
    fd_set efds = cluster->efds;
    int fd;
    
    processor->state = U_IDLE;				/* change this processor's state to idle */
    cluster->ready.idle = U_TRUE;			/* set the idle flag so that the processor gets woken up */

#ifdef __DEBUG_IO__
    for ( found = 0; found < FD_SETSIZE; found += 1 ) {
	if ( FD_ISSET( found, &(cluster->rfds) ) ) fprintf( stderr, "(0x%x) rfds %d is set\n", uThisTask(), found );
	if ( FD_ISSET( found, &(cluster->wfds) ) ) fprintf( stderr, "(0x%x) wfds %d is set\n", uThisTask(), found );
	if ( FD_ISSET( found, &(cluster->efds) ) ) fprintf( stderr, "(0x%x) efds %d is set\n", uThisTask(), found );
    } /* for */
#endif

#ifdef __DEBUG_IO__
    fprintf( stderr, "(0x%x) before select\n", uThisTask() );
#endif
    found = select( FD_SETSIZE, &rfds, &wfds, &efds, &timeout );
#ifdef __DEBUG_IO__
    fprintf( stderr, "(0x%x) after select, found:%d, errno:%d\n", uThisTask(), found, errno );
#endif

    processor->state = U_BUSY;				/* set processor state back to busy */
    cluster->ready.idle = U_FALSE;			/* set the cluster idle back to active */

    if ( found > 0 ) {					/* io is ready */
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) io is available, clearing masks and releasing blocked tasks\n", uThisTask() );
#endif
	
	uP( &(cluster->mutex) );			/* acquire the cluster */
	
	cluster->iopoller = 0;				/* nobody is the first io task */

	/* check to see which io operations were ready. */
	/* wake the tasks for those io operations which */
	/* are ready.  no need to clear the bits because */
	/* they are only local bits set by the select operation */

	for ( fd = 0; fd < FD_SETSIZE; fd += 1 ) {


	    if ( FD_ISSET( fd, &rfds ) ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

	    } else if ( FD_ISSET( fd, &wfds ) ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

	    } else if ( FD_ISSET( fd, &efds ) ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

	    } /* if */

	} /* for */
	
	/* must ensure that at least one task that does not */
	/* have io ready must be woken up so that at least */
	/* one task will resume the responsibilities of the poller.  */
	/* it may be the case that all tasks that wake up for io */
	/* will go onto do other things, and nobody will check for */
	/* io for those tasks that are still sleeping. */
	
	for ( fd = 0; fd < FD_SETSIZE; fd += 1 ) {

	    if ( cluster->iocount[fd] != 0 ) {

		for ( ; cluster->iocount[fd] != 0; cluster->iocount[fd] -= 1 ) {
		    uV( &(cluster->ioqueue[fd]) );
		} /* for */

		/* since we share one semaphore variable for all pending io */
		/* we wake up all tasks waiting for io on this fd and clear */
		/* all io pending bits */
		
		FD_CLR( fd, &cluster->rfds );
		FD_CLR( fd, &cluster->wfds );
		FD_CLR( fd, &cluster->efds );

		break;
		
	    } /* if */
	    
	} /* for */
	
	uV( &(cluster->mutex) );			/* release the cluster */
	
	return 1;					/* tell the caller io is ready */
	
    } else if ( found == 0 ) {
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) io is not available\n", uThisTask() );
#endif
	return 0;					/* tell the caller io is not ready */
    } else {
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) io is not available, errno:%d\n", uThisTask(), errno );
#endif
	if ( errno != EINTR ) {
	    uAbort( "uCheckStream(): internal error!" );
	} /* exit */
	return 0;					/* tell the caller io is not ready */
    } /* if */

} /* uCheckStream */

static void uSelectStream( uStream stream, fd_set *set ) {

    uCluster cluster = uThisCluster();

    uV( &(stream->mutex) );				/* release the stream */
    uP( &(cluster->mutex) );				/* acquire the cluster */

    FD_SET( fileno( stream->file ), set );		/* register interest in this event */

    if ( cluster->iopoller == 0 ) {			/* if this is the first io task */
	cluster->iopoller = uThisTask();		/* remember who the first io task is */
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) is the first io task\n", uThisTask() );
#endif
	for ( ;; ) {
#ifdef __DEBUG_IO__
	    fprintf( stderr, "(0x%x) adding %u to set 0x%x\n", uThisTask(), fileno( stream->file ), *set );
#endif
	    uV( &(cluster->mutex) );			/* release the cluster */
	    if ( uReadyTasks() == 0 ) {			/* if this is the only task running */
#ifdef __DEBUG_IO__
		fprintf( stderr, "(0x%x) no ready tasks, blocking\n", uThisTask() );
#endif
		if ( uCheckStream( 1 ) ) break;		/* wait for io, if it is ready, try io again */
	    } else {
#ifdef __DEBUG_IO__
		fprintf( stderr, "(0x%x) ready tasks, polling\n", uThisTask() );
#endif
		if ( uCheckStream( 0 ) ) break;		/* check for io, if it is ready, try io again */
	    } /* if */
#ifdef __DEBUG_IO__
	    fprintf( stderr, "(0x%x) no io ready, yielding\n", uThisTask() );
#endif
	    uYield();					/* give up the processor */
	    uP( &(cluster->mutex) );			/* acquire the cluster */
	} /* for */
    } else {						/* this is not the first io task */
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) is blocking for io\n", uThisTask() );
#endif
	cluster->iocount[(int)fileno( stream->file )] += 1;				/* remember that someone is blocked */
	uV( &(cluster->mutex) );			/* release the cluster */
	uP( &(cluster->ioqueue[(int)fileno( stream->file )]) );			/* wait for some io */
    } /* if */
    uP( &(stream->mutex) );				/* acquire the stream */
    
} /* uSelectStream */

int uFeof( uStream stream ) {
    return feof( stream->file );
} /* uFeof */

int uFerror( uStream stream ) {
    return ferror( stream->file );
} /* uFerror */

void uClearerr( uStream stream ) {
    clearerr( stream->file );
} /* uClearerr */

int uFileno( uStream stream ) {
    return fileno( stream->file );
} /* uFileno */

inline static void uInterEnable( int previous ) {
    /*
     * restores async interventions to the given state
     */

    uTask me = uThisTask();

    if ( previous ) {
	me->state &= ~U_DISABLE_INTER;
	if ( me->inters && me->icheckneeded ) {
	    /*
	     * Pending intervention(s) which haven't been checked; possibly deliver one.
	     */
	    uAsyncInterDeliver();
	} /* if */
    } /* if */
} /* uInterEnable */

inline static int uInterDisable( void ) {
    /*
     * Disables async interventions, returning old state.
     */

    uTask me = uThisTask();
    int enabled = (me->state & U_DISABLE_INTER) == 0;

    me->state |= U_DISABLE_INTER;
    return enabled;
} /* uInterDisable */

#ifndef __U_UNIXRC__
static volatile void uFopenError( uStream stream, char *path, char* type ) {
    uOpenExMsg msg;
    uException* except;

    msg.base.errno = errno;
    msg.path = path;
    msg.perms = type;
    msg.flags = 0;					/* unused for uFopen  */
    msg.mode = 0;					/* unused for uFopen  */

    switch( msg.base.errno ) {
      case EPERM:
	msg.base.msg = "uFopen: pathname contains high order bit set\n";
	except = &uBadPathEx;
	break;
      case ENOTDIR:
	msg.base.msg = "uFopen: a portion of the path is not a directory\n";
	except = &uBadPathEx;
	break;
      case ENOENT:
	msg.base.msg = "uFopen: file doesn't exist or component doesn't exist or path too long\n";
	except = &uBadPathEx;
	break;
      case EFAULT:
	msg.base.msg = "uFopen: path parameter points outside process address space\n";
	except = &uBadPathEx;
	break;
      case ELOOP:
	msg.base.msg = "uFopen: symbolic link loop\n";
	except = &uBadPathEx;
	break;
      case EOPNOTSUPP:
	msg.base.msg = "uFopen: Attempt to open socket\n";
	except = &uBadPathEx;
	break;
      case EACCES:
	msg.base.msg = "uFopen: wrong permissions on file or directory\n";
	except = &uNoPermsEx;
	break;
      case EISDIR:
	msg.base.msg = "uFopen: file is a directory, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EROFS:
	msg.base.msg = "uFopen: read only file system, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case ETXTBSY:
	msg.base.msg = "uFopen: shared text being executed, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EBUSY:
	msg.base.msg = "uFopen: special file already opened with exlusive access\n";
	except = &uNoPermsEx;
	break;
      case EMFILE:
	msg.base.msg = "uFopen: this process has too many files open\n";
	except = &uNoFilesEx;
	break;
      case ENFILE:
	msg.base.msg = "uFopen: too many files open in the system\n";
	except = &uNoFilesEx;
	break;
      case EIO:
	msg.base.msg = "uFopen: I/O failed\n";
	except = &uOpenIOFailedEx;
	break;
      case ENOSPC:
	msg.base.msg = "uFopen: no space on filesystem\n";
	except = &uOpenNoSpaceEx;
	break;
      case ENXIO:
	msg.base.msg = "uFopen: device doesn't exist or opening comm device with no delay and no carrier\n";
	except = &uOpenEx;
	break;
      default:
	msg.base.msg = "uFopen: unknown error\n";
	except = &uOpenEx;
	break;
    } /* switch */
    uRaise( NULL, except, &msg, sizeof(msg) );
} /* uFopenError */
#endif

uStream uFopen( char *path, char *type ) {
    uStream stream;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    stream = uCreateStream();				/* create new cluster */
    uStreamEnter( stream );				/* migrate to I/O cluster */

    stream->file = fopen( path, type );			/* open the stream */

    if ( stream->file == NULL ) {			/* operation failed */
	uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	uDestroyStream( stream );			/* clean up */
	uInterEnable( istat );				/* re-enable interventions */
#ifdef __U_UNIXRC__
	return NULL;
#else
	uFopenError( stream, path, type );		/* does not return */
#endif
    } /* exit */

    stream->block = uDoesStreamBlock( stream );
    uSetStreamAccess( stream );
    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return stream;
} /* uFopen */

#ifndef __U_UNIXRC__
static volatile void uFflushError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFflush: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFflush: no space on filesystem\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFflush: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFflush: writing to pipe w/o reader, or unconnected socket\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFflush: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      case EFBIG:
	msg.msg = "uFflush: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      default:
	msg.msg = "uFflush: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFflushError */
#endif

int uFflush( uStream stream ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {
	uClearerr( stream );
	code = fflush( stream->file );

	if ( code != EOF ) break;
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uFflushError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->wfds ) );

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uFflush */

#ifndef __U_UNIXRC__
static volatile void uFcloseError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFclose: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFclose: no space on filesystem\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFclose: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFclose: writing to pipe w/o reader, or unconnected socket\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFclose: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      case EFBIG:
	msg.msg = "uFclose: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      default:
	msg.msg = "uFclose: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* stream */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFcloseError */
#endif

int uFclose( uStream stream ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    /*
     * special case for the 3 standard files, as closing them would destroy the system cluster.
     */

    if ( stream == uStdin || stream == uStdout || stream == uStderr ) return 0;

    uStreamEnter( stream );				/* migrate to I/O cluster */
    uClearStreamAccess( stream );			/* reset blocking if set */
    uClearerr( stream );				/* reset error if set */

    code = fclose( stream->file );			/* close the stream */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uDestroyStream( stream );				/* clean up */
    uInterEnable( istat );				/* re-enable interventions */
    if ( code == EOF ) {				/* operation failed */
#ifndef __U_UNIXRC__
	uFcloseError( stream );				/* does not return */
#endif
    } /* exit */
    return code;
} /* uFclose */

#ifndef __U_UNIXRC__
static volatile void uFgetcError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFgetc: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uFgetc: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uFgetc: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFgetc: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFgetc: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFgetcError */
#endif

int uFgetc( uStream stream ) {
    int byte, terrno;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {

	if ( stream->block == U_FNDELAY_DEMAND ) uSetBlockFlag( stream );
	uClearerr( stream );
	byte = fgetc( stream->file );
	terrno = errno;
	if ( stream->block == U_FNDELAY_DEMAND ) uClearBlockFlag( stream );
	errno = terrno;

	if ( byte != EOF ) break;			/* if the read was successful, break */
	if ( feof( stream->file ) ) {			/* check for eof (EOF could imply an error) */
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* if */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uFgetcError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->rfds ) ); /* read failed to complete */

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
#ifndef __U_UNIXRC__
    if ( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uFgetc: End of File\n";
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );	/* does not return */
    } /* exit */
#endif
    return byte;
} /* uFgetc */

#ifndef __U_UNIXRC__
static volatile void uFgetsError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFgets: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uFgets: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uFgets: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFgets: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFgets: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFgetsError */
#endif

char *uFgets( char *str, int len, uStream stream ) {
    char *code;
    int terrno;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {

	if ( stream->block == U_FNDELAY_DEMAND ) uSetBlockFlag( stream );
	uClearerr( stream );
	code = fgets( str, len, stream->file );
	terrno = errno;
	if ( stream->block == U_FNDELAY_DEMAND ) uClearBlockFlag( stream );
	errno = terrno;

	if ( code != NULL ) break;			/* if the read was successful, break */
	if ( feof( stream->file ) ) {			/* check for eof */
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* exit */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uFgetsError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->rfds) ); /* read failed to complete */

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
#ifndef __U_UNIXRC__
    if ( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uFgets: End of File\n";
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );	/* does not return */
    } /* exit */
#endif
    return code;
} /* uFgets */

int uGetc( uStream stream ) {
    return( uFgetc( stream ) );
} /* uGetc */

int uGetchar( void ) {
    return( uFgetc( uStdin ) );
} /* uGetchar */

char *uGets( char *str, int len ) {
    return( uFgets( str, len, uStdin ) );
} /* uGets */

#ifndef __U_UNIXRC__
static volatile void uUngetcError( uStream stream ) {
    uReadWriteExMsg msg;

    msg.errno = 0;
    msg.msg = "uUngetc: Cannot push character back\n";

    uRaise( stream, &uReadWriteEx, &msg, sizeof(msg) );
} /* uUngetcError */
#endif

int uUngetc( int c, uStream stream ) {
    int byte;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */
    uClearerr( stream );

    byte = ungetc( c, stream->file );

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    if ( ferror( stream->file ) ) {			/* operation failed */
#ifndef __U_UNIXRC__
	uUngetcError( stream );				/* does not return */
#endif
    } /* exit */
    return byte;
} /* uUngetc */

#ifndef __U_UNIXRC__
static volatile void uFputcError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFputc: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFputc: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFputc: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFputc: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFputc: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uFputc: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFputc: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFputcError */
#endif

int uFputc( char byte, uStream stream ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {
	uClearerr( stream );

	code = fputc( byte, stream->file );

	if ( code != EOF ) break;			/* if the write was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uFputcError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->wfds ) ); /* write failed to complete */

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uFputc */

#ifndef __U_UNIXRC__
static volatile void uFputsError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFputs: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFputs: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFputs: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFputs: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFputs: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uFputs: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFputs: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFputsError */
#endif

int uFputs( char *str, uStream stream ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {
	uClearerr( stream );

	code = fputs( str, stream->file );

	if ( code != EOF ) break;			/* if the write was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uFputsError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->wfds ) ); /* write failed to complete */

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uFputs */

int uPutc( char byte, uStream stream ) {
    return( uFputc( byte, stream ) );
} /* uPutc */

int uPutchar( char byte ) {
    return( uFputc( byte, uStdout ) );
} /* uPutchar */

void uPuts( char *str ) {
    uFputs( str, uStdout );
    uFputc( '\n', uStdout );				/* to simulate puts() behaviour */
} /* uPuts */

#ifndef __U_UNIXRC__
static volatile void uDoScanError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uDoscan: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uDoscan: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uDoscan: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uDoscan: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uDoscan: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */

    uRaise( stream, except, &msg, sizeof(msg) );
} /* uDoScanError */
#endif

static int uDoscan( uStream stream, char *fmt, void *args ) {
    int count, terrno;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {
	if ( stream->block == U_FNDELAY_DEMAND ) uSetBlockFlag( stream );
	uClearerr( stream );
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) before doscan\n", uThisTask() );
#endif
	count = _doscan( stream->file, fmt, args );
	terrno= errno;
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x)  after doscan, count is %d, errno is %d\n", uThisTask(), count, errno );
#endif
	if ( stream->block == U_FNDELAY_DEMAND ) uClearBlockFlag( stream );
	errno= terrno;

	if ( count != -1 ) break;			/* if the read was successful, break */
	if ( feof( stream->file ) ) {			/* check for eof (EOF could imply an error) */
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) end of file\n", uThisTask() );
#endif
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* if */

	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __DEBUG_IO__
	fprintf( stderr, "(0x%x) error, errno is %d\n", uThisTask(), errno );
#endif
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uDoScanError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->rfds ) ); /* read failed to complete */

    } /* for */
    
    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
#ifndef __U_UNIXRC__
    if ( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uDoscan: End of File\n";
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );	/* does not return */
    } /* exit */
#endif
    return( count );
} /* uDoScan */

int uFscanf( uStream stream, char *fmt, ... ) {
    int count;
    va_list args;

    va_start( args, fmt );
    count = uDoscan( stream, fmt, args );
    va_end( args );

    return( count );
} /* uFscanf */

int uScanf( char *fmt, ... ) {
    int count;
    va_list args;

    va_start( args, fmt );
    count = uDoscan( uStdin, fmt, args );
    va_end( args );

    return( count );
} /* uScanf */

#ifndef __U_UNIXRC__
static volatile void uDoprntError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uDoprnt: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uDoprnt: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uDoprnt: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uDoprnt: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uDoprnt: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uDoprnt: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uDoprnt: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uDoprntError */
#endif

static int uDoprnt( char *fmt, void *args, uStream stream ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {
	uClearerr( stream );

	code = _doprnt( fmt, args, stream->file );

	if ( code != -1 ) break;			/* if the write was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uDoprntError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->wfds ) ); /* write failed to complete */

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uDoprnt */

int uFprintf( uStream stream, char *fmt, ... ) {
    int code;
    va_list args;

    va_start( args, fmt );
    code = uDoprnt( fmt, args, stream );
    va_end( args );

    return code;
} /* uFprintf */

int uPrintf( char *fmt, ... ) {
    int code;
    va_list args;

    va_start( args, fmt );
    code = uDoprnt( fmt, args, uStdout );
    va_end( args );

    return code;
} /* uPrintf */

#ifndef __U_UNIXRC__
static volatile void uFreadError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFread: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uFread: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uFread: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFread: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFread: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFreadError */
#endif

int uFread( char *buf, int size, int count, uStream stream ) {
    int code, terrno;
    int total = 0;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {

	if ( stream->block == U_FNDELAY_DEMAND ) uSetBlockFlag( stream );
	uClearerr( stream );
	code = fread( buf, size, count, stream->file );
	terrno = errno;
	if ( stream->block == U_FNDELAY_DEMAND ) uClearBlockFlag( stream );
	errno = terrno;

	total += code;
	buf += code * size;
	count -= code;
	
	if ( code != 0 ) break;				/* if the read was successful, break */
	if ( feof( stream->file ) ) {			/* check for eof (EOF could imply an error) */
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* if */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uFreadError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->rfds ) ); /* read failed to complete */

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
#ifndef __U_UNIXRC__
    if ( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uFread: End of File\n";
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );	/* does not return */
    } /* exit */
#endif
    return total;
} /* uFread */

#ifndef __U_UNIXRC__
static volatile void uFwriteError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;

    msg.errno = errno;

    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFwrite: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFwrite: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFwrite: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFwrite: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFwrite: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uFwrite: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFwrite: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFwriteError */
#endif

int uFwrite( char *buf, int size, int count, uStream stream ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */

    for ( ;; ) {
	uClearerr( stream );

	code = fwrite( buf, size, count, stream->file );

	if ( code != 0 ) break;				/* if the write was successful, break */
	if ( errno != EINTR && errno != U_EWOULDBLK ) {	/* if an error occurred, handle it */
#ifdef __U_UNIXRC__
	    break;
#else
	    uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	    uInterEnable( istat );			/* re-enable interventions */
	    uFwriteError( stream );			/* does not return */
#endif
	} /* exit */

	uSelectStream( stream, &(uThisCluster()->wfds ) ); /* write failed to complete */

    } /* for */

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return code;
} /* uFwrite */

#ifndef __U_UNIXRC__
static volatile void uFseekError( uStream stream ) {
    uReadWriteExMsg msg;

    msg.msg = "uFseek: seek failed\n";
    msg.errno = errno;

    uRaise( stream, &uReadWriteEx, &msg, sizeof(msg) );
} /* uFseekError */
#endif

int uFseek( uStream stream, long numbytes, int origin ) {
    int code;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */
    uClearerr( stream );

    code = fseek( stream->file, numbytes, origin );

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    if ( code == -1 ) {
#ifndef __U_UNIXRC__
	uFseekError( stream );				/* does not return */
#endif
    } /* exit */
    return code;
} /* uFseek */

void uRewind( uStream stream ) {
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */
    uClearerr( stream );

    rewind( stream->file );

    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
} /* uRewind */

#ifndef __U_UNIXRC__
static volatile void uFreopenError( char *path, char *perms, uStream stream ) {
    uOpenExMsg msg;
    uException* except;

    msg.base.errno = errno;
    msg.path = path;
    msg.perms = perms;
    msg.flags = 0;                      /* unused for uFopen  */
    msg.mode = 0;                       /* unused for uFopen  */

    switch( msg.base.errno ) {
      case EPERM:
	msg.base.msg = "uFreopen: pathname contains high order bit set\n";
	except = &uBadPathEx;
	break;
      case ENOTDIR:
	msg.base.msg = "uFreopen: a portion of the path is not a directory\n";
	except = &uBadPathEx;
	break;
      case ENOENT:
	msg.base.msg = "uFreopen: file doesn't exist or component doesn't exist or path too long\n";
	except = &uBadPathEx;
	break;
      case EFAULT:
	msg.base.msg = "uFreopen: path parameter points outside process address space\n";
	except = &uBadPathEx;
	break;
      case ELOOP:
	msg.base.msg = "uFreopen: symbolic link loop\n";
	except = &uBadPathEx;
	break;
      case EOPNOTSUPP:
	msg.base.msg = "uFreopen: Attempt to open socket\n";
	except = &uBadPathEx;
	break;
      case EACCES:
	msg.base.msg = "uFreopen: wrong permissions on file or directory\n";
	except = &uNoPermsEx;
	break;
      case EISDIR:
	msg.base.msg = "uFreopen: file is a directory, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EROFS:
	msg.base.msg = "uFreopen: read only file system, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case ETXTBSY:
	msg.base.msg = "uFreopen: shared text being executed, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EBUSY:
	msg.base.msg = "uFreopen: special file already opened with exlusive access\n";
	except = &uNoPermsEx;
	break;
      case EMFILE:
	msg.base.msg = "uFreopen: this process has too many files open\n";
	except = &uNoFilesEx;
	break;
      case ENFILE:
	msg.base.msg = "uFreopen: too many files open in the system\n";
	except = &uNoFilesEx;
	break;
      case EIO:
	msg.base.msg = "uFreopen: I/O failed\n";
	except = &uOpenIOFailedEx;
	break;
      case ENOSPC:
	msg.base.msg = "uFreopen: no space on filesystem\n";
	except = &uOpenNoSpaceEx;
	break;
      case ENXIO:
	msg.base.msg = "uFreopen: device doesn't exist or opening comm device with no delay and no carrier\n";
	except = &uOpenEx;
	break;
      default:
	msg.base.msg = "uFreopen: unknown open error\n";
	except = &uOpenEx;
	break;
    } /* switch */
    uRaise( NULL, except, &msg, sizeof(msg) );
} /* uFreopenError */
#endif

uStream uFreopen( char *path, char *perms, uStream stream ) {
    FILE *fp;
    int istat = uInterDisable();			/* disable interventions */
    uCluster prevClus = uThisCluster();			/* store previous cluster */

    uStreamEnter( stream );				/* migrate to I/O cluster */
    uClearStreamAccess( stream );
    uClearerr( stream );

    fp = freopen( path, perms, stream->file );

    if ( fp == NULL ) {					/* operation failed */
	uStreamLeave( stream, prevClus );		/* migrate back to previous cluster */
	uInterEnable( istat );				/* re-enable interventions */
#ifdef __U_UNIXRC__
	return NULL;
#else
	uFreopenError( path, perms, stream );		/* does not return */
#endif
    } /* exit */

    stream->file = fp;
    stream->block = uDoesStreamBlock( stream );
    uSetStreamAccess( stream );
    uStreamLeave( stream, prevClus );			/* migrate back to previous cluster */
    uInterEnable( istat );				/* re-enable interventions */
    return stream;
} /* uFreopen */

uStream uFinstall( FILE *file ) {
    uStream stream;

    stream = uMalloc( sizeof( uStreamD ) );		/* allocate a stream descriptor */
    stream->mutex = U_SEMAPHORE( 1 );			/* initialize the semaphore */
    stream->cluster = uThisCluster();			/* assign stream to the System cluster */
    stream->file = file;
    stream->block = uDoesStreamBlock( stream );
    uSetStreamAccess( stream );

    return stream;
} /* uFinstall */
 
void uFdestall( uStream stream ) {
    uFree( stream );
} /* uFdestall */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
