/*                               -*- Mode: C -*- 
 * 
 * SGI Graphics Interface Version 1.0, Copyright (C) Peter A. Buhr and Frank J. Henigman 1990
 * 
 * uSGIGraphics.c -- uSystem Graphics Interface for SGI
 * 
 * Author           : Frank J. Henigman
 * Created On       : Mon Jun 18 07:40:32 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Thu Aug 29 16:55:40 1991
 * Update Count     : 24
 */

#include <uSystem.h>
#include <uUnix.h>
#include <uSGIGraphics.h>

int __PrintFlag = 0;
#define PRINT if(__PrintFlag)uPrintf
    
#define UGR_QUEUE_SIZE		50			/* event queue size */
#define UGR_BLKQREAD_SIZE	600			/* the most events that we will try to read at once */
#ifndef UGR_TIME_SLICE
#  define UGR_TIME_SLICE	10			/* time slicing on graphics cluster */
#endif
    
    typedef struct list_element {				/* a list element */
	struct list_element *prev;				/* pointer to previous element */
	struct list_element *next;				/* pointer to next element */
    } uGrListElement;

typedef struct list_head {				/* a list */
    uGrListElement *head;				/* head of list */
    int size;						/* size of user data */
    int ( *cmp )( void *, void * );			/* data compare function */
} *uGrList;

typedef struct {					/* a device/window id pair */
    Device device;
    long wid;
} uGrDeviceWindow;

typedef struct {					/* client event queue */
    int head, tail;
    uGrEvent events[ UGR_QUEUE_SIZE ];
} uGrEventQueue;

typedef enum {						/* client's `read' status */
    s_noread,						/* does not want to read queue */
    s_read,						/* wants to read one event */
    s_blkread,						/* wants to read multiple events */
} uGrClientStatus;

typedef struct {					/* info needed to do a block read */
    uGrEvent *events;					/* array of events */
    short num_events;					/* size of array */
} uGrBlockReadInfo;

typedef struct {					/* info needed to tie a button */
    uGrDeviceWindow b;					/* the button and window to be tied */
    Device v1, v2;					/* the valuators to be tied */
} uGrTieInfo;

typedef struct {					/* info needed to `do a noise' on a valuator */
    uGrDeviceWindow v;					/* the valuator in question */
    short delta;					/* the noise value */
} uGrNoiseInfo;

typedef enum {						/* key to extra info stored in client device list */
    i_none,						/* no extra info */
    i_tie,						/* button tie info */
    i_noise,						/* valuator noise info */
} uGrDeviceInfoType;

typedef struct {					/* entry in client device list */
    uGrDeviceWindow devwid;				/* device and window id */
    uGrDeviceInfoType info;				/* type of extra info */
    union {						/* tie or noise extra info */
	struct {					/* valuators tied to the button */
	    Device v1, v2;
	} tie;
	struct {					/* noise info */
	    short previous;				/* previous value of this valuator */
	    short delta;				/* noise filter value for valuator */
	} noise;
    } u;
} uGrDeviceEntry;

typedef struct {					/* client record */
    uTask id;						/* client's task id */
    uGrList devices;					/* client's device list */
    uGrEventQueue queue;				/* event queue for this client */
    uGrClientStatus status;				/* indicates if/how client wants to read its queue */
    uGrBlockReadInfo bri;				/* block read information */
} uGrClient;

typedef enum {						/* message types to queue server */
    m_queue,
    m_unqueue,
    m_read,
    m_blkread,
    m_test,
    m_enter,
    m_reset,
    m_isqueued,
    m_tie,
    m_noise,
    m_stop,
} uGrMsgType;

typedef struct {					/* message to queue server */
    union {						/* THIS UNION MUST BE FIRST MEMBER IN STRUCT! */
	uGrEvent event;
	uGrDeviceWindow devwid;
	uGrBlockReadInfo bri;
	uGrTieInfo tie;
	uGrNoiseInfo noise;
    } u;
    uGrMsgType type;					/* type of message */
} uGrMsg;

typedef struct {					/* used to communicate between uGrSomeoneUses()... */
    Device device;					/* ...and uGrFindDevice() */
    int flag;
} uGrDeviceFlag;


/*
 * Global variables
 */

static uSemaphore uGrLock = U_SEMAPHORE(1);		/* graphics engine lock */
static uCluster uGrCluster = NULL;			/* graphics cluster */
static uCluster uGrUserCluster;				/* user cluster */
static uTask uGrQserverTask = NULL;			/* queue server task id */
static uGrList clients;					/* the list of clients */
#ifdef __U_DEBUG__
static uSemaphore uGrStatusLock = U_SEMAPHORE(1);	/* used in uGrOpen() and uGrClose() */
    static uTask uGrTask = NULL;				/* task that is currently doing graphics */
#endif
    
    
    static uGrList uGrListInit( int size, int ( *cmp )( void *, void * ) ) {
	
	/*
	 * Set up a new list.
	 */
	
	uGrList list;
	
	list = uMalloc( sizeof( struct list_head ) );
	list->head = NULL;
	list->size = size;
	list->cmp = cmp;
	return list;
    } /* uGrListInit */
    
    
    static void *uGrListAdd( uGrList list, void *data ) {
	
	/*
	 * Add an element to a list.
	 */
	
	uGrListElement *new;
	
	new = uMalloc( list->size + sizeof( uGrListElement ) );
	if ( list->head != NULL ) {
	    new->next = list->head;
	    list->head->prev = new;
	} else {
	    new->next = NULL;
	} /* if */
	new->prev = NULL;
	list->head = new;
	memcpy( (char *) new + sizeof( uGrListElement ), data, list->size );
	return (char *) new + sizeof( uGrListElement );
    } /* uGrListAdd */
    
    
    static void uGrListDelete( uGrList list, void *data ) {
	
	/*
	 * Delete the given element.
	 */
	
	uGrListElement *p;
	
	p = (uGrListElement *) ( (char *) data - sizeof( uGrListElement ) );
	if ( p->prev != NULL ) {
	    p->prev->next = p->next;
	} /* if */
	if ( p->next != NULL ) {
	    p->next->prev = p->prev;
	} /* if */
	if ( list->head == p ) {
	    list->head = p->next;
	} /* if */
	uFree( p );
    } /* uGrListDelete */
    
    
    static void *uGrListFind( uGrList list, void *data ) {
	
	/*
	 * Return a pointer to the given element.
	 */
	
	uGrListElement *i;
	
	for ( i = list->head; i != NULL  &&  ! list->cmp( data, (char *) i + sizeof( uGrListElement ) ); i = i->next );
	return i != NULL ? (char *) i + sizeof( uGrListElement ) : NULL;
    } /* uGrListFind */
    
    
    static void uGrListTraverse( uGrList list, void func( void *data, void *arg ), void *arg ) {
	
	/*
	 * Pass each data item in the list to a function.
	 */
	
	uGrListElement *i;
	
	for ( i = list->head; i != NULL; i = i->next ) {
	    func( (char *) i + sizeof( uGrListElement ), arg );
	} /* for */
    } /* uGrListTraverse */
    
    
    static int uGrListIsEmpty( uGrList list ) {
	
	/*
	 * Return non-zero if the given list is empty.
	 */
	
	return list->head == NULL;
    } /* uGrListIsEmpty */
    
    
    static void uGrQwatcher( uTask parent ) {
	
	/*
	 * Watch the Iris event queue and pass events to parent, the queue server.
	 * This task runs on its own cluster.
	 */
	
	static uGrEvent events[ UGR_BLKQREAD_SIZE ];
	static short data[ UGR_BLKQREAD_SIZE * 2 ];
	int i;
	int num;
	long wid;
	uGrBlockReadInfo bri;
	int stop;
	int yc;
	static char star[] = "O****************************************************************************************************";
	
#ifdef __U_DEBUG__
	uVerify();
#endif
	/*PRINT( "event (%d, %d, %d)\n", event.device, event.wid, event.value );*/
	wid = NULLWIN;					/* should really find out which window cursor is in */
	bri.events = events;
	for ( ;; ) {
	    yc = 0;
	    while ( ! qtest() ) {				/* spin while queue is empty */
		++yc;
		uYield();					/* yield to other tasks on graphics cluster */
	    } /* while */
	    num = blkqread( data, (short) ( UGR_BLKQREAD_SIZE * 2 ) );
	    bri.num_events = num / 2;
	    /*uPrintf( "%3d:%s\n", bri.num_events, yc>100?star:star+101-yc );*/
	    for ( i = 0; i < bri.num_events; i += 1 ) {	/* put data into events array */
		events[ i ].device = data[ i * 2 ];
		events[ i ].value = data[ i * 2 + 1 ];
		events[ i ].wid = wid;
		if ( events[ i ].device == INPUTCHANGE ) {	/* track the current window */
		    wid = events[ i ].value;
		} /* if */
	    } /* for */
	    uSend( parent, &stop, sizeof( stop ), &bri, sizeof( bri ) );
	    if ( stop ) {					/* parent tells us when to stop */
		uDie( NULL, 0 );
	    } /* exit */
	} /* for */
    } /* uGrQwatcher */
    
    
    static void uGrInitEventQueue( uGrEventQueue *q ) {
	
	/*
	 * Initialize an event queue.
	 */
	
	q->head = 0;
	q->tail = 0;
    } /* uGrInitEventQueue */
    
    
    static void uGrPushEvent( uGrEventQueue *q, uGrEvent e ) {
	
	/*
	 * Put an event in an event queue.  If the queue is full do nothing.
	 */
	
	if ( ( q->head + 1 ) % UGR_QUEUE_SIZE != q->tail ) {
	    q->events[ q->head ] = e;
	    q->head = ( q->head + 1 ) % UGR_QUEUE_SIZE;
	} /* if */
    } /* uGrPushEvent */
    
    
    static uGrEvent uGrPopEvent( uGrEventQueue *q ) {
	
	/*
	 * Remove the next event from an event queue.  If the queue is empty return
	 * an event with a zero in the device field.
	 */
	
	uGrEvent e;
	
	if ( q->head == q->tail ) {
	    e.device = NULLDEV;
	} else {
	    e = q->events[ q->tail ];
	    q->tail = ( q->tail + 1 ) % UGR_QUEUE_SIZE;
	} /* if */
	return e;
    } /* uGrPopEvent */
    
    
    static void uGrDoTie( uGrEventQueue *q, Device v ) {
	
	/*
	 * Put a tied event on the given queue.
	 */
	
	uGrEvent event;
	
	if ( v != NULLDEV ) {
	    event.device = v;
	    event.wid = NULLWIN;
	    event.value = getvaluator( v );
	    uGrPushEvent( q, event );
	} /* if */
    } /* uGrDoTie */
    
    
    static void uGrSaveEvent( uGrClient *c, uGrEvent *e ) {
	
	/*
	 * Add the given event to the queues of the clients who want it.
	 */
	
	uGrDeviceEntry de, *pde;
	void print_de(uGrClient*,void*);
	
	de.devwid.device = e->device;
	de.devwid.wid = e->wid;
	PRINT( "save (%d,%d)?\n", de.devwid.device, de.devwid.wid );
	pde = uGrListFind( c->devices, &de );		/* look for this device in the clients device list */
	if ( pde != NULL ) {				/* if the client wants the device... */
	    print_de(pde,NULL);
	    PRINT( "\n" );
	    PRINT( "yes - " );
	    switch ( pde->info ) {				/* ...see if any special processing is required */
	    case i_none:					/* nothing special so we... */
		PRINT( "pushing\n" );
		uGrPushEvent( &c->queue, *e );		/* ...add the event to the client's queue */
		break;
	    case i_noise:					/* the device has a noise filter */
		if ( ISTIMER( de.devwid.device ) ) {
		    pde->u.noise.previous += 1;
		    if ( pde->u.noise.previous == pde->u.noise.delta ) {
			PRINT( "enough timer events\n" );
			pde->u.noise.previous = 0;
			uGrPushEvent( &c->queue, *e );	/* there have been enough timer events so push */
		    } /* if */
		} else {					/* must be a valuator */
		    if ( abs( (int) ( e->value - pde->u.noise.previous ) ) >= pde->u.noise.delta ) {
			PRINT( "passed noise test\n" );
			pde->u.noise.previous = e->value;
			uGrPushEvent( &c->queue, *e );	/* value has changed enough so add the event */
		    } /* if */
		} /* if */
		break;
	    case i_tie:					/* the device is tied */
		PRINT( "tie\n" );
		uGrPushEvent( &c->queue, *e );		/* first add the button event to the queue */
		uGrDoTie( &c->queue, pde->u.tie.v1 );	/* push the first tied event */
		uGrDoTie( &c->queue, pde->u.tie.v2 );	/* push the second tied event */
		break;
	    default:PRINT("should not be here!\n");
	    } /* switch */
	} /* if */
    } /* uGrSaveEvent */
    
    
    static void uGrServe( uGrClient *c, void *arg ) {
	
	/*
	 * Serve a client - if it wants events and there are events, send them to the client.
	 */
	
	uGrEvent e;
	long num_read;
	
	switch ( c->status ) {
	case s_read:					/* client wants one event */
	    e = uGrPopEvent( &c->queue );			/* try to pop client's queue */
	    if ( e.device ) {				/* if there is an event... */
		uReply( c->id, &e, sizeof( e ) );		/* ...reply with the event */
		c->status = s_noread;			/* remember this client has been served */
	    } /* if */
	    break;
	case s_blkread:					/* client want multiple events */
	    num_read = 0;
	    do {
		e = uGrPopEvent( &c->queue );		/* try to pop client's queue */
		if ( e.device ) {				/* if there is an event... */
		    c->bri.events[ num_read ] = e;		/* ...stuff an event in the supplied array */
		    num_read += 1;
		    if ( num_read == c->bri.num_events ) {	/* stop if we have read enough */
			break;
		    } /* if */
		} /* if */
	    } while ( e.device );
	    if ( num_read > 0 ) {				/* if any events read, reply with the number */
		uReply( c->id, &num_read, sizeof( num_read ) );
		c->status = s_noread;			/* remember that this client has been served */
	    } /* if */
	    break;
	} /* switch */
    } /* uGrService */
    
    
    static int uGrCompareDevwid( uGrDeviceWindow *a, uGrDeviceWindow *b ) {
	
	/*
	 * Device/window compare function.
	 */
	
	return ( a->device == b->device  ||  a->device == NULLDEV  ||  b->device == NULLDEV )  &&  ( a->wid == b->wid  ||  a->wid == NULLWIN  ||  b->wid == NULLWIN );
    } /* uGrCompareDevice */
    
    
    static int uGrCompareDeviceEntry( uGrDeviceEntry *a, uGrDeviceEntry *b ) {
	
	/*
	 * Compare device entries.
	 */
	
	return uGrCompareDevwid( &a->devwid, &b->devwid );
    } /* uGrCompareDeviceEntry */
    
    
    static uGrClient *uGrLookupClient( uTask id ) {
	
	/*
	 * Look up a client or add it to the list.
	 */
	
	uGrClient *pc, c;
	
	c.id = id;
	pc = uGrListFind( clients, &c );
	if ( pc == NULL ) {
	    c.devices = uGrListInit( sizeof( uGrDeviceEntry ), uGrCompareDeviceEntry );
	    uGrInitEventQueue( &c.queue );
	    c.status = s_noread;
	    pc = uGrListAdd( clients, &c );
	    PRINT( "welcome client %#lx\n", pc );
	} /* if */
	return pc;
    } /* uGrLookupClient */
    
    
    static int uGrCompareClient( uGrClient *a, uGrClient *b ) {
	
	/*
	 * Client compare function.
	 */
	
	return a->id == b->id;
    } /* uGrCompareClient */
    
    
    static uGrFindDevice( uGrClient *c, uGrDeviceFlag *pdf ) {
	
	/*
	 * If the given client uses the given device set a flag.
	 */
	
	uGrDeviceWindow dw;
	
	dw.device = pdf->device;
	dw.wid = NULLWIN;
	if ( uGrListFind( c->devices, &dw ) != NULL ) {
	    pdf->flag = 1;
	} /* if */
	return;
    } /* uGrFindDevice */
    
    
    static int uGrSomeoneUses( Device dev ) {
	
	/*
	 * Return non-zero if the given device is used by any client
	 */
	
	uGrDeviceFlag df;
	
	df.device = dev;
	df.flag = 0;
	uGrListTraverse( clients, uGrFindDevice, &df );
	return df.flag;
    } /* uGrSomeoneUses */
    
    
    static void print_de( uGrDeviceEntry *a, void *arg ) {
	char s[99];
	switch(a->info){
	case i_none: sprintf(s,"none"); break;
	case i_tie: sprintf(s,"tie %d %d",a->u.tie.v1,a->u.tie.v2); break;
	case i_noise: sprintf(s,"noise %d",a->u.noise.delta); break;
	default: sprintf(s,"*BOGUS*");
	}
	PRINT( "(%d, %d) - %s   ", a->devwid.device, a->devwid.wid, s );
    }
    
    static char star[] = "************************************************************";
    
    static void print_client( uGrClient *a, void *arg ) {
	char *s;
	switch(a->status){
	case s_noread: s="noread"; break;
	case s_read: s="read"; break;
	case s_blkread: s="blkread"; break;
	}
	PRINT( "client %#lx\n", a );
	PRINT( "  id = %#lx\n", a->id );
	PRINT( "  status = %s\n", s );
	PRINT( "  list = " );
	uGrListTraverse( a->devices, print_de, NULL );
	PRINT( "\n" );
    }
    
    
    static void uGrQserver( void ) {
	
	/*
	 * The queue server responds to requests from clients to manipulate their
	 * event queues.
	 */
	
	uGrMsg msg;						/* received message */
	uTask id;						/* message sender id */
	uTask uGrQwatcherTask;				/* queue watcher task id */
	uCluster uGrQwatcherCluster;			/* queue watcher cluster */
	int stop;						/* flag indicates server and watcher will die */
	int isq;						/* result of `is queued' test */
	uGrEventQueue newq;
	uGrDeviceWindow dw;
	uGrDeviceEntry de, *pde;
	uGrClient *client;
	uGrEvent event;
	int i;
	
#ifdef __U_DEBUG__
	uVerify();
#endif
	clients = uGrListInit( sizeof( uGrClient ), uGrCompareClient );
	stop = 0;
	uGrQwatcherTask = NULL;
#ifdef UGR_WATCHER_CLUSTER
	uGrQwatcherCluster = uCreateCluster( 1, 0 );
#  ifndef UGR_DELAY_WATCHER
	uGrQwatcherTask = uLongEmit( uGrQwatcherCluster, 4000, uGrQwatcher, sizeof( uTask ), uThisTask() );
#  endif
#else
#  ifndef UGR_DELAY_WATCHER
	uGrQwatcherTask = uEmit( uGrQwatcher, uThisTask() );
#  endif
#endif
	
	for ( ;; ) {
	    if ( stop ) {					/* if stopping, make sure queue watcher isn't blocked */
		qenter( (short) 1, (short) 1 );
	    } /* if */
	    id = uReceive( &msg, sizeof( msg ) );
	    if ( id == uGrQwatcherTask ) {			/* this message is from queue watcher */
		uReply( id, &stop, sizeof( stop ) );
		if ( stop ) {
		    uAbsorb( uGrQwatcherTask, NULL, 0 );
#ifdef UGR_WATCHER_CLUSTER
		    uDestroyCluster( uGrQwatcherCluster );
#endif
		    uDie( NULL, 0 );
		} /* exit */
		/*PRINT( "(%d,%d) -> SaveEvent\n", msg.u.event.device, msg.u.event.wid );*/
		for ( i = 0; i < msg.u.bri.num_events; i += 1 ) {
		    uGrListTraverse( clients, uGrSaveEvent, msg.u.bri.events + i ); /* save event on the clients' queues */
		} /* for */
	    } else {					/* this message is from a client */
		if ( msg.type != m_stop ) {
		    client = uGrLookupClient( id );		/* for all but a stop message we look up the client */
		} /* if */
		PRINT( "message from client %#lx\n", client );
		switch ( msg.type ) {			/* deal with the message */
		case m_queue:				/* add a device to the list for this client */
		    uReply( id, NULL, 0 );
		    de.devwid = msg.u.devwid;
		    de.info = i_none;
		    PRINT( "queue (%d,%d)\n", de.devwid.device, de.devwid.wid );
		    if ( de.devwid.device == NULLDEV ) {	/* meaningless to queue NULLDEV so do nothing */
			break;
		    } /* if */
		    if ( de.devwid.wid == NULLWIN ) {	/* if queueing NULLWIN... */
			do {				/* ...remove all instances of this device... */
			    pde = uGrListFind( client->devices, &de );
			    if ( pde != NULL ) {
				uGrListDelete( client->devices, pde );
			    } /* if */
			} while ( pde != NULL );
			PRINT( "add NULLWIN\n" );
			uGrListAdd( client->devices, &de );	/* ...then add the only entry needed */
		    } else {				/* we are queueing non-NULLWIN */
			if ( uGrListFind( client->devices, &de ) != NULL ) {
			    break;				/* this device is already in list so don't add it */
			} /* if */
			de.devwid.wid = NULLWIN;
			pde = uGrListFind( client->devices, &de );
			if ( pde != NULL  &&  pde->devwid.device == NULLWIN ) {
			    break;				/* a NULLWIN for this device is in list so don't add */
			} /* if */
			PRINT( "add non-NULLWIN\n" );
			de.devwid.wid = msg.u.devwid.wid;
			uGrListAdd( client->devices, &de );	/* if we got this far we need to add to list */
		    } /* if */
		    if ( ! isqueued( de.devwid.device ) ) {
			qdevice( de.devwid.device );	/* queue this device if it isn't already queued */
			PRINT( "qdevice(%d)\n", de.devwid.device );
#ifdef UGR_DELAY_WATCHER
			if ( uGrQwatcherTask == NULL ) {
#  ifdef UGR_WATCHER_CLUSTER
			    uGrQwatcherTask = uLongEmit( uGrQwatcherCluster, 4000, uGrQwatcher, sizeof( uTask ), uThisTask() );
#  else
			    uGrQwatcherTask = uEmit( uGrQwatcher, uThisTask() );
#  endif
			} /* if */
#endif
		    } /* if */
		    break;
		case m_unqueue:				/* remove device(s) from the list for this client */
		    uReply( id, NULL, 0 );
		    de.devwid = msg.u.devwid;
		    PRINT( "unqueue (%d,%d)\n", de.devwid.device, de.devwid.wid );
		    do {					/* find and delete everything matching the device/wid */
			pde = uGrListFind( client->devices, &de );
			if ( pde != NULL ) {
			    PRINT( "deleting (%d,%d)\n", pde->devwid.device, pde->devwid.wid );
			    uGrListDelete( client->devices, pde );
			    if ( ! uGrSomeoneUses( pde->devwid.device ) ) {
				unqdevice( pde->devwid.device ); /* unqueue a device which no one uses */
				PRINT( "unqdevice(%d)\n", pde->devwid.device );
			    } /* if */
			} /* if */
		    } while ( pde != NULL );
		    if ( uGrListIsEmpty( client->devices ) ) { /* if client has no devices queued we forget him */
			PRINT( "bye bye client %#lx\n", client );
			uGrListDelete( clients, client );
		    } /* if */
		    break;
		case m_read:				/* remember client wants an event */
		    client->status = s_read;
		    break;
		case m_blkread:				/* remember client wants multiple events */
		    client->status = s_blkread;
		    break;
		case m_test:				/* pop client's queue and reply with the event */
		    event = uGrPopEvent( &client->queue );
		    uReply( id, &event, sizeof( event ) );
		    break;
		case m_reset:				/* clear events from the queue */
		    uReply( id, NULL, 0 );
		    uGrInitEventQueue( &newq );		/* initialize a new queue */
		    do {					/* transfer events we want to keep to a new queue */
			event = uGrPopEvent( &client->queue );
			if ( event.device ) {
			    dw.device = event.device;
			    dw.wid = event.wid;
			    if ( ! uGrCompareDevwid( &dw, &msg.u.devwid ) ) {
				uGrPushEvent( &client->queue, event );
			    } /* if */
			} /* if */
		    } while ( event.device );
		    client->queue = newq;			/* the new queue becomes the client's queue */
		    break;
		case m_isqueued:				/* search for the device and reply with the result */
		    de.devwid = msg.u.devwid;
		    isq = ( uGrListFind( client->devices, &de ) != NULL );
		    uReply( id, &isq, sizeof( isq ) );
		    break;
		case m_enter:				/* enter an event on the client's queue */
		    uReply( id, NULL, 0 );
		    uGrPushEvent( &client->queue, msg.u.event );
		    break;
		case m_tie:				/* tie valuator(s) to a button */
		    uReply( id, NULL, 0 );
		    de.devwid = msg.u.tie.b;
		    pde = uGrListFind( client->devices, &de );
		    if ( pde != NULL ) {
		    if ( msg.u.tie.v1 == NULLDEV  &&  msg.u.tie.v2 == NULLDEV ) {
			pde->info = i_tie;
			pde->u.tie.v1 = msg.u.tie.v1;
			pde->u.tie.v2 = msg.u.tie.v2;
		    } else {
			pde->info = i_none;
		    } /* if */
		} /* if */
		break;
	      case m_noise:
		uReply( id, NULL, 0 );
		de.devwid = msg.u.noise.v;
		pde = uGrListFind( client->devices, &de );
		if ( pde != NULL ) {
PRINT("setting noise(%d,%d) = %d\n", de.devwid.device, de.devwid.wid, msg.u.noise.delta);
		    pde->info = i_noise;
		    pde->u.noise.delta = msg.u.noise.delta;
		    if ( ISVALUATOR( msg.u.noise.v.device ) ) {
			pde->u.noise.previous = getvaluator( msg.u.noise.v.device );
		    } else {				/* must be a timer */
			pde->u.noise.previous = 0;
			noise( msg.u.noise.v.device, (short) 1 );
		    } /* if */
		} /* if */
		break;
	      case m_stop:				/* shut down the server if there are no clients */
		stop = uGrListIsEmpty( clients );
		uReply( id, &stop, sizeof( stop ) );	/* reply with flag that indicates if shutdown is ok */
		break;
	    } /* switch */
	} /* if */
uGrListTraverse( clients, print_client, NULL );
	uGrListTraverse( clients, uGrServe, NULL );	/* serve the clients */
    } /* for */
} /* uGrQserver */


#ifdef __U_DEBUG__
static void uGrCheckInputOpen( char *func_name ) {

    /*
     * Check if graphics input has been opened.
     */

    if ( uGrQserverTask == NULL ) {
	uAbort( "%s(): GRAPHICS INPUT IS NOT OPEN\n", func_name );
    } /* if */
} /* uGrCheckInputOpen */
#endif


void uGrQdevice( Device device, long wid ) {

    /*
     * Queue a device.
     */

    uGrMsg msg;

#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrQdevice" );
#endif
    msg.type = m_queue;
    msg.u.devwid.device = device;
    msg.u.devwid.wid = wid;
    uSend( uGrQserverTask, NULL, 0, &msg, sizeof( msg ) );
} /* uGrQqdevice */


void uGrUnqdevice( Device device, long wid ) {

    /*
     * Unqueue a device.
     */

    uGrMsg msg;

#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrUnqdevice" );
#endif
    msg.type = m_unqueue;
    msg.u.devwid.device = device;
    msg.u.devwid.wid = wid;
    uSend( uGrQserverTask, NULL, 0, &msg, sizeof( msg ) );
} /* uGrQunqdevice */


uGrEvent uGrQtest( void ) {

    /*
     * Do a non-blocking read of the queue.
     */

    uGrMsg msg;
    uGrEvent e;

#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrQtest" );
#endif
    msg.type = m_test;
    uSend( uGrQserverTask, &e, sizeof( e ), &msg, sizeof( msg ) );
    return e;
} /* uGrQtest */


uGrEvent uGrQread( void ) {

    /*
     * Do a blocking read of the queue.
     */

    uGrMsg msg;
    uGrEvent e;
    
#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrQread" );
#endif
    msg.type = m_read;
    uSend( uGrQserverTask, &e, sizeof( e ), &msg, sizeof( msg ) );
    return e;
} /* uGrQread */


void uGrQreset( Device device, long wid ) {

    /*
     * Empty the queue.
     */

    uGrMsg msg;
    
#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrQreset" );
#endif
    msg.type = m_reset;
    msg.u.devwid.device = device;
    msg.u.devwid.wid = wid;
    uSend( uGrQserverTask, NULL, 0, &msg, sizeof( msg ) );
} /* uGrQreset */


Boolean uGrIsqueued( Device device, long wid ) {

    /*
     * Ask if the given device has been queued.
     */

    uGrMsg msg;
    int isq;
    
#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrIsqueued" );
#endif
    msg.type = m_isqueued;
    msg.u.devwid.device = device;
    msg.u.devwid.wid = wid;
    uSend( uGrQserverTask, &isq, sizeof( isq ), &msg, sizeof( msg ) );
    return isq ? TRUE : FALSE;
} /* uGrIsqueued */


void uGrQenter( Device device, long wid, short value ) {

    /*
     * Add an event to the client's queue.
     */

    uGrMsg msg;

#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrQenter" );
#endif
    msg.type = m_enter;
    msg.u.event.device = device;
    msg.u.event.wid = wid;
    msg.u.event.value = value;
    uSend( uGrQserverTask, NULL, 0, &msg, sizeof( msg ) );
} /* uGrQenter */


long uGrBlkQread( uGrEvent *events, short num_events ) {

    /*
     * Read a block of entries from the client's queue.
     */

    uGrMsg msg;
    long num_read;

#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrBlkQread" );
    if ( num_events < 1 ) {
	uAbort( "uGrBlkQread(): ARRAY SIZE MUST BE AT LEAST ONE\n" );
    } /* if */
#endif
    msg.type = m_blkread;
    msg.u.bri.events = events;
    msg.u.bri.num_events = num_events;
    uSend( uGrQserverTask, &num_read, sizeof( num_read ), &msg, sizeof( msg ) );
    return num_read;
} /* uGrBlkQread */


void uGrTie( Device b, long wid, Device v1, Device v2 ) {

    /*
     * Tie or untie one or two valuators to a button and window.
     */

    uGrMsg msg;

#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrTie" );
    if ( ! ISBUTTON( b ) ) {
	uAbort( "uGrTie(): DEVICE %d IS NOT A BUTTON\n", b );
    } /* if */
    if ( ! ( ISVALUATOR( v1 )  ||  v1 == NULLDEV ) ) {
	uAbort( "uGrTie(): DEVICE %d IS NOT A VALUATOR\n", v1 );
    } /* if */
    if ( ! ( ISVALUATOR( v2 )  ||  v2 == NULLDEV ) ) {
	uAbort( "uGrTie(): DEVICE %d IS NOT A VALUATOR\n", v2 );
    } /* if */
#endif
    msg.type = m_tie;
    msg.u.tie.b.device = b;
    msg.u.tie.b.wid = wid;
    msg.u.tie.v1 = v1;
    msg.u.tie.v2 = v2;
    uSend( uGrQserverTask, NULL, 0, &msg, sizeof( msg ) );
} /* uGrTie */


void uGrNoise( Device v, long wid, short delta ) {

    /*
     * Filter valuator motion.  Also used to set timer event rate.
     */

    uGrMsg msg;

#ifdef __U_DEBUG__
    uGrCheckInputOpen( "uGrNoise" );
    if ( ! ( ISVALUATOR( v )  ||  ISTIMER( v ) ) ) {
	uAbort( "uGrNoise(): DEVICE %d IS NOT A VALUATOR OR A TIMER\n", v );
    } /* if */
    if ( delta < 1 ) {
	uAbort( "uGrNoise(): NOISE VALUE MUST BE AT LEAST 1\n" );
    } /* if */
#endif
    msg.type = m_noise;
    msg.u.noise.v.device = v;
    msg.u.noise.v.wid = wid;
    msg.u.noise.delta = delta;
    uSend( uGrQserverTask, NULL, 0, &msg, sizeof( msg ) );
} /* uGrTie */


void uGrOpen( char *code ) {

    /*
     * Open the graphics interface.
     */

    uClusterVars cv = U_CLUSTER_VARS();

#ifdef UGR_GRAPHICS_SPIN
    cv.Spin = UGR_GRAPHICS_SPIN;
#endif
    cv.TimeSlice = UGR_TIME_SLICE;
#ifdef __U_DEBUG__
    uP( &uGrStatusLock );
#endif
    if ( code[ 0 ] == 'w'  ||  code[ 1 ] == 'w' ) {
#ifdef __U_DEBUG__
        if ( uGrCluster != NULL ) {
	    uAbort( "uGrOpen(): GRAPHICS OUTPUT ALREADY OPENED\n" );
	} else {
#endif
	    uGrCluster = uLongCreateCluster( &cv );	/* create the graphics cluster */
#ifdef __U_DEBUG__
	} /* if */
#endif
    } /* if */

    if ( code[ 0 ] == 'r'  ||  code[ 1 ] == 'r' ) {
#ifdef __U_DEBUG__
	if ( uGrQserverTask != NULL ) {
	    uAbort( "uGrOpen(): GRAPHICS INPUT ALREADY OPENED\n" );
	} else {
#endif
	    if ( uGrCluster == NULL ) {			/* create graphics cluster if not done already */
		uGrCluster = uLongCreateCluster( &cv );	/* create the graphics cluster */
	    } /* if */
	    uGrQserverTask = uLongEmit( uGrCluster, 4000, uGrQserver, 0 ); /* start the queue server */
#ifdef __U_DEBUG__
	} /* if */
#endif
    } /* if */
#ifdef __U_DEBUG__
    uV( &uGrStatusLock );
#endif
} /* uGrOpen */


void uGrClose( char *code ) {

    /*
     * Close the graphics interface.
     */

    uGrMsg msg;						/* message to queue server */
    int stop;						/* reply from queue server */

#ifdef __U_DEBUG__
    uP( &uGrStatusLock );
#endif
    if ( code[ 0 ] == 'r'  ||  code[ 1 ] == 'r' ) {
#ifdef __U_DEBUG__
	if ( uGrQserverTask == NULL ) {
	    uAbort( "uGrClose(): GRAPHICS INPUT IS NOT OPEN\n" );
	} else {
#endif
	    msg.type = m_stop;				/* ask queue server to stop */
	    uSend( uGrQserverTask, &stop, sizeof( stop ), &msg, sizeof( msg ) );
	    if ( ! stop ) {
		uAbort( "uGrClose(): INPUT SERVER STILL HAS CLIENTS\n" );
	    } else {
		uAbsorb( uGrQserverTask, NULL, 0 );
		uGrQserverTask = NULL;
    	    } /* if */
#ifdef __U_DEBUG__
    	} /* if */
#endif
    } /* if */

    if ( code[ 0 ] == 'w'  ||  code[ 1 ] == 'w' ) {
#ifdef __U_DEBUG__
	if ( uGrCluster == NULL ) {
	    uAbort( "uGrClose(): GRAPHICS OUTPUT ALREADY CLOSED\n" );
	} else if ( uGrQserverTask != NULL ) {
	    uAbort( "uGrClose(): CANNOT CLOSE OUTPUT WITHOUT CLOSING INPUT" );
	} else {
#endif
	    uDestroyCluster( uGrCluster );
	    uGrCluster = NULL;
#ifdef __U_DEBUG__
	} /* if */
#endif
    } /* if */
#ifdef __U_DEBUG__
    uV( &uGrStatusLock );
#endif
} /* uGrClose */


void uGrAcquire( void ) {

    /*
     * Get exclusive control of the graphics device.
     */

#ifdef __U_DEBUG__
    if ( uGrCluster == NULL ) {
	uAbort( "uGrAcquire(): GRAPHICS OUTPUT DEVICE NOT OPENED\n" );
    } else if ( uGrTask == uThisTask() ) {
	uAbort( "uGrAcquire(): THIS TASK HAS ALREADY ACQUIRED THE GRAPHICS DEVICE\n" );
    } else {
#endif
	uP( &uGrLock );					/* acquire the graphics lock */
	uGrUserCluster = uThisCluster();		/* remember the user's cluster */
	uMigrate( uGrCluster );				/* migrate the user */
#ifdef __U_DEBUG__
	uGrTask = uThisTask();
    } /* if */
#endif
} /* uGrAcquire */


void uGrRelease( void ) {

    /*
     * Relinquish control of the graphics device.
     */

#ifdef __U_DEBUG__
    if ( uGrCluster == NULL ) {
	uAbort( "uGrRelease(): GRAPHICS OUTPUT DEVICE NOT OPENED\n" );
    } else if ( uGrTask != uThisTask() ) {
	uAbort ("uGrRelease(): THIS TASK HAS NOT ACQUIRED THE GRAPHICS DEVICE\n" );
    } else {
	uGrTask = NULL;
#endif
	uMigrate( uGrUserCluster );			/* migrate the user back to his cluster */
	uV( &uGrLock );					/* release the graphics lock */
#ifdef __U_DEBUG__
    } /* if */
#endif
} /* uGrRelease */


/* Local Variables: */
/* compile-command: "concc -work -multi -c uSGIGraphics.c" */
/* End: */
