/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.4.3, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uKernel.c -- The routines which form the nucleus of the concurrency system.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:13:53 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Sun Mar 28 12:17:22 1993
 * Update Count     : 698
 */

#define  __U_KERNEL__

#if defined( sparc )
/*
 * This is our version of the GNU ___builtin_saveregs for varargs.
 * It is included so that the uSystem programs can be compiled with
 * a compiler other than gcc, without requiring the GNU runtime library
 * to also be linked in.
 */

asm (".global ___builtin_saveregs");
asm ("___builtin_saveregs:");
asm ("st %i0,[%fp+68]");
asm ("st %i1,[%fp+72]");
asm ("st %i2,[%fp+76]");
asm ("st %i3,[%fp+80]");
asm ("st %i4,[%fp+84]");
asm ("retl");
asm ("st %i5,[%fp+88]");
#endif

#include "uUnix.h"
#include "uSystem.h"
#include "uMachine.i"
#include "uMalloc.i"

void *uMalloc( int size ) {
    void *addr = uLowMalloc( size );

#ifdef __U_DEBUG__
    if ( addr == NULL ) {
	uOutOfMemoryExMsg msg;

	msg.msg = "uMalloc: OUT OF MEMORY.\n";
	msg.size = size;

	uRaise( NULL, &uOutOfMemoryEx, &msg, sizeof( msg ) );
    } /* if */
#endif
    return addr;
} /* uMalloc */

void uFree( void *addr ) {
    uLowFree( addr );
} /* uFree */

void *uRealloc( void *addr, int size ) {
    addr = uLowRealloc( addr, size );

#ifdef __U_DEBUG__
    if ( addr == NULL ) {
	uNoExtendExMsg msg;

	msg.base.msg = "uRealloc: OUT OF MEMORY.\n";
	msg.base.size = size;
	msg.addr = addr;

	uRaise( NULL, &uNoExtendEx, &msg, sizeof( msg ) );
    } /* if */
#endif
    return addr;
} /* uRealloc */

#ifdef __U_MULTI__
static int uTerminate = U_FALSE;
static uLock uClusterLock = U_UNLOCKED;
static uCluster uClusterList = NULL;
static void uShutdown( int, int );
#endif /* __U_MULTI__ */

#include "uPeek.i"
#include "uQueue.i"

static uCluster uSystemCluster;

static void uExecuteTask( uTask task );  /* forwared declaration of a static function */

/*
 * The following variable is used to determine if the uSystem should
 * dump core when uAbort is called. The idea is, that only one uSystem
 * process (virtual processor) will dump core.  In this manner, if a
 * process dies with a segmentation fault, the core dump will show the
 * location of the segmentation fault.
 * Before calling abort(), uAbort sets this variable to the UNIX processor id
 * of the aborting processor.
 * The SIGCHLD handler also sets this variable to the UNIX process id of the
 * child, if the child died with a core dump.
 */
static int uCorePid = -1;

uCoroutine uThisCoroutine( void ) {
    return( uWorkCoroutine );
} /* uThisCoroutine */

uTask uThisTask( void ) {
    return( uWorkTask );
} /* uThisTask */

uProcessor uThisProcessor( void ) {
    return( uWorkProcessor );
} /* uThisProcessor */

uCluster uThisCluster( void ) {
    return( uWorkTask->cluster );
} /* uThisCluster */

void uLongSetStackSize( uCluster cluster, long size ) {
    cluster->stacksize = size;
} /* uLongSetStackSize */

long uLongGetStackSize( uCluster cluster ) {
    return( cluster->stacksize );
} /* uLongGetStackSize */

void uSetStackSize( long size ) {
    uLongSetStackSize( uThisCluster(), size );
} /* uSetStackSize */

long uLongGetTimeSlice( uCluster cluster ) {
    return( cluster->timeslice );
} /* uLongGetTimeSlice */

void uSetTimeSlice( long msecs ) {
    uLongSetTimeSlice( uThisCluster(), msecs );
} /* uSetTimeSlice */

long uGetTimeSlice( void ) {
    return( uLongGetTimeSlice( uThisCluster() ) );
} /* uGetTimeSlice */

void uLongSetSpin( uCluster cluster, long usecs ) {
    cluster->spin = usecs;
} /* uLongSetSpin */

long uLongGetSpin( uCluster cluster ) {
    return( cluster->spin );
} /* uLongGetSpin */

void uSetSpin( long usecs ) {
    uLongSetSpin( uThisCluster(), usecs );
} /* uSetSpin */

long uGetSpin( void ) {
    return( uLongGetSpin( uThisCluster() ) );
} /* uGetSpin */

void uLongSetArgLen( uCluster cluster, long len ) {
    cluster->arglen = len;
} /* uLongSetArgLen */

long uLongGetArgLen( uCluster cluster ) {
    return( cluster->arglen );
} /* uLongGetArgLen */

void uSetArgLen( long len ) {
    uLongSetArgLen( uThisCluster(), len );
} /* uSetArgLen */

long uGetArgLen( void ) {
    return( uLongGetArgLen( uThisCluster() ) );
} /* uGetArgLen */

int uLongReadyTasks( uCluster cluster ) {
    return( uTaskQueueLength( &(cluster->ready) )) ;
} /* uLongReadyTasks */
    
int uReadyTasks( void ) {
    return( uLongReadyTasks( uThisCluster() ) );
} /* uReadyTasks */

void uSetName( char *name ) {
    uWorkCoroutine->name = name;
} /* uSetName */

char *uGetName( uCoroutine cid ) {
    if ( cid->name == NULL ) {
 	sprintf( cid->hex, "0x%x", ( unsigned ) cid );
	cid->name = cid->hex;
    } /* if */
    return cid->name;
} /* uGetName */

void uSaveFixed( void ) {
    uWorkCoroutine->save = U_SAVE_FIXED;		/* save only the fixed point registers for this task */
} /* uSaveFixed */

void uSaveFloat( void ) {
    uWorkCoroutine->save = U_SAVE_FLOAT;		/* save fixed and floating pointer registers for this task */
} /* uSaveFloat */

long uGetStackSize( void ) {
    return( uLongGetStackSize( uThisCluster() ) );
} /* uGetStackSize */

static inline void uAcquireTask( uTask task ) {

    /*
     * This function allows a processor to acquire task for execution.
     */

    uAcquireLock( &(task->mutex) );
} /* uAcquireTask */

static inline void uReleaseTask( uTask task ) {

    /*
     * This function allows a processor to release a task
     * so that another processor may execute it.
     */

    uReleaseLock( &(task->mutex) );
} /* uReleaseTask */

static inline uTask uAllocTask( int space ) {

    /*
     * This function allocates a task, coroutine and stack.
     */

    void *start;
    uTask task;

    space = U_CEILING( space, sizeof(double) ) + sizeof(double); /* move value up to multiple of sizeof(double) */
    start = uMalloc( space + sizeof(struct uTaskD) );
#if __U_STACK_GROWS__DOWN__				/* stack grows down */
    task = (void *) U_FLOOR( (int) start + space, sizeof(double) ); /* move value down to multiple of sizeof(double) */
    *task = U_TASK();
    task->start = start;				/* free address */
    task->StackData.base = (void *) ( (int) task - sizeof( double ) ); /* prime stack for first push */
    task->StackData.limit = start;
    uPoke( task->StackData.limit, U_MAGIC );
#else							/* stack grows up */
    task = start;
    *task = U_TASK();
    task->start = start;				/* free address */
    task->StackData.base = (void *) U_CEILING( (int)start + sizeof(struct uTaskD), sizeof(double) );
    task->StackData.limit = start + space + sizeof(struct uTaskD) - sizeof(int);
    uPoke( task->StackData.limit, U_MAGIC );
#endif /* __U_STACK_GROWS__DOWN__ */
    task->cortn = task;					/* task is also a coroutine */

    return( task );
} /* uAllocTask */

static inline void uFreeTask( uTask task ) {

    /*
     * This function deallocates the resources of a task.
     */

    uFree( task->start );
} /* uFreeTask */

static inline uProcessor uAllocProcessor( void ) {

    /*
     * This function allocates a process descriptor.
     */

    uProcessor processor;

    processor = uMalloc( sizeof( struct uProcessorD ) ); /* allocate memory for a process descriptor */
    *processor = U_PROCESSOR();				/* initialize the processor descriptor */
    
    return( processor );				/* return a reference to the process */
} /* uAllocProcessor */

static inline void uFreeProcessor( uProcessor processor ) {

    /*
     * This function dellocates the resources of a process.
     */

    uFree( processor );					/* deallocate the processor descriptor */
} /* uFreeProcessor */

static inline void uAcquireClusterList( void ) {
#ifdef __U_MULTI__
    uAcquireLock( &uClusterLock );
#endif /* __U_MULTI__ */
} /* uAcquireClusterList */

static inline void uReleaseClusterList( void ) {
#ifdef __U_MULTI__
    uReleaseLock( &uClusterLock );
    if ( uTerminate ) {
	uAcquireLock( &uClusterLock );
	uShutdown( U_TRUE, U_TRUE );
	/* control does not return here */
    } /* if */
#endif /* __U_MULTI__ */
} /* uReleaseClusterList */

static inline uCluster uAllocCluster( void ) {

    /*
     * This function allocates a cluster descriptor.
     */

    uCluster cluster;
    int i;

    cluster = uMalloc( sizeof( struct uClusterD ) );	/* allocate memory for a cluster descriptor */
    *cluster = U_CLUSTER();				/* initialize the cluster descriptor */

    FD_ZERO( &(cluster->rfds) );			/* clear the read set */
    FD_ZERO( &(cluster->wfds) );			/* clear the write set */
    FD_ZERO( &(cluster->efds) );			/* clear the exception set */

    for ( i = 0; i < FD_SETSIZE; i += 1 ) {
	cluster->ioqueue[i] = U_SEMAPHORE( 0 );
	cluster->iocount[i] = 0;
    } /* for */

    return( cluster );					/* return a reference to the cluster descriptor */
} /* uAllocCluster */

static inline void uFreeCluster( uCluster cluster ) {

    /*
     * This function deallocates the resources of a cluster.
     */

    uFree( cluster );					/* deallocate the cluster descriptor */
} /* uFreeCluster */

#ifdef __U_MULTI__
static inline void uLinkCluster( uCluster cluster ) {
    uAcquireClusterList();

    cluster->link = uClusterList;
    uClusterList = cluster;

    uReleaseClusterList();
} /* uLinkCluster */

static void uUnlinkCluster( uCluster cluster ) {
    uCluster current, *old;

    uAcquireClusterList();

    current = uClusterList;
    old = &uClusterList;

    while ( current && current != cluster ) {
	old = &current->link;
	current = current->link;
    } /* while */

    if ( !current ) {
	/* Yipes! Cluster is not linked */
	uReleaseClusterList();
	uAbort( "uUnlinkCluster( 0x%x ) : CLUSTER IS NOT LINKED.\n", cluster ); 
    } /* if */

    *old = current->link;

    uReleaseClusterList();
} /* uUnlinkCluster */
#endif /* __U_MULTI__ */

#include "uStack.i"

void uLongSetTimeSlice( uCluster cluster, long msecs ) {
    long osecs;
    
    osecs = cluster->timeslice;				/* read the old slice */

    if ( msecs != osecs ) {				/* if the slice has changed */
	cluster->timeslice = msecs;			/* write the new slice */
#ifdef __U_MULTI__
	/* if the old slice was 0, or greater than the new slice, send a signal to each processor on this cluster */
	if ( osecs == 0 || osecs > msecs ) {
	    uProcessorStack *cpus;
	    uProcessor processor;

	    /*
	     * boink each of the processors so that they will update their time slice duration.
	     */

	    cpus = &(cluster->cpus);			/* select the processor stack associated with the task's cluster */
	    uAcquireProcessorStack( cpus );		/* lock the processor stack */
	    processor = uProcessorStackTop( cpus );
	    while ( processor != NULL ) {		/* go through the processor stack */
		kill( processor->pid, SIGALRM );	/* give it the signal kick */
		processor = uProcessorStackLink( processor );
	    } /* while */
	    uReleaseProcessorStack( cpus );		/* unlock the processor stack */
	} /* if */
#endif
    } /* if */
} /* uLongSetTimeSlice */

int uLongGetProcessors( uCluster cluster ) {
    return( uProcessorStackLength( &(cluster->cpus) ) );
} /* uLongGetProcessors */

void uSetProcessors( int num ) {
    uLongSetProcessors( uThisCluster(), num );
} /* uSetProcessors */

int uGetProcessors( void ) {
    return( uLongGetProcessors( uThisCluster() ) );
} /* uGetProcessors */

#ifdef __U_MULTI__
static void uShutdown( int killme, int msgprt ) {

    /*
     * This function goes through the list of clusters, destroying each one.
     */

    uCluster cluster;
    uProcessor processor;

    for ( cluster = uClusterList; cluster; cluster = cluster->link ) {
	if ( uTryAcquireLock( &(cluster->cpus.mutex) ) ) {
	    while ( !uProcessorStackEmpty( &cluster->cpus ) ) {
		processor = uPopProcessor( &cluster->cpus );
		if ( processor->pid != uCorePid && processor->pid != uWorkProcessor->pid ) {
		    kill( processor->pid, SIGKILL );
		    if ( msgprt ) {
#if __U_DEBUG__
			fprintf( stderr, "uShutdown(): Processor:0x%x (Unix pid:%d) was cleaned up on Cluster 0x%x.\n",
				processor, processor->pid, cluster );
#endif /* __U_DEBUG__ */
		    } /* if */
		} /* if */
	    } /* while */
	} /* if */
    } /* for */
    uReleaseLock( &uClusterLock );
    if ( killme ) {
	kill( uWorkProcessor->pid, SIGKILL );
    } /* if */
} /* uShutdown */
#endif /* __U_MULTI__ */

char *uSignalError[] = {				/* error messages associated with signals */
    "PROCESS EXITING",
    "HANGUP",
    "INTERRUPT",
    "QUIT",
    "ILLEGAL INSTRUCTION",
    "TRACE TRAP",
    "IOT INSTRUCTION",
    "EMT INSTRUCTION",
    "FLOATING POINT EXCEPTION",
    "KILL",
    "BUS ERROR",
    "SEGMENTATION VIOLATION",
    "BAD ARGUMENT TO SYSTEM CALL",
    "WRITE ON A PIPE WITH NO ONE TO READ IT",
    "ALARM CLOCK",
    "SOFTWARE TERMINATION SIGNAL",
    "URGENT CONDITION PRESENT ON SOCKET",
    "STOP",
    "STOP SIGNAL GENERATED FROM KEYBOARD",
    "CONTINUE AFTER STOP",
    "CHILD STATUS HAS CHANGED",
    "BACKGROUND READ ATTEMPTED FROM CONTROL TERMINAL",
    "BACKGROUND WRITE ATTEMPTED TO CONTROL TERMINAL",
    "IO IS POSSIBLE ON A DESCRIPTOR",
    "CPU TIME LIMIT EECEEDED",
    "FILE SIZE LIMIT EXCEEDED",
    "VIRTUAL TIME ALARM",
    "PROFILING TIMER ALARM",
    "USER BY PCI TO SIGNAL A WINDOW SIZE CHANGE",
    "USER DEFINED SIGNAL 1",
    "USER DEFINED SIGNAL 2"
    };

#ifdef __U_MULTI__
#include <sys/wait.h>
extern int wait3( union wait *, int, struct rusage * );

static void uSigChld( int a, int b, struct sigcontext *vsc ) {

    /*
     * This is a signal handler for the SIGCHLD signal.  It is installed by each
     * processor in the uSystem.  If any processor should die for any reason,
     * (floating point error, segmentation violation), its death will be caught by
     * its parent process.  When a process dies, its parent process receives the
     * SIGCHLD signal, and enters this handler.  If the process died abnormally,
     * then an error mesage is printed out and the uSystem error handling
     * facilities is invoked to handle the error.
     */

    int pid;
    union wait status;
    struct sigcontext *sc = vsc;
 
    pid = wait3( &status, WNOHANG, 0 );			/* find out who died and why they died */

    if ( pid != 0 && status.w_T.w_Retcode != 0 ) {	/* was there a problem ? */
	if ( status.w_T.w_Coredump ) {			/* dumped core, don't dump in uAbort */
	    uCorePid = pid;
	} /* if */
	if ( !uTerminate ) {
	    uAbort( "uSigChld(%d, %d, 0x%x) : PROCESSOR DIED, CAUSE OF DEATH WAS %s.\n", a, b, sc, uSignalError[status.w_T.w_Termsig] );
	} else {
	    if ( uTryAcquireLock( &uClusterLock ) ) {
		uShutdown( U_TRUE, U_TRUE );
		/* control never gets here */
	    } /* if */
	} /* if */
    } /* if */
} /* uSigChld */

static void uSigTerm() {

    /*
     * This is a signal handler for the SIGTERM signal.
     * The "kill" command will only affect the system cluster
     * processor, so we catch SIGTERM here and shut down all
     * the processors.
     */

    uTerminate = U_TRUE;
    if ( uTryAcquireLock( &uClusterLock ) ) {
	uShutdown( U_TRUE, U_TRUE );
	/* control never gets here */
    } /* if */
} /* uSigTerm */

#endif

#ifdef __U_MULTI__

#include "uCalibrate.h"
#include "uIdle.i"

#endif

static inline uTask uSelectTask( void ) {

    /*
     * This function selects a task from the ready queue associated with the
     * cluster on which the processor is executing.
     * If the processor has been scheduled for termination, this function
     * returns the null task reference, which terminates the processor.
     */
    
    uTask task;
    uTaskQueue *ready = &(uWorkProcessor->cluster->ready);

    for ( ;; ) {
      if ( uWorkProcessor->stop ) {			/* if the processor is scheduled for termination, */
	    return NULL;				/* return a reference to the null task. */
	} /* exit */
	uAcquireTaskQueue( ready );			/* lock the ready queue. */
      if ( ! uTaskQueueEmpty( ready ) ) {		/* if the ready queue is not empty, */
	    task = uDeleteTask( ready );		/* remove a task, */
	    uReleaseTaskQueue( ready );			/* unlock the ready queue, */
	    return task;				/* and return a reference to the task. */
	} /* exit */
	uReleaseTaskQueue( ready );			/* unlock the ready queue, */

#ifdef __U_MULTI__
	uIdle( ready );					/* idle the processor until work appears. */
#else
	uAbort( "uSelectTask() : NO READY TASK. SYSTEM IS EITHER DEADLOCKED OR ALL TASKS ARE WAITING.\n" );
#endif
	
    } /* for */
} /* uSelectTask */

static inline void uSave( void ) {

    /*
     * This function saves the context of a task or coroutine.
     */

    uStack stack = &(uWorkCoroutine->StackData);

    stack->pc = uReadReturnAddress();			/* save the return address */

    if ( uWorkCoroutine->cxtlist != NULL ) {		/* if user contexts to save... */
	register struct uCxtBase *p;

	for ( p = uWorkCoroutine->cxtlist; p != NULL; p = p->link ) {
	    p->saver( &(p->cxtarea) );
	} /* for */
    } /* if */

    /* Nothing but inline routine calls past this point */

    uWorkCoroutine->errno = errno;			/* save errno */
    uPushFixedRegs();					/* push the fixed point registers onto the task stack */

    if ( uWorkCoroutine->save == U_SAVE_FLOAT ) {	/* if performing a floating point save... */
	uPushFloatRegs();				/* push the floating point registers on the stack also */
    } /* if */

    stack->fp = uReadFramePointer();			/* save the current frame pointer */
    stack->sp = uReadStackPointer();			/* save the current stack pointer */
} /* uSave */

static inline void uRestore( void ) {

    /*
     * This function restores the context of a task or coroutine.
     */

    uStack stack = &(uWorkCoroutine->StackData);

    uWriteStackPointer( stack->sp );			/* restore the stack pointer */
    uWriteFramePointer( stack->fp );			/* restore the frame pointer */

    if ( uWorkCoroutine->save == U_SAVE_FLOAT ) {	/* if the floating point registers were saved ... */
	uPopFloatRegs();				/* pop the floating point registers from the stack */
    } /* if */

    uPopFixedRegs();					/* pop the fixed point registers from the stack */
    errno = uWorkCoroutine->errno;			/* restore errno */

    /* Ok to have non-inline routine calls past this point */

    if ( uWorkCoroutine->cxtlist != NULL ) {		/* if user contexts to restore... */
	struct uCxtBase *p;

	for ( p = uWorkCoroutine->cxtlist; p != NULL; p = p->link ) {
	    p->restorer( &(p->cxtarea) );
	} /* for */
    } /* if */

    uWriteReturnAddress( stack->pc );			/* restore the return address */
} /* uRestore */

static inline void uCallCoroutine( uCoroutine cortn ) {

    /*
     * This function initializes the context of a new coroutine.
     */

    uStack stack= &(cortn->StackData);

    uWriteStackPointer( stack->sp );			/* install the new task stack pointer */
    uWriteFramePointer( stack->fp );			/* install the new task frame pointer */

    uCallUsingStack( cortn->begin );			/* call the task, arguments are already on the stack */
} /* uCallCoroutine */

static inline void uCallTask( uTask task ) {

    /*
     * This function initializes the context of a new coroutine.
     */

    uCallCoroutine( task->cortn );
} /* uCallTask */

static void uScheduler( void ) {

    uTask task;

    while ( ( task = uSelectTask() ) != NULL ) {
	uAcquireTask( task );				/* acquire this task for execution */
	uExecuteTask( task );				/* execute this task */
	uReleaseTask( task );				/* release this task for execution */
    } /* while */
} /* uScheduler */

static inline void uBlockTask( void ) {

    uExecuteTask( uSchedulerTask );			/* resume the scheduler task */

    /* Check if an async intervention can be delivered */
    if( uWorkTask->inters && !(uWorkTask->state & U_DISABLE_INTER) && uWorkTask->icheckneeded )
	uAsyncInterDeliver();

} /* uBlockTask */

static inline void uWakeTask( uTask task ) {

    /*
     * This function adds a task to the ready queue associated with the
     * cluster on which the task is executing.
     */

    uTaskQueue *ready;
    
    ready = &(task->cluster->ready);			/* select the ready queue associated with the task's cluster */

    uAcquireTaskQueue( ready );				/* lock the ready queue */
    uInsertTask( ready, task );				/* add the task to the end of the ready queue */

#ifdef __U_MULTI__

    if ( ready->idle ) {				/* is there an idle processor that could execute this task ? */
	uProcessorStack *cpus;
	uProcessor processor;
	
	ready->idle = U_FALSE;				/* change ready queue idle state  */
	cpus = &(task->cluster->cpus);			/* select the processor stack associated with the task's cluster */
	uAcquireProcessorStack( cpus );			/* lock the processor stack */
	processor = uProcessorStackTop( cpus );
	while ( processor != NULL ) {			/* search processor stack for an idle processor */
	    if ( processor->state == U_IDLE ) {		/* found one? */
		processor->state = U_BUSY;		/* mark it busy */
		kill( processor->pid, SIGALRM );	/* wake it up, as it is probably sleeping */
	    } /* if */
	    processor = uProcessorStackLink( processor );
	} /* while */
	uReleaseProcessorStack( cpus );			/* unlock the processor stack */
	uReleaseTaskQueue( ready );			/* unlock the ready queue here so processor stack is guaranteed to be released */
    } else {
	uReleaseTaskQueue( ready );			/* unlock the ready queue */
    } /* if */

#else

    uReleaseTaskQueue( ready );				/* unlock the ready queue */

#endif
} /* uWakeTask */

volatile void uExit( int rc ) {
#ifdef __U_MULTI__
    uMigrate( uSystemCluster );				/* migrate to the system cluster */
    uShutdown( U_FALSE, U_FALSE );
#endif
    exit( rc );
} /* uExit */

volatile void uAbort( char *fmt, ... ) {

    /*
     * This function handles any errors that may appear during the execution of
     * uKernel code or user code.  Currently, this function simply
     * prints out a message indicating who caused the error, and what the
     * error was.  Then the uSystem shuts itself down, killing all remaining UNIX processes.
     */

    va_list argadr;

    extern int errno;

    va_start( argadr, fmt );

    fflush( stdout );
    fprintf( stderr, "*** uSYSTEM ERROR, " );
    _doprnt( fmt, argadr, stderr );
    fprintf( stderr, "*** UNIX ERRNO, %d\n", errno );
    fprintf( stderr, "Last uCoroutine to execute was %s.\n", uGetName( uThisCoroutine() ) );
    fprintf( stderr, "Last uTask to execute was 0x%x.\n", ( unsigned ) uThisTask() );
    fprintf( stderr, "uCoroutine and uTask lived on uCluster 0x%x.\n", ( unsigned )uThisCluster() );
    fflush( stderr );

    va_end( argadr );

    if( uCorePid == -1 ) {				/* if processor should core, prevent it from happening again */
	uCorePid = uWorkProcessor->pid;
    } /* if */

#ifdef __U_MULTI__
    uTerminate = U_TRUE;
    if ( uTryAcquireLock( &uClusterLock ) ) {
	uShutdown( U_FALSE, U_TRUE );
    } /* if */
#endif

    if ( uCorePid == uWorkProcessor->pid ) {
	abort();
    } else {						/* just exit */
	_exit( 1 );
    } /* if */
    /* control never reaches here */	 
} /* uAbort */

void uVerify( void ) {

    /*
     * This function checks the stack of a coroutine to make sure that the
     * stack has not overflowed.  If it has, the function causes an error.
     */

    uStack stack = &(uWorkCoroutine->StackData);

    if ( uWorkTask != uSchedulerTask ) {
	
	if ( uPeek( stack->limit ) != U_MAGIC ) {	/* magic word has been overwritten? */
	    uAbort( "uVerify() : STACK OVERFLOW, MAGIC WORD DESTROYED FOR COROUTINE: %s.\n", uGetName(uWorkCoroutine) );
	} /* if */

#if __U_STACK_GROWS__DOWN__				/* stack grows down */
	if ( uReadStackPointer() < stack->limit ) {	/* current stack pointer beyond stack limit? */
#else							/* stack grows up */
	if ( uReadStackPointer() >= stack->limit ) {	/* current stack pointer beyond stack limit? */
#endif /* __U_STACK_GROWS__DOWN__ */
	    uAbort( "uVerify() : STACK OVERFLOW, BEYOND END OF STACK FOR COROUTINE: %s.\n", uGetName(uWorkCoroutine) );
	} /* if */
    } /* if */
} /* uVerify */

static void uExecuteCoroutine( uCoroutine cortn ) {

    /*
     * This function switches the active thread of control from one task to another.
     */

#ifdef __U_DEBUG__
    uVerify();						/* verify that the stack is still okay */
#endif
    uSave();

    uWorkCoroutine = cortn;
    uWorkTask->cortn = cortn;
    
    if ( cortn->state & U_NEW ) {
	cortn->state &= ~U_NEW;
	cortn->state |= U_RUN;
	uCallCoroutine( cortn );
	cortn->cxtlist = NULL;
	uSuspendDie( NULL, 0 );
    } /* if */

    uRestore();
} /* uExecuteCoroutine */

static void uExecuteTask( uTask task ) {

    /*
     * This function switches the active thread of
     * control from one task to another.
     */

#ifdef __U_DEBUG__
    uVerify();						/* verify that the stack is still okay */
#endif
    uSave();

    uWorkTask = task;
    uWorkCoroutine = task->cortn;

    if ( task->state & U_NEW ) {
	task->state &= ~U_NEW;
	task->state |= U_RUN;
	task->cortn->state &= ~U_NEW;
	task->cortn->state |= U_RUN;
	uCallTask( task );
	task->cxtlist = NULL;
	uDie( NULL, 0 );
    } /* if */

    uRestore();
} /* uExecuteTask */

int uKerDisable( void ) {
    /*
     * This function disables async interventions.
     * It returns the previous state of async interventions:
     *    U_TRUE if interventions were enabled.
     *    U_FALSE if interventions were disabled.
     */
    
    register int enabled = (uWorkTask->state & U_DISABLE_INTER) == 0;
    
    uWorkTask->state |= U_DISABLE_INTER;

    return enabled;
} /* uKerDisable */

void uKerEnable( register int previous ) {
    /*
     * This function re-enables async interventions if previous is U_TRUE.
     */
    
    if( previous ) {
	uWorkTask->state &= ~U_DISABLE_INTER;
	if( uWorkTask->inters && uWorkTask->icheckneeded ) {
	    /*
	     * Pending intervention(s) which haven't been checked.
	     * possibly deliver one
	     */
	    uAsyncInterDeliver();
	} /* if */
    } /* if */
} /* uKerEnable */

void uP( uSemaphore *sem ) {

    /*
     * This function implements the P operation on a semaphore.
     */

    uAcquireTaskQueue( sem );				/* lock the semaphore */
    if ( sem->len < 0 ) {				/* if the semaphore count is less than zero, */
	sem->len += 1;					/* simply increment the semaphore count and */
	uReleaseTaskQueue( sem );			/* unlock the semaphore, */
    } else {
	uInsertTask( sem, uWorkTask );			/* otherwise, insert a task on the semaphore blocked queue and */
	uWorkTask->peedsem = sem;			/* record where it is blocked */
	uWorkTask->state |= U_PEED;			/* record that it is Pd */
	uReleaseTaskQueue( sem );			/* unlock the semaphore. */
	uBlockTask();					/* then call the kernel to block the current task */
    } /* if */

} /* uP */

void uV( uSemaphore *sem ) {

    /*
     * This function implements the V operation on a semaphore.
     */

    uTask task;

    uAcquireTaskQueue( sem );				/* lock the semaphore */
    if ( uTaskQueueEmpty( sem ) ) {			/* if the blocked queue is empty, */
	sem->len -= 1;					/* simply decrement the counter and */
	uReleaseTaskQueue( sem );			/* unlock the semaphore, */
    } else {
	task = uDeleteTask( sem );			/* otherwise, remove one task from the blocked queue, */
	uReleaseTaskQueue( sem );			/* unlock the semaphore and */
	task->state &= ~U_PEED;				/* task is running */
	task->peedsem = NULL;				/* no longer blocked */
	uWakeTask( task );				/* make the task eligible for execution. */
    } /* if */
} /* uV */

int uC( uSemaphore *sem ) {

    /*
     * This function implements the C operation on a semaphore.
     */

    return -uTaskQueueLength( sem );

} /* uC */

static inline uTask uDoEmit( uCluster cluster, long space, void (*begin)(), long arglen, void *argadr ) {

    uTask task;
    uStack stack;

    task = uAllocTask( space );
    task->cluster = cluster;				/* set the cluster of residence for this task */

    task->begin = begin;				/* task begins execution at this address */
    task->owner = task;					/* task is its own coroutine owner */
    task->cortn = task;					/* task's current coroutine is itself */
    
    stack = &(task->StackData);

    uMakeFrameUsingStack( stack, argadr, arglen );

    uWakeTask( task );

    return( task );
} /* uDoEmit */

uTask uEmit( void *begin, ... ) {

    /*
     * This function emits a task with the default values for cluster, stack size,
     * and argument length.
     */

    va_list argadr;
#ifndef __U_DEBUG__
    uTask task;

    va_start( argadr, begin );

#else
    volatile uTask task;
    void           *msgp;

    va_start( argadr, begin );

    uExcept {
#endif

	task = uDoEmit( uThisCluster(), uGetStackSize(), (void (*)())begin, uGetArgLen(), argadr );
	
	va_end( argadr );
	
#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp ) {
	    uEmitExMsg msg;

	    msg.msg = "uEmit: Cannot emit uTask due to memory allocation problems.\n";
	    msg.cluster = uThisCluster();
	    msg.space = uGetStackSize();
	    msg.begin = begin;
	    msg.arglen = uGetArgLen();

	    uRaise( uThisCluster(), &uEmitEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( task );					/* return a reference to the new task */
} /* uEmit */

uTask uLongEmit( uCluster where, long space, void *begin, long arglen, ... ) {

    /*
     * This function emits a task with
     * explicit values for cluster, stack size,
     * and argument length.
     */

    va_list argadr;
#ifndef __U_DEBUG__
    uTask task;

    va_start( argadr, arglen );
#else
    volatile uTask task;
    void           *msgp;

    va_start( argadr, arglen );

    uExcept {
#endif
	task = uDoEmit( where, space, (void (*)())begin, arglen, argadr );

	va_end( argadr );
#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp ) {
	    uEmitExMsg msg;

	    msg.msg = "uLongEmit: Cannot emit uTask due to memory allocation problems.\n";
	    msg.cluster = where;
	    msg.space = space;
	    msg.begin = begin;
	    msg.arglen = arglen;

	    uRaise( where, &uEmitEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( task );					/* return a reference to the new task */
} /* uLongEmit */

void uAbsorb( uTask dier, void *rbuf, int rlen ) {

    /*
     * This function waits for the termination of a task,
     * obtains the terminating result, and
     * deallocates the resources of the terminated task.
     */

    /*
     * Disable Async Interventions
     */
    int enabled = uKerDisable();

    uP( &(dier->done) );				/* wait for the task to complete */

#ifdef __U_DEBUG__

    if ( dier->ipc.slen > rlen ) {
	uAbsorbAreaTooShortExMsg msg;

	msg.base.msg = "uAbsorb: ABSORB MESSAGE AREA TOO SHORT TO RECEIVE ABSORBING TASK'S MESSAGE\n";
	msg.base.sbuf = dier->ipc.sbuf;
	msg.base.slen = dier->ipc.slen;
	msg.base.rbuf = rbuf;
	msg.base.rlen = rlen;
	msg.dier = dier;
	msg.absorber = uWorkTask;

	uV( &(dier->done) );	                        /* reset the semaphore so that the uAbsorb can be re-tried */

	/*
	 * Enable Async Interventions
	 */
	uKerEnable( enabled );

	uRaise( NULL, &uAbsorbAreaTooShortEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uCopy( dier->ipc.sbuf, rbuf, dier->ipc.slen );	/* copy the terminating result to the absorber */

    uAcquireTask( dier );

    uFreeTask( dier );

    /*
     * Enable Async Interventions
     */
    uKerEnable( enabled );

} /* uAbsorb */

volatile void uDie( void *sbuf, int slen ) {

    /*
     * This function terminates the execution of a task and returns a result to
     * an absorbing task.
     */

    uTask dier = uWorkTask;
    uStackBlock block = dier->eistack;
    uQdInter inter;

    dier->ipc.sbuf = sbuf;				/* copy the address and length of the terminating result */
    dier->ipc.slen = slen;				/* into my task descriptor */

    /* clean up tasks exception/intervention stack */
    while( block ) {
	switch( block->type ) {
	  case uIsExceptBlock:
	    /*
	     * If the data pointer is non-null then free the data.
	     */
	    if( block->u.uEPart.data ) {
		uFree( block->u.uEPart.data );
	    } /* if */
	    break;
	  case uIsDataBlock:
	    if( block->u.data ) {
		uFree( block->u.data );
		block->u.data = NULL;
	    } /* if */
	    break;
	  case uIsInterBlock:
	    /* do nothing */
	    break;
	} /* switch */

	block = block->prev;
    } /* while */

    uV( &(dier->done) );				/* signal the absorbing task */

    /*
     * clean out any pending async interventions
     * NOTE: There is still a window where an intervention might be
     * put this task before it is absorbed, but that is a design
     * problem with the program/task calling the intervention.
     */

    /* Grab task intervention data lock */
    uAcquireLock( &(dier->interlock) );
    while( dier->inters ) {
	inter = dier->inters;
	dier->inters = dier->inters->next;
	uFree( inter );
    } /* while */
    uReleaseLock( &(dier->interlock) );
    uBlockTask();					/* and block this task forever */
} /* uDie */

uTask uSend( uTask receiver, void *rbuf, int rlen, void *sbuf, int slen ) {

    /*
     * This function sends a message from one task to another, and waits for a reply.
     */

    uTask sender = uWorkTask;

    /*
     * Disable Async Interventions
     */
    int enabled = uKerDisable();

    sender->ipc.rbuf = rbuf;				/* move the address and length of the reply */
    sender->ipc.rlen = rlen;				/* into the task descriptor. */

    uP( &(receiver->ipc.snd) );				/* wait for the receiver task */

#ifdef __U_DEBUG__

    if ( slen > receiver->ipc.rlen ) {			/* if the send message is longer than receiver is willing to accept, */
	uSendMsgTooLongExMsg msg;

	msg.base.msg = "uSend: MESSAGE IS TOO LONG TO BE SENT FROM SENDING TASK\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = receiver->ipc.rbuf;
	msg.base.rlen = receiver->ipc.rlen;
	msg.sender = sender;
	msg.receiver = receiver;

	uV( &(receiver->ipc.snd) );			/* signal receiver to accept a new message */

	/*
	 * Enable Async Interventions
	 */
	uKerEnable( enabled );
	
	uRaise( NULL, &uSendMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uCopy( sbuf, receiver->ipc.rbuf, slen );		/* copy the message to the receiver */
    receiver->ipc.from = sender;			/* tell the receiver who sent the message */
    sender->ipc.to = receiver;				/* remember who the message is sent to */

    uV( &(receiver->ipc.rcv) );				/* signal the receiver to continue */
    uP( &(sender->ipc.rcv) );				/* wait for the reply message */

#ifdef __U_DEBUG__
    /*
     * If rlen is "-1", then the reply area was too short,
     * raise an exception
     */
    if( sender->ipc.rlen == -1 ) {
	uReplyAreaTooShortExMsg msg;

	msg.base.msg = "uSend: REPLY AREA IS TOO SHORT TO RECEIVE REPLY MESSAGE.\n";
	msg.base.sbuf = NULL;                     /* can't access these fields in the receiver at this time */
	msg.base.slen = 0;			  /* can't access these fields in the receiver at this time */
	msg.base.rbuf = rbuf;
	msg.base.rlen = rlen;
	msg.sender = sender;
	msg.replier = receiver;

	/*
	 * Enable Async Interventions
	 */
	uKerEnable( enabled );

	uRaise( NULL, &uReplyAreaTooShortEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    /*
     * Enable Async Interventions
     */
    uKerEnable( enabled );

    return( sender->ipc.from );				/* return the identity of the replier */
} /* uSend */

void uForward( uTask receiver, void *sbuf, int slen, uTask sender ) {

    /*
     * This function forwards a message from one task to another task.
     * The forwarding task may not reply to the message once it has
     * been forwarded.  The forward task must reply to the message,
     * or forward the message again.
     */

    int enabled = uKerDisable();

#ifdef __U_DEBUG__

    uTask forwarder = uWorkTask;
    
    if ( sender->ipc.to != forwarder ) {		/* if the sender did not send message to forwarder, error */
	uInvalidForwardExMsg msg;

	msg.base.base.msg = "uForward: FORWARDING TASK DID NOT RECEIVE MESSAGE FROM SPECIFIED RECEIVER TASK OR IT HAS ALREADY FORWARDED THE MESSAGE.\n";
	msg.base.base.sbuf = sbuf;
	msg.base.base.slen = slen;
	msg.base.base.rbuf = receiver->ipc.rbuf;
	msg.base.base.rlen = receiver->ipc.rlen;
	msg.base.sender = sender;
	msg.base.receiver = receiver;
	msg.forwarder = forwarder;

	/*
	 * Enable Async Interventions
	 */
	uKerEnable( enabled );

	uRaise( NULL, &uInvalidForwardEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uP( &(receiver->ipc.snd) );				/* wait for the receiver to be ready to receive a message */

#ifdef __U_DEBUG__

    if ( slen > receiver->ipc.rlen ) {			/* if message too long for the receiver, signal the */
	uForwardMsgTooLongExMsg msg;

	msg.base.msg = "uForward: MESSAGE IS TOO LONG TO BE FORWARDED FROM FORWARDING TASK\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = receiver->ipc.rbuf;
	msg.base.rlen = receiver->ipc.rlen;
	msg.sender = sender;
	msg.forwarder = forwarder;
	msg.receiver = receiver;

	uV( &(receiver->ipc.snd) );			/* signal receiver to accept a new message */

	/*
	 * Enable Async Interventions
	 */
	uKerEnable( enabled );

	uRaise( NULL, &uForwardMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uCopy( sbuf, receiver->ipc.rbuf, slen );		/* copy the message */

    receiver->ipc.from = sender;			/* let the receiver know who the message came from originally */
    sender->ipc.to = receiver;				/* update the sender's memory of who it sent the message to */

    uV( &(receiver->ipc.rcv) );				/* signal the receiver that it has now received a message */

    /*
     * Enable Async Interventions
     */
    uKerEnable( enabled );
    
} /* uForward */

uTask uReceive( void *rbuf, int rlen ) {

    /*
     * This function allows a task to receive a message
     * from any other task.
     */

    uTask receiver = uWorkTask;

    receiver->ipc.rbuf = rbuf;				/* copy the receiver's address and length of the */
    receiver->ipc.rlen = rlen;				/* receive buffer into its task descriptor */

    uV( &(receiver->ipc.snd) );				/* signal a sender that i am ready to receive a message */
    uP( &(receiver->ipc.rcv) );				/* wait for a message */

    return( receiver->ipc.from );			/* return the identity of the sender */
} /* uReceive */

void uReply( uTask sender, void *sbuf, int slen ) {

    /*
     * This function replies to a previously received message.
     */

    uTask replier = uWorkTask;

#ifdef __U_DEBUG__

    if ( sender->ipc.to != replier ) {			/* if the sender did not send a message to the replier, error */
	uNotReplyBlockedExMsg msg;

	msg.base.msg = "uReply: REPLYING TASK HAS NOT RECEIVED A MESSAGE FROM SENDER TASK OR IT HAS ALREADY BEEN REPLIED TO THAT TASK.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = sender->ipc.rbuf;
	msg.base.rlen = sender->ipc.rlen;
	msg.sender = replier;
	msg.receiver = sender;

	uRaise( NULL, &uNotReplyBlockedEx, &msg, sizeof( msg ) );
    } /* if */

    /* 
     * if the reply message is too long for the sender,
     * then raise an exception in the sender
     * Do this by setting the "sender->ipc.rlen" field to -1.
     * This is checked when the sender wakes up.  If it is -1,
     * an exception is raised in the sender.
     */
    if ( slen > sender->ipc.rlen ) {
	sender->ipc.rlen = -1;
    } else {
	uCopy( sbuf, sender->ipc.rbuf, slen );		/* copy the message */
    } /* if */
#else
    uCopy( sbuf, sender->ipc.rbuf, slen );		/* copy the message */
#endif

    sender->ipc.to = NULL;				/* clear the sender's to flag, so I cannot reply again */
    sender->ipc.from = replier;				/* tell the sender who finally replied to the message */

    uV( &(sender->ipc.rcv) );				/* signal the sender that a reply has been sent */

} /* uReply */

void uYield( void ) {

    /*
     * This function allows a task to explicitly give up its control
     * of a processor.
     */

    uWakeTask( uWorkTask );				/* put the task back on the ready queue */
    uBlockTask();					/* block this task temporarily */

} /* uYield */

uCluster uMigrate( uCluster new ) {

    /*
     * This function allows a task to migrate from one cluster to another.
     */

    uTask task = uThisTask();
    uCluster old = task->cluster;			/* remember old cluster */

    task->cluster = new;				/* change the task's cluster reference to the new cluster */
    uYield();						/* reschedule the task on the new cluster. */
    
    return old;						/* return a reference to the old cluster */
} /* uMigrate */

static inline uCoroutine uDoCocall( void *rbuf, int rlen, long space, void (*begin)(), long arglen, void *argadr ) {

    uCoroutine cortn;
    uStack stack;

    cortn = uAllocTask( space );

    cortn->begin = begin;				/* subroutine to be cocalled */

    stack = &(cortn->StackData);
    
    uMakeFrameUsingStack( stack, argadr, arglen );
    
    cortn->owner = uWorkTask;				/* task that owns this coroutine */

    uResume( cortn, rbuf, rlen, NULL, 0 );

    return( cortn );
} /* uDoCocall */

uCoroutine uCocall( void *rbuf, int rlen, void *begin, ... ) {

    va_list argadr;
#ifndef __U_DEBUG__
    uCoroutine cortn;

    va_start( argadr, begin );

#else
    volatile uCoroutine cortn;
    void                *msgp;

    va_start( argadr, begin );

    uExcept {
#endif

	cortn = uDoCocall( rbuf, rlen, uGetStackSize(), (void (*)())begin, uGetArgLen(), argadr );

	va_end( argadr );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp ) {
	    uCocallExMsg msg;

	    msg.msg = "uCocall: Cannot cocall uCoroutine due to memory allocation problems.\n";
	    msg.rbuf = rbuf;
	    msg.rlen = rlen;
	    msg.space = uGetStackSize();
	    msg.begin = begin;
	    msg.arglen = uGetArgLen();
	    
	    uRaise( NULL, &uCocallEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( cortn );

} /* uCocall */

uCoroutine uLongCocall( void *rbuf, int rlen, long space, void *begin, long arglen, ... ) {

    va_list argadr;
#ifndef __U_DEBUG__
    uCoroutine cortn;

    va_start( argadr, arglen );

#else
    volatile uCoroutine cortn;
    void                *msgp;
    
    va_start( argadr, arglen );

    uExcept {
#endif

	cortn = uDoCocall( rbuf, rlen, space, (void (*)())begin, arglen, argadr );

	va_end( argadr );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp ) {
	    uCocallExMsg msg;

	    msg.msg = "uLongCocall: Cannot cocall uCoroutine due to memory allocation problems.\n";
	    msg.rbuf = rbuf;
	    msg.rlen = rlen;
	    msg.space = space;
	    msg.begin = begin;
	    msg.arglen = arglen;
	    
	    uRaise( NULL, &uCocallEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( cortn );

} /* uLongCocall */

static inline void uDoResume( uCoroutine who, void *rbuf, int rlen, void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine;

    resumer->rbuf = rbuf;
    resumer->rlen = rlen;

    uCopy( sbuf, who->rbuf, slen );

    who->restarter = resumer;

    uExecuteCoroutine( who );

    who = resumer->restarter;

    if ( who->state == U_OLD ) {
	uFreeTask( who );
    } /* if */
} /* uDoResume */

void uResume( uCoroutine who, void *rbuf, int rlen, void *sbuf, int slen ) {

#ifdef __U_DEBUG__
    
    if ( uWorkTask != who->owner ) {
	uBadCoroutineExMsg msg;

	msg.base.msg = "uResume: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = who->rbuf;
	msg.base.rlen = who->rlen;
	msg.restarter = who;
	msg.thistask = uWorkTask;

	uRaise( NULL, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > who->rlen ) {
	uResumeMsgTooLongExMsg msg;

	msg.base.msg = "uResume: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = who->rbuf;
	msg.base.rlen = who->rlen;
	msg.resumed = who;
	msg.restarter = uWorkCoroutine;

	uRaise( NULL, &uResumeMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    who->resumer = uWorkCoroutine;

    uDoResume( who, rbuf, rlen, sbuf, slen );

} /* uResume */

void uSuspend( void *rbuf, int rlen, void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine->resumer;

#ifdef __U_DEBUG__
    
    if ( uWorkTask != resumer->owner ) {
	uBadCoroutineExMsg msg;

	msg.base.msg = "uSuspend: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = resumer->rbuf;
	msg.base.rlen = resumer->rlen;
	msg.restarter = resumer;
	msg.thistask = uWorkTask;

	uRaise( NULL, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > resumer->rlen ) {
	uSuspendMsgTooLongExMsg msg;

	msg.base.msg = "uSuspend: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = resumer->rbuf;
	msg.base.rlen = resumer->rlen;
	msg.resumed = resumer;
	msg.restarter = uWorkCoroutine;

	uRaise( NULL, &uSuspendMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uWorkCoroutine->resumer = NULL;			/* break link with resumer */

    uDoResume( resumer, rbuf, rlen, sbuf, slen );

} /* uSuspend */

static inline void uDoResumeDie( uCoroutine who, void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine;

    resumer->state = U_OLD;
    uCopy( sbuf, who->rbuf, slen );
    who->restarter = resumer;

    uExecuteCoroutine( who );
} /* uDoResumeDie */

void uResumeDie( uCoroutine who, void *sbuf, int slen ) {

#ifdef __U_DEBUG__
    
    if ( uWorkTask != who->owner ) {
	uBadCoroutineExMsg msg;

	msg.base.msg = "uResumeDie: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = who->rbuf;
	msg.base.rlen = who->rlen;
	msg.restarter = who;
	msg.thistask = uWorkTask;

	uRaise( NULL, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > who->rlen ) {
	uResumeMsgTooLongExMsg msg;

	msg.base.msg = "uResumeDie: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = who->rbuf;
	msg.base.rlen = who->rlen;
	msg.resumed = who;
	msg.restarter = uWorkCoroutine;

	uRaise( NULL, &uResumeMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uDoResumeDie( who, sbuf, slen );

} /* uResumeDie */

void uSuspendDie( void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine->resumer;

#ifdef __U_DEBUG__
    if ( uWorkTask != resumer->owner ) {
	uBadCoroutineExMsg msg;

	msg.base.msg = "uSuspendDie: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = resumer->rbuf;
	msg.base.rlen = resumer->rlen;
	msg.restarter = resumer;
	msg.thistask = uWorkTask;

	uRaise( NULL, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > resumer->rlen ) {
	uSuspendMsgTooLongExMsg msg;

	msg.base.msg = "uSuspendDie: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.base.sbuf = sbuf;
	msg.base.slen = slen;
	msg.base.rbuf = resumer->rbuf;
	msg.base.rlen = resumer->rlen;
	msg.resumed = resumer;
	msg.restarter = uWorkCoroutine;

	uRaise( NULL, &uSuspendMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uDoResumeDie( resumer, sbuf, slen );

} /* uSuspendDie */

#ifdef __U_MULTI__

static void uCPU( uProcessor processor ) {

    uTask task;

    processor->pid = getpid();				/* set pid of child process */
    uV( &(processor->done) );				/* signal that pid is set */

    task = uAllocTask( 0 );				/* allocate a task descriptor without a stack */
    task->cluster = processor->cluster;
    task->mutex = U_LOCKED;
    task->state = U_RUN;

    /*
     * Initialize the private variables for this UNIX process.
     */
    
    uWorkProcessor = processor;				/* the working processor is the allocated processor */
    uWorkTask = task;					/* the working task is the allocated task */
    uWorkCoroutine = task->cortn;
    uSchedulerTask = task;
    uSetName( "uScheduler" );

    /*
     * Now, this task must wait until it receives a message
     * indicating that this virtual processor must now
     * terminate execution.
     */
    
    uSetAlarm( uGetTimeSlice() );			/* set time slice to the cluster value */

    uScheduler();

    uSetAlarm( 0 );					/* turn off time slicing */

    uFreeTask( task );					/* deallocate the kernel task for this processor */
    
    uV( &(processor->done) );				/* signal that processor is finished */
    
    exit( 0 );						/* finally, this UNIX process terminates. */
} /* uCPU */

#endif

#ifdef __U_MULTI__

static uProcessor uEmitProcessor( uCluster cluster ) {

    /*
     * This function creates a new virtual processor on the specified cluster.
     * This function will return after the virtual processor
     * is created and just before the virtual processor will
     * start executing tasks.
     */

    volatile uProcessor processor;

#ifdef __U_DEBUG__
    void *msgp;
#endif

    int size;
    int pid;
    void *sp;
    void *fp;
    uStack stack;

    /*
     * Allocate and initialize the resources necessary
     * to get a new virtual processor operational.
     */

#ifdef __U_DEBUG__
    uExcept {
#endif

	processor = uAllocProcessor();			/* allocate a processor descriptor */

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp ) {
	    uCreateProcessorExMsg msg;
	    
	    msg.msg = "uLongSetProcessors: uProcessor creation failed due to memory allocation problems.\n";
	    msg.num_proc = uLongGetProcessors( cluster );
	    
	    uRaise( cluster, &uCreateProcessorEx, &msg, sizeof(msg) );
	} /* uWhen */
    } uEndExcept;
#endif

    processor->cluster = cluster;

    /*
     * Before a new UNIX process can be forked, the stack has to be switched to
     * the UNIX process associated with the processor that is executing this task.
     * If the processor stack is not used, the new UNIX process will be executing
     * on the current task's shared stack, which causes some major difficulties.
     */

    /*
     * Save the parameters of the task frame, and calculate the size of the frame.
     */

    sp = uReadStackPointer();				/* save the task stack pointer */
    fp = uReadFramePointer();				/* save the task frame pointer */
#ifdef __mips__
    size = 252;						/* mips doesn't use a frame pointer so fake some stack space  */
#else
    size = fp - sp;					/* calculate the size of this frame */
#endif

    /*
     * Save the frame of the current task.
     */
    
    stack = &(uWorkTask->StackData);			/* if the work task is the kernel task, */
    stack->sp = sp;					/* make sure the kernel task saves its sp and, */
    stack->fp = fp;					/* make sure the kernel task saves its fp */
    
    /*
     * Create a frame on the kernel task.
     */

    stack = &(uSchedulerTask->StackData);
    stack->sp -= sizeof( stack->fp );			/* make room to store the old frame pointer */
    uPoke( stack->sp, (int) stack->fp );		/* save the old frame pointer on the stack */
    stack->fp = stack->sp;				/* move the frame pointer to the new frame */
    stack->sp -= size;					/* make room on the stack for the frame */

    uCopy( sp, stack->sp, size );			/* copy the stack */

    /*
     * Now, the stack of the UNIX process is installed.
     */

    uWriteStackPointer( stack->sp );			/* write the new stack pointer */
    uWriteFramePointer( stack->fp );			/* write the new frame pointer */

    /*
     * Finally, a new UNIX process can be safely forked.
     */

    pid = uFork( uCPU, processor );

    /*
     * If the value of pid is zero, then this is the thread of the
     * new UNIX process.  Notice that pid will only be zero in the
     * BSD version of the uSystem.  The IRIX version of uFork()
     * actually starts a processor running the uCPU() function directly.
     */

    if ( pid == 0 ) {
	uCPU( processor );
    } /* if */

    /*
     * Remove the frame from the kernel task.
     */

    stack = &(uSchedulerTask->StackData);
    stack->sp = stack->fp;				/* move the UNIX stack pointer back a frame*/
    stack->fp = (void *) uPeek( stack->sp );		/* pop the old UNIX frame pointer from the stack */
    stack->sp += sizeof( stack->fp );			/* decrement the UNIX stack pointer */

    /*
     * Restore the frame of the working task.
     */

    stack = &(uWorkTask->StackData);
    sp = stack->sp;
    fp = stack->fp;

    uWriteStackPointer( sp );				/* write the task stack pointer */
    uWriteFramePointer( fp );				/* write the task frame pointer */

    uP( &(processor->done) );				/* wait for child process to set its pid */

#ifdef __U_DEBUG__
    if ( pid == -1 ) {					/* raise exception if fork failed */
	uCreateProcessorExMsg msg;

	uFreeProcessor( processor );

	msg.msg = "uLongSetProcessors: Cannot emit processor due to failure of Unix process creation.\n";
	msg.num_proc = uLongGetProcessors( cluster );

	uRaise( cluster, &uCreateProcessorEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    /*
     * Return a reference to the new process.
     */

    return( processor );
} /* uEmitProcessor */

#endif

#ifdef __U_MULTI__
    
static void uAbsorbProcessor( uProcessor processor ) {

    /*
     * This function absorbs a virtual processor.
     */

    processor->stop = U_TRUE;				/* schedule the processor for termination */
    kill( processor->pid, SIGALRM );			/* give it a nudge, to start termination */

    uP( &(processor->done) );				/* wait until it has terminated */

    uFreeProcessor( processor );			/* deallocate the processor */
} /* uAbsorbProcessor */

#endif

void uLongSetProcessors( uCluster cluster, int num ) {

#ifdef __U_MULTI__

    uProcessorStack *cpus = &(cluster->cpus);
    uProcessor processor;

#ifdef __U_DEBUG__
    if ( num < 0 ) {
	uCreateProcessorExMsg msg;

	msg.msg = "uLongSetProcessors: ATTEMPT TO SET NUMBER OF PROCESSORS TO VALUE LESS THAN 0\n";
	msg.num_proc = uLongGetProcessors( cluster );

	uRaise( cluster, &uCreateProcessorEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    while ( num > uProcessorStackLength( cpus ) ) {	/* increase the number of processors on this cluster */
	processor = uEmitProcessor( cluster );
	uAcquireProcessorStack( cpus );
	uPushProcessor( cpus, processor );
	uReleaseProcessorStack( cpus );
    } /* while */

    while ( num < uProcessorStackLength( cpus ) ) {	/* decrease the number of processors on this cluster */
	uAcquireProcessorStack( cpus );
	processor = uPopProcessor( cpus );
	uReleaseProcessorStack( cpus );
	uAbsorbProcessor( processor );
    } /* while */

#endif
} /* uLongSetProcessors */

uCluster uCreateCluster( int num, long time ) {

    /*
     * This function emits a new cluster.
     */

#ifdef __U_MULTI__
    
#ifndef __U_DEBUG__

    uCluster cluster;

#else

    volatile uCluster cluster = NULL;
    void              *msgp;

    uExcept {

#endif

	cluster = uAllocCluster();
	cluster->timeslice = time;			/* set the time slice duration for this new cluster */

	uLinkCluster( cluster );			/* link in the cluster */

	uLongSetProcessors( cluster, num );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to memory allocation problems\n";
	    msg.cv.TimeSlice = time;
	    msg.cv.ArgLen = U_ARG_LEN;
	    msg.cv.StackSize = U_STACK_SIZE;
	    msg.cv.Spin = U_SPIN;
	    msg.cv.Processors = num;

	    uRaise( NULL, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
	uWhen( cluster, &uCreateProcessorEx, &msgp ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to problems in uProcessor creation\n";
	    msg.cv.TimeSlice = time;
	    msg.cv.ArgLen = U_ARG_LEN;
	    msg.cv.StackSize = U_STACK_SIZE;
	    msg.cv.Spin = U_SPIN;
	    msg.cv.Processors = num;

	    uLongSetProcessors( cluster, 0 );
	    uUnlinkCluster( cluster );			/* unlink the cluster from the list */
	    uFree( cluster );

	    uRaise( NULL, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( cluster );					/* allocate and return a reference to the cluster descriptor */

#else

    return( uThisCluster() );				/* return a reference to the local cluster */

#endif
} /* uCreateCluster */

uCluster uLongCreateCluster( uClusterVars *cv ) {

    /*
     * This function emits a new cluster.
     */

#ifdef __U_MULTI__
    
#ifndef __U_DEBUG__

    uCluster cluster;

#else

    volatile uCluster cluster;
    void              *msgp;

    uExcept {

#endif

	cluster = uAllocCluster();
	cluster->timeslice = cv->TimeSlice;			/* set the time slice duration for this new cluster */
	cluster->spin = cv->Spin;				/* set the spin duration for this new cluster */
	cluster->stacksize = cv->StackSize;
	cluster->arglen = cv->ArgLen;

	/* link in the cluster */
	uLinkCluster( cluster );

	uLongSetProcessors( cluster, cv->Processors );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to memory allocation problems\n";
	    msg.cv = *cv;

	    uRaise( NULL, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
	uWhen( cluster, &uCreateProcessorEx, &msgp ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to problems in uProcessor creation\n";
	    msg.cv = *cv;

	    uLongSetProcessors( cluster, 0 );
	    uUnlinkCluster( cluster );			/* unlink the cluster from the list */
	    uFree( cluster );

	    uRaise( NULL, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( cluster );
    
#else

    return( uThisCluster() );				/* return a reference to the local cluster */

#endif
} /* uLongCreateCluster */

void uDestroyCluster( uCluster cluster ) {
    
    /*
     * This function destroys a cluster.
     */

#ifdef __U_MULTI__
    uLongSetProcessors( cluster, 0 );			/* remove all the processors */

    if ( uTaskQueueEmpty( &(cluster->ready) ) ) {	/* if no work on the ready queue */
	uUnlinkCluster( cluster );			/* unlink the cluster from the list */
	uFreeCluster( cluster );			/* free the cluster descriptor */
#ifdef __U_DEBUG__
    } else {
	uActiveTasksExMsg msg;

	msg = "uDestroyCluster: DESTROYING CLUSTER WITH ACTIVE TASKS\n";

	uRaise( cluster, &uActiveTasksEx, &msg, sizeof(msg) );
#endif
    } /* if */

#endif
} /* uDestroyCluster */

void uCxtInstall( struct uCxtBase *cxt ) {

    /*
     * Chain new user context save area onto coroutine/task
     */

    cxt->link = uThisCoroutine()->cxtlist;
    uThisCoroutine()->cxtlist = cxt;
} /* uCxtInstall */

static void uFirst( int argc, char *argv[], char *envp[] ) {

    void InstallStandardStream( void );
    void DestallStandardStream( void );
    
    int result = 0;
    uCluster cluster;

    uInstallStandardStream();				/* create the standard streams */

    cluster = uCreateCluster( 1, U_TIME_SLICE );	/* create the user cluster */

    uAbsorb( uLongEmit( cluster, uLongGetStackSize( cluster ), uStart, 0 ), NULL, 0 ); /* initialize the user cluster */

    /* execute the user application */
    uAbsorb( uLongEmit( cluster, uLongGetStackSize( cluster ), uMain, sizeof( argc ) + sizeof( argv ) + sizeof( envp ), argc, argv, envp ), &result, sizeof( result ) );

    uDestroyCluster( cluster );				/* destroy the user cluster */

    uDestallStandardStream();				/* destall the standard streams */

    uWorkProcessor->stop = U_TRUE;			/* schedule processor for termination */
    
    uDie( &result, sizeof( result ) );			/* return result from uMain */
} /* uFirst */

int uKernel( int argc, char **argv, char **envp ) {

    uProcessor uSystemProcessor;
    uTask uSystemTask;
    uTask uFirstTask;
    int result = 0;

    /*
     * Create and initialize hardware locks.
     */

    uCreateLocks();
    
    /*
     * Initialize the signal handlers for this UNIX process.
     * These signal handlers must be inherited by each UNIX process that
     * is subsequently forked.
     */

#ifdef __U_MULTI__
    uSignal( SIGCHLD, uSigChld );
    uSignal( SIGTERM, uSigTerm );
#endif
    uSignal( SIGALRM, uSigAlrm );
    
    /*
     * It is necessary to bootstrap the system cluster so that it works
     * in both uniprocessor and multiprocessor forms.
     */
    
    uSystemCluster = uAllocCluster();
#ifdef __U_MULTI__
    uLinkCluster( uSystemCluster );
#endif
    
    /*
     * It is necessary to bootstrap the system cluster's processor because it already exists.
     */

    uSystemProcessor = uAllocProcessor();		/* allocate the system processor, initialize it, */
    uSystemProcessor->cluster = uSystemCluster;		/* and add it to the system cluster */

    uSystemProcessor->pid = getpid();			/* retain the pid of this processor */

    uPushProcessor( &(uSystemCluster->cpus), uSystemProcessor ); /* push this processor on the system cluster */

    /*
     * It is necessary to bootstrap the system task on the system processor.
     */

    uSystemTask = uAllocTask( 0 );			/* allocate the system task and initialize it */
    uSystemTask->cluster = uSystemCluster;
    uSystemTask->mutex = U_LOCKED;			/* since this task is self starting acquire the task */
    uSystemTask->state = U_RUN;				/* and set its state to running */

    uWorkProcessor = uSystemProcessor;			/* set the local variables for the UNIX process */
    uWorkTask = uSystemTask;
    uWorkCoroutine = uWorkTask->cortn;
    uSchedulerTask = uWorkTask;
    uSetName( "uScheduler" );

#ifdef __U_MULTI__					/* for efficiency reasons, the alarm is turned off on */
    uSetTimeSlice( 0 );					/* the system cluster in the multiprocessor version, */
#endif							/* don't want all those signals being generated. */
    uSetAlarm( uGetTimeSlice() );			/* start the alarm */
    uFirstTask = uEmit( uFirst, argc, argv, envp );	/* emit the system task */
    uScheduler();					/* call the scheduler to execute other tasks */
    uAbsorb( uFirstTask, &result, sizeof( result ) );	/* absorb the system task */
    uSetAlarm( 0 );					/* stop the alarm */

#ifdef __U_MULTI__
    uAcquireClusterList();				/* get control of the cluster list */
    uShutdown( U_FALSE, U_TRUE );			/* shut down but don't kill me */
    uTerminate = U_TRUE;				/* indicate shutdown */
#endif							/* don't want all those signals being generated. */

    uFreeTask( uSystemTask );				/* free remaining system resources */
    uFreeProcessor( uSystemProcessor );
    uFreeCluster( uSystemCluster );

    uDestroyLocks();					/* destroy hardware locks */

    return( result );					/* return the value from the user defined main */
} /* uKernel */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
