/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.4.3, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uKernel.h -- Declaration of objects that the uSystem makes available to the user.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:23:30 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Sun Mar 28 14:21:17 1993
 * Update Count     : 249
 */

#ifndef __U_KERNELH__
#define __U_KERNELH__

/*
 * USER CODE MAY BE WRITTEN TO DEPEND ON THE FOLLOWING MACROS AND TYPE DEFINITIONS
 * BUT NOT ON THEIR INTERNAL STRUCTURE:
 *
 *   NULL, uSemaphore, U_SEMAPHORE, uTask, uCoroutine, uCluster,
 *   uClusterVars, U_CLUSTER_VARS
 *
 * NO USER CODE SHOULD BE WRITTEN TO DEPEND ON ANY OTHER MACROS OR TYPE DEFINITIONS.
 */

/*
 * Macro definitions.
 */

#ifndef NULL
#define NULL 0
#endif
#define U_NULL 0

#define U_LOCKED ( (uLock) 1 )
#define U_UNLOCKED ( (uLock) 0 )

#define U_FALSE 0
#define U_TRUE 1

/*
 * Definition of a lock.
 */

typedef int uLock;

/*
 * Definition of a queue of tasks, and its associated constructor.
 */

/* WARNING: monitor preprocessor depends on uTaskQueue */

typedef struct uTaskQueue {
    uLock mutex;					/* lock to ensure mutual exclusion */
    int len;						/* number of tasks on queue */
    int idle;						/* flag for idle processors */
    struct uTaskD *head;				/* pointer to first task in queue */
    struct uTaskD *tail;				/* pointer to last task in queue */
} uTaskQueue;

#ifdef __GNUC__
#define U_TASK_QUEUE() ( (uTaskQueue) { U_UNLOCKED, 0, U_FALSE, NULL } )
#else
#define U_TASK_QUEUE() { U_UNLOCKED, 0, U_FALSE, NULL }
#endif
    
/*
 * Definition of a semaphore, and its associated constructor.
 */

typedef uTaskQueue uSemaphore;    

#ifdef __GNUC__
#define U_SEMAPHORE( V ) ( (uTaskQueue) { U_UNLOCKED, -(V), U_FALSE, NULL } )
#else
#define U_SEMAPHORE( V ) { U_UNLOCKED, -(V), U_FALSE, NULL }
#endif
    
/*
 * Definition of cluster variable control block, and its associated constructor.
 */

typedef struct uClusterVars {
    long Processors;
    long TimeSlice;
    long Spin;
    long StackSize;
    long ArgLen;
} uClusterVars;

#ifdef __GNUC__
#define U_CLUSTER_VARS() ( (uClusterVars) { 1, U_TIME_SLICE, U_SPIN, U_STACK_SIZE, U_ARG_LEN } )
#else
#define U_CLUSTER_VARS() { 1, U_TIME_SLICE, U_SPIN, U_STACK_SIZE, U_ARG_LEN }
#endif
    
typedef struct uTaskD *uTask, *uCoroutine;
typedef struct uProcessorD *uProcessor;
typedef struct uClusterD *uCluster;

/*
 * These enumeration constants are used to differentiate between
 * an exception block, and an regular intervention block, since both
 * occur on the same per-coroutine stack.
 * The third value marks a block which only contains a reference
 * to memory to be freed when the block is traversed (by a raised
 * exception only).
 */
typedef enum { uIsExceptBlock, uIsInterBlock, uIsDataBlock } uExceptInterEnum;

/*
 * Definition of an exception. Exceptions can _only_ be initialized.  They
 * cannot be changed after they are initialized.
 */
typedef const struct _uException {
    const struct _uException *parent;			/* constant pointer to a constant uException */
} uException;

/*
 * Definition of the type for an intervention.  It points to a
 * string containing the name as supplied by the creator of the
 * intervention.
 */
typedef const char *uInterName;

/*
 * This defines the elements of the Exception and Intervention stack for a coroutine.
 */

typedef struct _uStackBlockD {
    uExceptInterEnum     type;
    struct _uStackBlockD *prev;
    union {
	struct {
	    uException     *exception;
	    void           *object;
	    void           *data;
	    int            len;
	    void           *return_addr;
	    int            reraise;
	    jmp_buf        context;
	} uEPart;
	struct {
	    uInterName *name;
	    void       *fp;
	} uIPart;
	/*
	 * If the following field is non-NULL, it stores memory
	 * which must be freed when an exception is raised and
	 * this block is traversed.
	 * (if the type is uIsDataBlock)
	 */
	void       *data;
    } u;
} uStackBlockD, *uStackBlock;

/*
 * structures for user specified contexts
 */

struct uCxtBase {					/* base type for all context areas */
    int size;
    void (*saver)( void * );
    void (*restorer)( void * );
    struct uCxtBase *link;
    char cxtarea[1];
};

#define uCxtDeclare( Name, Size, Saver, Restorer )	\
struct uCxtD##Name {	\
    int size;	\
    void (*saver)( void * );	\
    void (*restorer)( void * );	\
    struct uCxt##Name *link;	\
    char cxtarea[Size];	\
} uCxt_##Name = { Size, (void (*)(void *))Saver, (void (*)(void *))Restorer }

#define uCxt( Name ) (struct uCxtBase *)(&(uCxt_##Name))

#ifdef __U_KERNEL__

#include <sys/types.h>					/* Some cluster variables depend on some definitions */

/*
 * Definition of a stack of processors, and its associated constructor.
 */

typedef struct uProcessorStack {
    uLock mutex;					/* lock to ensure mutual exclusion */
    int len;						/* number of processors on stack */
    struct uProcessorD *top;				/* pointer to processor on top of stack */
} uProcessorStack;

#ifdef __GNUC__
#define U_PROCESSOR_STACK() ( (uProcessorStack) { U_UNLOCKED, 0, NULL } )
#else
#define U_PROCESSOR_STACK() { U_UNLOCKED, 0, NULL }
#endif
    
/*
 * Definition of a stack descriptor, and its associated constructor.
 */

typedef struct uStackD {
    void *base;						/* pointer to base of stack */
    void *limit;					/* pointer to limit of stack */
    void *sp;						/* current stack pointer */
    void *fp;						/* current frame pointer */
    void *pc;						/* return address */
} *uStack;

/*
 * Definition of an ipc descriptor, and its associated constructor.
 */

typedef struct uMsgD {
    uSemaphore snd;					/* semaphore used by tasks sending messages */
    uSemaphore rcv;					/* semaphore used by task receiving message */
    struct uTaskD *to;					/* pointer to task receiving message */
    struct uTaskD *from;				/* pointer to task sending message */
    void *sbuf;						/* address of send or reply message */
    int slen;						/* length of send or reply message */
    void *rbuf;						/* address of receive message */
    int rlen;						/* length of receive message */
} *uMsg;

#ifdef __GNUC__
#define U_MSG() ( (struct uMsgD) { U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ), NULL } )
#else
#define U_MSG() { U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ), NULL }
#endif __GNUC__
    
/*
 * Definition of a pending intervention descriptor, and its associated constructor.
 */

typedef struct uQdInterD {
    uInterName *pinter;					/* Pending async intervention "name" */
    void *interdata;					/* Pending async intervention data */
    int interlen;					/* Pending async intervention data length */
    struct uQdInterD *next;				/* next in queue (bag?) */
} *uQdInter;

#ifdef __GNUC__
#define U_QDINTER() ( (struct uQdInterD ) { NULL, NULL, 0, NULL } )
#else
#define U_QDINTER() { NULL, NULL, 0, NULL }
#endif __GNUC__

/*
 * Definition of a task descriptor, and its associated constructor.
 */

struct uTaskD {
    uLock mutex;					/* provide mutual exclusion on task execution */
    int errno;						/* errno for this task */
    int state;						/* current state of task or coroutine */
    int save;						/* are the floating point registers saved */
    struct uCxtBase *cxtlist;				/* list of user defined contexts */
    uSemaphore done;					/* done semaphore */
    uSemaphore *peedsem;				/* semaphore being peed on */
    struct uMsgD ipc;					/* IPC information */
    char *name;						/* name of the task */
    uStackBlock eistack;				/* Exception and Intervention stack for the coroutine */

    uLock interlock;					/* lock protecting async intervention data */
    uQdInter inters;					/* queue (bag?) of pending interventions */
    int icheckneeded;					/* if a check should be made by the delivery routine */

    int rlen;						/* length of resume message buffer */
    void *rbuf;						/* address of resume message buffer */
    void *start;					/* address for freeing task storage */
    struct uStackD StackData;				/* stack information */
    struct uClusterD *cluster;				/* cluster task is executing on */
    struct uTaskD *cortn;				/* current coroutine executing */
    void (*begin)();					/* pointer to subroutine started as a coroutine */
    struct uTaskD *owner;				/* task that created this coroutine */
    struct uTaskD *restarter;				/* last coroutine to restart coroutine, used to deallocate resources */
    struct uTaskD *resumer;				/* last cocalling or resuming coroutine */
    struct uTaskD *link;				/* link for ready queue */
    char hex[11];					/* area big enough to hold 8 digit hex value, leading 0x and trailing \0 */
};

#ifdef __GNUC__
#define U_TASK() ( (struct uTaskD) { U_UNLOCKED, 0, U_NEW, U_SAVE_FIXED, NULL, U_SEMAPHORE( 0 ), NULL, U_MSG(), \
					 NULL, NULL, U_UNLOCKED, NULL, 0, 0, NULL } )
#else
#define U_TASK() { U_UNLOCKED, 0, U_NEW, U_SAVE_FIXED, NULL, U_SEMAPHORE( 0 ), NULL, U_MSG(), \
		       NULL, NULL, U_UNLOCKED, NULL, 0, 0, NULL }
#endif __GNUC__

#define U_SAVE_FIXED 0
#define U_SAVE_FLOAT 1

#define U_NEW		0x1
#define U_RUN		0x2
#define U_OLD		0x4
#define U_PEED		0x8
#define U_DISABLE_INTER	0x10

/*
 * Definition of a processor descriptor, and its associated constructor.
 */

struct uProcessorD {
    int stop;						/* stop flag */
    int state;						/* state */
    uSemaphore done;					/* done semaphore */
    int pid;						/* process id */
    struct uClusterD *cluster;				/* pointer to cluster on which this processor executes */
    struct uProcessorD *link;				/* link used to chain processors together on processor stack */
};

#ifdef __GNUC__
#define U_PROCESSOR() ( (struct uProcessorD) { U_FALSE, U_BUSY, U_SEMAPHORE( 0 ) } )
#else
#define U_PROCESSOR() { U_FALSE, U_BUSY, U_SEMAPHORE( 0 ) }
#endif __GNUC__
    
#define U_BUSY 0
#define U_IDLE 1

/*
 * Definition of a cluster descriptor, and its associated constructor.
 */

struct uClusterD {
    uCluster link;					/* link used to chain clusters together on cluster list */
							/* head of list is in the System cluster */
    long arglen;					/* default argument size */
    long stacksize;					/* default stack size */
    long timeslice;					/* default processor time slice */
    long spin;						/* default processor spin before sleep time */
    struct uTaskQueue ready;				/* ready queue */
    struct uProcessorStack cpus;			/* list of processors for this cluster */
    uSemaphore mutex;					/* mutual exclusion on cluster changes */
    struct uTaskD *iopoller;				/* who is the task polling for io */
    uSemaphore ioqueue[FD_SETSIZE];			/* list of blocked io tasks */
    unsigned int iocount[FD_SETSIZE];			/* number of blocked io tasks */
    fd_set rfds;					/* reading file descriptor mask */
    fd_set wfds;					/* write file descriptor mask */
    fd_set efds;					/* exceptional condition pending mask */
};

#ifdef __GNUC__
#define U_CLUSTER() ( (struct uClusterD) { NULL, U_ARG_LEN, U_STACK_SIZE, U_TIME_SLICE, U_SPIN, U_TASK_QUEUE(), U_PROCESSOR_STACK(), U_SEMAPHORE( 1 ), 0 } )
#else
#define U_CLUSTER() { NULL, U_ARG_LEN, U_STACK_SIZE, U_TIME_SLICE, U_SPIN, U_TASK_QUEUE(), U_PROCESSOR_STACK(), U_SEMAPHORE( 1 ), 0 }
#endif __GNUC__

#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>

#define U_MAGIC 0x12345678

#ifdef sgi

/* declare private variables to be a part of the private data section of each process */

#define uSchedulerTask (* (uTask *) ((int)PRDA))
#define uWorkProcessor (* (uProcessor *) ((int)PRDA + sizeof( uTask )))
#define uWorkTask (* (uTask *) ((int)PRDA + sizeof( uTask ) + sizeof( uProcessor )))
#define uWorkCoroutine (* (uCoroutine *) ((int)PRDA + sizeof( uTask ) + sizeof( uProcessor ) + sizeof( uTask )))

#else

extern uTask uSchedulerTask;				/* reference to the uSystem task */
extern uProcessor uWorkProcessor;			/* reference to the currently active processor */
extern uTask uWorkTask;					/* reference to the currently active task */
extern uCoroutine uWorkCoroutine;			/* reference to the currently active coroutine */

#endif
    
#endif __U_KERNEL__

/*
 * These macros return the ceiling and floor of values V of multiples of value A.
 * In other words, if A is 8, and V is 6, U_CEILING returns 8, while U_FLOOR returns 0.
 * If V is already a multiple of A, both macros return the same value.
 */
    
#define U_CEILING( V, A ) ( ( (V) + (A) - 1 ) & ~ ( (A) - 1 ) )
#define U_FLOOR( V, A ) ( (V) & ~ ( (A) - 1 ) )

/*
 * These are the functions that the uKernel provides.
 */

extern volatile void uExit( int );
extern volatile void uAbort( char *, ... );
/* Backwards compatibility */
#define uError uAbort

extern void uP( uSemaphore * );
extern void uV( uSemaphore * );
extern int uC( uSemaphore * );

extern uCluster uCreateCluster( int, long );
extern uCluster uLongCreateCluster( uClusterVars * );
extern void uDestroyCluster( uCluster );
extern uCluster uThisCluster( void );

/*
 * The prototypes for these routines are getting more and more difficult to
 * specify because of changes in ANSI C.  The problem is that void (*)() no
 * longer matches any function type because of the promotion rules. Why isn't
 * void(*)(...) allowed?  I wish ANSI would get this right.
 */

#if defined( __cplusplus ) || defined( c_plusplus )
    /*
     * Now C++ will not match a function pointer with void *
     */
    extern uTask uEmit( ... );			/* C++ 1.2.1 bug prevents the 1st parameter from being void (*)() */
    extern uTask uLongEmit( uCluster, long, void (*)(), long, ... );
    extern uCoroutine uCocall( void *, int, void (*)(), ... );
    extern uCoroutine uLongCocall( void *, int, long, void (*)(), long, ... );
#else
    extern uTask uEmit( void *, ... );
    extern uTask uLongEmit( uCluster, long, void *, long, ... );
    extern uCoroutine uCocall( void *, int, void *, ... );
    extern uCoroutine uLongCocall( void *, int, long, void *, long, ... );
    extern void uMain();
#endif

extern void uAbsorb( uTask, void *, int );
extern volatile void uDie( void *, int );
extern uTask uThisTask( void );

extern void uSuspend( void *, int, void *, int );
extern void uResume( uCoroutine, void *, int, void *, int );
extern void uSuspendDie( void *, int );
extern void uResumeDie( uCoroutine, void *, int );
extern uCoroutine uThisCoroutine( void );

extern uTask uSend( uTask, void *, int, void *, int );
extern uTask uReceive( void *, int );
extern void uReply( uTask, void *, int );
extern void uForward( uTask, void *, int, uTask );

extern uCluster uMigrate( uCluster );

extern void uVerify( void );

extern void uYield( void );

extern void uLongSetTimeSlice( uCluster, long );
extern long uLongGetTimeSlice( uCluster );
extern void uSetTimeSlice( long );
extern long uGetTimeSlice( void );

extern void uLongSetSpin( uCluster, long );
extern long uLongGetSpin( uCluster );
extern void uSetSpin( long );
extern long uGetSpin( void );

extern void uLongSetStackSize( uCluster, long );
extern long uLongGetStackSize( uCluster );
extern void uSetStackSize( long );
extern long uGetStackSize( void );

extern void uLongSetArgLen( uCluster, long );
extern long uLongGetArgLen( uCluster );
extern void uSetArgLen( long );
extern long uGetArgLen( void );

extern uProcessor uThisProcessor( void );
extern void uLongSetProcessors( uCluster, int );
extern int uLongGetProcessors( uCluster );
extern void uSetProcessors( int );
extern int uGetProcessors( void );

extern void uSetName( char * );
extern char *uGetName( uCoroutine );

extern int uLongReadyTasks( uCluster );
extern int uReadyTasks( void );

extern void uSaveFixed( void );
extern void uSaveFloat( void );

extern int uKernel( int, char *[], char *[] );

#endif
