/* Copyright (c) 1989  P. A. Buhr */

/*
  Second Reader/Writer Problem (P.J. Courtois, F. Heymans, D.L. Parnas)
  
  Method: Allow multiple readers but only one writer for a particular
     resource. Writers have priority over readers.
  Problem: As long as writers keep coming along, no reader can get in,
     hence readers could starve.
  */

#include <uSystem.h>
#include <uDelay.h>

long random( void );

/* shared variables for process communication */

uSemaphore LockCsRead = U_SEMAPHORE( 1 ), LockCsWrite = U_SEMAPHORE( 1 ), PriorityLock = U_SEMAPHORE( 1 );
int rdcnt = 0, wrtcnt = 0;
uSemaphore rd = U_SEMAPHORE( 1 ), wrt = U_SEMAPHORE( 1 );

void Reader( void ) {
    /*
      This lock is necessary to ensure priority for writers. Without this
      extra lock, there is the possibility that a writer and one or more
      readers will be simultaneously waiting for a uV(rd) to be done by a
      reader. In that event, it is not possible to guarantee priority to 
      the writer. Hence, all readers wait here and enter the write check
      code one at a time.
      */
    uP( &PriorityLock );
    /*
      A reader waits here until the previous reader has checked
      if there are any writers, and if not, proceeds on to the read-
      code. This causes a serial check of the write semaphore by
      each reader, so that should a writer come along, it can P on
      the read semaphore and stop any new readers from entering the
      check on the write semaphore. Thus, when all existing readers
      finish in the read-code, the new reader will be waiting on
      the read semaphore and the writer will be able to enter the
      write-code. In this way, the writers have priority to make
      readers wait until after the writer has finished.
      */
    uP( &rd );
    uP( &LockCsRead );
    rdcnt += 1;
    if ( rdcnt == 1 ) {					/* 1st reader ? */
	uP( &wrt );					/* wait for writer to finish */
    } /* if */
    uV( &LockCsRead );
    uV( &rd ); 
    uV( &PriorityLock );

    uPrintf( "Reader:%x, simultaneous readers:%d\n", uThisTask( ), rdcnt );
    uDelay( 2 );					/* pretend to read */

    uP( &LockCsRead );
    rdcnt -= 1;
    if ( rdcnt == 0 ) {					/* all readers finished ? */
	uV( &wrt );					/* signal writers */
    } /* if */
    uV( &LockCsRead );

    uDie( NULL, 0 );
} /* Reader */

void Writer( void ) {
    uP( &LockCsWrite );
    wrtcnt += 1;
    if ( wrtcnt == 1 ) {				/* 1st writer ? */
	uP( &rd );					/* wait for readers to finish and */
							/* prevent any new readers from starting */
    } /* if */	
    uV( &LockCsWrite );
    
    /*
      serialize writers, each writer waits here until there is no writer in 
      the write-code.
      */
    
    uP( &wrt );

    uPrintf( "Writer:%x\n", uThisTask( ) );
    uDelay( 5 );					/* pretend to write */

    uV( &wrt );
    
    uP( &LockCsWrite );
    wrtcnt -= 1;
    if ( wrtcnt == 0 ) {				/* writers finished */
	uV( &rd );					/* allow readers to gain control */
    } /* if */
    uV( &LockCsWrite );

    uDie( NULL, 0 );
} /* endwrite */

#define NoOfRW 20

void uMain( ) {
    uTask rw[NoOfRW];
    int i;
    
    for ( i = 0; i < NoOfRW; i += 1 ) {
	if ( random() % 10 < 7 ) {			/* 70 % chance of creating a reader */
	    rw[i] = uEmit( Reader );			/* start reader */
	} else {
	    rw[i] = uEmit( Writer );			/* start writer */
	} /* if */
	uDelay( 5 );
    } /* for */
    
    for ( i = 0; i < NoOfRW; i += 1 ) {
	uAbsorb( rw[i], NULL, 0 );			/* wait for completion of readers */
    } /* for */

    uPrintf( "successful execution\n" );
} /* uMain */
