/*
 * Producer-consumer problem using controllers to prevent having to buffer
 * producer or consumer requests in the buffer manager.
 */

#include <uSystem.h>
#include <uDelay.h>

#define BufferElemType int

struct message {
    char msgtype;
    BufferElemType elem;
};

void ProdCntl( uTask bufman ) {
    uTask prod;
    struct message msg;

    msg.msgtype = 'P';
    
    for ( ;; ) {					/* main receive loop */
	prod = uReceive( &(msg.elem), sizeof(msg.elem) );
	uSend( bufman, NULL, 0, &msg, sizeof(msg) );
	uReply( prod, NULL, 0 );
    } /* for */
} /* ProdCntl */

void ConsCntl( uTask bufman ) {
    uTask cons;
    struct message msg;
    BufferElemType elem;

    msg.msgtype = 'C';
    
    for ( ;; ) {					/* main receive loop */
	cons = uReceive( NULL, 0 );
	uSend( bufman, &elem, sizeof(elem), &msg, sizeof(msg) );
	uReply( cons, &elem, sizeof(elem) );
    } /* for */
} /* ConsCntl */

void BufManager( ) {
#   define QueueSize 3
    
    struct queue {
	int front;					/* position of front of queue */
	int back;					/* position of back of queue */
	BufferElemType queue[QueueSize + 1];		/* queue of integers */
    }; /* queue */

    struct queue buffer = { 0, 1 };			/* bounded buffer, initialize front & back */

    uTask worker;
    struct message msg;

    uTask WaitingProd = NULL, WaitingCons = NULL;

    for ( ;; ) {					/* main receive loop */
	worker = uReceive( &msg, sizeof(msg) );
	if ( msg.msgtype == 'P' ) {			/* producer message ? */
	    buffer.queue[buffer.back] = msg.elem;	/* insert element in queue */
	    buffer.back = (buffer.back + 1) % (QueueSize + 1);
	    if ( WaitingCons != NULL ) {		/* consumer waiting ? */
		buffer.front = (buffer.front + 1) % (QueueSize + 1);
		msg.elem = buffer.queue[buffer.front]; /* remove element from queue */
		uReply( WaitingCons, &(msg.elem), sizeof(msg.elem) ); /* restart consumer */
		WaitingCons = NULL;
	    } /* if */
	    if (buffer.front == buffer.back) {		/* buffer full ? */
		WaitingProd = worker;			/* no reply and remember producer */
	    } else {
		uReply( worker, NULL, 0 );		/* restart producer */
	    } /* if */
	} else {					/* consumer message */
	    if ((buffer.front + 1) % (QueueSize + 1) == buffer.back) { /* buffer empty ? */
		WaitingCons = worker;
	    } else {
		buffer.front = (buffer.front + 1) % (QueueSize + 1);
		msg.elem = buffer.queue[buffer.front];	/* remove element from queue */
		uReply( worker, &(msg.elem), sizeof(msg.elem) ); /* restart consumer */
		if ( WaitingProd != NULL ) {
		    uReply( WaitingProd, NULL, 0 );	/* restart producer */
		    WaitingProd = NULL;
		} /* if */
	    } /* if */
	}
    } /* for */
} /* BufManager */

extern int random( void );

void Producer( uTask prodcntl ) {
    BufferElemType elem;

    elem = random() % 100 + 1;
    uSend( prodcntl, NULL, 0, &elem, sizeof(elem) );
    uPrintf( " Producer:%x elem:%d\n", uThisTask(), elem );

    uDie( NULL, 0 );
} /* Producer */

void Consumer( uTask conscntl ) {
    BufferElemType elem;

    uSend( conscntl, &elem, sizeof(elem), NULL, 0 );
    uPrintf( "Consumer :%x elem:%d\n", uThisTask(), elem );

    uDie( NULL, 0 );
} /* Pawn */

#define NoOfPCs 40

void uMain( ) {
    int i, Prods, Conss;
    uTask bufman, prod, cons;
    uTask PidConss[NoOfPCs / 2], PidProds[NoOfPCs / 2];

    bufman = uEmit( BufManager );			/* create buffer manager */
    prod = uEmit( ProdCntl, bufman );			/* create the producer */
    cons = uEmit( ConsCntl, bufman );			/* create the consumer */

    for ( Prods = 0, Conss = 0; Prods + Conss < NoOfPCs; ) {
	if ( random() % 10 <= 3 ) {			/* create more producers at the start */
	    if ( Conss < NoOfPCs / 2 ) {
		PidConss[Conss] = uEmit( Consumer, cons );
		Conss += 1;
	    } /* if */
	} else {
	    if ( Prods < NoOfPCs / 2 ) {
		PidProds[Prods] = uEmit( Producer, prod );
		Prods += 1;
	    } /* if */
	} /* if */
	uDelay( 5 );
    } /* for */
    
    for ( i = 0; i < NoOfPCs / 2; i += 1 ) {		/* absorb the consumers */
	uAbsorb( PidConss[i], NULL, 0 );
    } /* for */
    
    for ( i = 0; i < NoOfPCs / 2; i += 1 ) {		/* absorb the producers */
	uAbsorb( PidProds[i], NULL, 0 );
    } /* for */

    uPrintf( "successful execution\n" );
} /* uMain */
