//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Philipp E. Lim 1996
// 
// uRealTime.h -- 
// 
// Author           : Philipp E. Lim
// Created On       : Fri Jul 19 16:34:59 1996
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Oct  4 17:30:28 2006
// Update Count     : 114
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#ifndef __U_REALTIME_H__
#define __U_REALTIME_H__

#pragma __U_NOT_USER_CODE__


//#include <uDebug.h>
#include <cstring>					// access: ffs


// provide default definition for real-time monitor
#define uRealTimeMonitor _Mutex<uPrioritySeq,uPrioritySeq> class

// Faked for now, since Events aren't implemented yet.
class uEvent {
  public:
    uEvent() {};
};


//######################### Priority Scheduling #########################


#define __U_MAX_NUMBER_PRIORITIES__ 32


template<class List, class Node> class uPriorityScheduleQueue : public uBaseSchedule<Node> {
  protected:
    List objects[__U_MAX_NUMBER_PRIORITIES__];
    unsigned int mask;					// allow access to all queue flags
    unsigned int verCount;
  public:
    uPriorityScheduleQueue() {
	verCount = mask = 0;
    } // uPriorityScheduleQueue::uPriorityScheduleQueue

    virtual bool empty() const {
	return mask == 0;
    } // uPriorityScheduleQueue::empty

    virtual Node *head() const {
	int highestPriority = ffs( mask ) - 1;

	if ( highestPriority >= 0 ) {
	    Node *node = objects[highestPriority].head();
	    return node;
	} else {
	    return NULL;
	} // if
    } // uPriorityScheduleQueue::head

    virtual void add( Node *node ) {
	int priority = getActivePriority( node->task() );
#ifdef __U_DEBUG__
	assert( 0 <= priority && priority <= __U_MAX_NUMBER_PRIORITIES__ - 1 );
#endif // __U_DEBUG__
	objects[priority].add( node );
	mask |= 1ul << priority;
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uPriorityScheduleQueue &)0x%p.add( 0x%p ) task %.256s (0x%p) adding   task %.256s (0x%p) with priority %d on cluster 0x%p\n",
		   this, node, uThisTask().getName(), &uThisTask(), node->task().getName(), &node->task(), priority, &uThisCluster() );
#endif // __U_DEBUG_H__
    } // uPriorityScheduleQueue::add

    virtual Node *drop() {
	int highestPriority = ffs( mask ) - 1;

	if ( highestPriority >= 0 ) {
	    Node *node = objects[highestPriority].drop();
	    if ( objects[highestPriority].empty() ) {
		mask &= ~ ( 1ul << highestPriority );
	    } // if
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uPriorityScheduleQueue &)0x%p.drop( 0x%p ) task %.256s (0x%p) removing task %.256s (0x%p) with priority %d on cluster 0x%p\n",
		       this, node, uThisTask().getName(), &uThisTask(), node->task().getName(), &node->task(), highestPriority, &uThisCluster() );
#endif // __U_DEBUG_H__
	    return node;
	} else {
	    return NULL;
	} // if
    } // uPriorityScheduleQueue::drop

    virtual bool checkPriority( Node &, Node & ) {
	return false;
    } // uPriorityScheduleQueue::checkPriority

    virtual void resetPriority( Node &, Node & ) {
    } // uPriorityScheduleQueue::resetPriority

    virtual void addInitialize( uBaseTaskSeq & ) {
    } // uPriorityScheduleQueue::addInitialize

    virtual void removeInitialize( uBaseTaskSeq & ) {
    } // uPriorityScheduleQueue::removeInitialize

    virtual void rescheduleTask( uBaseTaskDL *, uBaseTaskSeq & ) {
    } // uPriorityScheduleQueue::rescheduleTask
}; // uPriorityScheduleQueue


class uPrioritySeq : public uBasePrioritySeq {
  protected:
    uBaseTaskSeq objects[__U_MAX_NUMBER_PRIORITIES__];
    unsigned int mask;					// allow access to all queue flags
    uBaseTask *oldInheritTask;
    int afterEntry( uBaseTask *owner );
  public:
    uPrioritySeq();
    virtual bool empty() const;
    virtual uBaseTaskDL *head() const;
    virtual int add( uBaseTaskDL *node, uBaseTask *owner );
    virtual uBaseTaskDL *drop();
    virtual void remove( uBaseTaskDL *node );
    virtual void onAcquire( uBaseTask &oldOwner );
    virtual void onRelease( uBaseTask &owner );
}; // uPrioritySeq


template<class List, class Node> class uPriorityScheduleSeq : public uPriorityScheduleQueue<List, Node> {
    using uPriorityScheduleQueue<List, Node>::objects;
    using uPriorityScheduleQueue<List, Node>::mask;
  public:
    virtual bool checkPriority( Node &owner, Node &calling ) {
	return getActivePriority( owner.task() ) > getActivePriority( calling.task() );
    } // uPriorityScheduleSeq::checkPriority

    virtual void resetPriority( Node &owner, Node &calling ) {
	if ( owner.listed() ) {
	    remove( &owner );
	    setActivePriority( owner.task(), calling.task() );
	    add( &owner );
	} else {
	    setActivePriority( owner.task(), calling.task() );
	} // if
    } // uPriorityScheduleSeq::resetPriority

    virtual void remove( Node *node ) {
	int priority = getActivePriority( node->task() );
	objects[priority].remove( node );
	if ( objects[priority].empty() ) {
	    mask &= ~ ( 1ul << priority );
	} // if
    } // uPriorityScheduleSeq::remove
}; // uPriorityScheduleSeq


//######################### uRealTimeBaseTask #########################


class uRealTimeBaseTask : public uBaseTask {
    class VerCount : public uSeqable {
      public:
	int version;
	uCluster *cluster;
    }; // VerCount

    uDuration deadline;
    uSequence<VerCount> verCountSeq;			// list of scheduler version counts with associated cluster 
  protected: 
    uTime firstActivateTime;
    uEvent firstActivateEvent;
    uTime endTime;
  public:
    uRealTimeBaseTask( uCluster &cluster = uThisCluster() );
    uRealTimeBaseTask( uTime firstActivateTask_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uRealTimeBaseTask( uEvent firstActivateEvent_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uRealTimeBaseTask( uTime firstActivateTask_, uEvent firstActivateEvent_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    ~uRealTimeBaseTask();
    virtual uDuration getDeadline() const;
    virtual uDuration setDeadline( uDuration deadline_ );

    // These members should be private but cannot be because they are
    // referenced from user code.

    virtual int getVersion( uCluster &cluster );
    virtual int setVersion( uCluster &cluster, int version );
}; // uRealTimeBaseTask


//######################### uPeriodicBaseTask #########################


class uPeriodicBaseTask : public uRealTimeBaseTask {
  protected:
    uDuration period;
  public:
    uPeriodicBaseTask( uDuration period_, uCluster &cluster = uThisCluster() );
    uPeriodicBaseTask( uDuration period_, uTime firstActivateTask_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uPeriodicBaseTask( uDuration period_, uEvent firstActivateEvent_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uPeriodicBaseTask( uDuration period_, uTime firstActivateTask_, uEvent firstActivateEvent_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uDuration getPeriod() const;
    uDuration setPeriod( uDuration period_ );
}; // uPeriodicBaseTask


//######################### uSporadicBaseTask #########################


class uSporadicBaseTask : public uRealTimeBaseTask {
  protected:
    uDuration frame;
  public:
    uSporadicBaseTask( uDuration frame_, uCluster &cluster = uThisCluster() );
    uSporadicBaseTask( uDuration frame_, uTime firstActivateTask_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uSporadicBaseTask( uDuration frame_, uEvent firstActivateEvent_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uSporadicBaseTask( uDuration frame_, uTime firstActivateTask_, uEvent firstActivateEvent_, uTime endTime_, uDuration deadline_, uCluster &cluster = uThisCluster() );
    uDuration getFrame() const;
    uDuration setFrame( uDuration frame_ );
}; // uSporadicBaseTask


//######################### uRealTimeCluster #########################


class uRealTimeCluster : public uCluster {
  public:
    uRealTimeCluster( uBaseSchedule<uBaseTaskDL> &rq, int size = uDefaultStackSize(), const char *name = "uRealTimeCluster" );
    uRealTimeCluster( uBaseSchedule<uBaseTaskDL> &rq, const char *name );
    ~uRealTimeCluster();
}; // uRealTimeCluster


#pragma __U_USER_CODE__

#endif // __U_REALTIME_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
