//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 1994
// 
// uSocket.h -- Nonblocking UNIX Socket I/O
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 17:04:36 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Sep  6 11:45:30 2007
// Update Count     : 329
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#ifndef __U_SOCKET_H__
#define __U_SOCKET_H__


#include <uFile.h>
//#include <uDebug.h>


#pragma __U_NOT_USER_CODE__


#include <cstdio>
#include <netdb.h>					// MAXHOSTNAMELEN on Solaris
#include <sys/param.h>					// MAXHOSTNAMELEN on BSD
#include <sys/socket.h>
#include <sys/un.h>

#if defined( __freebsd__ )
#include <netinet/in.h>					// struct sockaddr_in
#endif // __freebsd__


//######################### uSocket #########################


class uSocket {
    friend class uSocketServer;				// access: access, uSocket, ~uSocket
    friend class uSocketAccept;				// access: access
    friend class uSocketClient;				// access: access, uSocket, ~uSocket

    uIOaccess access;

    const int domain, type, protocol;			// copies for information purposes

    uSocket( uSocket & );				// no copy
    uSocket &operator=( uSocket & );			// no assignment

    void createSocket( int domain, int type, int protocol );

    uSocket( int domain, int type, int protocol ) : domain( domain ), type( type ), protocol( protocol ) {
	createSocket( domain, type, protocol );
    } // uSocket::uSocket

    virtual ~uSocket();
  public:
    _Event Failure : public uIOFailure {
	const uSocket &socket_;
      protected:
	Failure( const uSocket &s, const char *const msg );
      public:
	const uSocket &socket() const;
	virtual void defaultTerminate() const;
    }; // uSocket::Failure
    friend _Event Failure;				// access: access

    _Event OpenFailure : public uSocket::Failure {
      protected:
	const int domain;
	const int type;
	const int protocol;
      public:
	OpenFailure( const uSocket &socket, const int domain, const int type, const int protocol, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocket::OpenFailure

    _Event CloseFailure : public uSocket::Failure {
      public:
	CloseFailure( const uSocket &socket, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocket::CloseFailure
    friend _Event CloseFailure;				// access: access
}; // uSocket


//######################### uSocketIO #########################


class uSocketIO : public uFileIO {
  protected:
    struct inetAddr : sockaddr_in {
	int retcode;
	char name_[MAXHOSTNAMELEN+1];
	void getinetAddr( unsigned short port, const char *name );

	inetAddr( unsigned short port, const char *name = "" ) {
	    strcpy( name_, name );			// copy name
	    getinetAddr( port, name );
	} // uSocketIO::inetAddr::inetAddr
    }; // inetAddr

    struct sockaddr *saddr;				// default send/receive address
    socklen_t saddrlen;					// size of send address
    socklen_t baddrlen;					// size of address buffer (UNIX/INET)

    virtual void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) = 0;
    virtual void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) = 0;

    uSocketIO( uIOaccess &acc, struct sockaddr *saddr ) : uFileIO( acc ), saddr( saddr ) {
    } // uSocketIO::uSocketIO
  public:
    _Mutex const struct sockaddr *getsockaddr() {	// must cast result to sockaddr_in or sockaddr_un
	return saddr;
    } // uSocketIO::getsockname

    _Mutex int getsockname( struct sockaddr *name, socklen_t *len ) {
	return ::getsockname( access.fd, name, len );
    } // uSocketIO::getsockname

    _Mutex int getpeername( struct sockaddr *name, socklen_t *len ) {
	return ::getpeername( access.fd, name, len );
    } // uSocketIO::getpeername

    int send( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int sendto( char *buf, int len, struct sockaddr *to, socklen_t tolen, int flags = 0, uDuration *timeout = NULL );

    int sendto( char *buf, int len, int flags = 0, uDuration *timeout = NULL ) {
	return sendto( buf, len, saddr, saddrlen, flags, timeout );
    } // uSocketIO::sendto

    int sendmsg( const struct msghdr *msg, int flags = 0, uDuration *timeout = NULL );
    int recv( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int recvfrom( char *buf, int len, struct sockaddr *from, socklen_t *fromlen, int flags = 0, uDuration *timeout = NULL );

    int recvfrom( char *buf, int len, int flags = 0, uDuration *timeout = NULL ) {
	saddrlen = baddrlen;				// set to receive buffer size
	return recvfrom( buf, len, saddr, &saddrlen, flags, timeout );
    } // uSocketIO::recvfrom

    int recvmsg( struct msghdr *msg, int flags = 0, uDuration *timeout = NULL );
}; // uSocketIO


//######################### uSocketServer #########################


_Monitor uSocketServer : public uSocketIO {
    friend class uSocketAccept;				// access: socket, acceptor, unacceptor, ~uSocketAccept

    int acceptorCnt;					// number of simultaneous acceptors using server
    uSocket socket;					// one-to-one correspondance between server and socket

    void acceptor() {
	uAtomicInc( acceptorCnt );
    } // uSocketServer::acceptor

    void unacceptor() {
	uAtomicDec( acceptorCnt );
    } // uSocketServer::unacceptor

    void createSocketServer1( const char *name, int type, int protocol, int backlog );
    void createSocketServer2( unsigned short port, int type, int protocol, int backlog );
    void createSocketServer3( unsigned short *port, int type, int protocol, int backlog );
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );

    void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void openFailure( const unsigned short port, const char *const name, int domain, int type, int protocol, int backlog, const char *const msg );
  public:
    _Event Failure : public uSocket::Failure {
	const uSocketServer &server_;
      protected:
	Failure( const uSocketServer &s, const char *const msg );
      public:
	const uSocketServer &server() const;
	int fileDescriptor() const;
	virtual void defaultTerminate() const;
    }; // uSocketServer::Failure
    friend _Event Failure;				// access: socket

    _Event OpenFailure : public uSocketServer::Failure {
      protected:
	const unsigned short port;
	char name_[uEHMMaxName];
	const int domain;
	const int type;
	const int protocol;
	const int backlog;
      public:
	OpenFailure( const uSocketServer &server, const unsigned short port, const char *const name, int domain, int type, int protocol, int backlog, const char *const msg );
	const char *name() const;			// return socket name, or NULL when hostname is used
	virtual void defaultTerminate() const;
    }; // uSocketServer::OpenFailure

    _Event CloseFailure : public uSocketServer::Failure {
	const int acceptorCnt;
      public:
	CloseFailure( const uSocketServer &server, const int acceptorCnt, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketServer::CloseFailure

    _Event ReadFailure : public uSocketServer::Failure {
      protected:
	const char *buf;
	const int len;
	const int flags;
	const struct sockaddr *from;
	const socklen_t *fromlen;
	const uDuration *timeout;
      public:
	ReadFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketServer::ReadFailure

    _Event ReadTimeout : public uSocketServer::ReadFailure {
      public:
	ReadTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketServer::ReadTimeout

    _Event WriteFailure : public uSocketServer::Failure {
      protected:
	const char *buf;
	const int len;
	const int flags;
	const struct sockaddr *to;
	const int tolen;
	const uDuration *timeout;
      public:
	WriteFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketServer::WriteFailure

    _Event WriteTimeout : public uSocketServer::WriteFailure {
      public:
	WriteTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketServer::WriteTimeout


    // AF_UNIX
    uSocketServer( const char *name, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 ) :
	    uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
	createSocketServer1( name, type, protocol, backlog );
    } // uSocketServer::uSocketServer

    // AF_INET, local host
    uSocketServer( unsigned short port, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 ) :
	    uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
	createSocketServer2( port, type, protocol, backlog );
    } // uSocketServer::uSocketServer

    uSocketServer( unsigned short *port, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 ) :
	    uSocketIO( socket.access, (sockaddr *)new inetAddr( 0 ) ), socket( AF_INET, type, protocol ) {
	createSocketServer3( port, type, protocol, backlog );
    } // uSocketServer::uSocketServer

    virtual ~uSocketServer() {
	if ( acceptorCnt != 0 ) {
	    if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, acceptorCnt, "closing socket server with outstanding acceptor(s)" );
	} // if
	delete saddr;
    } // uSocketServer::~uSocketServer

    void setClient( struct sockaddr *addr, socklen_t len ) {
#ifdef __U_DEBUG__
	if ( len > baddrlen ) {
	    uAbort( "(uSocketServer &)0x%p.setClient( addr:0x%p, len:%d ) : New server address to large.", this, addr, len );
	} // if
#endif // __U_DEBUG__
	memcpy( saddr, addr, len );
    } // uSocketServer::setClient

    void getClient( struct sockaddr *addr, socklen_t *len ) {
#ifdef __U_DEBUG__
	if ( *len < baddrlen ) {
	    uAbort( "(uSocketServer &)0x%p.getClient( addr:0x%p, len:%d ) : Area for server address to small.", this, addr, *len );
	} // if
#endif // __U_DEBUG__
	*len = baddrlen;
	memcpy( addr, saddr, baddrlen );
    } // uSocketServer::getServer
}; // uSocketServer


//######################### uSocketAccept #########################


_Monitor uSocketAccept : public uSocketIO {
    uSocketServer &socketserver;			// many-to-one correspondence among acceptors and server
    uIOaccess access;
    bool openAccept;					// current open accept for socket ?
    uDuration *timeout;
    struct sockaddr *adr;
    socklen_t *len;

    const uSocket &sock() const {
	return socketserver.socket;
    } // uSocketAccept::sock

    void createSocketAcceptor( uDuration *timeout, struct sockaddr *adr, socklen_t *len );
    int close_();
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );

    void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *op );
  public:
    _Event Failure : public uSocket::Failure {
	const uSocketAccept &acceptor_;
      protected:
	Failure( const uSocketAccept &acceptor, const char *const msg );
      public:
	const uSocketAccept &acceptor() const;
	const uSocketServer &server() const;
	int fileDescriptor() const;
	virtual void defaultTerminate() const;
    }; // uSocketAccept::Failure

    _Event OpenFailure : public Failure {
      protected:
	const uDuration *timeout;
	const struct sockaddr *adr;
	const socklen_t *len;
      public:
	OpenFailure( const uSocketAccept &acceptor, const uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketAccept::OpenFailure

    _Event OpenTimeout : public OpenFailure {
      public:
	OpenTimeout( const uSocketAccept &acceptor, const uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketAccept::OpenTimeout

    _Event CloseFailure : public Failure {
      public:
	CloseFailure( const uSocketAccept &acceptor, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketAccept::CloseFailure

    _Event ReadFailure : public Failure {
      protected:
	const char *buf;
	const int len;
	const int flags;
	const struct sockaddr *from;
	const socklen_t *fromlen;
	const uDuration *timeout;
      public:
	ReadFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketAccept::ReadFailure

    _Event ReadTimeout : public ReadFailure {
      public:
	ReadTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketAccept::ReadTimeout

    _Event WriteFailure : public Failure {
      protected:
	const char *buf;
	const int len;
	const int flags;
	const struct sockaddr *to;
	const int tolen;
	const uDuration *timeout;
      public:
	WriteFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketAccept::WriteFailure

    _Event WriteTimeout : public WriteFailure {
      public:
	WriteTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketAccept::WriteTimeout


    uSocketAccept( uSocketServer &s, struct sockaddr *adr = NULL, socklen_t *len = NULL ) :
	    uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( NULL ), adr( adr ), len( len ) {
	createSocketAcceptor( NULL, adr, len );
	socketserver.acceptor();
    } // uSocketAccept::uSocketAccept

    uSocketAccept( uSocketServer &s, uDuration *timeout, struct sockaddr *adr = NULL, socklen_t *len = NULL ) :
	    uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( timeout ), adr( adr ), len( len ) {
	createSocketAcceptor( timeout, adr, len );
	socketserver.acceptor();
    } // uSocketAccept::uSocketAccept

    uSocketAccept( uSocketServer &s, bool doAccept, struct sockaddr *adr = NULL, socklen_t *len = NULL ) :
	    uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( NULL ), adr( adr ), len( len ) {
	if ( doAccept ) {
	    createSocketAcceptor( NULL, adr, len );
	} // if
	socketserver.acceptor();
    } // uSocketAccept::uSocketAccept

    uSocketAccept( uSocketServer &s, uDuration *timeout, bool doAccept, struct sockaddr *adr = NULL, socklen_t *len = NULL ) :
	    uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( timeout ), adr( adr ), len( len ) {
	if ( doAccept ) {
	    createSocketAcceptor( timeout, adr, len );
	} // if
	socketserver.acceptor();
    } // uSocketAccept::uSocketAccept


    virtual ~uSocketAccept() {
	socketserver.unacceptor();
	if ( openAccept && close_() == -1 ) {
	    if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, "unable to close socket acceptor" );
	} // if
    } // uSocketAccept::~uSocketAccept

    void accept( uDuration *timeout ) {
	close();
	uSocketAccept::timeout = timeout;
	createSocketAcceptor( timeout, adr, len );
    } // uSocketAccept::accept

    void accept() {
	accept( timeout );				// use global timeout
    } // uSocketAccept::accept

    void close() {
	if ( openAccept && close_() == -1 ) {
	    _Throw CloseFailure( *this, "unable to close socket acceptor" );
	} // if
    } // uSocketAccept::close
}; // uSocketAccept


//######################### uSocketClient #########################


_Monitor uSocketClient : public uSocketIO {
    uSocket socket;					// one-to-one correspondence between client and socket
    char *tmpnm;					// temporary file for communicate with UNIX datagram

    void connectionOriented( const char *name, unsigned short port, const int domain, uDuration *timeout, int type, int protocol );
    void createSocketClient1( const char *name, uDuration *timeout, int type, int protocol );
    void createSocketClient2( unsigned short port, const char *name, uDuration *timeout, int type, int protocol );
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );

    void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void openFailure( const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
    void openTimeout( const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
  public:
    _Event Failure : public uSocket::Failure {
	const uSocketClient &client_;
      protected:
	Failure( const uSocketClient &client, const char *const msg );
      public:
	const uSocketClient &client() const;
	int fileDescriptor() const;
	virtual void defaultTerminate() const;
    }; // uSocketClient::Failure

    _Event OpenFailure : public uSocketClient::Failure {
      protected:
	unsigned short port;
	char name_[uEHMMaxName];
	const uDuration *timeout;
	const int domain;
	const int type;
	const int protocol;
      public:
	OpenFailure( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
	const char *name() const;
	virtual void defaultTerminate() const;
    }; // uSocketClient::OpenFailure

    _Event OpenTimeout : public uSocketClient::OpenFailure {
      public:
	OpenTimeout( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketClient::OpenTimeout

    _Event CloseFailure : public uSocketClient::Failure {
      public:
	CloseFailure( const uSocketClient &acceptor, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketClient::CloseFailure

    _Event ReadFailure : public uSocketClient::Failure {
      protected:
	const char *buf;
	const int len;
	const int flags;
	const struct sockaddr *from;
	const socklen_t *fromlen;
	const uDuration *timeout;
      public:
	ReadFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketClient::ReadFailure

    _Event ReadTimeout : public uSocketClient::ReadFailure {
      public:
	ReadTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketClient::ReadTimeout

    _Event WriteFailure : public uSocketClient::Failure {
      protected:
	const char *buf;
	const int len;
	const int flags;
	const struct sockaddr *to;
	const int tolen;
	const uDuration *timeout;
      public:
	WriteFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketClient::WriteFailure

    _Event WriteTimeout : public uSocketClient::WriteFailure {
      public:
	WriteTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uSocketClient::WriteTimeout

    // AF_UNIX
    uSocketClient( const char *name, int type = SOCK_STREAM, int protocol = 0 ) :
	    uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
	createSocketClient1( name, NULL, type, protocol );
    } // uSocketClient::uSocketClient

    uSocketClient( const char *name, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 ) :
	    uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
	createSocketClient1( name, timeout, type, protocol );
    } // uSocketClient::uSocketClient

    // AF_INET, local host
    uSocketClient( unsigned short port, int type = SOCK_STREAM, int protocol = 0 ) :
	    uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
	createSocketClient2( port, ((inetAddr *)saddr)->name_, NULL, type, protocol );
    } // uSocketClient::uSocketClient

    uSocketClient( unsigned short port, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 ) :
	    uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
	createSocketClient2( port, ((inetAddr *)saddr)->name_, timeout, type, protocol );
    } // uSocketClient::uSocketClient

    // AF_INET, other host
    uSocketClient( unsigned short port, const char *name, int type = SOCK_STREAM, int protocol = 0 ) :
	    uSocketIO( socket.access, (sockaddr *)new inetAddr( port, name ) ), socket( AF_INET, type, protocol ) {
	createSocketClient2( port, name, NULL, type, protocol );
    } // uSocketClient::uSocketClient

    uSocketClient( unsigned short port, const char *name, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 ) :
	    uSocketIO( socket.access, (sockaddr *)new inetAddr( port, name ) ), socket( AF_INET, type, protocol ) {
	createSocketClient2( port, name, timeout, type, protocol );
    } // uSocketClient::uSocketClient

    virtual ~uSocketClient();

    void setServer( struct sockaddr *addr, socklen_t len ) {
#ifdef __U_DEBUG__
	if ( len > baddrlen ) {
	    uAbort( "(uSocketClient &)0x%p.setServer( addr:0x%p, len:%d ) : New server address to large.", this, addr, len );
	} // if
#endif // __U_DEBUG__
	memcpy( saddr, addr, len );
    } // uSocketClient::setServer

    void getServer( struct sockaddr *addr, socklen_t *len ) {
#ifdef __U_DEBUG__
	if ( *len < baddrlen ) {
	    uAbort( "(uSocketClient &)0x%p.getServer( addr:0x%p, len:%d ) : Area for server address to small.", this, addr, *len );
	} // if
#endif // __U_DEBUG__
	*len = baddrlen;
	memcpy( addr, saddr, baddrlen );
    } // uSocketClient::getServer
}; // uSocketClient


#pragma __U_USER_CODE__

#endif // __U_SOCKET_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
