//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 1994
// 
// uSocket.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 17:06:26 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Sep  6 11:43:57 2007
// Update Count     : 887
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>
#include <uSocket.h>

//#include <uDebug.h>

#include <cstring>					// strerror, memset
#include <unistd.h>					// read, write, close, etc.


#ifndef SUN_LEN
#define SUN_LEN(su) (sizeof(*(su)) - sizeof((su)->sun_path) + strlen((su)->sun_path))
#endif


//######################### uSocket #########################


void uSocket::createSocket( int domain, int type, int protocol ) {
    for ( ;; ) {
	access.fd = ::socket( domain, type, protocol );
      if ( access.fd != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( access.fd == -1 ) {
        // The exception should be caught by constructor in
	// uSocketServer or uSocketClient and throw a different exception
	// indicating the failure to create a server or a client.  However, it
	// cannot be implemented because the initialization list in a
	// constructor cannot be encapsulated in a try block.  Actually, there
	// is a feature in c++ allowing catch clause to handle exception
	// propoagated out of the initialization list but g++ does not have it
	// yet.

        _Throw OpenFailure( *this, domain, type, protocol, "unable to open socket" );
    } // if

    // Sockets always treated as non-blocking.
    access.poll.setStatus( uPoll::AlwaysPoll );
    access.poll.setPollFlag( access.fd );

    // Sockets always have local-address reuse enable.
    const int enable = 1;				// 1 => enable option
    if ( setsockopt( access.fd, SOL_SOCKET, SO_REUSEADDR,  &enable, sizeof(enable) ) == -1 ) {
        _Throw OpenFailure( *this, domain, type, protocol, "unable to set socket-option" );
    } // if
} // uSocket::createSocket


uSocket::~uSocket() {
    int retcode;

    uThisCluster().closeFD( access.fd );
    for ( ;; ) {
	retcode = ::close( access.fd );
      if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( retcode == -1 ) {
        if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, "unable to close socket" );
    } // if
} // uSocket::~uSocket


//######################### uSocketIO #########################


void uSocketIO::inetAddr::getinetAddr( unsigned short port, const char *name ) {
    if ( name[0] != '\0' ) {				// name ?
#if defined( __darwin__ ) || defined( __freebsd__ )
	struct addrinfo hints = { 0, AF_INET, 0, 0, 0, 0, 0, 0 };
	struct addrinfo *addr;
	int rc;
	rc = getaddrinfo( name, NULL, &hints, &addr );
	if ( rc != 0 ) {
	    retcode = 1;
	    return;
	} // if
	memcpy( &sin_addr.s_addr, addr->ai_addr, addr->ai_addrlen );
	freeaddrinfo( addr );
#else
	hostent hp;
	int herrno = 0;
	char buffer[8 * 1024];				// best guess at size
	hostent *rc;

	for ( int i = 0; i < 5; i += 1 ) {		// N tries
#if defined( __solaris__ ) || defined( __irix__ )
	    rc = gethostbyname_r( name, &hp, buffer, sizeof(buffer), &herrno );
#elif defined( __linux__ )
	    gethostbyname_r( name, &hp, buffer, sizeof(buffer), &rc, &herrno );
#else
	    #error uC++ : internal error, unsupported architecture
#endif
	  if ( rc != 0 ) break;				// success ?
	  if ( herrno != TRY_AGAIN ) break;		// failure ?
	    sleep( 1 );					// TRY_AGAIN => delay and try again
	} // for
	if ( rc == 0 ) {
	    retcode = 1;
	    return;
	} // if
	memcpy( &sin_addr.s_addr, hp.h_addr, hp.h_length );
#endif // __darwin__ || __freebsd__
    } else {
	sin_addr.s_addr = htonl( INADDR_ANY );
    } // if

    retcode = 0;
    sin_family = AF_INET;
    sin_port = htons( port );
    memset( &sin_zero, '\0', sizeof(sin_zero) );
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketIO &)0x%p::inetAddr.getinetAddr name:%s, sin_family:%d sin_port:%d (%x), sin_addr:%d.%d.%d.%d\n",
	       this, name, sin_family, sin_port, sin_port,
	       sin_addr._S_un._S_un_b.s_b1, sin_addr._S_un._S_un_b.s_b2, sin_addr._S_un._S_un_b.s_b3, sin_addr._S_un._S_un_b.s_b4 );
#endif // __U_DEBUG_H__
} // uSocketIO::inetAddr::getinetAddr


int uSocketIO::send( char *buf, int len, int flags, uDuration *timeout ) {
    int slen;

    struct Send : public uIOClosure {
	char *buf;
	int len;
	int flags;

	int action( int rwe ) { return ::send( access.fd, buf, len, flags ); }
	Send( uIOaccess &access, int &slen, char *buf, int len, int flags ) : uIOClosure( access, slen ), buf( buf ), len( len ), flags( flags ) {}
    } sendClosure( access, slen, buf, len, flags );

    sendClosure.wrapper( 0 );
    if ( slen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    writeFailure( buf, len, flags, NULL, 0, timeout, "send" );
	} // if
	int mask = uCluster::writeSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( sendClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( sendClosure, mask, &t ) == 0 ) { // timeout ?
		writeTimeout( buf, len, flags, NULL, 0, timeout, "send" );
	    } // if
	} // if
    } // if

    return slen;
} // uSocketIO::send


int uSocketIO::sendto( char *buf, int len, struct sockaddr *to, socklen_t tolen, int flags, uDuration *timeout ) {
    int slen;

    struct Sendto : public uIOClosure {
	char *buf;
	int len;
	int flags;
	struct sockaddr *to;
	socklen_t tolen;

	int action( int rwe ) { return ::sendto( access.fd, buf, len, flags, to, tolen ); }
	Sendto( uIOaccess &access, int &slen, char *buf, int len, int flags, struct sockaddr *to, socklen_t tolen ) :
	    uIOClosure( access, slen ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ) {}
    } sendtoClosure( access, slen, buf, len, flags, to, tolen );

    sendtoClosure.wrapper( 0 );
    if ( slen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    writeFailure( buf, len, flags, NULL, 0, timeout, "sendto" );
	} // if
	int mask = uCluster::writeSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( sendtoClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( sendtoClosure, mask, &t ) == 0 ) { // timeout ?
		writeTimeout( buf, len, flags, NULL, 0, timeout, "sendto" );
	    } // if
	} // if
    } // if

    return slen;
} // uSocketIO::sendto


int uSocketIO::recv( char *buf, int len, int flags, uDuration *timeout ) {
    int rlen;

    struct Recv : public uIOClosure {
	char *buf;
	int len;
	int flags;

	int action( int rwe ) { return ::recv( access.fd, buf, len, flags ); }
	Recv( uIOaccess &access, int &rlen, char *buf, int len, int flags ) : uIOClosure( access, rlen ), buf( buf ), len( len ), flags( flags ) {}
    } recvClosure( access, rlen, buf, len, flags );

    recvClosure.wrapper( 0 );
    if ( rlen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    readFailure( buf, len, flags, NULL, NULL, timeout, "recv" );
	} // if
	int mask = uCluster::readSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( recvClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( recvClosure, mask, &t ) == 0 ) { // timeout ?
		readTimeout( buf, len, flags, NULL, NULL, timeout, "recv" );
	    } // if
	} // if
    } // if

    return rlen;
} // uSocketIO::recv


int uSocketIO::recvfrom( char *buf, int len, struct sockaddr *from, socklen_t *fromlen, int flags, uDuration *timeout ) {
    int rlen;

    struct Recvfrom : public uIOClosure {
	char *buf;
	int len;
	int flags;
	struct sockaddr *from;
	socklen_t *fromlen;

	int action( int rwe ) {
	    int rlen;
#if defined( __irix__ )
	    // TEMPORARY: sets fromlen to 0 when EWOULDBLOCK
	    int tmp;
	    if ( fromlen != 0 ) tmp = *fromlen;		// save *fromlen, as it may be set to 0 after each attempt
#endif // __irix__
	    rlen = ::recvfrom( access.fd, buf, len, flags, from, fromlen );
#if defined( __irix__ )
	    // TEMPORARY: sets fromlen to 0 when EWOULDBLOCK
	    if ( fromlen != 0 && *fromlen == 0 ) *fromlen = tmp; // reset *fromlen after each attempt
#endif // __irix__
	    return rlen;
	}
	Recvfrom( uIOaccess &access, int &rlen, char *buf, int len, int flags, struct sockaddr *from, socklen_t *fromlen ) :
	    uIOClosure( access, rlen ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ) {}
    } recvfromClosure( access, rlen, buf, len, flags, from, fromlen );

    recvfromClosure.wrapper( 0 );
    if ( rlen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    readFailure( buf, len, flags, from, fromlen, timeout, "recvfrom" );
	} // if
	int mask = uCluster::readSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( recvfromClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( recvfromClosure, mask, &t ) == 0 ) { // timeout ?
		readTimeout( buf, len, flags, from, fromlen, timeout, "recvfrom" );
	    } // if
	} // if
    } // if

    return rlen;
} // uSocketIO::recvfrom


int uSocketIO::recvmsg( struct msghdr *msg, int flags, uDuration *timeout ) {
    int rlen;

    struct Recvmsg : public uIOClosure {
	struct msghdr *msg;
	int flags;

	int action( int rwe ) { return ::recvmsg( access.fd, msg, flags ); }
	Recvmsg( uIOaccess &access, int &rlen, struct msghdr *msg, int flags ) : uIOClosure( access, rlen ), msg( msg ), flags( flags ) {}
    } recvmsgClosure( access, rlen, msg, flags );

    recvmsgClosure.wrapper( 0 );
    if ( rlen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    readFailure( (const char *)msg, 0, flags, NULL, NULL, timeout, "recvmsg" );
	} // if
	int mask = uCluster::readSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( recvmsgClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( recvmsgClosure, mask, &t ) == 0 ) { // timeout ?
		readTimeout( (const char *)msg, 0, flags, NULL, NULL, timeout, "recvmsg" );
	    } // if
	} // if
    } // if

    return rlen;
} // uSocketIO::recvmsg


//######################### uSocketServer #########################


void uSocketServer::createSocketServer1( const char *name, int type, int protocol, int backlog ) {
    int retcode;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.createSocketServer1 attempting binding to name:%s\n", this, name );
#endif // __U_DEBUG_H__

    if ( strlen( name ) >= sizeof(((sockaddr_un *)saddr)->sun_path) ) {
        openFailure( 0, name, AF_UNIX, type, protocol, backlog, "socket name too long" );
    } // if

    ((sockaddr_un *)saddr)->sun_family = AF_UNIX;
    strcpy( ((sockaddr_un *)saddr)->sun_path, name );
    saddrlen = SUN_LEN( (sockaddr_un *)saddr );
    baddrlen = sizeof(sockaddr_un);

    for ( ;; ) {
	retcode = ::bind( socket.access.fd, saddr, saddrlen );
      if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( retcode == -1 ) {
        openFailure( 0, name, AF_UNIX, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    retcode = ::listen( socket.access.fd, backlog );
	  if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
	} // for
	if ( retcode == -1 ) {
	    openFailure( 0, name, AF_UNIX, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.createSocketServer1 binding to name:%s\n", this, name );
#endif // __U_DEBUG_H__
} // uSockeServer::createSocketServer1


void uSocketServer::createSocketServer2( unsigned short port, int type, int protocol, int backlog ) {
    int retcode;

    if ( ((inetAddr *)saddr)->retcode > 0 ) {		// problem generating internet address ?
	openFailure( port, ((inetAddr *)saddr)->name_, AF_INET, type, protocol, backlog, "internet name lookup failure" );
    } // if

    baddrlen = saddrlen = sizeof(sockaddr_in);

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.createSocketServer2 attempting binding to port:%d, name:%s\n", this, port, ((inetAddr *)saddr)->name_ );
#endif // __U_DEBUG_H__

    for ( ;; ) {
	retcode = ::bind( socket.access.fd, saddr, saddrlen );
      if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( retcode == -1 ) {
	openFailure( port, ((inetAddr *)saddr)->name_, AF_INET, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    retcode = ::listen( socket.access.fd, backlog );
	  if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
	} // for
	if ( retcode == -1 ) {
	    openFailure( port, ((inetAddr *)saddr)->name_, AF_INET, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.createSocketServer2 binding to port:%d, name:%s\n", this, port, ((inetAddr *)saddr)->name_ );
#endif // __U_DEBUG_H__
} // uSocketServer::createSocketServer2


void uSocketServer::createSocketServer3( unsigned short *port, int type, int protocol, int backlog ) {
    createSocketServer2( 0, type, protocol, backlog );	// 0 port number => select an used port

    getsockname( saddr, &saddrlen );			// insert unsed port number into address ("bind" does not do it)
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.createSocketServer3 binding to port:%d\n", this, ntohs( ((sockaddr_in *)saddr)->sin_port ) );
#endif // __U_DEBUG_H__
    *port = ntohs( ((sockaddr_in *)saddr)->sin_port );
} // uSocketServer::createSocketServer3


void uSocketServer::readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcpy( msg, "socket " ); strcat( msg, op ); strcat( msg, " fails" );
    _Throw uSocketServer::ReadFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::readFailure


void uSocketServer::readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::ReadTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::readTimeout


void uSocketServer::writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketServer::WriteFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::writeFailure


void uSocketServer::writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::WriteTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::writeTimeout


void uSocketServer::readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketServer::ReadFailure( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketServer::readFailure


void uSocketServer::readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::ReadTimeout( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketServer::readTimeout


void uSocketServer::writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketServer::WriteFailure( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketServer::writeFailure


void uSocketServer::writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::WriteTimeout( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketServer::writeTimeout


void uSocketServer::openFailure( const unsigned short port, const char *const name, int domain, int type, int protocol, int backlog, const char *const msg ) { // prevent expanding _Event at call site to reduce stack size
    _Throw OpenFailure( *this, port, name, domain, type, protocol, backlog, msg );
} // uSocketServer::openFailure


//######################### uSocketAccept #########################


void uSocketAccept::createSocketAcceptor( uDuration *timeout, struct sockaddr *adr, socklen_t *len ) {
    struct Accept : public uIOClosure {
	struct sockaddr *adr;
	socklen_t *len;

	int action( int rwe ) {
	    int fd, tmp;
#ifdef KNOT
	    accept_syscalls += 1;
#endif // KNOT
	    if ( len != NULL ) tmp = *len;		// save *len, as it may be set to 0 after each attempt
	    fd = ::accept( access.fd, adr, len );
	    if ( len != NULL && *len == 0 ) *len = tmp;	// reset *len after each attempt
	    return fd;
	}
	Accept( uIOaccess &access, int &fd, struct sockaddr *adr, socklen_t *len ) : uIOClosure( access, fd ), adr( adr ), len( len ) {}
    } acceptClosure( socketserver.access, access.fd, adr, len );

    baddrlen = saddrlen = socketserver.saddrlen;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept before accept\n", this );
#endif // __U_DEBUG_H__

    acceptClosure.wrapper( 0 );
    if ( access.fd == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    _Throw OpenFailure( *this, timeout, adr, len, "unable to accept connection on socket server" );
	} // if
	int mask = uCluster::readSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( acceptClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( acceptClosure, mask, &t ) == 0 ) { // timeout ?
		_Throw OpenTimeout( *this, timeout, adr, len, "timeout during accept" );
	    } // if
	} // if
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept after accept fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__

    // On some UNIX systems the file descriptor created by accept inherits the
    // non-blocking characteristic from the base socket; on other system this
    // does not seem to occur, so explicitly set the file descriptor to
    // non-blocking.

    access.poll.setStatus( uPoll::AlwaysPoll );
    access.poll.setPollFlag( access.fd );
    openAccept = true;
} // uSocketAccept::createSocketAcceptor


int uSocketAccept::close_() {
    int retcode;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.close, enter fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__

    uThisCluster().closeFD( access.fd );
    for ( ;; ) {
	retcode = ::close( access.fd );
      if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.close, exit fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__
    openAccept = false;

    return retcode;
} // uSocketAccept::close_


void uSocketAccept::readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::ReadFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::readFailure


void uSocketAccept::readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::ReadTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::readTimeout


void uSocketAccept::writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::WriteFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::writeFailure


void uSocketAccept::writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::WriteTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::writeTimeout


void uSocketAccept::readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::ReadFailure( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketAccept::readFailure


void uSocketAccept::readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::ReadTimeout( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketAccept::readTimeout


void uSocketAccept::writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::WriteFailure( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketAccept::writeFailure


void uSocketAccept::writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::WriteTimeout( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketAccept::writeTimeout


//######################### uSocketClient #########################


void uSocketClient::connectionOriented( const char *name, unsigned short port, const int domain, uDuration *timeout, int type, int protocol ) {
    int retcode;

    struct Connect : public uIOClosure {
	struct sockaddr *adr;
	socklen_t len;
	bool doConnect;

	int action( int rwe ) { return doConnect ? ::connect( access.fd, adr, len ) : 0; }
	Connect( uIOaccess &access, int &retcode, struct sockaddr *adr, socklen_t len ) : uIOClosure( access, retcode ), adr( adr ), len( len ), doConnect( true ) {}
    } connectClosure( socket.access, retcode, saddr, saddrlen );

    connectClosure.wrapper( 0 );

    if ( retcode == -1 ) {
	if ( errno == U_EWOULDBLOCK ) {
	    // poll for free port
	    int mask = uCluster::readSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( connectClosure, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( connectClosure, mask, &t ) == 0 ) { // timeout ?
		    openTimeout( port, name, timeout, domain, type, protocol, "timeout during connect" );
		} // if
	    } // if
	} // if
	if ( retcode == -1 ) {
	    if ( errno == EINPROGRESS ) {
		// waiting for a non-blocking connect
		connectClosure.doConnect = false;
		int mask = uCluster::writeSelect;
		if ( timeout == NULL ) {
		    uThisCluster().select( connectClosure, mask );
		} else {
		    timeval t = *timeout;		// convert to timeval for select
		    if ( uThisCluster().select( connectClosure, mask, &t ) == 0 ) { // timeout ?
			openTimeout( port, name, timeout, domain, type, protocol, "timeout during connect" );
		    } // if
		} // if
		socklen_t retcodeLen = sizeof(retcode);
		int status = ::getsockopt( socket.access.fd, SOL_SOCKET, SO_ERROR, &retcode, &retcodeLen );
		if ( status == -1 ) {
		    openFailure( port, name, timeout, domain, type, protocol, "unable to connect to socket" );
		} // if
	    } else {
		openFailure( port, name, timeout, domain, type, protocol, "unable to connect to socket" );
	    } // if
	} // if
    } // if
} // uSocketClient::connectionOriented


void uSocketClient::createSocketClient1( const char *name, uDuration *timeout, int type, int protocol ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.createSocketClient1 attempting connection to name:%s\n", this, name );
#endif // __U_DEBUG_H__

    if ( strlen( name ) >= sizeof(((sockaddr_un *)saddr)->sun_path) ) {
	openFailure( 0, name, timeout, AF_UNIX, type, protocol, "socket name too long" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	strcpy( ((sockaddr_un *)saddr)->sun_path, name );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );
	baddrlen = sizeof(sockaddr_un);

	connectionOriented( name, 0, AF_UNIX, timeout, type, protocol );
    } else {
	int retcode;

	// Create temporary file for the client to communicate through.
	tmpnm = tempnam( NULL, "uC++" );
	if ( strlen( tmpnm ) >= sizeof(((sockaddr_un *)saddr)->sun_path) ) {
	    openFailure( 0, tmpnm, timeout, AF_UNIX, type, protocol, "socket temporary name too long" );
	} // if

	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	strcpy( ((sockaddr_un *)saddr)->sun_path, tmpnm );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );

	for ( ;; ) {
	    retcode = ::bind( socket.access.fd, saddr, saddrlen );
	  if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
	} // for
	if ( retcode == -1 ) {
	    openFailure( 0, name, timeout, AF_UNIX, type, protocol, "unable to bind name to socket" );
	} // if

	// Now set default address specified by user.
	strcpy( ((sockaddr_un *)saddr)->sun_path, name );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );
	baddrlen = sizeof(sockaddr_un);
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.createSocketClient1 connection to name:%s\n", this, name );
#endif // __U_DEBUG_H__
} // uSocketClient::createSocketClient1


void uSocketClient::createSocketClient2( unsigned short port, const char *name, uDuration *timeout, int type, int protocol ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.createSocketClient2 attempting connection to port:%d, name:%s\n", this, port, name );
#endif // __U_DEBUG_H__

    if ( ((inetAddr *)saddr)->retcode > 0 ) {		// problem generating internet address ?
        openFailure( port, ((inetAddr *)saddr)->name_, timeout, AF_INET, type, protocol, "internet name lookup failure" );
    } // if

    baddrlen = saddrlen = sizeof(sockaddr_in);

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	connectionOriented( name, port, AF_INET, timeout, type, protocol );
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.createSocketClient2 connection to port:%d, name:%s\n", this, port, name );
#endif // __U_DEBUG_H__
} // uSocketClient::createSocketClient2


void uSocketClient::readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::ReadFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::readFailure


void uSocketClient::readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::ReadTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::readTimeout


void uSocketClient::writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::WriteFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::writeFailure


void uSocketClient::writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::WriteTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::writeTimeout


void uSocketClient::readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::ReadFailure( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketClient::readFailure


void uSocketClient::readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::ReadTimeout( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketClient::readTimeout


void uSocketClient::writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::WriteFailure( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketClient::writeFailure


void uSocketClient::writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::WriteTimeout( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketClient::writeTimeout


void uSocketClient::openFailure( const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) { // prevent expanding _Event at call site to reduce stack size
    _Throw OpenFailure( *this, port, name, timeout, domain, type, protocol, msg );
} // uSocketClient::openFailure


void uSocketClient::openTimeout( const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) { // prevent expanding _Event at call site to reduce stack size
    _Throw OpenTimeout( *this, port, name, timeout, domain, type, protocol, msg );
} // uSocketClient::openTimeout


uSocketClient::~uSocketClient() {
    if ( saddr->sa_family == AF_UNIX && socket.type == SOCK_DGRAM ) {
	// Remove temporary file created for communication.
	int retcode = unlink( tmpnm );
	delete tmpnm;
	if ( retcode == -1 ) {
	    delete saddr;
	    if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, "unlink failed for temporary pipe" );
	} // if
    } // if
    delete saddr;
} // uSocketClient::~uSocketClient


//######################### uSocket (cont) #########################


uSocket::Failure::Failure( const uSocket &socket, const char *const msg ) : uIOFailure( msg ), socket_( socket ) {}

const uSocket &uSocket::Failure::socket() const { return socket_; }

void uSocket::Failure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p, %.256s.", &socket(), message() );
} // uSocket::Failure::defaultTerminate


uSocket::OpenFailure::OpenFailure( const uSocket &socket, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocket::Failure( socket, msg ), domain( domain ), type( type ), protocol( protocol ) {}

void uSocket::OpenFailure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p.uSocket( domain:%d, type:%d, protocol:%d ), %.256s.\nError(%d) : %s.",
	    &socket(), domain, type, protocol, message(), errNo(), strerror( errNo() ) );
} // uSocket::OpenFailure::defaultTerminate


uSocket::CloseFailure::CloseFailure( const uSocket &socket, const char *const msg ) : uSocket::Failure( socket, msg ) {}

void uSocket::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p.~uSocket(), %.256s %d.\nError(%d) : %s.",
	    &socket(), message(), socket().access.fd, errNo(), strerror( errNo() ) );
} // uSocket::CloseFailure::defaultTerminate


//######################### uSocketServer (cont) #########################


uSocketServer::Failure::Failure( const uSocketServer &server, const char *const msg ) : uSocket::Failure( server.socket, msg ), server_( server ) {
} // uSocketServer::Failure::Failure

const uSocketServer &uSocketServer::Failure::server() const {
    return server_;
} // uSocketServer::Failure::server

int uSocketServer::Failure::fileDescriptor() const {
    return server().access.fd;
} // uSocketServer::Failure::fileDescriptor

void uSocketServer::Failure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p, %.256s.\nError(%d) : %s.",
	    &server(), message(), errNo(), strerror( errNo() ) );
} // uSocketServer::Failure::defaultTerminate


uSocketServer::OpenFailure::OpenFailure( const uSocketServer &server, unsigned short port, const char *name, int domain, int type, int protocol, int backlog, const char *const msg ) :
	uSocketServer::Failure( server, msg ), port( port ), domain( domain ), type( type ), protocol( protocol ), backlog( backlog ) {
    uEHM::strncpy( name_, name, uEHMMaxName );		// copy file name
} // uSocketServer::OpenFailure::OpenFailure

const char *uSocketServer::OpenFailure::name() const { return name_; }

void uSocketServer::OpenFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.uSocketServer( port:%d, name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d, backlog:%d ), %.256s.\nError(%d) : %s.",
	    &server(), port, name(), &uThisCluster(), domain, type, protocol, backlog, message(), errNo(), strerror( errNo() ) );
} // uSocketServer::OpenFailure::defaultTerminate


uSocketServer::CloseFailure::CloseFailure( const uSocketServer &server, int acceptorCnt, const char *const msg ) :
	uSocketServer::Failure( server, msg ), acceptorCnt( acceptorCnt ) {
} // uSocketServer::CloseFailure::CloseFailure

void uSocketServer::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.~uSocketServer(), %.256s, %d acceptor(s) outstanding.",
	    &server(), message(), acceptorCnt );
} // uSocketServer::CloseFailure::defaultTerminate


uSocketServer::ReadFailure::ReadFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketServer::ReadFailure::ReadFailure

void uSocketServer::ReadFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::ReadFailure::defaultTerminate


uSocketServer::ReadTimeout::ReadTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::ReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketServer::ReadTimeout::ReadTimeout

void uSocketServer::ReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketServer::ReadTimeout::defaultTerminate


uSocketServer::WriteFailure::WriteFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketServer::WriteFailure::WriteFailure

void uSocketServer::WriteFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::WriteFailure::defaultTerminate


uSocketServer::WriteTimeout::WriteTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::WriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketServer::WriteTimeout::WriteTimeout

void uSocketServer::WriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketServer::WriteTimeout::defaultTerminate


//######################### uSocketAccept (cont) #########################


uSocketAccept::Failure::Failure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocket::Failure( acceptor.sock(), msg ), acceptor_( acceptor ) {}

const uSocketAccept &uSocketAccept::Failure::acceptor() const {
    return acceptor_;
} // uSocketAccept::Failure::acceptor

const uSocketServer &uSocketAccept::Failure::server() const {
    return acceptor_.socketserver;
} // uSocketAccept::Failure::server

int uSocketAccept::Failure::fileDescriptor() const {
    return acceptor().access.fd;
} // uSocketAccept::Failure::fileDescriptor

void uSocketAccept::Failure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p( server:0x%p ), %.256s.", &acceptor(), &server(), message() );
} // uSocketAccept::Failure::defaultTerminate


uSocketAccept::OpenFailure::OpenFailure( const uSocketAccept &acceptor, const class uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg ) :
	uSocketAccept::Failure( acceptor, msg ), timeout( timeout ), adr( adr ), len( len ) {}

void uSocketAccept::OpenFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s, error(%d) : %s",
	    &acceptor(), &server(), timeout, adr, len, message(), errNo(), strerror( errNo() ) );
} // uSocketAccept::OpenFailure::defaultTerminate


uSocketAccept::OpenTimeout::OpenTimeout( const uSocketAccept &acceptor, const class uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg ) :
	uSocketAccept::OpenFailure( acceptor, timeout, adr, len, msg ) {}

void uSocketAccept::OpenTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s",
	    &acceptor(), &server(), timeout, adr, len, message() );
} // uSocketAccept::OpenTimeout::defaultTerminate


uSocketAccept::CloseFailure::CloseFailure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocketAccept::Failure( acceptor, msg ) {}

void uSocketAccept::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.~uSocketAccept(), %.256s.\nError(%d) : %s.",
	    &acceptor(), message(), errNo(), strerror( errNo() ) );
} // uSocketAccept::CloseFailure::defaultTerminate


uSocketAccept::ReadFailure::ReadFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketAccept::ReadFailure::ReadFailure

void uSocketAccept::ReadFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::ReadFailure::defaultTerminate


uSocketAccept::ReadTimeout::ReadTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::ReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketAccept::ReadTimeout::ReadTimeout

void uSocketAccept::ReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketAccept::ReadTimeout::defaultTerminate


uSocketAccept::WriteFailure::WriteFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketAccept::WriteFailure::WriteFailure

void uSocketAccept::WriteFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::WriteFailure::defaultTerminate


uSocketAccept::WriteTimeout::WriteTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::WriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketAccept::WriteTimeout::WriteTimeout

void uSocketAccept::WriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketAccept::WriteTimeout::defaultTerminate


//######################### uSocketClient (cont) #########################


uSocketClient::Failure::Failure( const uSocketClient &client, const char *const msg ) : uSocket::Failure( client.socket, msg ), client_( client ) {
} // uSocketClient::Failure::Failure

const uSocketClient &uSocketClient::Failure::client() const {
    return client_;
} // uSocketClient::Failure::client

int uSocketClient::Failure::fileDescriptor() const {
    return client().access.fd;
} // uSocketClient::Failure::fileDescriptor

void uSocketClient::Failure::defaultTerminate() const {
	uAbort( "(uSocketClient &)0x%p, %.256s.",
		&client(), message() );
} // uSocketClient::Failure::defaultTerminate


uSocketClient::OpenFailure::OpenFailure( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::Failure( client, msg ), port( port ), timeout( timeout ), domain( domain ), type( type ), protocol( protocol ) {
    uEHM::strncpy( name_, name, uEHMMaxName );		// copy file name
} // uSocketClient::OpenFailure::OpenFailure

const char *uSocketClient::OpenFailure::name() const { return name_; }

void uSocketClient::OpenFailure::defaultTerminate() const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.\nError(%d) : %s.",
	       &client(), name(), &uThisCluster(), timeout, domain, type, protocol, message(), errNo(), strerror( errNo() ) );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.\nError(%d) : %s.",
	       &client(), port, name(), &uThisCluster(), timeout, domain, type, protocol, message(), errNo(), strerror( errNo() ) );
    } // if
} // uSocketClient::OpenFailure::defaultTerminate


uSocketClient::OpenTimeout::OpenTimeout( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::OpenFailure( client, port, name, timeout, domain, type, protocol, msg ) {
} // uSocketClient::OpenTimeout::OpenTimeout

void uSocketClient::OpenTimeout::defaultTerminate() const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.",
		&client(), name(), &uThisCluster(), domain, type, protocol, message() );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.",
		&client(), port, name(), &uThisCluster(), timeout, domain, type, protocol, message() );
    } // if
} // uSocketClient::OpenTimeout::defaultTerminate


uSocketClient::CloseFailure::CloseFailure( const uSocketClient &client, const char *const msg ) :
	uSocketClient::Failure( client, msg ) {}

void uSocketClient::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.~uSocketClient(), %.256s.\nError(%d) : %s.",
	    &client(), message(), errNo(), strerror( errNo() ) );
} // uSocketClient::CloseFailure::defaultTerminate


uSocketClient::ReadFailure::ReadFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketClient::ReadFailure::ReadFailure

void uSocketClient::ReadFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::ReadFailure::defaultTerminate


uSocketClient::ReadTimeout::ReadTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::ReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketClient::ReadTimeout::ReadTimeout

void uSocketClient::ReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketClient::ReadTimeout::defaultTerminate


uSocketClient::WriteFailure::WriteFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketClient::WriteFailure::WriteFailure

void uSocketClient::WriteFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::WriteFailure::defaultTerminate


uSocketClient::WriteTimeout::WriteTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::WriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketClient::WriteTimeout::WriteTimeout

void uSocketClient::WriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketClient::WriteTimeout::defaultTerminate


// Local Variables: //
// compile-command: "gmake install" //
// End: //
