//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 2003
// 
// uSemaphore.cc -- 
// 
// Author           : Peter A. Buhr
// Created On       : Thu Nov 20 17:17:52 2003
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Sep 26 14:33:48 2007
// Update Count     : 61
// 
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>

namespace UPP {
//######################### TimedWaitHandler #########################


    uSemaphore::TimedWaitHandler::TimedWaitHandler( uBaseTask &task, uSemaphore &semaphore ) : semaphore( semaphore ) {
	This = &task;
	timedout = false;
    } // uSemaphore::TimedWaitHandler::TimedWaitHandler

    uSemaphore::TimedWaitHandler::TimedWaitHandler( uSemaphore &semaphore ) : semaphore( semaphore ) {
	This = NULL;
	timedout = false;
    } // uSemaphore::TimedWaitHandler::TimedWaitHandler

    void uSemaphore::TimedWaitHandler::uHandler() {
	semaphore.waitTimeout( *This, *this );
    } // uSemaphore::TimedWaitHandler::uHandler


//######################### uSemaphore #########################


    void uSemaphore::waitTimeout( uBaseTask &task, TimedWaitHandler &h ) {
	// This uSemaphore member is called from the kernel, and therefore, cannot
	// block, but it can spin.

	spin.acquire();
	if ( task.entryRef.listed() ) {			// is task on queue
	    uBaseTask &task = waiting.dropHead()->task(); // remove task at head of waiting list
	    h.timedout = true;
	    count += 1;					// adjust the count to reflect the wake up
	    spin.release();
	    task.wake();				// wake up task
	} else {
	    spin.release();
	} // if
    } // uSemaphore::waitTimeout


    void uSemaphore::P() {				// wait on a semaphore
	spin.acquire();
	count -= 1;
	if ( count < 0 ) {
	    waiting.addTail( &(uThisTask().entryRef) );	// queue current task
#ifdef KNOT
	    g_io_lock += 1;
#endif // KNOT
	    uProcessorKernel::schedule( &spin );	// atomically release spin lock and block
	} else {
	    spin.release();
	} // if
    } // uSemaphore::P


    bool uSemaphore::P( uDuration duration ) {		// wait on a semaphore
	return P( uThisProcessor().getClock().getTime() + duration );
    } // uSemaphore::P


    bool uSemaphore::P( uTime time ) {			// wait on a semaphore
	spin.acquire();
	count -= 1;
	if ( count < 0 ) {
	    uBaseTask &task = uThisTask();		// optimization
	    TimedWaitHandler handler( task, *this );	// handler to wake up blocking task
	    uEventNode timeoutEvent( task, handler, time, 0 );
	    timeoutEvent.executeLocked = true;
	    timeoutEvent.add();
	    waiting.addTail( &(task.entryRef) );	// queue current task
	    uProcessorKernel::schedule( &spin );	// atomically release spin lock and block
	    // count is incremented in waitTimeout for timeout
	    timeoutEvent.remove();
	    return ! handler.timedout;
	} else {
	    spin.release();
	    return true;
	} // if
    } // uSemaphore::P


    void uSemaphore::P( uSemaphore &s ) {		// wait on a semaphore and release another
	spin.acquire();
	if ( &s == this ) {				// perform operation on self ?
	    if ( count < 0 ) {				// V my semaphore
		waiting.dropHead()->task().wake();	// remove task at head of waiting list
	    } // if
	    count += 1;
	} else {
	    s.V();					// V other semaphore
	} // if

	count -= 1;					// now P my semaphore
	if ( count < 0 ) {
	    waiting.addTail( &(uThisTask().entryRef) );	// block current task
	    uProcessorKernel::schedule( &spin );	// atomically release spin lock and block
	} else {
	    spin.release();
	} // if
    } // uSemaphore::P


    bool uSemaphore::P( uSemaphore &s, uDuration duration ) { // wait on semaphore and release another
	return P( s, uThisProcessor().getClock().getTime() + duration );
    } // uSemaphore::P


    bool uSemaphore::P( uSemaphore &s, uTime time ) {	// wait on semaphore and release another
	spin.acquire();
	if ( &s == this ) {				// perform operation on self ?
	    if ( count < 0 ) {				// V my semaphore
		waiting.dropHead()->task().wake();	// remove task at head of waiting list
	    } // if
	    count += 1;
	} else {
	    s.V();					// V other semaphore
	} // if

	count -= 1;					// now P my semaphore
	if ( count < 0 ) {
	    uBaseTask &task = uThisTask();		// optimization
	    TimedWaitHandler handler( task, *this );	// handler to wake up blocking task
	    uEventNode timeoutEvent( task, handler, time, 0 );
	    timeoutEvent.executeLocked = true;
	    timeoutEvent.add();
	    waiting.addTail( &(task.entryRef) );	// queue current task
	    uProcessorKernel::schedule( &spin );	// atomically release spin lock and block
	    // count is incremented in waitTimeout for timeout
	    timeoutEvent.remove();
	    return ! handler.timedout;
	} else {
	    spin.release();
	    return true;
	} // if
    } //  uSemaphore::P


    bool uSemaphore::TryP() {				// conditionally wait on a semaphore
	spin.acquire();
	if ( count > 0 ) {
	    count -= 1;
	    spin.release();
	    return true;
	} // if
	spin.release();
	return false;
    } // uSemaphore::TryP


    void uSemaphore::V() {				// signal semaphore
	// special form to handle the case where the woken task deletes the semaphore storage
	uBaseTaskDL *task;
	spin.acquire();
	count += 1;
	if ( count <= 0 ) {
	    task = waiting.dropHead();			// remove task at head of waiting list
#ifdef KNOT
	    g_io_lock -= 1;
#endif // KNOT
	    spin.release();
	    task->task().wake();			// make new owner
	} else {
	    spin.release();
	} // if
    } // uSemaphore::V


    void uSemaphore::V( unsigned int times ) {		// signal semaphore
	spin.acquire();
	for ( int i = times; i > 0; i -= 1 ) {
	    if ( count >= 0 ) {
		count += i;
		break;
	    } // if
	    count += 1;
	    waiting.dropHead()->task().wake();		// remove task at head of waiting list and make new owner
	} // for
	spin.release();
    } // uSemaphore::V
} // UPP

// Local Variables: //
// compile-command: "gmake install" //
// End: //
