//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 1994
// 
// uMachContext.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Fri Feb 25 15:46:42 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Sat Sep 22 14:18:08 2007
// Update Count     : 641
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>
#include <uAlign.h>
#include <uProfiler.h>

//#include <uDebug.h>

#include <cstdio>					// fprintf
#include <cerrno>

#if defined( __sparc__ ) || defined( __mips__ ) || defined( __x86_64__ ) || defined( __ia64__ )
extern "C" void uInvokeStub( uMachContext * );
#endif // __sparc__ || __mips__

#if defined( __sparc__ )
#if defined(  __solaris__ )
#include <sys/stack.h>
#else
#include <machine/trap.h>
#include <machine/asm_linkage.h>
#include <machine/psl.h>
#endif // __solaris__
#endif // __sparc__


using namespace UPP;


extern "C" void pthread_deletespecific_( void * );	// see pthread simulation


#define uMinStackSize 1000				// minimum feasible stack size in bytes


void uMachContext::invokeCoroutine( uBaseCoroutine &This ) { // magically invoke the "main" of the most derived class
    // Called from the kernel when starting a coroutine or task so must switch
    // back to user mode.

    This.setState( uBaseCoroutine::Active );		// set state of next coroutine to active
    THREAD_GETMEM( This )->enableInterrupts();

    if ( uThisTask().profileActive && uProfiler::uProfiler_postallocateMetricMemory ) {
	(*uProfiler::uProfiler_postallocateMetricMemory)( uProfiler::profilerInstance, uThisTask() );
    } // if
    // also appears in uBaseCoroutine::uContextSw2
    if ( ! THREAD_GETMEM( disableInt ) && uThisTask().profileActive && uProfiler::uProfiler_registerCoroutineUnblock ) {
	(*uProfiler::uProfiler_registerCoroutineUnblock)( uProfiler::profilerInstance, uThisTask() );
    } // if

    // At this point, execution is on the stack of the new coroutine or task
    // that has just been switched to by the kernel.  Therefore, interrupts can
    // legitimately occur now.

    try {
	This.corStarter();				// moved from uCoroutineMain to allow recursion on "main"
	uEHM::poll();					// cancellation checkpoint
	This.main();					// start coroutine's "main" routine
    } catch ( uBaseCoroutine::UnwindStack &u ) {
	u.exec_dtor = false;				// defuse the bomb or otherwise unwinding will continue
	This.notHalted = false;				// terminate coroutine
    } catch( uBaseCoroutine::UnhandledException &ex ) {
	This.handleUnhandled( &ex );
    } catch ( uBaseEvent &ex ) {
	This.handleUnhandled( &ex );
    } catch( ... ) {					// unknown exception ?
	This.handleUnhandled();
    } // try

    // check outside handler so exception is freed before suspending
    if ( ! This.notHalted ) {				// exceptional ending ?
	This.suspend();					// restart last resumer, which should immediately propagate the nonlocal exception
    } // if
    This.corFinish();
} // uMachContext::invokeCoroutine


void uMachContext::invokeTask( uBaseTask &This ) {	// magically invoke the "main" of the most derived class
    // Called from the kernel when starting a coroutine or task so must switch
    // back to user mode.

#if defined(__U_MULTI__)
    THREAD_SETMEM( activeTask, ((uBaseTask *)&This) );
#endif

    errno = 0;						// reset errno for each task
    This.currCoroutine->setState( uBaseCoroutine::Active ); // set state of next coroutine to active
    This.setState( uBaseTask::Running );
    THREAD_GETMEM( This )->enableInterrupts();

    if ( uThisTask().profileActive && uProfiler::uProfiler_postallocateMetricMemory ) {
	(*uProfiler::uProfiler_postallocateMetricMemory)( uProfiler::profilerInstance, uThisTask() );
    } // if

    // At this point, execution is on the stack of the new coroutine or task
    // that has just been switched to by the kernel.  Therefore, interrupts can
    // legitimately occur now.

    uPthreadable *pthreadable = dynamic_cast< uPthreadable * > (&This);

    try {
	try {
	    uEHM::poll();				// cancellation checkpoint
	    This.main();				// start task's "main" routine
	} catch( uBaseCoroutine::UnwindStack &evt ) {
	    evt.exec_dtor = false;			// defuse the unwinder	
	} catch( uBaseEvent &ex ) {
	    ex.defaultTerminate();
	    cleanup( This );				// preserve current exception
	} catch( ... ) {
	    if ( ! This.cancelInProgress() ) {
		cleanup( This );			// preserve current exception
		// CONTROL NEVER REACHES HERE!
	    } // if
	    if ( pthreadable ) {
		pthreadable->stop_unwinding = true;	// prevent continuation of unwinding
	    } // if
#if defined (__irix__)					// remove if appropriate
	    uPthreadable::unwind_lock.release();	// release lock acquired in Pthreadable::do_unwind()
#endif // __irix__
	} // try
    } catch (...) {
	uEHM::terminate();				// if defaultTerminate or std::terminate throws exception
    } // try
    // NOTE: this code needs further protection as soon as asynchronous cancellation is supported

    // Clean up storage associated with the task for pthread thread-specific
    // data, e.g., exception handling associates thread-specific data with any
    // task.

    if ( This.pthreadData != NULL ) {
	pthread_deletespecific_( This.pthreadData );
    } // if

    This.notHalted = false;
    This.setState( uBaseTask::Terminate );

    This.getSerial().leave2();
    // CONTROL NEVER REACHES HERE!
    assert( false );
} // uMachContext::invokeTask


void uMachContext::cleanup( uBaseTask &This ) {
    try {
	std::terminate();				// call task terminate routine
    } catch( ... ) {					// control should not return
    } // try

    if ( This.pthreadData != NULL ) {			// see above for explanation
	pthread_deletespecific_( This.pthreadData );
    } // if

    uEHM::terminate();					// call abort terminate routine
} // uMachContext::cleanup


void uMachContext::extraSave() {
    // Don't need to test extras bit, it is sufficent to check context list

    uContext *context;
    for ( uSeqIter<uContext> iter(additionalContexts); iter >> context; ) {
	context->save();
    } // for
} // uMachContext::extraSave


void uMachContext::extraRestore() {
    // Don't need to test extras bit, it is sufficent to check context list

    uContext *context;
    for ( uSeqIter<uContext> iter(additionalContexts); iter >> context; ) {
	context->restore();
    } // for
} // uMachContext::extraRestore


unsigned int uMachContext::magic() {
    return 0x12345678;
} // uMachContext::magic


void uMachContext::startHere( void (*uInvoke)( uMachContext & ) ) {
/*******************************************************
  ,-----------------.
  |                 |
  | __U_CONTEXT_T__ |
  |                 |
  `-----------------'   <--- context
  magic number (16 bytes)
  ,-----------------. \ <--- base (stack grows down)
  |                 | |
  |    task stack   | } size
  |                 | |
  `-----------------' / <--- limit
  magic number (16 bytes)
0                       <--- storage
*******************************************************/

#if defined( __i386__ )

    // set base and limits of stack

    limit = (char *)storage + 16;
    base = (char *)limit + size;
    context = (char *)limit + size + 16;

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	void *uFixedRegisters[3];			// fixed registers ebx, edi, esi
	void *uReturn;					// where to go on return from uSwitch
	void *uDummyReturn;
	void *uArgument;
    };

    ((uContext_t *)context)->SP = (char *)base - sizeof( uFakeStack );
    ((uContext_t *)context)->FP = NULL;			// terminate stack with NULL fp

    ((uFakeStack *)(((uContext_t *)context)->SP))->uDummyReturn = NULL;
    ((uFakeStack *)(((uContext_t *)context)->SP))->uArgument = this; // argument to uInvoke
    ((uFakeStack *)(((uContext_t *)context)->SP))->uReturn = rtnAdr( (void (*)())uInvoke );
#else
    if ( ::getcontext( (ucontext *)context ) == -1 ) {	// initialize ucontext area
	uAbort( "internal error, getcontext failed" );
    } // if
    ((ucontext *)context)->uc_stack.ss_sp = (char *)limit;
    ((ucontext *)context)->uc_stack.ss_size = size; 
    ((ucontext *)context)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)context, (void (*)())uInvoke, 2, this );
    // TEMPORARY: stack is incorrectly initialized to allow stack walking
    ((ucontext *)context)->uc_mcontext.gregs[ REG_EBP ] = 0; // terminate stack with NULL fp
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)context)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#elif defined( __x86_64__ )

    // set base and limits of stack

    limit = (char *)storage + 16;
    base = (char *)limit + size;
    context = (char *)limit + size + 16;

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	void *uFixedRegisters[5];			// fixed registers rbx, r12, r13, r14, r15
	void *uReturn;					// where to go on return from uSwitch
	void *uDummyReturn;				// NULL return address to provide proper alignment
    };

    ((uContext_t *)context)->SP = (char *)base - sizeof( uFakeStack );
    ((uContext_t *)context)->FP = NULL;			// terminate stack with NULL fp

    ((uFakeStack *)(((uContext_t *)context)->SP))->uDummyReturn = NULL;
    ((uFakeStack *)(((uContext_t *)context)->SP))->uReturn = rtnAdr( (void (*)())uInvokeStub );
    ((uFakeStack *)(((uContext_t *)context)->SP))->uFixedRegisters[0] = this;
    ((uFakeStack *)(((uContext_t *)context)->SP))->uFixedRegisters[1] = rtnAdr( (void (*)())uInvoke );
#else
    // makecontext is difficult to support.  See http://sources.redhat.com/bugzilla/show_bug.cgi?id=404
    #error uC++ : internal error, swapcontext cannot be used on the x86_64 architecture
#endif

#elif defined( __ia64__ )

    // set base and limits of stack

    limit = (char *)storage + 16;
    base = (char *)limit + size;
    context = (char *)limit + size + 16;

#if ! defined( __U_SWAPCONTEXT__ )

    struct uFakeStack {
	void *scratch1[2];				// abi-mandated scratch area
	struct uPreservedState {
	    void *b5;
	    void *b4;
	    void *b3;
	    void *b2;
	    void *b1;
	    void *b0;
	    void *pr;
	    void *lc;
	    void *pfs;
	    void *fpsr;
	    void *unat;
	    void *spill_unat;
	    void *rnat;
	    void *r7;
	    void *r6;
	    void *r5;
	    void *r4;
	    void *r1;
	} preserved;
	void *scratch2[2];				// abi-mandated scratch area
    };

    ((uContext_t *)context)->SP = (char *)base - sizeof( uFakeStack );
    ((uContext_t *)context)->BSP = (char*)limit + 16;
    sigemptyset( &((uContext_t *)context)->sigMask );
#ifdef __U_MULTI__
    sigaddset( &((uContext_t *)context)->sigMask, SIGALRM );
#endif // __U_MULTI__
    memset( ((uContext_t *)context)->SP, 0, sizeof( uFakeStack ) );
    ((uFakeStack *)(((uContext_t *)context)->SP))->preserved.b0 = rtnAdr( (void (*)())uInvokeStub );
    ((uFakeStack *)(((uContext_t *)context)->SP))->preserved.r1 = gpAdr( (void (*)())uInvokeStub );
#if defined( __INTEL_COMPILER )
    ((uFakeStack *)(((uContext_t *)context)->SP))->preserved.fpsr = (void*)__getReg( _IA64_REG_AR_FPSR );
#else
    asm ( "mov.m %0 = ar.fpsr" : "=r" (((uFakeStack *)(((uContext_t *)context)->SP))->preserved.fpsr) );
#endif // __INTEL_COMPILER
    ((uFakeStack *)(((uContext_t *)context)->SP))->preserved.r4 = this;
    ((uFakeStack *)(((uContext_t *)context)->SP))->preserved.b1 = rtnAdr( (void (*)())uInvoke );
    ((uFakeStack *)(((uContext_t *)context)->SP))->preserved.b2 = NULL; // null terminate for stack walking
#else
    if ( ::getcontext( (ucontext *)context ) == -1 ) {	// initialize ucontext area
	uAbort( "internal error, getcontext failed" );
    } // if
    ((ucontext *)context)->uc_stack.ss_sp = (char *)limit;
    ((ucontext *)context)->uc_stack.ss_size = size;
    ((ucontext *)context)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)context, (void (*)())uInvoke, 2, this );

    // TEMPORARY: getcontext doesn't get the signal mask properly
    if ( sigprocmask( SIG_BLOCK, NULL, (sigset_t*)&(((ucontext *)context)->uc_sigmask) ) == -1 ) {
        uAbort( "internal error, sigprocmask" );
    } // if
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)context)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#elif defined( __sparc__ )

    // set base and limits of stack

    limit = (char *)storage + 16;
    base = (char *)limit + size;
    context = (char *)limit + size + 16;

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	void *uLocalRegs[8];
	void *uInRegs[8];
#if defined( _ILP32 )
	void *uStructRetAddress;
#endif
	void *uCalleeRegArgs[6];
    };

    ((uContext_t *)context)->FP = (char *)(SA( (unsigned long)base - STACK_ALIGN + 1 ) - SA( MINFRAME ) - STACK_BIAS );
    ((uContext_t *)context)->SP = (char *)((uContext_t *)context)->FP - SA( MINFRAME );
    ((uContext_t *)context)->PC = (char *)rtnAdr( (void (*)())uInvokeStub ) - 8;

    ((uFakeStack *)((char *)((uContext_t *)context)->FP + STACK_BIAS))->uInRegs[0] = this; // argument to uInvoke
    ((uFakeStack *)((char *)((uContext_t *)context)->FP + STACK_BIAS))->uInRegs[1] = (void *)uInvoke; // uInvoke routine
    ((uFakeStack *)((char *)((uContext_t *)context)->FP + STACK_BIAS))->uInRegs[6] = NULL; // terminate stack with NULL fp
#else
    if ( ::getcontext( (ucontext *)context ) == -1 ) {	// initialize ucontext area
	uAbort( " : internal error, getcontext failed" );
    } // if
    ((ucontext *)context)->uc_stack.ss_sp = (char *)base - 8;	// TEMPORARY: -8 for bug in Solaris (fixed in Solaris 4.10)
    ((ucontext *)context)->uc_stack.ss_size = size; 
    ((ucontext *)context)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)context, (void (*)(...))uInvoke, 2, this );
    // TEMPORARY: stack is incorrectly initialized to allow stack walking
    *((int *)base - 8) = 0;				// terminate stack with NULL fp
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)context)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#elif defined( __mips__ )

#if ! defined( __U_SWAPCONTEXT__ )
    struct uFakeStack {
	unsigned long long uFixedRegisters[11];		// fixed registers r16-r23,r30,r31,gp
	void *uDummyAlign[1];				// make multiple of 16
    };

    limit = (char *)storage + 16;
    base = (char *)limit + size;
    context = (char *)limit + size + 16;

    ((uContext_t *)context)->SP = (char *)base - sizeof( uFakeStack );

    ((uFakeStack *)((uContext_t *)context)->SP)->uFixedRegisters[0] = reinterpret_cast<unsigned long long>( this ); // argument to uInvoke (r16)
    ((uFakeStack *)((uContext_t *)context)->SP)->uFixedRegisters[1] = reinterpret_cast<unsigned long long>( rtnAdr( (void (*)())uInvoke ) );	// uInvoke routine (r17)
    ((uFakeStack *)((uContext_t *)context)->SP)->uFixedRegisters[8] = 0; // terminate stack with NULL fp (r30)
    ((uFakeStack *)((uContext_t *)context)->SP)->uFixedRegisters[9] = reinterpret_cast<unsigned long long>( rtnAdr( (void (*)())uInvokeStub ) ); // (r31)

#else
    if ( ::getcontext( (ucontext *)context ) == -1 ) {	// initialize ucontext area
	uAbort( "internal error, getcontext failed" );
    } // if
    ((ucontext *)context)->uc_stack.ss_sp = (char *)base - 8;	// TEMPORARY: -8 for bug in Solaris (fixed in Solaris 4.10)
    ((ucontext *)context)->uc_stack.ss_size = size; 
    ((ucontext *)context)->uc_stack.ss_flags = 0;
    ::makecontext( (ucontext *)context, (void (*)())uInvoke, 2, this );
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ )
    sigaddset( (sigset_t*)&((ucontext *)context)->uc_sigmask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__
#endif

#else
    #error uC++ : internal error, unsupported architecture
#endif

    // set magic words on stack

#ifdef __U_DEBUG__
    *(unsigned int *)((char *)limit - 16) = magic();
#endif // __U_DEBUG__
    *(unsigned int *)base = magic();
} // uMachContext::startHere


void uMachContext::createContext( void *storage_, unsigned int storageSize ) { // used by all constructors
    if ( storage_ != NULL ) {
#ifdef __U_DEBUG__
	if ( ((size_t)storage_ & (uAlign() - 1)) != 0 ) { // multiple of uAlign ?
	    uAbort( "Stack storage 0x%p for task/coroutine must be aligned on %d byte boundary.", storage_, (int)uAlign() );
	} // if
#endif // __U_DEBUG__
	userStack = true;
	storage = storage_;
    } else {
	userStack = false;
	// use malloc because "new" handles the overflow
	storage = malloc( uCeiling( sizeof(__U_CONTEXT_T__), 16 ) + storageSize + 32 );
	if ( storage == NULL ) {			// allocate stack from heap
	    uAbort( "Attempt to allocate %d bytes of storage for coroutine or task execution-state but insufficient memory available.", storageSize );
	} // if
    } // if
#ifdef __U_DEBUG__
    if ( storageSize < uMinStackSize ) {		// below minimum stack size ?
	uAbort( "Stack size %d provides less than minimum of %d bytes for a stack.", storageSize, uMinStackSize );
    } // if
#endif // __U_DEBUG__
    size = storageSize;
    extras.allExtras = 0;
} // uMachContext::createContext


void *uMachContext::stackPointer() const {
    if ( &uThisCoroutine() == this ) {			// accessing myself ?
	void *sp;					// use my current stack value
#if defined( __i386__ )
	asm("movl %%esp,%0" : "=m" (sp) : );
#elif defined( __x86_64__ )
	asm("movq %%rsp,%0" : "=m" (sp) : );
#elif defined( __ia64__ )
#if defined( __INTEL_COMPILER )
	sp = (void*)__getReg( _IA64_REG_SP );
#else
	asm("mov %0 = r12" : "=r" (sp) : );
#endif // __INTEL_COMPILER
#elif defined( __sparc__ )
	asm("mov %%sp,%0" : "=r" (sp) : );
	sp = (void*)((char*)sp + STACK_BIAS);
#elif defined( __mips__ )
#ifdef _ABIN32
	asm("sw $sp,%0" : "=m" (sp) : );
#elif _ABI64
	asm("sd $sp,%0" : "=m" (sp) : );
#else
	#error uC++ : internal error, unsupported architecture
#endif
#else
	#error uC++ : internal error, unsupported architecture
#endif
	return sp;
    } else {						// accessing another coroutine
#if defined( __U_SWAPCONTEXT__ )
	return (void *)(((ucontext_t *)context)->uc_mcontext.
#if defined( __i386__ )
#if defined( __linux__ )
	    gregs[REG_ESP]);
#elif defined( __freebsd__ )
	    mc_esp;
#else
	    #error uC++ : internal error, unsupported architecture
#endif // __linux__
#elif defined( __linux__ ) && defined( __x86_64__ )
	    gregs[REG_RSP]);
#elif defined( __linux__ ) && defined( __ia64__ )
	    sc_gr[12]);
#elif defined( __solaris__ ) && defined( __sparc__ )
	    (void*)((char*)gregs[REG_SP] + STACK_BIAS));
#elif defined( __irix__ ) &&  defined( __mips__ )
	    gregs[CTX_SP]);
#else
	    #error uC++ : internal error, unsupported architecture
#endif
#else
#if defined( __sparc__ )
	return (void*)((char*)((uContext_t *)context)->SP + STACK_BIAS);
#else
	return ((uContext_t *)context)->SP;
#endif // __sparc__
#endif
    } // if
} // uMachContext::stackPointer


#if defined( __ia64__ )
void *uMachContext::registerStackPointer() const {
#if defined( __U_SWAPCONTEXT__ )
    if ( &uThisCoroutine() == this ) {			// accessing myself ?
#if defined( __INTEL_COMPILER )
	return (void*)__getReg( _IA64_REG_AR_BSP );
#else
	void *bsp;					// use my current stack value
	asm("mov %0 = ar.bsp" : "=r" (bsp) : );
	return bsp;
#endif // __INTEL_COMPILER
    } else {						// accessing another coroutine
	return (void *)(((ucontext_t *)context)->uc_mcontext.sc_ar_bsp);
    } // if
#else
    return ((uContext_t *)context)->BSP;
#endif
} // uMachContext::registerStackPointer
#endif // __ia64__


ptrdiff_t uMachContext::stackFree() const {
    return (char *)stackPointer() - (char *)limit;
} // uMachContext::stackFree


ptrdiff_t uMachContext::stackUsed() const {
    ptrdiff_t amt = (char *)base - (char *)stackPointer();
    return amt < 0 ? -amt : amt;			// abs, for stacks growing up
} // uMachContext::stackUsed


void uMachContext::verify() {
    // Ignore the boot task as it uses the UNIX stack.
    if ( storage == ((uBaseTask *)uKernelModule::bootTask)->storage ) return;

    if ( stackPointer() < limit ) {
	uAbort( "Stack overflow detected: stack pointer 0x%p below limit 0x%p.\n"
		"Possible cause is allocation of large stack frame(s) and/or deep call stack.",
		stackPointer(), limit );
    } else if ( (char *)stackPointer() + 4 * 1024 < limit ) {
	fprintf( stderr, "WARNING within 4K of stack limit: stack pointer 0x%p and limit 0x%p.\n"
		 "Possible cause is allocation of large stack frame(s) and/or deep call stack.\n",
		 stackPointer(), limit );
    } else if ( stackPointer() > base ) {
	uAbort( "Stack underflow detected.\n"
		"Possible cause is corrupted stack frame via overwriting memory." );
    } else if ( ( *(unsigned int *)base != magic() )
#ifdef __U_DEBUG__
		|| ( *(unsigned int *)((char *)limit - 16) != magic() )
#endif // __U_DEBUG__
	       ) {
	uAbort( "Stack corruption detected.\n"
		"Possible cause is corrupted stack frame via overwriting memory." );
#if defined( __ia64__ )
    } else if ( registerStackPointer() >= stackPointer() ) {
        // on ia64 the stack grows from both ends; when the two stack pointers cross, we have overflow
	uAbort( "Stack overflow detected: stack pointer 0x%p at or below register stack pointer 0x%p.\n"
		"Possible cause is allocation of large stack frame(s) and/or deep call stack.",
		stackPointer(), registerStackPointer() );
#endif // __ia64__
    } // if
} // uMachContext::verify


void *uMachContext::rtnAdr( void (*rtn)() ) {
#if defined( __linux__ ) && defined( __ia64__ )
    if ( ! rtn ) return NULL;

    struct TOC {
	void *rtn;
	void *toc;
    };

    return ((TOC *)rtn)->rtn;
#else
    return (void *)rtn;
#endif
} // uMachContext::rtnAdr


#if defined( __linux__ ) && defined( __ia64__ )
void *uMachContext::gpAdr( void (*rtn)() ) {
    if ( ! rtn ) return NULL;

    struct TOC {
	void *rtn;
	void *toc;
    };

    return ((TOC *)rtn)->toc;
} // uMachContext::gpAdr
#endif


// Local Variables: //
// compile-command: "gmake install" //
// End: //
