//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 1994
// 
// uCluster.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Mon Mar 14 17:34:24 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Sep 11 08:26:35 2007
// Update Count     : 476
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__


#include <uC++.h>
#include <uIOcntl.h>
#include <uProfiler.h>
//#include <uDebug.h>

#if defined( __solaris__ ) && ! defined( __U_PTHREAD__ )
#include <sys/lwp.h>					// needed for _lwp_kill
#endif // __solaris__ && ! __U_PTHREAD__


using namespace UPP;


//######################### uCluster #########################


void uCluster::wakeProcessor( uPid_t pid ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "uCluster::wakeProcessor: waking processor %lu\n", (unsigned long)pid );
#endif // __U_DEBUG_H__

#if defined( __U_PTHREAD__ )
    uRealPthread::pthread_kill( pid, SIGUSR1 );
#elif defined( __solaris__ )
    _lwp_kill( pid, SIGUSR1 );
#else
    kill( pid, SIGUSR1 );				// wake it up, as it is probably sleeping
#endif
} // uCluster::wakeProcessor


void uCluster::processorPause() {
    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );

    // Check the ready queue to make sure that no task managed to slip onto the
    // queue since the processor last checked.

    readyIdleTaskLock.acquire();
    if ( readyTasksEmpty() && uThisProcessor().external.empty() ) {
	// stop generating SIGALRM signals on this processor until woken up

	uThisProcessor().setContextSwitchEvent( 0 );	// turn off context-switching

	// Block any SIGALRM signals from arriving.
	sigset_t new_mask, mask;
	sigemptyset( &new_mask );
	sigaddset( &new_mask, SIGALRM );
	sigaddset( &new_mask, SIGUSR1 );
	if ( sigprocmask( SIG_BLOCK, &new_mask, &mask ) == -1 ) {
	    uAbort( "internal error, sigprocmask" );
	} // if

	if ( THREAD_GETMEM( inKernelRF ) ) {		// in kernel roll-forward flag on ?
	        readyIdleTaskLock.release();
		if ( sigprocmask( SIG_SETMASK, &mask, NULL ) == -1 ) {
		    uAbort( "internal error, sigprocmask" );
		} // if
		uKernelModule::rollForward( true );	// make sure to do chores
	} else {
	    idleProcessors.addTail( &(uThisProcessor().idleRef) );
	    readyIdleTaskLock.release();

	    // Install the old signal mask and wait for a signal to arrive.

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uCluster &)0x%p.processorPause, before sigpause\n", this );
#endif // __U_DEBUG_H__
	    sigsuspend( &mask );
	    if ( sigprocmask( SIG_SETMASK, &mask, NULL ) == -1 ) {
		uAbort( "internal error, sigprocmask" );
	    } // if
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uCluster &)0x%p.processorPause, after sigpause\n", this );
#endif // __U_DEBUG_H__

	    // A UNIX process may be woken by any signal, e.g. SIGCHLD, so it is
	    // necessary to check and remove the processor from the idle queue.
	    // Normally a processor is removed in makeTaskReady.

	    if ( uThisProcessor().idle() ) {
		readyIdleTaskLock.acquire();
		if ( uThisProcessor().idle() ) {
		    idleProcessors.remove( &(uThisProcessor().idleRef) );
		} // if
		readyIdleTaskLock.release();
	    } // if

	    // Just woken up after an alarm but in kernel/library code, so no
	    // actual popping from the event list took place in the sigalarm
	    // handler (i.e., signal alarm handler just ignored the interrupt).
	    // Therefore, do a roll-forward to ensure that any necessary events
	    // are popped from the event list.

	    uKernelModule::rollForward( true );
	} // if

	// Reset the context-switch

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uCluster &)0x%p.processorPause, reset timeslice:%d\n", this, uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__
	uThisProcessor().setContextSwitchEvent( uThisProcessor().getPreemption() );
    } else {
	readyIdleTaskLock.release();
    } // if

    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );
} // uCluster::processorPause


void uCluster::makeProcessorIdle( uProcessor &processor ) {
    readyIdleTaskLock.acquire();
    idleProcessors.addTail( &(processor.idleRef) );
    readyIdleTaskLock.release();
} // uCluster::makeProcessorIdle


void uCluster::makeProcessorActive( uProcessor &processor ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.makeProcessorActive(1)\n", this );
#endif // __U_DEBUG_H__
    readyIdleTaskLock.acquire();
    if ( processor.idle() ) {				// processor on idle queue ?
	idleProcessors.remove( &(processor.idleRef) );
    } // if
    readyIdleTaskLock.release();
} // uCluster::makeProcessorActive


void uCluster::makeProcessorActive() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.makeProcessorActive(2)\n", this );
#endif // __U_DEBUG_H__
    readyIdleTaskLock.acquire();
    if ( ! readyTasks->empty() && ! idleProcessors.empty() ) {
	uPid_t pid = idleProcessors.dropHead()->processor().pid;
	readyIdleTaskLock.release();			// don't hold lock while sending SIGALRM
	wakeProcessor( pid );
    } else {
	readyIdleTaskLock.release();
    } // if
} // uCluster::makeProcessorActive


void uCluster::makeTaskReady( uBaseTask &readyTask ) {
    readyIdleTaskLock.acquire();
    if ( &readyTask.bound != NULL ) {			// task bound to a specific processor ?
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uCluster &)0x%p.makeTaskReady(1): task %.256s (0x%p) makes task %.256s (0x%p) ready\n",
		  this, uThisTask().getName(), &uThisTask(), readyTask.getName(), &readyTask );
#endif // __U_DEBUG_H__
	uProcessor *p = &readyTask.bound;		// optimization
	p->external.addTail( &(readyTask.readyRef) );	// add task to end of special ready queue
#ifdef __U_MULTI__
	if ( p->idle() ) {				// processor on idle queue ?
	    idleProcessors.remove( &(p->idleRef) );
	    uPid_t pid = p->pid;
	    readyIdleTaskLock.release();		// don't hold lock while sending SIGALRM
	    wakeProcessor( pid );
	} else {
	    readyIdleTaskLock.release();
	} // if
#else
	readyIdleTaskLock.release();
#endif // __U_MULTI__
    } else {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uCluster &)0x%p.makeTaskReady(2): task %.256s (0x%p) makes task %.256s (0x%p) ready\n",
		   this, uThisTask().getName(), &uThisTask(), readyTask.getName(), &readyTask );
#endif // __U_DEBUG_H__
#ifdef __U_MULTI__
	// Wake up an idle processor if the ready task is migrating to another
	// cluster with idle processors or if the ready task is on the same
	// cluster but the ready queue of that cluster is not empty. This check
	// prevents a single task on a cluster, which does a yield, from
	// unnecessarily waking up a processor that has no work to do.

	if ( ! idleProcessors.empty() && ( &uThisCluster() != this || ! readyTasks->empty() ) ) {
	    readyTasks->add( &(readyTask.readyRef) ); // add task to end of cluster ready queue
	    uPid_t pid = idleProcessors.dropHead()->processor().pid;
	    readyIdleTaskLock.release();		// don't hold lock while sending SIGALRM
	    wakeProcessor( pid );
	} else {
	    readyTasks->add( &(readyTask.readyRef) ); // add task to end of cluster ready queue
	    readyIdleTaskLock.release();
	} // if
#else
	readyTasks->add( &(readyTask.readyRef) );	// add task to end of cluster ready queue
	readyIdleTaskLock.release();
#endif // __U_MULTI__
    } // if
} // uCluster::makeTaskReady


void uCluster::readyTaskRemove( uBaseTaskDL *node ) {
    readyIdleTaskLock.acquire();
    readyTasks->remove( node );
    readyIdleTaskLock.release();
} // uCluster::readyTaskRemove


uBaseTask &uCluster::readyTaskTryRemove() {
    // Select a task from the ready queue of this cluster if there are no ready
    // tasks, return the nil pointer.

    uBaseTask *task;

    readyIdleTaskLock.acquire();
    if ( ! readyTasksEmpty() ) {
	task = &(readyTasks->drop()->task());
    } else {
	task = NULL;
    } // if
    readyIdleTaskLock.release();
    return *task;
} // uCluster::readyTaskTryRemove


void uCluster::taskAdd( uBaseTask &task ) {
    readyIdleTaskLock.acquire();
    tasksOnCluster.addTail( &(task.clusterRef) );
    if ( &task.bound == NULL ) readyTasks->addInitialize( tasksOnCluster ); // processor task is not part of normal initialization
    readyIdleTaskLock.release();
} // uCluster::taskAdd


void uCluster::taskRemove( uBaseTask &task ) {
    readyIdleTaskLock.acquire();
    tasksOnCluster.remove( &(task.clusterRef) );
    if ( &task.bound == NULL ) readyTasks->removeInitialize( tasksOnCluster ); // processor task is not part of normal initialization
    readyIdleTaskLock.release();
} // uCluster::taskRemove


void uCluster::taskReschedule( uBaseTask &task ) {
    readyIdleTaskLock.acquire();
    readyTasks->rescheduleTask( &(task.clusterRef), tasksOnCluster );
    readyIdleTaskLock.release();
} // uCluster::taskReschedule


void uCluster::processorAdd( uProcessor &processor ) {
    processorsOnClusterLock.acquire();
    processorsOnCluster.addTail( &(processor.processorRef) );
    processorsOnClusterLock.release();
} // uCluster::processorAdd


void uCluster::processorRemove( uProcessor &processor ) {
    processorsOnClusterLock.acquire();
    processorsOnCluster.remove( &(processor.processorRef) );
    processorsOnClusterLock.release();
} // uCluster::processorRemove


void uCluster::closeFD( int fd ) {
    NBIO->closeFD( fd );
} // uCluster::closeFD


void uCluster::createCluster( unsigned int stackSize, const char *name ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.createCluster\n", this );
#endif // __U_DEBUG_H__

#ifdef __U_DEBUG__
#ifdef __U_MULTI__
    debugIgnore = false;
#else
    debugIgnore = true;
#endif // __U_MULTI__
#endif // __U_DEBUG__

    setName( name );
    setStackSize( stackSize );

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif // __U_LOCALDEBUGGER_H__

    uKernelModule::globalClusterLock->acquire();
    uKernelModule::globalClusters->addTail( &globalRef );
    uKernelModule::globalClusterLock->release();

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->createCluster( *this );
#endif // __U_LOCALDEBUGGER_H__

    if ( readyTasks == NULL ) {
	readyTasks = new uDefaultScheduler;
	defaultReadyTasks = true;
    } else {
	defaultReadyTasks = false;
    } // if

#ifdef __U_MULTI__
    NBIO = new uNBIO;
#endif // __U_MULTI__

    if ( uProfiler::uProfiler_registerCluster ) {
	(*uProfiler::uProfiler_registerCluster)( uProfiler::profilerInstance, *this );
    } // if
} // uCluster::createCluster


uCluster::uCluster( unsigned int stackSize, const char *name ) : globalRef( *this ), readyTasks( NULL ) {
    createCluster( stackSize, name );
} // uCluster::uCluster


uCluster::uCluster( const char *name ) : globalRef( *this ), readyTasks( NULL ) {
    createCluster( uDefaultStackSize(), name );
} // uCluster::uCluster


uCluster::uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, unsigned int stackSize, const char *name ) : globalRef( *this ), readyTasks( &ReadyQueue ) {
    createCluster( stackSize, name );
} // uCluster::uCluster


uCluster::uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, const char *name ) : globalRef( *this ), readyTasks( &ReadyQueue ) {
    createCluster( uDefaultStackSize(), name );
} // uCluster::uCluster


uCluster::~uCluster() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.~uCluster\n", this );
#endif // __U_DEBUG_H__

    if ( uProfiler::uProfiler_deregisterCluster ) {
	(*uProfiler::uProfiler_deregisterCluster)( uProfiler::profilerInstance, *this );
    } // if

#ifdef __U_MULTI__
    delete NBIO;
#endif // __U_MULTI__

    uProcessorDL *pr;
    for ( uSeqIter<uProcessorDL> iter(processorsOnCluster); iter >> pr; ) {
	if ( pr->processor().getDetach() ) {		// detached ?
	    delete &pr->processor();			// delete detached processor
#ifdef __U_DEBUG__
	} else {
	    // Must check for processors before tasks because each processor has a
	    // processor task, and hence, there is always a task on the cluster.
	    uAbort( "Attempt to delete cluster %.256s (0x%p) with processor 0x%p still on it.\n"
		    "Possible cause is the processor has not been deleted.",
		    getName(), this, &(pr->processor()) );
#endif // __U_DEBUG__
	} // if
    } // for

#ifdef __U_DEBUG__
    uBaseTaskDL *tr;
    tr = tasksOnCluster.head();
    if ( tr != NULL ) {
	uAbort( "Attempt to delete cluster %.256s (0x%p) with task %.256s (0x%p) still on it.\n"
		"Possible cause is the task has not been deleted.",
		getName(), this, tr->task().getName(), &(tr->task()) );
    } // if
#endif // __U_DEBUG__

    if ( defaultReadyTasks ) {				// delete if cluster allocated it
	delete readyTasks;
    } // if

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->destroyCluster( *this );
#endif // __U_LOCALDEBUGGER_H__

    uKernelModule::globalClusterLock->acquire();
    uKernelModule::globalClusters->remove( &globalRef );
    uKernelModule::globalClusterLock->release();
} // uCluster::~uCluster


void uCluster::taskResetPriority( uBaseTask &owner, uBaseTask &calling ) { // TEMPORARY
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.taskResetPriority, owner:0x%p, calling:0x%p, owner's cluster:0x%p\n", this, &owner, &calling, owner.currCluster );
#endif // __U_DEBUG_H__
    readyIdleTaskLock.acquire();
    if ( &uThisCluster() == owner.currCluster ) {
	if ( readyTasks->checkPriority( owner.readyRef, calling.readyRef ) ) {
	    readyTasks->resetPriority( owner.readyRef, calling.readyRef );
	} // if
    } // if
    readyIdleTaskLock.release();
} // uCluster::taskResetPriority


void uCluster::taskSetPriority( uBaseTask &owner, uBaseTask &calling ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCluster &)0x%p.taskSetPriority, owner:0x%p, calling:0x%p, owner's cluster:0x%p\n", this, &owner, &calling, owner.currCluster );
#endif // __U_DEBUG_H__
    readyIdleTaskLock.acquire();
    readyTasks->resetPriority( owner.readyRef, calling.readyRef );
    readyIdleTaskLock.release();
} // uCluster::taskSetPriority


const int uCluster::readSelect = 1;
const int uCluster::writeSelect = 2;
const int uCluster::exceptSelect = 4;


int uCluster::select( int fd, int rwe, timeval *timeout ) {
    int retcode;					// create fake structures for null closure

    uIOaccess access;
    access.fd = fd;
    access.poll.setStatus( uPoll::NeverPoll );

    struct Select : public uIOClosure {
	int action( int rwe ) { return 0; }
	Select( uIOaccess &access, int retcode ) : uIOClosure( access, retcode ) {}
    } selectClosure( access, retcode );

    return NBIO->select( selectClosure, rwe, timeout );
} // uCluster::select


// Local Variables: //
// compile-command: "gmake install" //
// End: //
