//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.1, Copyright (C) Peter A. Buhr 1994
// 
// uSocket.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 17:06:26 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Feb  8 12:46:19 2006
// Update Count     : 868
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>
#include <uSocket.h>

//#include <uDebug.h>

#include <cstring>					// strerror, memset
#include <unistd.h>					// read, write, close, etc.
#include <sys/socket.h>
#include <sys/un.h>


#ifndef SUN_LEN
#define SUN_LEN(su) (sizeof(*(su)) - sizeof((su)->sun_path) + strlen((su)->sun_path))
#endif


//######################### uSocket #########################


void uSocket::uCreateSocket( int domain, int type, int protocol ) {
    for ( ;; ) {
	access.fd = ::socket( domain, type, protocol );
      if ( access.fd != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( access.fd == -1 ) {
        // The exception should be caught by constructor in
	// uSocketServer or uSocketClient and throw a different exception
	// indicating the failure to create a server or a client.  However, it
	// cannot be implemented because the initialization list in a
	// constructor cannot be encapsulated in a try block.  Actually, there
	// is a feature in c++ allowing catch clause to handle exception
	// propoagated out of the initialization list but g++ does not have it
	// yet.

        _Throw OpenFailure( *this, domain, type, protocol, "unable to open socket" );
    } // if

    // Sockets always treated as non-blocking.
    access.poll.setStatus( uPoll::AlwaysPoll );
    access.poll.setPollFlag( access.fd );

    // Sockets always have local-address reuse enable.
    const int enable = 1;				// 1 => enable option
    if ( setsockopt( access.fd, SOL_SOCKET, SO_REUSEADDR,  &enable, sizeof(enable) ) == -1 ) {
        _Throw OpenFailure( *this, domain, type, protocol, "unable to set socket-option" );
    } // if
} // uSocket::uCreateSocket


uSocket::uSocket( int domain, int type, int protocol ) : domain( domain ), type( type ), protocol( protocol ) {
    uCreateSocket( domain, type, protocol );
} // uSocket::uSocket


uSocket::~uSocket() {
    int code;

    uThisCluster().closeFD( access.fd );
    for ( ;; ) {
	code = ::close( access.fd );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
        if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, "unable to close socket" );
    } // if
} // uSocket::~uSocket


//######################### uSocketIO #########################


void uSocketIO::inetAddr::getinetAddr( unsigned short port, const char *name ) {
    if ( name[0] != '\0' ) {				// no name ?
#if defined( __darwin__ ) || defined( __freebsd__ )
	struct addrinfo hints = { 0, AF_INET, 0, 0, 0, 0, 0, 0 };
	struct addrinfo *addr;
	int rc;
	rc = getaddrinfo( name, NULL, &hints, &addr );
	if ( rc != 0 ) {
	    code = 1;
	    return;
	} // if
	memcpy( &sin_addr.s_addr, addr->ai_addr, addr->ai_addrlen );
	freeaddrinfo( addr );
#else
	hostent hp;
	int herrno = 0;
	char buffer[8 * 1024];				// best guess at size
	hostent *rc;

	for ( int i = 0; i < 5; i += 1 ) {		// N tries
#if defined( __solaris__ ) || defined( __irix__ )
	    rc = gethostbyname_r( name, &hp, buffer, sizeof(buffer), &herrno );
	    if ( rc != 0 ) break;			// success ?
#elif defined( __linux__ )
	    gethostbyname_r( name, &hp, buffer, sizeof(buffer), &rc, &herrno );
	    if ( rc != 0 ) break;			// success ?
#else
	    #error uC++ : internal error, unsupported architecture
#endif
	    if ( herrno != TRY_AGAIN ) break;		// failure ?
	    sleep( 1 );					// TRY_AGAIN => delay and try again
	} // for
	if ( rc == 0 ) {
	    code = 1;
	    return;
	} // if
	memcpy( &sin_addr.s_addr, hp.h_addr, hp.h_length );
#endif // __darwin__ || __freebsd__
    } else {
	sin_addr.s_addr = htonl( INADDR_ANY );
    } // if

    code = 0;
    sin_family = AF_INET;
    sin_port = htons( port );
    memset( &sin_zero, '\0', sizeof(sin_zero) );
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketIO &)0x%p::inetAddr.getinetAddr name:%s, sin_family:%d sin_port:%d (%x), sin_addr:%d.%d.%d.%d\n",
	       this, name, sin_family, sin_port, sin_port,
	       sin_addr._S_un._S_un_b.s_b1, sin_addr._S_un._S_un_b.s_b2, sin_addr._S_un._S_un_b.s_b3, sin_addr._S_un._S_un_b.s_b4 );
#endif // __U_DEBUG_H__
} // uSocketIO::inetAddr::getinetAddr


uSocketIO::inetAddr::inetAddr( unsigned short port, const char *name ) {
    strcpy( _name, name );				// copy name
    getinetAddr( port, name );
} // uSocketIO::inetAddr::inetAddr


int uSocketIO::_send( char *buf, int len, int flags ) {
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.setPollFlag( access.fd );
    int code = ::send( access.fd, buf, len, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.clearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_send


uSocketIO::uSocketIO( uIOaccess &acc, sockaddr *saddr ) : uFileIO( acc ), saddr( saddr ) {
} // uSocketIO::uSocketIO


const struct sockaddr *uSocketIO:: getsockaddr() {
    return saddr;
} // uSocketIO::getsockname


int uSocketIO::getsockname( sockaddr *name, socklen_t *len ) {
    return ::getsockname( access.fd, name, len );
} // uSocketIO::getsockname


int uSocketIO::getpeername( sockaddr *name, socklen_t *len ) {
    return ::getpeername( access.fd, name, len );
} // uSocketIO::getpeername


int uSocketIO::send( char *buf, int len, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _send( buf, len, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::writeSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( access.fd, mask, &t ) == 0 ) { // timeout ?
		    writeTimeout( buf, len, flags, NULL, 0, timeout, "send" );
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	writeFailure( buf, len, flags, NULL, 0, timeout, "send" );
    } // if

    return code;
} // uSocketIO::send


int uSocketIO::_sendto( char *buf, int len, int flags, sockaddr *to, socklen_t tolen ) {
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.setPollFlag( access.fd );
    int code = ::sendto( access.fd, buf, len, flags, (sockaddr *)to, tolen );
    int terrno = errno;					// preserve errno across calls
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketIO &)0x%p._sendto attempting send to fd:%d, buf:0x%p, len:%d, flags:0x%x, to:0x%p, tolen:%d code:%d errno:%d\n",
	       this, access.fd, buf, len, flags, to, tolen, code, errno );
#endif // __U_DEBUG_H__
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.clearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_sendto


int uSocketIO::sendto( char *buf, int len, int flags, uDuration *timeout ) {
    return sendto( buf, len, saddr, saddrlen, flags, timeout );
} // uSocketIO::sendto


int uSocketIO::sendto( char *buf, int len, struct sockaddr *to, socklen_t tolen, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _sendto( buf, len, flags, to, tolen );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::writeSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( access.fd, mask, &t ) == 0 ) { // timeout ?
		    writeTimeout( buf, len, flags, NULL, 0, timeout, "sendto" );
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	writeFailure( buf, len, flags, NULL, 0, timeout, "sendto" );
    } // if

    return code;
} // uSocketIO::sendto


int uSocketIO::_recv( char *buf, int len, int flags ) {
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.setPollFlag( access.fd );
    int code = ::recv( access.fd, buf, len, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.clearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_recv


int uSocketIO::recv( char *buf, int len, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _recv( buf, len, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::readSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( access.fd, mask, &t ) == 0 ) { // timeout ?
		    readTimeout( buf, len, flags, NULL, NULL, timeout, "recv" );
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	readFailure( buf, len, flags, NULL, NULL, timeout, "recv" );
    } // if

    return code;
} // uSocketIO::recv


int uSocketIO::_recvfrom( char *buf, int len, int flags, sockaddr *from, socklen_t *fromlen ) {
#if defined( __irix__ )
    // TEMPORARY: sets fromlen to 0 when EWOULDBLOCK
    int tmp;
    if ( fromlen != 0 ) tmp = *fromlen;			// save *fromlen, as it may be set to 0 after each attempt
#endif // __irix__
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.setPollFlag( access.fd );
    int code = ::recvfrom( access.fd, buf, len, flags, (sockaddr *)from, (socklen_t *) fromlen );
    int terrno = errno;					// preserve errno across calls
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketIO &)0x%p._recvfrom attempting read from fd:%d, buf:0x%p, len:%d, flags:0x%x, from:0x%p, fromlen:0x%p, code:%d, errno:%d\n",
	       this, access.fd, buf, len, flags, from, fromlen, code, errno );
#endif // __U_DEBUG_H__
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.clearPollFlag( access.fd );
    errno = terrno;
#if defined( __irix__ )
    // TEMPORARY: sets fromlen to 0 when EWOULDBLOCK
    if ( fromlen != 0 && *fromlen == 0 ) *fromlen = tmp; // reset *fromlen after each attempt
#endif // __irix__
    return code;
} // uSocketIO::_recvfrom


int uSocketIO::recvfrom( char *buf, int len, int flags, uDuration *timeout ) {
    saddrlen = baddrlen;				// set to receive buffer size
    return recvfrom( buf, len, saddr, &saddrlen, flags, timeout );
} // uSocketIO::recvfrom


int uSocketIO::recvfrom( char *buf, int len, struct sockaddr *from, socklen_t *fromlen, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _recvfrom( buf, len, flags, from, fromlen );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::readSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( access.fd, mask, &t ) == 0 ) { // timeout ?
		    readTimeout( buf, len, flags, from, fromlen, timeout, "recvfrom" );
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	readFailure( buf, len, flags, from, fromlen, timeout, "recvfrom" );
    } // if

    return code;
} // uSocketIO::recvfrom


int uSocketIO::_recvmsg( struct msghdr *msg, int flags ) {
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.setPollFlag( access.fd );
    int code = ::recvmsg( access.fd, msg, flags );
    int terrno = errno;					// preserve errno across calls
    if ( access.poll.getStatus() == uPoll::PollOnDemand ) access.poll.clearPollFlag( access.fd );
    errno = terrno;
    return code;
} // uSocketIO::_recvmsg


int uSocketIO::recvmsg( struct msghdr *msg, int flags, uDuration *timeout ) {
    int code;

    for ( ;; ) {
	code = _recvmsg( msg, flags );
      if ( code != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::readSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( access.fd, mask, &t ) == 0 ) { // timeout ?
		    readTimeout( (const char *)msg, 0, flags, NULL, NULL, timeout, "recvmsg" );
		} // if
	    } // if
	} // if
    } // for
    if ( code == -1 ) {
	readFailure( (const char *)msg, 0, flags, NULL, NULL, timeout, "recvmsg" );
    } // if

    return code;
} // uSocketIO::recvmsg


//######################### uSocketServer #########################


void uSocketServer::acceptor() {
    acceptorCnt += 1;
} // uSocketServer::acceptor


void uSocketServer::unacceptor() {
    acceptorCnt -= 1;
} // uSocketServer::unacceptor


int uSocketServer::_accept( struct sockaddr *adr, socklen_t *len ) {
    int tmp = 0;
    if ( len != NULL ) tmp = *len;			// save *len, as it may be set to 0 after each attempt
    if ( socket.access.poll.getStatus() == uPoll::PollOnDemand ) socket.access.poll.setPollFlag( socket.access.fd );
    int fd = ::accept( socket.access.fd, (sockaddr *)adr, len );
    int terrno = errno;					// preserve errno across calls
    if ( socket.access.poll.getStatus() == uPoll::PollOnDemand ) socket.access.poll.clearPollFlag( socket.access.fd );
    errno = terrno;
    if ( len != NULL && *len == 0 ) *len = tmp;		// reset *len after each attempt
    return fd;
} // uSocketServer::_accept


void uSocketServer::uCreateSocketServer1( const char *name, int type, int protocol, int backlog ) {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer1 attempting binding to name:%s\n", this, name );
#endif // __U_DEBUG_H__

    if ( strlen( name ) >= sizeof(((sockaddr_un *)saddr)->sun_path) ) {
        _Throw OpenFailure( *this, 0, name, AF_UNIX, type, protocol, backlog, "socket name too long" );
    } // if

    ((sockaddr_un *)saddr)->sun_family = AF_UNIX;
    strcpy( ((sockaddr_un *)saddr)->sun_path, name );
    saddrlen = SUN_LEN( (sockaddr_un *)saddr );
    baddrlen = sizeof(sockaddr_un);

    for ( ;; ) {
	code = ::bind( socket.access.fd, saddr, saddrlen );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
        _Throw OpenFailure( *this, 0, name, AF_UNIX, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    code = ::listen( socket.access.fd, backlog );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    _Throw OpenFailure( *this, 0, name, AF_UNIX, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer1 binding to name:%s\n", this, name );
#endif // __U_DEBUG_H__
} // uSockeServer::uCreateSocketServer1


void uSocketServer::uCreateSocketServer2( unsigned short port, int type, int protocol, int backlog ) {
    int code;

    if ( ((inetAddr *)saddr)->code > 0 ) {		// problem generating internet address ?
        _Throw OpenFailure( *this, port, ((inetAddr *)saddr)->_name, AF_INET, type, protocol, backlog, "internet name lookup failure" );
    } // if

    baddrlen = saddrlen = sizeof(sockaddr_in);

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer2 attempting binding to port:%d, name:%s\n", this, port, ((inetAddr *)saddr)->_name );
#endif // __U_DEBUG_H__

    for ( ;; ) {
	code = ::bind( socket.access.fd, saddr, saddrlen );
      if ( code != -1 ) break;
      if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code == -1 ) {
	_Throw OpenFailure( *this, port, ((inetAddr *)saddr)->_name, AF_INET, type, protocol, backlog, "unable to bind name to socket" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    code = ::listen( socket.access.fd, backlog );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    _Throw OpenFailure( *this, port, ((inetAddr *)saddr)->_name, AF_INET, type, protocol, backlog, "unable to listen on socket" );
	} // if
    } // if

    acceptorCnt = 0;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer2 binding to port:%d, name:%s\n", this, port, ((inetAddr *)saddr)->_name );
#endif // __U_DEBUG_H__
} // uSocketServer::uCreateSocketServer2


void uSocketServer::uCreateSocketServer3( unsigned short *port, int type, int protocol, int backlog ) {
    uCreateSocketServer2( 0, type, protocol, backlog );	// 0 port number => select an used port

    getsockname( saddr, &saddrlen );			// insert unsed port number into address ("bind" does not do it)
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketServer &)0x%p.uCreateSocketServer3 binding to port:%d\n", this, ntohs( ((sockaddr_in *)saddr)->sin_port ) );
#endif // __U_DEBUG_H__
    *port = ntohs( ((sockaddr_in *)saddr)->sin_port );
} // uSocketServer::uCreateSocketServer3


void uSocketServer::readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcpy( msg, "socket " ); strcat( msg, op ); strcat( msg, " fails" );
    _Throw uSocketServer::ReadFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::readFailure


void uSocketServer::readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::ReadTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::readTimeout


void uSocketServer::writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketServer::WriteFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::writeFailure


void uSocketServer::writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::WriteTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketServer::writeTimeout


void uSocketServer::readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketServer::ReadFailure( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketServer::readFailure


void uSocketServer::readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::ReadTimeout( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketServer::readTimeout


void uSocketServer::writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketServer::WriteFailure( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketServer::writeFailure


void uSocketServer::writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketServer::WriteTimeout( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketServer::writeTimeout


uSocketServer::uSocketServer( const char *name, int type, int protocol, int backlog ) :
	uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketServer1( name, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::uSocketServer( unsigned short port, int type, int protocol, int backlog ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketServer2( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::uSocketServer( unsigned short *port, int type, int protocol, int backlog ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( 0 ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketServer3( port, type, protocol, backlog );
} // uSocketServer::uSocketServer


uSocketServer::~uSocketServer() {
    delete saddr;
    if ( acceptorCnt != 0 ) {
	if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, acceptorCnt, "closing socket server with outstanding acceptor(s)" );
    } // if
} // uSocketServer::~uSocketServer


void uSocketServer::setClient( struct sockaddr *addr, socklen_t len ) {
#ifdef __U_DEBUG__
    if ( len > baddrlen ) {
	uAbort( "(uSocketServer &)0x%p.setClient( addr:0x%p, len:%d ) : New server address to large.", this, addr, len );
    } // if
#endif // __U_DEBUG__
    memcpy( saddr, addr, len );
} // uSocketServer::setClient


void uSocketServer::getClient( struct sockaddr *addr, socklen_t *len ) {
#ifdef __U_DEBUG__
    if ( *len < baddrlen ) {
	uAbort( "(uSocketServer &)0x%p.getClient( addr:0x%p, len:%d ) : Area for server address to small.", this, addr, *len );
    } // if
#endif // __U_DEBUG__
    *len = baddrlen;
    memcpy( addr, saddr, baddrlen );
} // uSocketServer::getServer


//######################### uSocketAccept #########################


const uSocket &uSocketAccept::sock() const {
    return socketserver.socket;
} // uSocketAccept::sock


void uSocketAccept::uCreateSocketAcceptor( uDuration *timeout, struct sockaddr *adr, socklen_t *len ) {
    baddrlen = saddrlen = socketserver.saddrlen;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept before accept\n", this );
#endif // __U_DEBUG_H__

    for ( ;; ) {
	access.fd = socketserver._accept( adr, len );
      if ( access.fd != -1 ) break;
      if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	if ( errno == U_EWOULDBLOCK ) {
	    int mask = uCluster::readSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( socketserver.socket.access.fd, mask );
	    } else {
		timeval t = *timeout;		// convert to timeval for select
		if ( uThisCluster().select( socketserver.socket.access.fd, mask, &t ) == 0 ) { // timeout ?
#ifdef __U_DEBUG_H__
		    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept timeout fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__
		    _Throw OpenTimeout( *this, timeout, adr, len, "timeout during accept" );
		} // if
	    } // if
	} // if
    } // for

    if ( access.fd == -1 ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept accept failure fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__
	_Throw OpenFailure( *this, timeout, adr, len, "unable to accept connection on socket server" );
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.uSocketAccept after accept fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__

    // On some UNIX systems the file descriptor created by accept inherits the
    // non-blocking characteristic from the base socket; on other system this
    // does not seem to occur, so explicitly set the file descriptor to
    // non-blocking.

    access.poll.setStatus( uPoll::AlwaysPoll );
    access.poll.setPollFlag( access.fd );
    openAccept = true;
} // uSocketAccept::uCreateSocketAcceptor


int uSocketAccept::_close() {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.close, enter fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__

    uThisCluster().closeFD( access.fd );
    for ( ;; ) {
	code = ::close( access.fd );
	if ( code != -1 ) break;
	if ( errno != EINTR ) break;			// timer interrupt ?
    } // for
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketAccept &)0x%p.close, exit fd:%d\n", this, access.fd );
#endif // __U_DEBUG_H__
    openAccept = false;

    return code;
} // uSocketAccept::_close


void uSocketAccept::readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::ReadFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::readFailure


void uSocketAccept::readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::ReadTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::readTimeout


void uSocketAccept::writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::WriteFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::writeFailure


void uSocketAccept::writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::WriteTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketAccept::writeTimeout


void uSocketAccept::readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::ReadFailure( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketAccept::readFailure


void uSocketAccept::readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::ReadTimeout( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketAccept::readTimeout


void uSocketAccept::writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketAccept::WriteFailure( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketAccept::writeFailure


void uSocketAccept::writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketAccept::WriteTimeout( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketAccept::writeTimeout


uSocketAccept::uSocketAccept( uSocketServer &s, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( NULL ), adr( adr ), len( len ) {
    uCreateSocketAcceptor( NULL, adr, len );
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::uSocketAccept( uSocketServer &s, uDuration *timeout, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( timeout ), adr( adr ), len( len ) {
    uCreateSocketAcceptor( timeout, adr, len );
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::uSocketAccept( uSocketServer &s, bool doAccept, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( NULL ), adr( adr ), len( len ) {
    if ( doAccept ) {
	uCreateSocketAcceptor( NULL, adr, len );
    } // if
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::uSocketAccept( uSocketServer &s, uDuration *timeout, bool doAccept, struct sockaddr *adr, socklen_t *len ) :
	uSocketIO( access, s.saddr ), socketserver( s ), openAccept( false ), timeout( timeout ), adr( adr ), len( len ) {
    if ( doAccept ) {
	uCreateSocketAcceptor( timeout, adr, len );
    } // if
    socketserver.acceptor();
} // uSocketAccept::uSocketAccept


uSocketAccept::~uSocketAccept() {
    socketserver.unacceptor();
    if ( openAccept && _close() == -1 ) {
	if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, "unable to close socket acceptor" );
    } // if
} // uSocketAccept::~uSocketAccept


void uSocketAccept::accept() {
    accept( timeout );					// use global timeout
} // uSocketAccept::accept


void uSocketAccept::accept( uDuration *timeout ) {
    close();
    uSocketAccept::timeout = timeout;
    uCreateSocketAcceptor( timeout, adr, len );
} // uSocketAccept::accept


void uSocketAccept::close() {
    if ( openAccept && _close() == -1 ) {
	_Throw CloseFailure( *this, "unable to close socket acceptor" );
    } // if
} // uSocketAccept::close


//######################### uSocketClient #########################


void uSocketClient::uCreateSocketClient1( const char *name, uDuration *timeout, int type, int protocol ) {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient1 attempting connection to name:%s\n", this, name );
#endif // __U_DEBUG_H__

    if ( strlen( name ) >= sizeof(((sockaddr_un *)saddr)->sun_path) ) {
	_Throw OpenFailure( *this, 0, name, timeout, AF_UNIX, type, protocol, "socket name too long" );
    } // if

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	strcpy( ((sockaddr_un *)saddr)->sun_path, name );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );
	baddrlen = sizeof(sockaddr_un);

	if ( socket.access.poll.getStatus() == uPoll::PollOnDemand ) socket.access.poll.setPollFlag( socket.access.fd );
	code = ::connect( socket.access.fd, saddr, saddrlen );
	int terrno = errno;				// preserve errno across calls
	if ( socket.access.poll.getStatus() == uPoll::PollOnDemand ) socket.access.poll.clearPollFlag( socket.access.fd );
	errno = terrno;

	int status = 0;
	if ( code == -1 && ( errno == EINPROGRESS || errno == EINTR || errno == U_EWOULDBLOCK ) ) {
	    int mask = uCluster::writeSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( socket.access.fd, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
		    _Throw OpenTimeout( *this, 0, name, timeout, AF_UNIX, type, protocol, "timeout during connect" );
		} // if
	    } // if
	    socklen_t codeLen = sizeof(code);
	    status = ::getsockopt( socket.access.fd, SOL_SOCKET, SO_ERROR, &code, &codeLen );
	} // if

	if ( status == -1 || code == -1 ) {
	    _Throw OpenFailure( *this, 0, name, timeout, AF_UNIX, type, protocol, "unable to connect to socket" );
	} // if
    } else {
	// Create temporary file for the client to communicate through.
	tmpnm = tempnam( NULL, "uC++" );
	if ( strlen( tmpnm ) >= sizeof(((sockaddr_un *)saddr)->sun_path) ) {
	    _Throw OpenFailure( *this, 0, tmpnm, timeout, AF_UNIX, type, protocol, "socket temporary name too long" );
	} // if

	((sockaddr_un *)saddr)->sun_family = AF_UNIX;
	strcpy( ((sockaddr_un *)saddr)->sun_path, tmpnm );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );

	for ( ;; ) {
	    code = ::bind( socket.access.fd, saddr, saddrlen );
	  if ( code != -1 ) break;
	  if ( errno != EINTR ) break;			// timer interrupt ?
	} // for
	if ( code == -1 ) {
	    _Throw OpenFailure( *this, 0, name, timeout, AF_UNIX, type, protocol, "unable to bind name to socket" );
	} // if

	// Now set default address specified by user.
	strcpy( ((sockaddr_un *)saddr)->sun_path, name );
	saddrlen = SUN_LEN( (sockaddr_un *)saddr );
	baddrlen = sizeof(sockaddr_un);
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient1 connection to name:%s\n", this, name );
#endif // __U_DEBUG_H__
} // uSocketClient::uCreateSocketClient1


void uSocketClient::uCreateSocketClient2( unsigned short port, const char *name, uDuration *timeout, int type, int protocol ) {
    int code;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient2 attempting connection to port:%d, name:%s\n", this, port, name );
#endif // __U_DEBUG_H__

    if ( ((inetAddr *)saddr)->code > 0 ) {		// problem generating internet address ?
        _Throw OpenFailure( *this, port, ((inetAddr *)saddr)->_name, timeout, AF_INET, type, protocol, "internet name lookup failure" );
    } // if

    baddrlen = saddrlen = sizeof(sockaddr_in);

    if ( type != SOCK_DGRAM ) {				// connection-oriented
	for ( ;; ) {
	    if ( socket.access.poll.getStatus() == uPoll::PollOnDemand ) socket.access.poll.setPollFlag( socket.access.fd );
	    code = ::connect( socket.access.fd, saddr, saddrlen );
	    int terrno = errno;			// preserve errno across calls
	    if ( socket.access.poll.getStatus() == uPoll::PollOnDemand ) socket.access.poll.clearPollFlag( socket.access.fd );
	    errno = terrno;
	  if ( code != -1 ) break;
	  if ( ! ( errno == EINTR || errno == U_EWOULDBLOCK ) ) break;
	    if ( errno == U_EWOULDBLOCK ) {
		int mask = uCluster::readSelect;
		if ( timeout == NULL ) {
		    uThisCluster().select( socket.access.fd, mask );
		} else {
		    timeval t = *timeout;		// convert to timeval for select
		    if ( uThisCluster().select( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
			_Throw OpenTimeout( *this, port, name, timeout, AF_INET, type, protocol, "timeout during connect" );
		    } // if
		} // if
	    } // if
	} // for
	// Alternate approach for waiting for a non-blocking connect.
	if ( code == -1 && errno == EINPROGRESS ) {
	    int mask = uCluster::writeSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( socket.access.fd, mask );
	    } else {
		timeval t = *timeout;		// convert to timeval for select
		if ( uThisCluster().select( socket.access.fd, mask, &t ) == 0 ) { // timeout ?
		    _Throw OpenTimeout( *this, port, name, timeout, AF_INET, type, protocol, "timeout during connect" );
		} // if
	    } // if
	    code = 0;
	} // if
	
	if ( code == -1 ) {
	    _Throw OpenFailure( *this, port, name, timeout, AF_INET, type, protocol, "unable to connect to socket" );
	} // if
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSocketClient &)0x%p.uCreateSocketClient2 connection to port:%d, name:%s\n", this, port, name );
#endif // __U_DEBUG_H__
} // uSocketClient::uCreateSocketClient2


void uSocketClient::readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::ReadFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::readFailure


void uSocketClient::readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::ReadTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::readTimeout


void uSocketClient::writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::WriteFailure( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::writeFailure


void uSocketClient::writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::WriteTimeout( *this, buf, len, 0, NULL, 0, timeout, msg );
} // uSocketClient::writeTimeout


void uSocketClient::readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::ReadFailure( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketClient::readFailure


void uSocketClient::readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::ReadTimeout( *this, buf, len, flags, from, fromlen, timeout, msg );
} // uSocketClient::readTimeout


void uSocketClient::writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "socket " ), op ), " fails" );
    _Throw uSocketClient::WriteFailure( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketClient::writeFailure


void uSocketClient::writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during socket " ), op );
    _Throw uSocketClient::WriteTimeout( *this, buf, len, flags, to, tolen, timeout, msg );
} // uSocketClient::writeTimeout


uSocketClient::uSocketClient( const char *name, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, NULL, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( const char *name, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new sockaddr_un ), socket( AF_UNIX, type, protocol ) {
    uCreateSocketClient1( name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, ((inetAddr *)saddr)->_name, NULL, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, ((inetAddr *)saddr)->_name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, const char *name, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port, name ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, name, NULL, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::uSocketClient( unsigned short port, const char *name, uDuration *timeout, int type, int protocol ) :
	uSocketIO( socket.access, (sockaddr *)new inetAddr( port, name ) ), socket( AF_INET, type, protocol ) {
    uCreateSocketClient2( port, name, timeout, type, protocol );
} // uSocketClient::uSocketClient


uSocketClient::~uSocketClient() {
    if ( saddr->sa_family == AF_UNIX && socket.type == SOCK_DGRAM ) {
	// Remove temporary file created for communication.
	int code = unlink( tmpnm );
	delete tmpnm;
	if ( code == -1 ) {
	    delete saddr;
	    if ( ! std::uncaught_exception() ) _Throw CloseFailure( *this, "unlink failed for temporary pipe" );
	} // if
    } // if
    delete saddr;
} // uSocketClient::~uSocketClient


void uSocketClient::setServer( struct sockaddr *addr, socklen_t len ) {
#ifdef __U_DEBUG__
    if ( len > baddrlen ) {
	uAbort( "(uSocketClient &)0x%p.setServer( addr:0x%p, len:%d ) : New server address to large.", this, addr, len );
    } // if
#endif // __U_DEBUG__
    memcpy( saddr, addr, len );
} // uSocketClient::setServer


void uSocketClient::getServer( struct sockaddr *addr, socklen_t *len ) {
#ifdef __U_DEBUG__
    if ( *len < baddrlen ) {
	uAbort( "(uSocketClient &)0x%p.getServer( addr:0x%p, len:%d ) : Area for server address to small.", this, addr, *len );
    } // if
#endif // __U_DEBUG__
    *len = baddrlen;
    memcpy( addr, saddr, baddrlen );
} // uSocketClient::getServer


//######################### uSocket (cont) #########################


uSocket::Failure::Failure( const uSocket &socket, const char *const msg ) : uIOFailure( msg ), _socket( socket ) {}

const uSocket &uSocket::Failure::socket() const { return _socket; }

void uSocket::Failure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p, %.256s.", &socket(), message() );
} // uSocket::Failure::defaultTerminate


uSocket::OpenFailure::OpenFailure( const uSocket &socket, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocket::Failure( socket, msg ), domain( domain ), type( type ), protocol( protocol ) {}

void uSocket::OpenFailure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p.uSocket( domain:%d, type:%d, protocol:%d ), %.256s.\nError(%d) : %s.",
	    &socket(), domain, type, protocol, message(), errNo(), strerror( errNo() ) );
} // uSocket::OpenFailure::defaultTerminate


uSocket::CloseFailure::CloseFailure( const uSocket &socket, const char *const msg ) :	uSocket::Failure( socket, msg ) {}

void uSocket::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocket &)0x%p.~uSocket(), %.256s %d.\nError(%d) : %s.",
	    &socket(), message(), socket().access.fd, errNo(), strerror( errNo() ) );
} // uSocket::CloseFailure::defaultTerminate


//######################### uSocketServer (cont) #########################


uSocketServer::Failure::Failure( const uSocketServer &server, const char *const msg ) : uSocket::Failure( server.socket, msg ), _server( server ) {
} // uSocketServer::Failure::Failure

const uSocketServer &uSocketServer::Failure::server() const {
    return _server;
} // uSocketServer::Failure::server

int uSocketServer::Failure::fileDescriptor() const {
    return server().access.fd;
} // uSocketServer::Failure::fileDescriptor

void uSocketServer::Failure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p, %.256s.\nError(%d) : %s.",
	    &server(), message(), errNo(), strerror( errNo() ) );
} // uSocketServer::Failure::defaultTerminate


uSocketServer::OpenFailure::OpenFailure( const uSocketServer &server, unsigned short port, const char *name, int domain, int type, int protocol, int backlog, const char *const msg ) :
	uSocketServer::Failure( server, msg ), port( port ), domain( domain ), type( type ), protocol( protocol ), backlog( backlog ) {
    strcpy( _name, name );				// copy file name
} // uSocketServer::OpenFailure::OpenFailure

const char *uSocketServer::OpenFailure::name() const { return _name; }

void uSocketServer::OpenFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.uSocketServer( port:%d, name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d, backlog:%d ), %.256s.\nError(%d) : %s.",
	    &server(), port, name(), &uThisCluster(), domain, type, protocol, backlog, message(), errNo(), strerror( errNo() ) );
} // uSocketServer::OpenFailure::defaultTerminate


uSocketServer::CloseFailure::CloseFailure( const uSocketServer &server, int acceptorCnt, const char *const msg ) :
	uSocketServer::Failure( server, msg ), acceptorCnt( acceptorCnt ) {
} // uSocketServer::CloseFailure::CloseFailure

void uSocketServer::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.~uSocketServer(), %.256s, %d acceptor(s) outstanding.",
	    &server(), message(), acceptorCnt );
} // uSocketServer::CloseFailure::defaultTerminate


uSocketServer::ReadFailure::ReadFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketServer::ReadFailure::ReadFailure

void uSocketServer::ReadFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::ReadFailure::defaultTerminate


uSocketServer::ReadTimeout::ReadTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::ReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketServer::ReadTimeout::ReadTimeout

void uSocketServer::ReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketServer::ReadTimeout::defaultTerminate


uSocketServer::WriteFailure::WriteFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketServer::WriteFailure::WriteFailure

void uSocketServer::WriteFailure::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketServer::WriteFailure::defaultTerminate


uSocketServer::WriteTimeout::WriteTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketServer::WriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketServer::WriteTimeout::WriteTimeout

void uSocketServer::WriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketServer &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &server(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketServer::WriteTimeout::defaultTerminate


//######################### uSocketAccept (cont) #########################


uSocketAccept::Failure::Failure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocket::Failure( acceptor.sock(), msg ), _acceptor( acceptor ) {}

const uSocketAccept &uSocketAccept::Failure::acceptor() const {
    return _acceptor;
} // uSocketAccept::Failure::acceptor

const uSocketServer &uSocketAccept::Failure::server() const {
    return _acceptor.socketserver;
} // uSocketAccept::Failure::server

int uSocketAccept::Failure::fileDescriptor() const {
    return acceptor().access.fd;
} // uSocketAccept::Failure::fileDescriptor

void uSocketAccept::Failure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p( server:0x%p ), %.256s.", &acceptor(), &server(), message() );
} // uSocketAccept::Failure::defaultTerminate


uSocketAccept::OpenFailure::OpenFailure( const uSocketAccept &acceptor, const class uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg ) :
	uSocketAccept::Failure( acceptor, msg ), timeout( timeout ), adr( adr ), len( len ) {}

void uSocketAccept::OpenFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s, error(%d) : %s",
	    &acceptor(), &server(), timeout, adr, len, message(), errNo(), strerror( errNo() ) );
} // uSocketAccept::OpenFailure::defaultTerminate


uSocketAccept::OpenTimeout::OpenTimeout( const uSocketAccept &acceptor, const class uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg ) :
	uSocketAccept::OpenFailure( acceptor, timeout, adr, len, msg ) {}

void uSocketAccept::OpenTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.uSocketAccept( server:0x%p, timeout:0x%p, adr:0x%p, len:0x%p ), %.256s",
	    &acceptor(), &server(), timeout, adr, len, message() );
} // uSocketAccept::OpenTimeout::defaultTerminate


uSocketAccept::CloseFailure::CloseFailure( const uSocketAccept &acceptor, const char *const msg ) :
	uSocketAccept::Failure( acceptor, msg ) {}

void uSocketAccept::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.~uSocketAccept(), %.256s.\nError(%d) : %s.",
	    &acceptor(), message(), errNo(), strerror( errNo() ) );
} // uSocketAccept::CloseFailure::defaultTerminate


uSocketAccept::ReadFailure::ReadFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketAccept::ReadFailure::ReadFailure

void uSocketAccept::ReadFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::ReadFailure::defaultTerminate


uSocketAccept::ReadTimeout::ReadTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::ReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketAccept::ReadTimeout::ReadTimeout

void uSocketAccept::ReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketAccept::ReadTimeout::defaultTerminate


uSocketAccept::WriteFailure::WriteFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketAccept::WriteFailure::WriteFailure

void uSocketAccept::WriteFailure::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketAccept::WriteFailure::defaultTerminate


uSocketAccept::WriteTimeout::WriteTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketAccept::WriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketAccept::WriteTimeout::WriteTimeout

void uSocketAccept::WriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketAccept &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &acceptor(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketAccept::WriteTimeout::defaultTerminate


//######################### uSocketClient (cont) #########################


uSocketClient::Failure::Failure( const uSocketClient &client, const char *const msg ) : uSocket::Failure( client.socket, msg ), _client( client ) {
} // uSocketClient::Failure::Failure

const uSocketClient &uSocketClient::Failure::client() const {
    return _client;
} // uSocketClient::Failure::client

int uSocketClient::Failure::fileDescriptor() const {
    return client().access.fd;
} // uSocketClient::Failure::fileDescriptor

void uSocketClient::Failure::defaultTerminate() const {
	uAbort( "(uSocketClient &)0x%p, %.256s.",
		&client(), message() );
} // uSocketClient::Failure::defaultTerminate


uSocketClient::OpenFailure::OpenFailure( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::Failure( client, msg ), port( port ), timeout( timeout ), domain( domain ), type( type ), protocol( protocol ) {
    strcpy( _name, name );				// copy file name
} // uSocketClient::OpenFailure::OpenFailure

const char *uSocketClient::OpenFailure::name() const { return _name; }

void uSocketClient::OpenFailure::defaultTerminate() const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.\nError(%d) : %s.",
	       &client(), name(), &uThisCluster(), timeout, domain, type, protocol, message(), errNo(), strerror( errNo() ) );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.\nError(%d) : %s.",
	       &client(), port, name(), &uThisCluster(), timeout, domain, type, protocol, message(), errNo(), strerror( errNo() ) );
    } // if
} // uSocketClient::OpenFailure::defaultTerminate


uSocketClient::OpenTimeout::OpenTimeout( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg ) :
	uSocketClient::OpenFailure( client, port, name, timeout, domain, type, protocol, msg ) {
} // uSocketClient::OpenTimeout::OpenTimeout

void uSocketClient::OpenTimeout::defaultTerminate() const {
    if ( domain == AF_UNIX ) {
	uAbort( "(uSocketClient &)0x%p.uSocketClient( name:\"%s\", cluster:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.",
		&client(), name(), &uThisCluster(), domain, type, protocol, message() );
    } else { // AF_INET
	uAbort( "(uSocketClient &)0x%p.uSocketClient( port:%d, name:\"%s\", cluster:0x%p, timeout:0x%p, domain:%d, type:%d, protocol:%d ) : %.256s.",
		&client(), port, name(), &uThisCluster(), timeout, domain, type, protocol, message() );
    } // if
} // uSocketClient::OpenTimeout::defaultTerminate


uSocketClient::CloseFailure::CloseFailure( const uSocketClient &client, const char *const msg ) :
	uSocketClient::Failure( client, msg ) {}

void uSocketClient::CloseFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.~uSocketClient(), %.256s.\nError(%d) : %s.",
	    &client(), message(), errNo(), strerror( errNo() ) );
} // uSocketClient::CloseFailure::defaultTerminate


uSocketClient::ReadFailure::ReadFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), from( from ), fromlen( fromlen ), timeout( timeout ) {
} // uSocketClient::ReadFailure::ReadFailure

void uSocketClient::ReadFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::ReadFailure::defaultTerminate


uSocketClient::ReadTimeout::ReadTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::ReadFailure( sa, buf, len, flags, from, fromlen, timeout, msg ) {
} // uSocketClient::ReadTimeout::ReadTimeout

void uSocketClient::ReadTimeout::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.read/recv/recvfrom( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketClient::ReadTimeout::defaultTerminate


uSocketClient::WriteFailure::WriteFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::Failure( sa, msg ), buf( buf ), len( len ), flags( flags ), to( to ), tolen( tolen ), timeout( timeout ) {
} // uSocketClient::WriteFailure::WriteFailure

void uSocketClient::WriteFailure::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uSocketClient::WriteFailure::defaultTerminate


uSocketClient::WriteTimeout::WriteTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg ) :
	uSocketClient::WriteFailure( sa, buf, len, flags, to, tolen, timeout, msg ) {
} // uSocketClient::WriteTimeout::WriteTimeout

void uSocketClient::WriteTimeout::defaultTerminate() const {
    uAbort( "(uSocketClient &)0x%p.write/send/sendto( buf:0x%p, len:%d, flags:%d, timeout:0x%p ) : %.256s for file descriptor %d.",
	    &client(), buf, len, flags, timeout, message(), fileDescriptor() );
} // uSocketClient::WriteTimeout::defaultTerminate


// Local Variables: //
// compile-command: "gmake install" //
// End: //
