//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.1, Copyright (C) Peter A. Buhr 1994
// 
// uFile.h -- Nonblocking UNIX I/O library
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 16:38:54 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Jan 24 18:06:21 2006
// Update Count     : 122
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#ifndef __U_FILE_H__
#define __U_FILE_H__


#include <uIOcntl.h>
#include <uIOEvent.h>


#pragma __U_NOT_USER_CODE__


#include <fcntl.h>					// open, mode flags
#include <sys/stat.h>					// stat


//######################### uFile #########################


class uFile {
    friend class uFileWrapper;
    friend class uFileAccess;

    char *name;
    int accessCnt;

    void uCreateFile( const char *name );
    _Mutex void access();
    _Mutex void unaccess();
  public:
    uFile( const char *name );
    virtual ~uFile();

    const char *getName() const;
    void status( struct stat &buf );

    _DualEvent Failure;
    _DualEvent TerminateFailure;
    _DualEvent StatusFailure;
}; // uFile


//######################### uFileIO #########################


class uFileIO {
    int _read( char *buf, int len );
    int _readv( const struct iovec *iov, int iovcnt );
    int _write( char *buf, int len );
    int _writev( const struct iovec *iov, int iovcnt );
  protected:
    uIOaccess &access;

    virtual void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;
    virtual void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;

    uFileIO( uIOaccess &acc );
    virtual ~uFileIO();
  public:
    int read( char *buf, int len, uDuration *timeout = NULL );
    int readv( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );
    _Mutex int write( char *buf, int len, uDuration *timeout = NULL );
    int writev( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );
    int fd();
}; // uFileIO


//######################### uFileAccess #########################


class uFileAccess : public uFileIO {		// monitor
    template< class char_t, class traits > friend class std::basic_filebuf; // access: constructor

    uFile &ufile;
    uIOaccess access;

    uFileAccess( int fd, uFile &f );
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
  public:
    uFileAccess( uFile &f, int flags, int mode = 0644 );
    _Mutex virtual ~uFileAccess();

    off_t lseek( off_t offset, int whence );
    int fsync();

    _DualEvent Failure;
    friend _DualEvent Failure;
    _DualEvent OpenFailure;
    _DualEvent CloseFailure;
    _DualEvent SeekFailure;
    _DualEvent SyncFailure;
    _DualEvent ReadFailure;
    _DualEvent ReadTimeout;
    _DualEvent WriteFailure;
    _DualEvent WriteTimeout;
}; // uFileAccess


//######################### uFile (cont) #########################


_DualEvent uFile::Failure : public uIOFailure {
    const uFile &f;
    char name[uEHMMaxName + sizeof("...")];		// room for "...\0"
  public:
    Failure( const uFile &f, const char *const msg );
    ~Failure();
    const uFile &file() const;
    const char *getName() const;
    virtual void defaultTerminate() const;
}; // uFile::Failure


_DualEvent uFile::TerminateFailure : public uFile::Failure {
    const int accessCnt;
  public:
    TerminateFailure( const uFile &f, const int accessCnt, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFile::TerminateFailure


_DualEvent uFile::StatusFailure : public uFile::Failure {
    const struct stat &buf;
  public:
    StatusFailure( const uFile &f, const struct stat &buf, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFile::StatusFailure


//######################### uFileAccess (cont) #########################


_DualEvent uFileAccess::Failure : public uFile::Failure {
    const uFileAccess &fa;
    int fd;
  public:
    Failure( const uFileAccess &fa, const char *const msg );
    const uFileAccess &fileAccess() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uFileAccess::Failure


_DualEvent uFileAccess::OpenFailure : public uFileAccess::Failure {
    const int flags;
    const int mode;
  public:
    OpenFailure( uFileAccess &fa, int flags, int mode, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::OpenFailure


_DualEvent uFileAccess::CloseFailure : public uFileAccess::Failure {
  public:
    CloseFailure( uFileAccess &fa, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::CloseFailure


_DualEvent uFileAccess::SeekFailure : public uFileAccess::Failure {
    const off_t offset;
    const int whence;
  public:
    SeekFailure( const uFileAccess &fa, const off_t offset, const int whence, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::SeekFailure


_DualEvent uFileAccess::SyncFailure : public uFileAccess::Failure {
  public:
    SyncFailure( const uFileAccess &fa, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::SyncFailure


_DualEvent uFileAccess::ReadFailure : public uFileAccess::Failure {
  protected:
    const char *buf;
    const int len;
    const uDuration *timeout;
  public:
    ReadFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uFileAccess::ReadFailure


_DualEvent uFileAccess::ReadTimeout : public uFileAccess::ReadFailure {
  public:
    ReadTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
}; // uFileAccess::ReadTimeout


_DualEvent uFileAccess::WriteFailure : public uFileAccess::Failure {
  protected:
    const char *buf;
    const int len;
    const uDuration *timeout;
  public:
    WriteFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    virtual void defaultResume() const;			// handle special case when errno == EIO
    virtual void defaultTerminate() const;
}; // uFileAccess::WriteFailure


_DualEvent uFileAccess::WriteTimeout : public uFileAccess::WriteFailure {
  public:
    WriteTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
}; // uFileAccess::WriteTimeout


#pragma __U_USER_CODE__

#endif // __U_FILE_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
