//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.1, Copyright (C) Peter A. Buhr 1996
// 
// uBaseTask.cc -- 
// 
// Author           : Peter A. Buhr
// Created On       : Mon Jan  8 16:14:20 1996
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Sep 21 18:15:32 2006
// Update Count     : 263
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 

#define __U_KERNEL__
#define __U_PROFILE__
#define __U_PROFILEABLE_ONLY__


#include <uC++.h>
#include <uProfiler.h>
//#include <uDebug.h>


//######################### uBaseTask #########################


void uBaseTask::createTask( uCluster &cluster ) {
#ifdef __U_DEBUG__
    currSerialOwner = this;
    currSerialCount = 1;
    currSerialLevel = 0;
#endif // __U_DEBUG__
    state = Start;
    recursion = mutexRecursion = 0;
    currCluster = &cluster;				// remember the cluster task is created on
    currCoroutine = this;				// the first coroutine that a task executes is itself
    acceptedCall = NULL;				// no accepted mutex entry yet
    priority = activePriority = 0;
    inheritTask = this;

    // exception handling

    terminateRtn = uEHM::terminate;			// initialize default terminate routine

    // profiling

    profileActive = false;				// can be read before uTaskConstructor is called

    // debugging
#if __U_LOCALDEBUGGER_H__
    DebugPCandSRR = NULL;
    uProcessBP = false;					// used to prevent triggering breakpoint while processing one
#endif // __U_LOCALDEBUGGER_H__

    // pthreads

    pthreadData = NULL;
} // uBaseTask::createTask


uBaseTask::uBaseTask( uCluster &cluster, uProcessor &processor ) : uBaseCoroutine( cluster.getStackSize() ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( processor ) {
    createTask( cluster );
} // uBaseTask::uBaseTask


void uBaseTask::setState( uBaseTask::State s ) {
    state = s;

    if ( profileActive && uProfiler::uProfiler_registerTaskExecState ) { 
	(*uProfiler::uProfiler_registerTaskExecState)( uProfiler::profilerInstance, *this, state ); 
    } // if
} // uBaseTask::setState


void uBaseTask::wake() {
    setState( Ready );					// task is marked available for execution
    currCluster->makeTaskReady( *this );		// put the task on the ready queue of the cluster
} // uBaseTask::wake


uBaseTask::uBaseTask( uCluster &cluster ) : uBaseCoroutine( cluster.getStackSize() ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
    createTask( cluster );
} // uBaseTask::uBaseTask


uCluster &uBaseTask::migrate( uCluster &cluster ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uBaseTask &)0x%p.migrate, from cluster:0x%p to cluster:0x%p\n", this, &uThisCluster(), &cluster );
#endif // __U_DEBUG_H__

    assert( &bound == NULL );

#ifdef __U_DEBUG__
    if ( this != &uThisTask() ) {
	uAbort( "Attempt to migrate task %.256s (0x%p) to cluster %.256s (0x%p).\n"
		"A task may only migrate itself to another cluster.",
		getName(), this, cluster.getName(), &cluster );
    } // if
#endif // __U_DEBUG__

    // A simple optimization: migrating to the same cluster that the task is
    // currently executing on simply returns the value of the current cluster.
    // Therefore, migrate does not always produce a context switch.

  if ( &cluster == currCluster ) return cluster;

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif // __U_LOCALDEBUGGER_H__

    // Remove the task from the list of tasks that live on this cluster,
    // and add it to the list of tasks that live on the new cluster.

    uCluster &prevCluster = *currCluster;		// save for return

    if ( profileActive && uProfiler::uProfiler_registerTaskMigrate ) {	// task registered for profiling ?              
	(*uProfiler::uProfiler_registerTaskMigrate)( uProfiler::profilerInstance, *this, prevCluster, cluster );
    } // if

    // Interrupts are disabled because once the task is removed from a cluster
    // it is dangerous for it to be placed back on that cluster during an
    // interrupt.  Therefore, interrupts are disabled until the task is on its
    // new cluster.

    THREAD_GETMEM( This )->disableInterrupts();

    prevCluster.taskRemove( *this );			// remove from current cluster
    currCluster = &cluster;				// change task's notion of which cluster it is executing on
    cluster.taskAdd( *this );				// add to new cluster

    THREAD_GETMEM( This )->enableInterrupts();

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->migrateULThread( cluster );
#endif // __U_LOCALDEBUGGER_H__

    // Force a context switch so the task is scheduled on the new cluster.

    yield();

    return prevCluster;					// return reference to previous cluster
} // uBaseTask::migrate


void uBaseTask::uYieldNoPoll() {
    assert( ! THREAD_GETMEM( disableIntSpin ) );
#ifdef __U_DEBUG__
    if ( this != &uThisTask() ) {
	uAbort( "Attempt to yield the execution of task %.256s (0x%p) by task %.256s (0x%p).\n"
		"A task may only yield itself.",
		getName(), this, uThisTask().getName(), &uThisTask() );
    } // if
#endif // __U_DEBUG__

    uSCHEDULE( this );					// find someone else to execute; wake on kernel stack
} // uBaseTask::uYieldNoPoll


void uBaseTask::yield( unsigned int times ) {
    for ( ; times > 0 ; times -= 1 ) {
	yield();
    } // for
} // uBaseTask::yield


void uBaseTask::uYieldYield( unsigned int times ) {	// inserted by translator for -yield
    // Calls to uYieldYield can be inserted in any inlined routine, which can
    // than be called from a uWhen clause, resulting in an attempt to context
    // switch while holding a spin lock. To ensure assert checking in normal
    // usages of yield, this check cannot be inserted in yield.

    if ( ! THREAD_GETMEM( disableIntSpin ) ) {
	for ( ; times > 0 ; times -= 1 ) {
	    uYieldNoPoll();
	} // for
    } // if
} // uBaseTask::uYieldYield


void uBaseTask::uYieldInvoluntary() {
    assert( ! THREAD_GETMEM( disableIntSpin ) );
#ifdef __U_DEBUG__
    if ( this != &uThisTask() ) {
	uAbort( "Attempt to yield the execution of task %.256s (0x%p) by task %.256s (0x%p).\n"
		"A task may only yield itself.",
		getName(), this, uThisTask().getName(), &uThisTask() );
    } // if
#endif // __U_DEBUG__

    // Are the uC++ kernel memory allocation hooks active?
    if ( profileActive && uProfiler::uProfiler_preallocateMetricMemory ) {
	// create a preallocated memory array on the stack
	void *ptrs[U_MAX_METRICS];

	(*uProfiler::uProfiler_preallocateMetricMemory)( uProfiler::profilerInstance, ptrs, *this );

	THREAD_GETMEM( This )->disableInterrupts();
	(*uProfiler::uProfiler_setMetricMemoryPointers)( uProfiler::profilerInstance, ptrs, *this ); // force task to use local memory array
	activeProcessorKernel->schedule( this );	// find someone else to execute; wake on kernel stack
	(*uProfiler::uProfiler_resetMetricMemoryPointers)( uProfiler::profilerInstance, *this );     // reset task to use its native memory array
	THREAD_GETMEM( This )->enableInterrupts();

	// free any blocks of memory not used by metrics
	for ( int metric = 0; metric < uProfiler::profilerInstance->numMemoryMetrics; metric += 1 ) {
	    free( ptrs[metric] );
	} // for
    } else {
	THREAD_GETMEM( This )->disableInterrupts();
	activeProcessorKernel->schedule( this );	// find someone else to execute; wake on kernel stack
	THREAD_GETMEM( This )->enableInterrupts();
    } // if
} // uBaseTask::uYieldInvoluntary


void uBaseTask::uSleep( uTime time ) {
#ifdef __U_DEBUG__
    if ( this != &uThisTask() ) {
	uAbort( "Attempt to put task %.256s (0x%p) to sleep.",
		getName(), this );
    } // if
#endif // __U_DEBUG__

  if ( time <= activeProcessorKernel->kernelClock.getTime() ) return;

    uWakeupHndlr handler( *this );			// handler to wake up blocking task
    uEventNode uRTEvent( *this, handler, time );	// event node for event list
    uRTEvent.add( true );
} // uBaseTask::uSleep


void uBaseTask::uSleep( uDuration duration ) {
    uSleep( activeProcessorKernel->kernelClock.getTime() + duration );
} // uBaseTask::uSleep


void uBaseTask::profileActivate( uBaseTask &task ) {
    if ( ! profileTaskSamplerInstance ) {		// already registered for profiling ?
	if ( uProfiler::uProfiler_registerTask ) {	// task registered for profiling ? 
	    (*uProfiler::uProfiler_registerTask)( uProfiler::profilerInstance, *this, *serial, task );
	    profileActive = true;
	} // if
    } else {
	profileActive = true;
    } // if 
} // uBaseTask::profileActivate


void uBaseTask::profileActivate() {
    profileActivate( *(uBaseTask *)0 );
} // uBaseTask::profileActivate


void uBaseTask::profileInactivate() {
    profileActive = false;
} // uBaseTask::profileInactivate


void uBaseTask::printCallStack() const {
    if ( profileTaskSamplerInstance ) {
	(*uProfiler::uProfiler_printCallStack)( profileTaskSamplerInstance );
    } // if
} // uBaseTask::printCallStack


// Local Variables: //
// compile-command: "gmake install" //
// End: //
