//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.1, Copyright (C) Peter A. Buhr 2002
// 
// PThread.cc -- 
// 
// Author           : Peter A. Buhr
// Created On       : Thu Jan 17 17:06:03 2002
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Sep 21 21:13:17 2006
// Update Count     : 111
// 

#ifdef __U_CPLUSPLUS__
#include <uC++.h>
#endif

#include <stdio.h>
#include <pthread.h>
#include <stdlib.h>										// prototype: rand
#include <limits.h>
#include <errno.h>

#if ! defined( __freebsd__ )
extern "C" int pthread_yield( void );					// may not be part of vendor's pthread package
#endif // __freebsd__


class MutexMem {
	pthread_mutex_t &mutex;
  public:
	MutexMem( pthread_mutex_t &mutex ) : mutex( mutex ) {
		pthread_mutex_lock( &mutex );
	}
	~MutexMem() {
		pthread_mutex_unlock( &mutex );
	}
};

template <class ELEMTYPE> class BoundedBuffer {
	pthread_mutex_t mutex;
	int front, back, count;
	ELEMTYPE Elements[20];
	pthread_cond_t Full, Empty;							// waiting consumers & producers
  public:
	BoundedBuffer() {
		front = back = count = 0;
		pthread_mutex_init( &mutex, NULL );
		pthread_cond_init( &Full, NULL );
		pthread_cond_init( &Empty, NULL );
	}
	~BoundedBuffer() {
		pthread_mutex_lock( &mutex );
		pthread_cond_destroy( &Empty );
		pthread_cond_destroy( &Full );
		pthread_mutex_destroy( &mutex );
	}
	int query() { return count; }

	void insert( ELEMTYPE elem ) {
		MutexMem lock( mutex );
		while ( count == 20 ) pthread_cond_wait( &Empty, &mutex ); // block producer
		Elements[back] = elem;
		back = ( back + 1 ) % 20;
		count += 1;
		pthread_cond_signal( &Full );					// unblock consumer
	}
	ELEMTYPE remove() {
		MutexMem lock( mutex );
		while ( count == 0 ) pthread_cond_wait( &Full, &mutex ); // block consumer
		ELEMTYPE elem = Elements[front];
		front = ( front + 1 ) % 20;
		count -= 1;
		pthread_cond_signal( &Empty );					// unblock producer
		return elem;
	}
};

void *producer( void *arg ) {
	BoundedBuffer<int> &buf = *(BoundedBuffer<int> *)arg;
	const int NoOfItems = rand() % 40;
	int item;

	for ( int i = 1; i <= NoOfItems; i += 1 ) {			// produce a bunch of items
		item = rand() % 100 + 1;						// produce a random number
		printf( "Producer:0x%lx, value:%d\n", (unsigned long)pthread_self(), item );
		buf.insert( item );								// insert element into queue
	} // for
	printf( "Producer:0x%lx is finished\n", (unsigned long)pthread_self() );
	return (void *)1;
} // producer

void *consumer( void *arg ) {
	BoundedBuffer<int> &buf = *(BoundedBuffer<int> *)arg;
	int item;

	for ( ;; ) {										// consume until a negative element appears
		item = buf.remove();							// remove from front of queue
		printf( "Consumer:0x%lx, value:%d\n", (unsigned long)pthread_self(), item );
	  if ( item == -1 ) break;
	} // for
	printf( "Consumer:0x%lx is finished\n", (unsigned long)pthread_self() );
	return (void *)0;
} // consumer


void *Worker1( void *arg ) {
#ifdef __U_CPLUSPLUS__
	int stacksize = uThisTask().stackSize();
	printf( "Worker1, stacksize:%d\n", stacksize );
#endif
	for ( int i = 0; i < 100000; i += 1 ) {}
	printf( "Worker1, ending\n" );
	return (void *)0;
} // Worker1

void *Worker2( void *arg ) {
    pthread_key_t key[PTHREAD_KEYS_MAX];
    unsigned long int i;
    for ( i = 0; i < 60; i += 1 ) {
		if ( pthread_key_create( &key[i], NULL ) != 0 ) {
			printf( "Create key\n" );
			exit( -1 );
		} // if
		printf( "0x%lx, i:%lu key:%d\n", (unsigned long)pthread_self(), i, key[i] );
    } // for
    for ( i = 0; i < 60; i += 1 ) {
		if ( pthread_setspecific( key[i], (const void *)i ) != 0 ) {
			printf( "Set key\n" );
			exit( -1 );
		} // if
    } // for
    for ( i = 0; i < 60; i += 1 ) {
		if ( (unsigned long int)pthread_getspecific( key[i] ) != i ) {
			printf( "Get key\n" );
			exit( -1 );
		} // if
    } // for
    pthread_exit( NULL );
	return (void *)0;
} // Worker2


int check = 0;

void clean1( void *) {
	printf("Executing top cleanup\n");
	check -= 1;
}

void clean2( void *) {
	printf("Executing bottom cleanup\n");
	check -= 3333;
}

class RAII {
  public:
    RAII() {
		check += 1;
		printf("Executing RAII constructor\n");
    }
    ~RAII() {
		check -= 1;
		printf("Executing RAII destructor\n");
    }
}; // RAII

void *cancellee( void *arg ) {
	int i;
	pthread_setcanceltype( PTHREAD_CANCEL_DEFERRED, NULL );
	for ( ;; ) {
		pthread_cleanup_push( clean2, NULL );
		check += 3333;
		{
			RAII rai;				
			for ( i = 0; i < 10; i += 1 ) {				
				pthread_cleanup_push( clean1, NULL );			
				check += 1;
				pthread_testcancel();					// wait to be cancelled			
				pthread_cleanup_pop( 1 );
			} // for
		}
		pthread_cleanup_pop( 1 );
	} // for
	abort();											// CONTROL NEVER REACHES HERE!
} // cancellee

void *canceller( void *arg ) {
	pthread_t cancellee = *(pthread_t *)arg;
	printf( "0x%lx before cancel\n", (unsigned long)pthread_self() );
	pthread_cancel( cancellee );
	pthread_join( cancellee, NULL );
	printf( "0x%lx after cancel\n", (unsigned long)pthread_self() );
	printf( "check value: %d\n", check );
#if defined( __U_CPLUSPLUS__ ) || defined( __linux__ )
	if ( check != 0 ) uAbort( "not all destructors called" );
#else
	// since g++ will not execute the destructor on cancellation
	if ( check != 1 ) uAbort( "not all destructors called" );
#endif
	return NULL;
} // canceller


void *spinner( void *arg ) {
	printf( "Spinner starting\n" );
	for ( ;; ) {
		pthread_self();									// just spin for awhile
	} // for
} // spinner


#ifdef __U_CPLUSPLUS__
void uMain::main() {
#else
int main() {
#endif
	const int NoOfCons = 20, NoOfProds = 30;
	BoundedBuffer<int> buf;								// create a buffer monitor
	pthread_t cons[NoOfCons];							// pointer to an array of consumers
	pthread_t prods[NoOfProds];							// pointer to an array of producers

	// parallelism

	pthread_setconcurrency( 4 );

	// create/join and mutex/condition test

	printf( "create/join and mutex/condition test\n\n" );

	for ( int i = 0; i < NoOfCons; i += 1 ) {			// create consumers
		if ( pthread_create( &cons[i], NULL, consumer, &buf ) != 0 ) {
			printf( "create thread failure, errno:%d\n", errno );
			exit( -1 );
		} // if
	} // for
	for ( int i = 0; i < NoOfProds; i += 1 ) {			// 	create producers
		if ( pthread_create( &prods[i], NULL, producer, &buf ) != 0 ) {
			printf( "create thread failure\n" );
			exit( -1 );
		} // if
	} // for

	void *result;
	for ( int i = 0; i < NoOfProds; i += 1 ) {			// wait for producers to end
		if ( pthread_join( prods[i], &result ) != 0 ) {
			printf( "join thread failure\n" );
			exit( -1 );
		} // if
		if ( result != (void *)1 ) {
			printf( "bad return value\n" );
			exit( -1 );
		} // if
		printf( "join prods[%d]:0x%lx result:0x%p\n", i, (unsigned long)prods[i], result );
	} // for

	for ( int i = 0; i < NoOfCons; i += 1 ) {			// terminate each consumer
		buf.insert( -1 );
	} // for

	for ( int i = 0; i < NoOfCons; i += 1 ) {			// wait for consumer to end
		if ( pthread_join( cons[i], &result ) != 0 ) {
			printf( "join thread failure\n" );
			exit( -1 );
		} // if
		if ( result != (void *)0 ) {
			printf( "bad return value\n" );
			exit( -1 );
		} // if
		printf( "join cons[%d]:0x%lx result:0x%p\n", i, (unsigned long)cons[i], result );
	} // for

	// thread attribute test

	printf( "\n\nthread attribute test\n\n" );

	pthread_attr_t attr;
	pthread_attr_init( &attr );
	pthread_attr_setstacksize( &attr, 64 * 1000 );
	pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_DETACHED );
	size_t stacksize;
	pthread_attr_getstacksize( &attr, &stacksize );

	pthread_t worker1, worker2;

	if ( pthread_create( &worker1, &attr, Worker1, NULL ) != 0 ) {
		printf( "create thread failure\n" );
		exit( -1 );
	} // if

	pthread_attr_destroy( &attr );

	// thread specific data test

	printf( "\n\nthread specific data test\n\n" );

    if ( pthread_create( &worker1, NULL, Worker2, NULL ) != 0 ) {
		printf( "create thread failure\n" );
		exit( -1 );
	} // if
    if ( pthread_create( &worker2, NULL, Worker2, NULL ) != 0 ) {
		printf( "create thread failure\n" );
		exit( -1 );
	} // if
    if ( pthread_join( worker1, NULL ) != 0 ) {
		printf( "join thread failure\n" );
		exit( -1 );
	} // if
    if ( pthread_join( worker2, NULL ) != 0 ) {
		printf( "join thread failure\n" );
		exit( -1 );
	} // if

	// thread cancel test

#if defined( __U_BROKEN_CANCEL__ )
	printf( "\n\ncancellation test skipped -- pthread cancellation not supported on this system\n\n" );
#else
	printf( "\n\nthread cancellation test\n\n" );

	// check uMain's pthread emulation
	pthread_setcanceltype( PTHREAD_CANCEL_DEFERRED, NULL );
	pthread_cleanup_push(clean2, NULL);
	check += 3333;
	{
		RAII rai;				
		pthread_cleanup_push( clean1, NULL );			
		check += 1;
		pthread_testcancel();							// check to be cancelled (not gonna happen)		
		pthread_cleanup_pop( 1 );
	}
	pthread_cleanup_pop( 1 );

    if ( check != 0 ) uAbort( "not all destructors called" );


    if ( pthread_create( &worker1, NULL, cancellee, NULL ) != 0 ) {
		printf( "create thread failure\n" );
		exit( -1 );
	} // if
	printf( "cancellee %d\n", worker1 );
    if ( pthread_create( &worker2, NULL, canceller, &worker1 ) != 0 ) {
		printf( "create thread failure\n" );
		exit( -1 );
	} // if
	// worker1 joined in canceller
    if ( pthread_join( worker2, NULL ) != 0 ) {
		printf( "join thread failure\n" );
		exit( -1 );
	} // if
#endif // __U_BROKEN_CANCEL__

	printf( "successful completion\n" );
} // main


// Local Variables: //
// compile-command: "g++ -g Pthread.cc -lpthread" //
// tab-width: 4 //
// End: //
