//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.0, Copyright (C) Peter A. Buhr 1994
// 
// uSocket.h -- Nonblocking UNIX Socket I/O
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 17:04:36 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Feb  8 12:46:54 2006
// Update Count     : 310
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#ifndef __U_SOCKET_H__
#define __U_SOCKET_H__


#include <uFile.h>
//#include <uDebug.h>


#pragma __U_NOT_USER_CODE__


#include <cstdio>
#include <netdb.h>					// MAXHOSTNAMELEN on Solaris
#include <sys/param.h>					// MAXHOSTNAMELEN on BSD

#if defined( __freebsd__ )
#include <netinet/in.h>					// struct sockaddr_in
#include <sys/socket.h>					// SOCK_STREAM
#endif // __freebsd__


//######################### uSocket #########################


class uSocket {
    friend class uSocketServer;				// access: access, uSocket, ~uSocket
    friend class uSocketAccept;				// access: access
    friend class uSocketClient;				// access: access, uSocket, ~uSocket

    uIOaccess access;

    const int domain, type, protocol;			// copies for information purposes

    uSocket( uSocket & );				// no copy
    uSocket &operator=( uSocket & );			// no assignment

    void uCreateSocket( int domain, int type, int protocol );
    uSocket( int domain, int type, int protocol );	// creation restricted to friends
    virtual ~uSocket();
  public:
    _DualEvent Failure;
    friend _DualEvent Failure;				// access: access
    _DualEvent OpenFailure;
    _DualEvent CloseFailure;
    friend _DualEvent CloseFailure;			// access: access
}; // uSocket


//######################### uSocketIO #########################


class uSocketIO : public uFileIO {
    int _send( char *buf, int len, int flags );
    int _sendto( char *buf, int len, int flags, struct sockaddr *to, socklen_t tolen );
    int _sendmsg( const struct msghdr *msg, int flags );
    int _recv( char *buf, int len, int flags );
    int _recvfrom( char *buf, int len, int flags, struct sockaddr *from, socklen_t *fromlen );
    int _recvmsg( struct msghdr *msg, int flags );
  protected:
    struct inetAddr : sockaddr_in {
	int code;
	char _name[MAXHOSTNAMELEN+1];
	void getinetAddr( unsigned short port, const char *name );
	inetAddr( unsigned short port, const char *name = "" );
    }; // inetAddr

    struct sockaddr *saddr;				// default send/receive address
    socklen_t saddrlen;					// size of send address
    socklen_t baddrlen;					// size of address buffer (UNIX/INET)

    virtual void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) = 0;
    virtual void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op ) = 0;

    uSocketIO( uIOaccess &acc, struct sockaddr *saddr );
  public:
    _Mutex const struct sockaddr *getsockaddr();	// must cast result to sockaddr_in or sockaddr_un
    _Mutex int getsockname( struct sockaddr *name, socklen_t *len );
    _Mutex int getpeername( struct sockaddr *name, socklen_t *len );
    int send( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int sendto( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int sendto( char *buf, int len, struct sockaddr *to, socklen_t tolen, int flags = 0, uDuration *timeout = NULL );
    int sendmsg( const struct msghdr *msg, int flags = 0, uDuration *timeout = NULL );
    int recv( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int recvfrom( char *buf, int len, int flags = 0, uDuration *timeout = NULL );
    int recvfrom( char *buf, int len, struct sockaddr *from, socklen_t *fromlen, int flags = 0, uDuration *timeout = NULL );
    int recvmsg( struct msghdr *msg, int flags = 0, uDuration *timeout = NULL );
}; // uSocketIO


//######################### uSocketServer #########################


_Monitor uSocketServer : public uSocketIO {
    friend class uSocketAccept;				// access: socket, _accept, acceptor, unacceptor, ~uSocketAccept

    int acceptorCnt;					// number of simultaneous acceptors using server
    uSocket socket;					// one-to-one correspondance between server and socket

    _Mutex void acceptor();
    _Mutex void unacceptor();
    int _accept( struct sockaddr *adr, socklen_t *len );
    void uCreateSocketServer1( const char *name, int type, int protocol, int backlog );
    void uCreateSocketServer2( unsigned short port, int type, int protocol, int backlog );
    void uCreateSocketServer3( unsigned short *port, int type, int protocol, int backlog );
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );

    void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
  public:
    // AF_UNIX
    uSocketServer( const char *name, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 );
    // AF_INET, local host
    uSocketServer( unsigned short port, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 );
    uSocketServer( unsigned short *port, int type = SOCK_STREAM, int protocol = 0, int backlog = 10 );
    virtual ~uSocketServer();

    void setClient( struct sockaddr *addr, socklen_t len );
    void getClient( struct sockaddr *addr, socklen_t *len );

    _DualEvent Failure;
    friend _DualEvent Failure;				// access: socket
    _DualEvent OpenFailure;
    _DualEvent CloseFailure;
    _DualEvent ReadFailure;
    _DualEvent ReadTimeout;
    _DualEvent WriteFailure;
    _DualEvent WriteTimeout;
}; // uSocketServer


//######################### uSocketAccept #########################


_Monitor uSocketAccept : public uSocketIO {
    uSocketServer &socketserver;			// many-to-one correspondence among acceptors and server
    uIOaccess access;
    bool openAccept;					// current open accept for socket ?
    uDuration *timeout;
    struct sockaddr *adr;
    socklen_t *len;

    const uSocket &sock() const;
    void uCreateSocketAcceptor( uDuration *timeout, struct sockaddr *adr, socklen_t *len );
    int _close();
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );

    void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *op );
  public:
    uSocketAccept( uSocketServer &s, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    uSocketAccept( uSocketServer &s, uDuration *timeout, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    uSocketAccept( uSocketServer &s, bool doAccept, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    uSocketAccept( uSocketServer &s, uDuration *timeout, bool doAccept, struct sockaddr *adr = NULL, socklen_t *len = NULL );
    virtual ~uSocketAccept();

    void accept();
    void accept( uDuration *timeout );
    void close();

    _DualEvent Failure;
    _DualEvent OpenFailure;
    _DualEvent OpenTimeout;
    _DualEvent CloseFailure;
    _DualEvent ReadFailure;
    _DualEvent ReadTimeout;
    _DualEvent WriteFailure;
    _DualEvent WriteTimeout;
}; // uSocketAccept


//######################### uSocketClient #########################


_Monitor uSocketClient : public uSocketIO {
    uSocket socket;					// one-to-one correspondence between client and socket
    char *tmpnm;					// temporary file for communicate with UNIX datagram

    void uCreateSocketClient1( const char *name, uDuration *timeout, int type, int protocol );
    void uCreateSocketClient2( unsigned short port, const char *name, uDuration *timeout, int type, int protocol );
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );

    void readTimeout( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void readFailure( const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const op );
  public:
    // AF_UNIX
    uSocketClient( const char *name, int type = SOCK_STREAM, int protocol = 0 );
    uSocketClient( const char *name, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 );
    // AF_INET, local host
    uSocketClient( unsigned short port, int type = SOCK_STREAM, int protocol = 0 );
    uSocketClient( unsigned short port, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 );
    // AF_INET, other host
    uSocketClient( unsigned short port, const char *name, int type = SOCK_STREAM, int protocol = 0 );
    uSocketClient( unsigned short port, const char *name, uDuration *timeout, int type = SOCK_STREAM, int protocol = 0 );
    virtual ~uSocketClient();

    void setServer( struct sockaddr *addr, socklen_t len );
    void getServer( struct sockaddr *addr, socklen_t *len );

    _DualEvent Failure;
    friend _DualEvent Failure;
    _DualEvent OpenFailure;
    _DualEvent OpenTimeout;
    _DualEvent CloseFailure;
    _DualEvent ReadFailure;
    _DualEvent ReadTimeout;
    _DualEvent WriteFailure;
    _DualEvent WriteTimeout;
}; // uSocketClient


//######################### uSocket (cont) #########################


_DualEvent uSocket::Failure : public uIOFailure {
    const uSocket &_socket;
  public:
    Failure( const uSocket &s, const char *const msg );
    const uSocket &socket() const;
    virtual void defaultTerminate() const;
}; // uSocket::Failure


_DualEvent uSocket::OpenFailure : public uSocket::Failure {
  protected:
    const int domain;
    const int type;
    const int protocol;
  public:
    OpenFailure( const uSocket &socket, const int domain, const int type, const int protocol, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocket::OpenFailure


_DualEvent uSocket::CloseFailure : public uSocket::Failure {
  public:
    CloseFailure( const uSocket &socket, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocket::CloseFailure


//######################### uSocketServer (cont) #########################


_DualEvent uSocketServer::Failure : public uSocket::Failure {
    const uSocketServer &_server;
  public:
    Failure( const uSocketServer &s, const char *const msg );
    const uSocketServer &server() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uSocketServer::Failure


_DualEvent uSocketServer::OpenFailure : public uSocketServer::Failure {
  protected:
    const unsigned short port;
    char _name[FILENAME_MAX];
    const int domain;
    const int type;
    const int protocol;
    const int backlog;
  public:
    OpenFailure( const uSocketServer &server, const unsigned short port, const char *const name, int domain, int type, int protocol, int backlog, const char *const msg );
    const char *name() const;			// return socket name, or NULL when hostname is used
    virtual void defaultTerminate() const;
}; // uSocketServer::OpenFailure


_DualEvent uSocketServer::CloseFailure : public uSocketServer::Failure {
    const int acceptorCnt;
  public:
    CloseFailure( const uSocketServer &server, const int acceptorCnt, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::CloseFailure


_DualEvent uSocketServer::ReadFailure : public uSocketServer::Failure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *from;
    const socklen_t *fromlen;
    const uDuration *timeout;
  public:
    ReadFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::ReadFailure


_DualEvent uSocketServer::ReadTimeout : public uSocketServer::ReadFailure {
  public:
    ReadTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::ReadTimeout


_DualEvent uSocketServer::WriteFailure : public uSocketServer::Failure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *to;
    const int tolen;
    const uDuration *timeout;
  public:
    WriteFailure( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::WriteFailure


_DualEvent uSocketServer::WriteTimeout : public uSocketServer::WriteFailure {
  public:
    WriteTimeout( const uSocketServer &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketServer::WriteTimeout


//######################### uSocketAccept (cont) #########################


_DualEvent uSocketAccept::Failure : public uSocket::Failure {
    const uSocketAccept &_acceptor;
  public:
    Failure( const uSocketAccept &acceptor, const char *const msg );
    const uSocketAccept &acceptor() const;
    const uSocketServer &server() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uSocketAccept::Failure


_DualEvent uSocketAccept::OpenFailure : public uSocketAccept::Failure {
  protected:
    const uDuration *timeout;
    const struct sockaddr *adr;
    const socklen_t *len;
  public:
    OpenFailure( const uSocketAccept &acceptor, const uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::OpenFailure


_DualEvent uSocketAccept::OpenTimeout : public uSocketAccept::OpenFailure {
  public:
    OpenTimeout( const uSocketAccept &acceptor, const uDuration *timeout, const struct sockaddr *adr, const socklen_t *len, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::OpenTimeout


_DualEvent uSocketAccept::CloseFailure : public uSocketAccept::Failure {
  public:
    CloseFailure( const uSocketAccept &acceptor, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::CloseFailure


_DualEvent uSocketAccept::ReadFailure : public uSocketAccept::Failure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *from;
    const socklen_t *fromlen;
    const uDuration *timeout;
  public:
    ReadFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::ReadFailure


_DualEvent uSocketAccept::ReadTimeout : public uSocketAccept::ReadFailure {
  public:
    ReadTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::ReadTimeout


_DualEvent uSocketAccept::WriteFailure : public uSocketAccept::Failure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *to;
    const int tolen;
    const uDuration *timeout;
  public:
    WriteFailure( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::WriteFailure


_DualEvent uSocketAccept::WriteTimeout : public uSocketAccept::WriteFailure {
  public:
    WriteTimeout( const uSocketAccept &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketAccept::WriteTimeout


//######################### uSocketClient (cont) #########################


_DualEvent uSocketClient::Failure : public uSocket::Failure {
    const uSocketClient &_client;
  public:
    Failure( const uSocketClient &client, const char *const msg );
    const uSocketClient &client() const;
    int fileDescriptor() const;
    virtual void defaultTerminate() const;
}; // uSocketClient::Failure


_DualEvent uSocketClient::OpenFailure : public uSocketClient::Failure {
  protected:
    unsigned short port;
    char _name[FILENAME_MAX];
    const uDuration *timeout;
    const int domain;
    const int type;
    const int protocol;
  public:
    OpenFailure( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
    const char *name() const;
    virtual void defaultTerminate() const;
}; // uSocketClient::OpenFailure


_DualEvent uSocketClient::OpenTimeout : public uSocketClient::OpenFailure {
  public:
    OpenTimeout( const uSocketClient &client, const unsigned short port, const char *const name, uDuration *timeout, const int domain, const int type, const int protocol, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::OpenTimeout


_DualEvent uSocketClient::CloseFailure : public uSocketClient::Failure {
  public:
    CloseFailure( const uSocketClient &acceptor, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::CloseFailure


_DualEvent uSocketClient::ReadFailure : public uSocketClient::Failure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *from;
    const socklen_t *fromlen;
    const uDuration *timeout;
  public:
    ReadFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::ReadFailure


_DualEvent uSocketClient::ReadTimeout : public uSocketClient::ReadFailure {
  public:
    ReadTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *from, const socklen_t *fromlen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::ReadTimeout


_DualEvent uSocketClient::WriteFailure : public uSocketClient::Failure {
  protected:
    const char *buf;
    const int len;
    const int flags;
    const struct sockaddr *to;
    const int tolen;
    const uDuration *timeout;
  public:
    WriteFailure( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::WriteFailure


_DualEvent uSocketClient::WriteTimeout : public uSocketClient::WriteFailure {
  public:
    WriteTimeout( const uSocketClient &sa, const char *buf, const int len, const int flags, const struct sockaddr *to, const int tolen, const uDuration *timeout, const char *const msg );
    virtual void defaultTerminate() const;
}; // uSocketClient::WriteTimeout


#pragma __U_USER_CODE__

#endif // __U_SOCKET_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
