//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.0, Copyright (C) Peter A. Buhr 2001
// 
// pthread.cc -- 
// 
// Author           : Peter A. Buhr
// Created On       : Sun Dec  9 21:38:53 2001
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Sep 25 17:08:03 2006
// Update Count     : 728
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 

// see http://nptl.bullopensource.org/Tests/Optimization-level-in-nptl.html

#define __U_KERNEL__
#include <uC++.h>
#include <uSystemTask.h>
#include <uSequence.h>
#include <csignal>					// access: sigset_t
#include <cerrno>					// access: EBUSY, ETIMEDOUT
#include <pthread.h>
#include <limits.h>					// access: PTHREAD_KEYS_MAX
#include <uStack.h>

//#include <uDebug.h>

#if defined( __solaris__ )
#include <thread.h>
#endif // __solaris__

#if defined( __irix__ )					// TEMPORARY
#undef PTHREAD_KEYS_MAX
#define PTHREAD_KEYS_MAX _POSIX_THREAD_KEYS_MAX
extern "C" long _sysconf( int );			// TEMPORARY: unnecessary
#endif // __irix__

#define NOT_A_PTHREAD ((pthread_t)-2)			// used as return from pthread_self for non-pthread tasks

static struct Pthread_attr_t {			// thread attributes
    int contentionscope;
    int detachstate;
    size_t stacksize;
    void *stackaddr;
    int policy;
    int inheritsched;
    struct sched_param param;
} u_pthread_attr_defaults = { PTHREAD_SCOPE_SYSTEM,
			      PTHREAD_CREATE_JOINABLE,
			      __U_DEFAULT_STACK_SIZE__,
			      NULL,
			      0,
			      PTHREAD_EXPLICIT_SCHED,
			      { 0 },
};


struct Pthread_values : public uSeqable {		// thread specific data
    bool in_use;
    void *value;
};
struct Pthread_keys {					// all of these fields are initialized with zero
    bool in_use;
    void (*destructor)( void * );
    uSequence<Pthread_values> threads;
};
// Create storage separately to ensure no constructors are called.
char u_pthread_keys_storage[sizeof(Pthread_keys) * PTHREAD_KEYS_MAX] __attribute__((aligned (16))) = {0};
#define u_pthread_keys ((Pthread_keys *)u_pthread_keys_storage)

static pthread_mutex_t u_pthread_keys_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t u_pthread_once_lock = PTHREAD_MUTEX_INITIALIZER;

struct Pthread_kernel_threads : public uColable {
    uProcessor processor;
    Pthread_kernel_threads() : processor( true ) {}
};
uStack<Pthread_kernel_threads> u_pthreads_kernel_threads;
static bool u_pthreads_kernel_threads_zero = false;	// set to zero ?
static int u_pthreads_no_kernel_threads = 1;		// minimum of one processor per cluster
static pthread_mutex_t u_pthread_concurrency_lock = PTHREAD_MUTEX_INITIALIZER;


//######################### uPThread #########################


_Task Pthread {						// private type
  public:
    Pthread_attr_t attr;				// pthread attributes
    void *(*start_func)( void * );
    void *arg, *joinval;				// thread parameter and join value

    pthread_t pid;					// only used if pthread_t is too small
    bool finished;					// indicates if thread's main terminated

    static unsigned int pthreadCount;
    static uOwnerLock pthreadCountLock;

    // cancellation

    uBaseCoroutine::Cleanup cleanup_handlers;

    _Nomutex static void do_unwind( uBaseCoroutine::Cleanup &cleanup_handlers, uBaseCoroutine &cor ) {
#if defined( __irix__ )
	while ( cleanup_handlers != NULL ) {		// execute cleanups
	    cleanup_handlers->__ptch_func(cleanup_handlers->__ptch_arg);
	    cleanup_handlers = cleanup_handlers->__ptch_next;
	} // while	
#else   
	for ( uBaseCoroutine::PthreadCleanup *buf = cleanup_handlers.drop(); buf != 0; buf = cleanup_handlers.drop() ) { // execute cleanups
	    buf->routine(buf->args);			// execute 
	} // for	  
#endif
	cor.unwindStack();				// Exception caught in "main" routine of current task
    } // Pthread::do_unwind

#if ! defined( __irix__ )
    template<typename Buffer> _Nomutex static void cleanup_push( void (*routine)(void *), void *args, Buffer *buffer ) {
	_STATIC_ASSERT_ (sizeof(*buffer) >= sizeof(uBaseCoroutine::PthreadCleanup));
	uBaseCoroutine::PthreadCleanup *buf = new(buffer) uBaseCoroutine::PthreadCleanup; // use the buffer data structure (on the stack) to store information
	buf->routine = routine;
	buf->args = args;

	uBaseCoroutine *c = &uThisCoroutine();
	Pthread *p = dynamic_cast<Pthread *>(c);
	if ( p != NULL ) {
	    p->cleanup_handlers.add( buf );
	} else {
	    uMain *m = dynamic_cast<uMain *>(c); // uMain has to pass as a pthread task... 
	    if ( m != NULL ) {
		m->cleanup_handlers.add(buf);
#ifdef __U_DEBUG__
	    } else {
		uAbort( "pthread_cleanup_push performed on something other than a pthread task" );	    
#endif // __U_DEBUG__
	    } // if
	} // if
    } // cleanup_push

    // specialization for freebsd
    _Nomutex static void cleanup_push( void (*routine)(void *), void *args ) {
	uBaseCoroutine::PthreadCleanup *buffer = (uBaseCoroutine::PthreadCleanup*)malloc( sizeof( uBaseCoroutine::PthreadCleanup ) );
	cleanup_push( routine, args, buffer );
    } // cleanup_push

    _Nomutex static void *cleanup_pop( int ex ) {
	uBaseCoroutine::PthreadCleanup *buf;

	uBaseCoroutine *c = &uThisCoroutine();
	Pthread *p = dynamic_cast<Pthread *>(c);
	if ( p != NULL ) {
	    assert( ! p->cleanup_handlers.empty() );
	    buf = p->cleanup_handlers.drop();
	} else {
	    uMain *m = dynamic_cast<uMain *>(c); // uMain has to pass as a pthread task... 
	    if ( m != NULL ) {
		assert( ! m->cleanup_handlers.empty() );
		buf = m->cleanup_handlers.drop();
#ifdef __U_DEBUG__
	    } else {
		uAbort( "pthread_cleanup_pop performed on something other than a pthread task" );	    
#endif // __U_DEBUG__
	    } // if
	} // if
	if ( ex ) {					// execute the handler?
	    buf->routine( buf->args );
	} // if
	return buf;
    } // cleanup_pop

#else

    // must be called only from within uMain or a Pthread    
    _Nomutex static uBaseCoroutine::Cleanup *cancel_list() {
	uBaseCoroutine *c = &uThisCoroutine();
	uMain *m = dynamic_cast<uMain *>(c);
	if ( m != NULL ) {
	    return &m->cleanup_handlers;
	} // if
	Pthread *p = dynamic_cast<Pthread *>(c);
	if ( p != NULL ) {
	    return &p->cleanup_handlers;
	} // if
	uAbort( "pthread_cleanup_push performed on something other than a pthread task" );	    
    } // __pthread_cancel_list
    
#endif //#if !defined( __irix__ )

    void main() {
	try {
	    joinval = (*start_func)( arg );
	} catch( uBaseCoroutine::UnwindStack &evt ) {	// cancellation or exit
	    evt.exec_dtor = false;			// defuse the unwinder		    
	} // try

	if ( attr.detachstate == PTHREAD_CREATE_DETACHED ) {
	    uKernelModule::systemTask->pthreadDetachEnd( *this );
	} // if

	finished = true;
    } // Pthread::main

    Pthread( void * (*start_func)( void * ), void *arg, const pthread_attr_t *attr, size_t stacksize ) :
		uBaseTask( stacksize ), start_func( start_func ), arg( arg ) {
#if defined( __irix__ )
	cleanup_handlers = NULL;
#endif
	if ( attr == NULL ) {				// construct default attribute
	    Pthread::attr = u_pthread_attr_defaults;	// copy default attributes
	    Pthread::attr.stacksize = stacksize;
	} else {
	    Pthread::attr = **((Pthread_attr_t **)attr); // copy attributes
	} // if

	finished = false;

	pthreadCountLock.acquire();
	pthreadCount += 1;
	pthreadCountLock.release();
    } // Pthread::Pthread

    ~Pthread() {
	pthreadCountLock.acquire();
	pthreadCount -= 1;
	pthreadCountLock.release();
    } // Pthread::~Pthread

    // Accepted implicitly after thread terminates and task becomes a monitor.
    void *join() {
	return joinval;
    } // Pthread::join
}; // Pthread


unsigned int Pthread::pthreadCount;
uOwnerLock Pthread::pthreadCountLock;

extern "C" void _pthread_shutdown() {
    if ( Pthread::pthreadCount > 0 ) exit( 0 );
} // _pthread_shutdown


//######################### PthreadPid #########################


class PthreadPid {
    template<bool selector>
    class Impl {
	// empty implementation -- one of the specializations is always used
    };
  public:
    static int create( pthread_t *new_thread_id, const pthread_attr_t *attr, void * (*start_func)( void * ), void *arg, size_t stacksize );
    static void join( pthread_t threadID );
    static Pthread *lookup( pthread_t threadID );
    static pthread_t rev_lookup( Pthread *addr );
    static void destroy();
}; // PthreadPid


template<>
class PthreadPid::Impl<true> {
    // best implementation, when PublicPid is large enough to hold a Pid
    friend class PthreadPid;

    static int create( pthread_t *new_thread_id, const pthread_attr_t *attr, void * (*start_func)( void * ), void *arg, size_t stacksize ) {
	*new_thread_id = (pthread_t)(long)new Pthread( start_func, arg, attr, stacksize );
	return 0;
    } // PthreadPid::Impl::create

    static void join( pthread_t threadID ) {
	delete lookup( threadID );
    } // PthreadPid::Impl::join

    static Pthread *lookup( pthread_t threadID ) {
	return (Pthread *)threadID;
    } // PthreadPid::lookup

    static pthread_t rev_lookup( Pthread *addr ) {
	return (pthread_t)(long)addr;
    } // PthreadPid::rev_lookup
    
    static void destroy() {
    } // PthreadPid::Impl::destroy
}; // PthreadPid::Impl

template<>
class PthreadPid::Impl<false> {
    // fall-back implementation when PublicLock is too small

#   define U_POWER( n ) (1 << n)
#   define U_PID_DIRECTORY_SIZE 8			// bits
#   define U_PID_NODE_SIZE 8				// bits (must be > 1)
#   define U_PID_MAX U_POWER(U_PID_DIRECTORY_SIZE + U_PID_NODE_SIZE)
#   define U_PID_NODE_NUMBER (U_POWER(U_PID_NODE_SIZE) - 2) // leave room for malloc header

    friend class PthreadPid;

    _STATIC_ASSERT_( U_PID_DIRECTORY_SIZE + U_PID_NODE_SIZE < sizeof(unsigned int) * 8 );

    static pthread_mutex_t create_lock;
    static Pthread **directory[U_POWER(U_PID_DIRECTORY_SIZE)];
    static unsigned int free_elem;
    static bool first;
    static unsigned int size;

    static void init_block( unsigned int dir ) {
	Pthread **block = directory[ dir ];
	int i;

	dir <<= U_PID_NODE_SIZE;
	for ( i = 0; i < U_PID_NODE_NUMBER - 1; i += 1 ) {
	    block[ i ] = (Pthread *)(long)(dir | i + 1);
	} // for
	// insert "NULL" value at end of free list
	block[ i ] = (Pthread *)(U_POWER(U_PID_DIRECTORY_SIZE + U_PID_NODE_SIZE) | dir);
    } // PthreadPid::Impl::init_block

    static unsigned int alloc( Pthread *tid ) {
	unsigned int block = free_elem & (U_POWER(U_PID_NODE_SIZE) - 1),
	    dir = free_elem >> U_PID_NODE_SIZE;

	if ( free_elem >= U_POWER(U_PID_DIRECTORY_SIZE + U_PID_NODE_SIZE) ) {
	    dir = (U_POWER(U_PID_DIRECTORY_SIZE) - 1) & dir; // remove high-order end-marker
	    if ( dir == (U_POWER(U_PID_DIRECTORY_SIZE) - 1) ) {
		return UINT_MAX;
	    } else {
		if ( first ) {
		    first = false;
		} else {
		    dir += 1;
		} // if
		directory[ dir ] = new Pthread *[U_PID_NODE_NUMBER];
		init_block( dir );
		free_elem = dir << U_PID_NODE_SIZE;
	    } // if
	} // if
	    
	unsigned int rtn = free_elem;
	free_elem = (unsigned int)(long)directory[ dir ][ block ];
	directory[ dir ][ block ] = tid;

	return rtn;
    } // PthreadPid::Impl::alloc

    static void free( unsigned int pid ) {
	unsigned int block = pid & (U_POWER(U_PID_NODE_SIZE) - 1),
	    dir = pid >> U_PID_NODE_SIZE;
    
	directory[ dir ][ block ] = (Pthread *)(long)free_elem;
	free_elem = pid;
    } // PthreadPid::Impl::free

    static int create( pthread_t *new_thread_id, const pthread_attr_t *attr, void * (*start_func)( void * ), void *arg, size_t stacksize ) {
	pthread_mutex_lock( &create_lock );
      if ( size == U_PID_MAX ) {
	    pthread_mutex_unlock( &create_lock );
	    return EAGAIN;
	} // if
	Pthread *tid = new Pthread( start_func, arg, attr, stacksize );
	tid->pid = (pthread_t)alloc( tid );		// not called if no space remaining
	size += 1;
	pthread_mutex_unlock( &create_lock );
	*new_thread_id = tid->pid;
	return 0;
    } // PthreadPid::Impl::create
    
    static void join( pthread_t threadID ) {
	delete PthreadPid::lookup( threadID );
	pthread_mutex_lock( &create_lock );
	free( (unsigned)threadID );
	size -= 1;
	pthread_mutex_unlock( &create_lock );
    } // PthreadPid::Impl::join

    static Pthread *lookup( pthread_t pid ) {
	return directory[ (unsigned)pid >> U_PID_DIRECTORY_SIZE ][ (unsigned)pid & (U_POWER(U_PID_NODE_SIZE) - 1) ];
    } // PthreadPid::lookup

    static pthread_t rev_lookup( Pthread *addr ) {
	return (((Pthread *)addr)->pid);
    } // PthreadPid::rev_lookup
    
    static void destroy() {
	for ( int i = 0; i < U_POWER(U_PID_DIRECTORY_SIZE) && directory[ i ] != NULL; i += 1 ) {
	    delete [] directory[ i ];
	} // for
    } // PthreadPid::Impl::destroy
}; // PthreadPid::Impl


pthread_mutex_t PthreadPid::Impl<false>::create_lock = PTHREAD_MUTEX_INITIALIZER;
Pthread **PthreadPid::Impl<false>::directory[U_POWER(U_PID_DIRECTORY_SIZE)];
unsigned int PthreadPid::Impl<false>::free_elem = U_POWER(U_PID_DIRECTORY_SIZE + U_PID_NODE_SIZE);
bool PthreadPid::Impl<false>::first = true;
unsigned int PthreadPid::Impl<false>::size = 0;


inline int PthreadPid::create( pthread_t *new_thread_id, const pthread_attr_t *attr, void * (*start_func)( void * ), void *arg, size_t stacksize ) {
    return Impl< sizeof(pthread_t) >= sizeof(Pthread *) >::create( new_thread_id, attr, start_func, arg, stacksize );
} // PthreadPid::create

inline void PthreadPid::join( pthread_t threadID ) {
    Impl< sizeof(pthread_t) >= sizeof(Pthread *) >::join( threadID );
} // PthreadPid::join

inline Pthread *PthreadPid::lookup( pthread_t threadID ) {
    return Impl< sizeof(pthread_t) >= sizeof(Pthread *) >::lookup( threadID );
} // PthreadPid::lookup

inline pthread_t PthreadPid::rev_lookup( Pthread *addr ) {
    return Impl< sizeof(pthread_t) >= sizeof(Pthread *) >::rev_lookup( addr );
} // PthreadPid::rev_lookup

inline void PthreadPid::destroy() {
    return Impl< sizeof(pthread_t) >= sizeof(Pthread *) >::destroy();
} // PthreadPid::destroy


//######################### PthreadLock #########################


class PthreadLock {
    template<typename Lock, bool selector>
    class Impl {
	// empty implementation -- one of the specializations is always used
    };
  public:
    template<typename Lock, class PublicLock>
    static void init( PublicLock *lock ) {
	Impl< Lock, sizeof( PublicLock ) >= sizeof( Lock ) >::init( lock );
    } // PthreadLock::init

    template<typename Lock, class PublicLock>
    static void destroy( PublicLock *lock ) {
	Impl< Lock, sizeof( PublicLock ) >= sizeof( Lock ) >::destroy( lock );
    } // PthreadLock::destroy

    template<typename Lock, class PublicLock>
    static Lock *get( PublicLock *lock ) {
	return Impl< Lock, sizeof( PublicLock ) >= sizeof( Lock ) >::get( lock );
    } // PthreadLock::get
}; // PthreadLock


template<typename Lock>
class PthreadLock::Impl<Lock, true> {
    // best implementation, when PublicLock is large enough to hold a Lock
    friend class PthreadLock;

    template< class PublicLock >
    static void init( PublicLock *lock ) {
	new( lock ) Lock;				// run constructor on supplied storage
    } // PthreadLock::Impl::init

    template< class PublicLock >
    static void destroy( PublicLock *lock ) {
	((Lock *)lock)->~Lock();			// run destructor on supplied storage
    } // PthreadLock::Impl::destroy
    
    template< class PublicLock >
    static Lock *get( PublicLock *lock ) {
	return (Lock*)lock;
    } // PthreadLock::Impl::get
}; // PthreadLock::Impl

template<typename Lock>
class PthreadLock::Impl<Lock, false> {
    // fall-back implementation when PublicLock is too small
    friend class PthreadLock;

#ifdef __solaris__
    // store pointer in the second word to avoid Solaris magic number
    struct storage {
	uint64_t dummy;					// contains Solaris magic number
	Lock *lock;
    };
#else
    struct storage {
	Lock *lock;
    };
#endif

    template< class PublicLock >
    static void init( PublicLock *lock ) {
#ifdef __solaris__
	((storage *)lock)->dummy = 0;			// zero Solaris magic number
#endif
	((storage *)lock)->lock = new Lock;
    } // PthreadLock::Impl::init
    
    template< class PublicLock >
    static void destroy( PublicLock *lock ) {
	delete ((storage *)lock)->lock;
    } // PthreadLock::Impl::destroy
    
    template< class PublicLock >
    static Lock *get( PublicLock *lock ) {
	Lock *l = ((storage *)lock)->lock;		// optimization
	if ( l == NULL ) {
	    init( lock );
	    l = ((storage *)lock)->lock;		// get the newly initialized pointer
	} // if
	assert( l != NULL );
	return l;
    } // PthreadLock::Impl::get
}; // PthreadLock::Impl


//######################### pthread simulation #########################


extern "C" {
    //######################### Creation #########################


    void _pthread_pid_destroy( void ) {			// see uKernelBoot::finishup
	PthreadPid::destroy();
    } // _pthread_pid_destroy


    int pthread_create( pthread_t *new_thread_id, const pthread_attr_t *attr, void * (*start_func)( void * ), void *arg ) __THROW {
	size_t stacksize = ( attr != NULL ) ? (*(Pthread_attr_t **)attr)->stacksize : uDefaultStackSize();
	return PthreadPid::create( new_thread_id, attr, start_func, arg, stacksize );
    } // pthread_create

    int pthread_attr_init( pthread_attr_t *attr ) __THROW {
	// storage for pthread_attr_t must be >= void *
	_STATIC_ASSERT_( sizeof(pthread_attr_t) >= sizeof(void *) );
	*((Pthread_attr_t **)attr) = new Pthread_attr_t;
	**((Pthread_attr_t **)attr) = u_pthread_attr_defaults; // set all fields to zero
	return 0;
    } //  pthread_attr_init

    int pthread_attr_destroy( pthread_attr_t *attr ) __THROW {
	delete *((Pthread_attr_t **)attr);
	return 0;
    } // pthread_attr_destroy

    int pthread_attr_setscope( pthread_attr_t *attr, int contentionscope ) __THROW {
	(*(Pthread_attr_t **)attr)->contentionscope = contentionscope;
	return 0;
    } // pthread_attr_setscope

    int pthread_attr_getscope( const pthread_attr_t *attr, int *contentionscope ) __THROW {
	*contentionscope = (*(Pthread_attr_t **)attr)->contentionscope;
	return 0;
    } // pthread_attr_getscope

    int pthread_attr_setdetachstate( pthread_attr_t *attr, int detachstate ) __THROW {
	(*(Pthread_attr_t **)attr)->detachstate = detachstate;
	return 0;
    } // pthread_attr_setdetachstate

    int pthread_attr_getdetachstate( const pthread_attr_t *attr, int *detachstate ) __THROW {
	*detachstate = (*(Pthread_attr_t **)attr)->detachstate;
	return 0;
    } // pthread_attr_getdetachstate

    int pthread_attr_getguardsize( const pthread_attr_t *attr, size_t *guardsize ) __THROW {
	guardsize = 0;					// guard area is not provided
	return 0;
    } // pthread_attr_getguardsize

    int pthread_attr_setguardsize( pthread_attr_t *attr, size_t guardsize ) __THROW {
	return 0;
    } // pthread_attr_setguardsize

    int pthread_attr_setstacksize( pthread_attr_t *attr, size_t stacksize ) __THROW {
	(*(Pthread_attr_t **)attr)->stacksize = stacksize;
	return 0;
    } // pthread_attr_setstacksize

    int pthread_attr_getstacksize( const pthread_attr_t *attr, size_t *stacksize ) __THROW {
	*stacksize = (*(Pthread_attr_t **)attr)->stacksize;
	return 0;
    } // pthread_attr_getstacksize

    int pthread_attr_setstackaddr( pthread_attr_t *attr, void *stackaddr ) __THROW {
	(*(Pthread_attr_t **)attr)->stackaddr = stackaddr;
	return ENOSYS;					// unsupported
    } // pthread_attr_setstackaddr

    int pthread_attr_getstackaddr( const pthread_attr_t *attr, void **stackaddr ) __THROW {
	*stackaddr = (*(Pthread_attr_t **)attr)->stackaddr;
	return ENOSYS;					// unsupported
    } // pthread_attr_getstackaddr

    int pthread_attr_setstack( pthread_attr_t *attr, void *stackaddr, size_t stacksize ) __THROW {
	(*(Pthread_attr_t **)attr)->stackaddr = stackaddr;
	(*(Pthread_attr_t **)attr)->stacksize = stacksize;
	return ENOSYS;					// unsupported
    } // pthread_attr_setstack

    int pthread_attr_getstack( const pthread_attr_t *attr, void **stackaddr, size_t *stacksize ) __THROW {
	*stackaddr = (*(Pthread_attr_t **)attr)->stackaddr;
	*stacksize = (*(Pthread_attr_t **)attr)->stacksize;
	return 0;
    } // pthread_attr_getstack

    // Initialize thread attribute *attr with attributes corresponding to the
    // already running thread threadID. It shall be called on unitialized attr
    // and destroyed with pthread_attr_destroy when no longer needed.
    int pthread_getattr_np( pthread_t threadID, pthread_attr_t *attr ) __THROW { // GNU extension
	assert( threadID != NOT_A_PTHREAD );
	// race condition during copy
	pthread_attr_init( attr );
	**((Pthread_attr_t **)attr) = PthreadPid::lookup( threadID )->attr; // copy all fields
	return 0;
    } // pthread_getattr_np

    int pthread_attr_setschedpolicy( pthread_attr_t *attr, int policy ) __THROW {
	(*(Pthread_attr_t **)attr)->policy = policy;
	return ENOSYS;					// unsupported
    } // pthread_attr_setschedpolicy

    int pthread_attr_getschedpolicy( const pthread_attr_t *attr, int *policy ) __THROW {
	*policy = (*(Pthread_attr_t **)attr)->policy;
	return ENOSYS;
    } // pthread_attr_getschedpolicy

    int pthread_attr_setinheritsched( pthread_attr_t *attr, int inheritsched ) __THROW {
	(*(Pthread_attr_t **)attr)->inheritsched = inheritsched;
	return ENOSYS;
    } // pthread_attr_setinheritsched

    int pthread_attr_getinheritsched( const pthread_attr_t *attr, int *inheritsched ) __THROW {
	*inheritsched = (*(Pthread_attr_t **)attr)->inheritsched;
	return ENOSYS;
    } // pthread_attr_getinheritsched

    int pthread_attr_setschedparam( pthread_attr_t *attr, const struct sched_param *param ) __THROW {
	(*(Pthread_attr_t **)attr)->param = *param;
	return ENOSYS;
    } // pthread_attr_setschedparam

    int pthread_attr_getschedparam( const pthread_attr_t *attr, struct sched_param *param ) __THROW {
	*param = ((Pthread_attr_t *)attr)->param;
	return ENOSYS;
    } // pthread_attr_getschedparam

#if defined( __freebsd__ )
    void pthread_yield( void ) {			// GNU extension
	uThisTask().uYieldNoPoll();
	pthread_testcancel();				// pthread_yield is a cancellation point
    } // pthread_yield
#else
    int pthread_yield( void ) __THROW {			// GNU extension
	uThisTask().uYieldNoPoll();
	pthread_testcancel();				// pthread_yield is a cancellation point
	return 0;
    } // pthread_yield
#endif // __freebsd__


    //######################### Exit #########################


    void pthread_exit( void *status ) {
	uBaseCoroutine *c = &uThisCoroutine();
	Pthread *p = dynamic_cast<Pthread *>(c);
	if ( p != NULL ) {
	    p->joinval = status;
	    Pthread::Pthread::do_unwind(p->cleanup_handlers, *p);
	    // CONTROL NEVER REACHES HERE!
#ifdef __U_DEBUG__
	    assert( false );
#endif // __U_DEBUG__
	} // if
	uMain *m = dynamic_cast<uMain *>(c); // uMain has to pass as a pthread task... 
	if ( m != NULL ) {
	    m->joinval = status;
	    Pthread::do_unwind(m->cleanup_handlers, *m);
	    // CONTROL NEVER REACHES HERE!
#ifdef __U_DEBUG__
	    assert( false );
#endif // __U_DEBUG__
	} // if
#ifdef __U_DEBUG__
	uAbort( "pthread_exit performed on non-pthread task or while executing a uC++ coroutine" );
	// CONTROL NEVER REACHES HERE!
#ifdef __U_DEBUG__
	assert( false );
#endif // __U_DEBUG__
#endif // __U_DEBUG__
    } // pthread_exit

    int pthread_join( pthread_t threadID, void **status ) {
	assert( threadID != NOT_A_PTHREAD );
	pthread_testcancel();				// pthread_join is a cancellation point
	Pthread *p = PthreadPid::lookup( threadID );
      if ( p->attr.detachstate != PTHREAD_CREATE_JOINABLE ) return EINVAL;
      if ( p == &uThisTask() ) return EDEADLK; 
	// join() is only necessary for status; otherwise synchronization occurs at delete.
	if ( status != NULL ) *status = p->join();
	PthreadPid::join( threadID );
	return 0;
    } // pthread_join

    int pthread_tryjoin_np( pthread_t threadID, void **status ) __THROW { // GNU extension
	Pthread *p = PthreadPid::lookup( threadID );
      if ( p->attr.detachstate != PTHREAD_CREATE_JOINABLE ) return EINVAL;
      if ( p == &uThisTask() ) return EDEADLK;
      	// race condition during check
      if ( ! p->finished ) return EBUSY;		// thread not finished ?
	if ( status != NULL ) *status = p->join();
	PthreadPid::join( threadID );
	return 0;
    } // pthread_tryjoin_np

    int pthread_timedjoin_np( pthread_t threadID, void **status, const struct timespec *abstime ) { // GNU extension
	uAbort( "pthread_timedjoin_np : not implemented" );
	return 0;
    } // pthread_timedjoin_np

    int pthread_detach( pthread_t threadID ) __THROW {
	// There is always a race condition in setting the detach flag, even if
	// "detach" is accessed with mutual exclusion.
	PthreadPid::lookup( threadID )->attr.detachstate = PTHREAD_CREATE_DETACHED;
	return 0;
    } // pthread_detach


    //######################### Parallelism #########################


    void _pthread_delete_kernel_threads() __THROW { // see uMain::~uMain
	Pthread_kernel_threads *p;
	for ( uStackIter<Pthread_kernel_threads> iter( u_pthreads_kernel_threads ); iter >> p; ) {
	    delete p;
	} // for
    } // _pthread_delete_kernel_threads

    int pthread_getconcurrency( void ) __THROW {	// XOPEN extension
	return u_pthreads_kernel_threads_zero ? 0 : u_pthreads_no_kernel_threads;
    } // pthread_getconcurrency

    int pthread_setconcurrency( int new_level ) __THROW { // XOPEN extension
      if ( new_level < 0 ) return EINVAL;
      if ( new_level == 0 ) {
	  u_pthreads_kernel_threads_zero = true;	// remember set to zero, but ignore
	  return 0;					// uC++ does not do kernel thread management
	} // exit
	u_pthreads_kernel_threads_zero = false;
	pthread_mutex_lock( &u_pthread_concurrency_lock );
	for ( ; new_level > u_pthreads_no_kernel_threads; u_pthreads_no_kernel_threads += 1 ) { // add processors ?
	    u_pthreads_kernel_threads.push( new Pthread_kernel_threads );
	} // for
	for ( ; new_level < u_pthreads_no_kernel_threads; u_pthreads_no_kernel_threads -= 1 ) { // remove processors ?
	    Pthread_kernel_threads *p = u_pthreads_kernel_threads.pop();
	    delete p;
	} // for
	pthread_mutex_unlock( &u_pthread_concurrency_lock );
	return 0;
    } // pthread_setconcurrency


    //######################### Thread Specific Data #########################


    void _pthread_deletespecific( void *pthreadData ) __THROW { // see uMachContext::invokeTask
	pthread_mutex_lock( &u_pthread_keys_lock );
	Pthread_values *values = (Pthread_values *)pthreadData;

	// If, after all the destructors have been called for all non-NULL
	// values with associated destructors, there are still some non-NULL
	// values with associated destructors, then the process is
	// repeated. If, after at least PTHREAD_DESTRUCTOR_ITERATIONS
	// iterations of destructor calls for outstanding non-NULL values,
	// there are still some non-NULL values with associated destructors,
	// implementations may stop calling destructors, or they may continue
	// calling destructors until no non-NULL values with associated
	// destructors exist, even though this might result in an infinite
	// loop.
				   
	bool destcalled = true;
	for ( int attempts = 0; attempts < PTHREAD_DESTRUCTOR_ITERATIONS && destcalled ; attempts += 1 ) {
	    destcalled = false;
	    for ( int i = 0; i < PTHREAD_KEYS_MAX; i += 1 ) { // remove each value from key list
#ifdef __U_DEBUG_H__
		if ( values[i].in_use ) {
		    uDebugPrt( "_pthread_deletespecific, value[%d].in_use:%d, value:0x%p\n",
			       i, values[i].in_use, values[i].value );
		} // if
#endif // __U_DEBUG_H__
		if ( values[i].in_use ) {
		    Pthread_values &entry = values[i];
		    entry.in_use = false;
		    u_pthread_keys[i].threads.remove( &entry );
		    if ( u_pthread_keys[i].destructor != NULL && entry.value != NULL ) {
			void *data = entry.value;
			entry.value = NULL;
#ifdef __U_DEBUG_H__
			uDebugPrt( "_pthread_deletespecific, task:0x%p, destructor:0x%p, value:0x%p begin\n",
				   &uThisTask(), u_pthread_keys[i].destructor, data );
#endif // __U_DEBUG_H__
			destcalled = true;
			pthread_mutex_unlock( &u_pthread_keys_lock );
			u_pthread_keys[i].destructor( data );
			pthread_mutex_lock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
			uDebugPrt( "_pthread_deletespecific, task:0x%p, destructor:0x%p, value:0x%p end\n",
				   &uThisTask(), u_pthread_keys[i].destructor, data );
#endif // __U_DEBUG_H__
		    } // if
		} // if
	    } // for
	} // for
	pthread_mutex_unlock( &u_pthread_keys_lock );
	delete [] values;
    } // _pthread_deletespecific


    int pthread_key_create( pthread_key_t *key, void (*destructor)( void * ) ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_create(key:0x%p, destructor:0x%p) enter task:0x%p\n", key, destructor, &uThisTask() );
#endif // __U_DEBUG_H__
	pthread_mutex_lock( &u_pthread_keys_lock );
	for ( int i = 0; i < PTHREAD_KEYS_MAX; i += 1 ) {
 	    if ( ! u_pthread_keys[i].in_use ) {
		u_pthread_keys[i].in_use = true;
		u_pthread_keys[i].destructor = destructor;
		pthread_mutex_unlock( &u_pthread_keys_lock );
		*key = i;
#ifdef __U_DEBUG_H__
		uDebugPrt( "pthread_key_create(key:%d, destructor:0x%p) exit task:0x%p\n", *key, destructor, &uThisTask() );
#endif // __U_DEBUG_H__
		return 0;
	    } // if
	} // for
	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_create(key:0x%p, destructor:0x%p) try again!!! task:0x%p\n", key, destructor, &uThisTask() );
#endif // __U_DEBUG_H__
	return EAGAIN;
    } // pthread_key_create

    int pthread_key_delete( pthread_key_t key ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_delete(key:0x%x) enter task:0x%p\n", key, &uThisTask() );
#endif // __U_DEBUG_H__
	pthread_mutex_lock( &u_pthread_keys_lock );
      if ( key >= PTHREAD_KEYS_MAX || ! u_pthread_keys[key].in_use ) {
	    pthread_mutex_unlock( &u_pthread_keys_lock );
	    return EINVAL;
	} // if
	u_pthread_keys[key].in_use = false;
	u_pthread_keys[key].destructor = NULL;
	
	// Remove key from all threads with a value.

	Pthread_values *p;
	uSequence<Pthread_values> &head = u_pthread_keys[key].threads;
	for ( uSeqIter<Pthread_values> iter( head ); iter >> p; ) {
	    p->in_use = false;
	    head.remove( p ); 
	} // for

	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_key_delete(key:0x%x) exit task:0x%p\n", key, &uThisTask() );
#endif // __U_DEBUG_H__
	return 0;
    } // pthread_key_delete

    int pthread_setspecific( pthread_key_t key, const void *value ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_setspecific(key:0x%x, value:0x%p) enter task:0x%p\n", key, value, &uThisTask() );
#endif // __U_DEBUG_H__
	uBaseTask &t = uThisTask();

	Pthread_values *values;
	if ( t.pthreadData == NULL ) {
	    values = new Pthread_values[PTHREAD_KEYS_MAX];
	    t.pthreadData = values;
	    for ( int i = 0; i < PTHREAD_KEYS_MAX; i += 1 ) {
		values[i].in_use = false;
	    } // for
	} else {
	    values = (Pthread_values *)t.pthreadData;
	} // if

	pthread_mutex_lock( &u_pthread_keys_lock );
      if ( key >= PTHREAD_KEYS_MAX || ! u_pthread_keys[key].in_use ) {
	    pthread_mutex_unlock( &u_pthread_keys_lock );
	    return EINVAL;
	} // if

	Pthread_values &entry = values[key];
	if ( ! entry.in_use ) {
	    entry.in_use = true;
	    u_pthread_keys[key].threads.add( &entry );
	} // if
	entry.value = (void *)value;
	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_setspecific(key:0x%x, value:0x%p) exit task:0x%p\n", key, value, &uThisTask() );
#endif // __U_DEBUG_H__
	return 0;
    } // pthread_setspecific

    void *pthread_getspecific( pthread_key_t key ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_getspecific(key:0x%x) enter task:0x%p\n", key, &uThisTask() );
#endif // __U_DEBUG_H__
      if ( key >= PTHREAD_KEYS_MAX ) return NULL;

	uBaseTask &t = uThisTask();
      if ( t.pthreadData == NULL ) return NULL;

	pthread_mutex_lock( &u_pthread_keys_lock );
	Pthread_values &entry = ((Pthread_values *)t.pthreadData)[key];
      if ( ! entry.in_use ) {
	    pthread_mutex_unlock( &u_pthread_keys_lock );
	    return NULL;
	} // if
	void *value = entry.value;
	pthread_mutex_unlock( &u_pthread_keys_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "0x%p = pthread_getspecific(key:0x%x) exit task:0x%p\n", value, key, &uThisTask() );
#endif // __U_DEBUG_H__
	return value;
    } // pthread_getspecific


    //######################### Signal #########################


    int pthread_sigmask( int how, const sigset_t *set, sigset_t *oset ) __THROW {
	return 0;
    } // pthread_sigmask

    int pthread_kill( pthread_t thread, int sig ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_kill( 0x%lx, %d )\n", thread, sig );
#endif // __U_DEBUG_H__
	assert( thread != NOT_A_PTHREAD );
	if ( sig == 0 ) {
	    return 0;
	} else {
	    uAbort( "pthread_kill : not implemented" );
	} // if
	return 0;
    } // pthread_kill


    //######################### ID #########################


    pthread_t pthread_self( void ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_self enter task:0x%p\n", &uThisTask() );
#endif // __U_DEBUG_H__
#ifdef __U_DEBUG__
	Pthread *p = dynamic_cast<Pthread *>(&uThisTask());
      if ( p == NULL ) {
	    return NOT_A_PTHREAD;
	} // if
#endif // __U_DEBUG__
	return PthreadPid::rev_lookup( (Pthread *)&uThisTask() );
    } // pthread_self

#if ! defined( __irix__ )				// TEMPORARY: done with macro: bad dog!
    int pthread_equal( pthread_t t1, pthread_t t2 ) __THROW {
	return t1 == t2;
    } // pthread_equal
#endif // ! __irix__

    int pthread_once( pthread_once_t *once_control, void (*init_routine)( void ) ) __OLD_THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_once(once_control:0x%p, init_routine:0x%p) enter task:0x%p\n", once_control, init_routine, &uThisTask() );
#endif // __U_DEBUG_H__
	// storage for pthread_once_t must be >= int and initialized to zero by
	// PTHREAD_ONCE_INIT
	_STATIC_ASSERT_( sizeof(pthread_once_t) >= sizeof(int) );
	pthread_mutex_lock( &u_pthread_once_lock );
	if ( *((int *)once_control) == 0 ) {
	    init_routine();
	    *((int *)once_control) = 1;
	} // if
	pthread_mutex_unlock( &u_pthread_once_lock );
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_once(once_control:0x%p, init_routine:0x%p) exit task:0x%p\n", once_control, init_routine, &uThisTask() );
#endif // __U_DEBUG_H__
	return 0;
    } // pthread_once


    //######################### Scheduling #########################


    int pthread_setschedparam( pthread_t thread, int policy, const struct sched_param *param ) __THROW {
	uAbort( "pthread_setschedparam : not implemented" );
	return 0;
    } // pthread_setschedparam
    int pthread_getschedparam( pthread_t thread, int *policy, struct sched_param *param ) __THROW {
	uAbort( "pthread_getschedparam : not implemented" );
	return 0;
    } // pthread_getschedparam


    //######################### Cancellation #########################


    int pthread_cancel( pthread_t threadID ) __OLD_THROW {
	assert( threadID != NOT_A_PTHREAD );
	Pthread *p = PthreadPid::lookup( threadID );
	p->cancel();
	return 0;
    } // pthread_cancel

    int pthread_setcancelstate( int state, int *oldstate ) __OLD_THROW {
      if ( state != PTHREAD_CANCEL_ENABLE && state != PTHREAD_CANCEL_DISABLE ) return EINVAL;
        uBaseCoroutine *c = &uThisCoroutine();
#ifdef __U_DEBUG__
	Pthread *p = dynamic_cast<Pthread *>(c);
	uMain *m = dynamic_cast<uMain *>(c);
	if ( p == NULL && m == NULL ) uAbort( "pthread_setcancelstate performed on non-pthread task  or while executing a uC++ coroutine" );
#endif // __U_DEBUG__
	if ( oldstate != NULL ) *oldstate = c->getCancelState();
	c->setCancelState( (uBaseCoroutine::CancellationState)state );      
	return 0;
    } // pthread_setcancelstate

    int pthread_setcanceltype( int type, int *oldtype ) __OLD_THROW {
	// currnetly do not support asynchronous cancellation
      if ( type != PTHREAD_CANCEL_DEFERRED && type != PTHREAD_CANCEL_ASYNCHRONOUS ) return EINVAL;
	uBaseCoroutine *c = &uThisCoroutine();
#ifdef __U_DEBUG__
	Pthread *p = dynamic_cast<Pthread *>(c);
	uMain *m = dynamic_cast<uMain *>(c);
	if ( p == NULL && m == NULL ) uAbort( "pthread_setcancelstate performed on non-pthread task or while executing a uC++ coroutine" );
#endif // __U_DEBUG__
	if ( oldtype != NULL ) *oldtype = c->getCancelType();
	c->setCancelType( (uBaseCoroutine::CancellationType)type );
	return 0;
    } // pthread_setcanceltype

    void pthread_testcancel( void ) __OLD_THROW {
	uBaseCoroutine *c = &uThisCoroutine();
	Pthread *p = dynamic_cast<Pthread *>(c);
	uMain *m = dynamic_cast<uMain *>(c);		// uMain has to pass as a pthread task... 
	if ( p == NULL && m == NULL ) return;
      if ( c->getCancelState() == (uBaseCoroutine::CancellationState)PTHREAD_CANCEL_DISABLE || ! c->cancelled() ) return;
#if defined( __U_BROKEN_CANCEL__ )
	uAbort( "pthread cancellation is not supported on this system" );
#else
	pthread_exit( PTHREAD_CANCELED );
#endif // __U_BROKEN_CANCEL__
	// CONTROL NEVER REACHES HERE
#ifdef __U_DEBUG__
	assert( false );
#endif // __U_DEBUG__
    } // pthread_testcancel


#if defined( __solaris__ )
  asm(
" .file \"pthread.cc\"\n\
   .section \".text\"\n\
   .align 4\n\
   .global _getfp\n\
   .type _getfp,#function\n\
_getfp:\n\
     retl\n\
     mov %fp, %o0\n\
   .size _getfp,(.-_getfp)"
	);
#endif

extern "C" {
#if ! defined( __irix__ )

#if defined( __solaris__ ) || defined( __linux__ )
#if defined( __solaris__ )
    void __pthread_cleanup_push( void (*routine)(void *), void *args, caddr_t fp, _cleanup_t *buffer ) {
#elif defined( __linux__ )
    void _pthread_cleanup_push( struct _pthread_cleanup_buffer *buffer, void (*routine) (void *), void *args ) __THROW {
#endif // __solaris__ || __linux__
	Pthread::cleanup_push( routine, args, buffer );
#elif defined( __freebsd__ )
    void pthread_cleanup_push( void (*routine) (void *), void *args ) {
	Pthread::cleanup_push( routine, args );
#else
    #error uC++ : internal error, unsupported architecture
#endif // __solaris__
    } // pthread_cleanup_push

#if defined( __solaris__ ) || defined( __linux__ )
#if defined( __solaris__ )
    void __pthread_cleanup_pop( int ex, _cleanup_t *buffer ) {
#elif defined( __linux__ )
    void _pthread_cleanup_pop ( struct _pthread_cleanup_buffer *buffer, int ex ) __THROW {
#endif // __solaris__
	Pthread::cleanup_pop( ex );
#elif defined( __freebsd__ )
    void pthread_cleanup_pop( int ex ) {
	free( Pthread::cleanup_pop( ex ) );
#endif // __solaris__ || __linux__
    } // pthread_cleanup_pop

#else

    __pthread_cncl_hdlr **__pthread_cancel_list() {
	return Pthread::cancel_list();
    } // __pthread_cancel_list

#endif //#if !defined( __irix__ )
} // extern "C"


    //######################### Mutex #########################


    int pthread_mutex_init( pthread_mutex_t *mutex, const pthread_mutexattr_t *attr ) __THROW {
	PthreadLock::init< uOwnerLock >( mutex );
	return 0;
    } // pthread_mutex_init

    int pthread_mutex_destroy( pthread_mutex_t *mutex ) __THROW {
	PthreadLock::destroy< uOwnerLock >( mutex );
	return 0;
    } // pthread_mutex_destroy

    int pthread_mutex_lock( pthread_mutex_t *mutex ) __THROW {
	if ( ! uKernelModule::kernelModuleInitialized ) {
	    uKernelModule::startup();
	} // if

      if ( THREAD_GETMEM( disableInt) ) return 0; // TEMPORARY: profiler allocating memory from the kernel issue
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_mutex_lock(mutex:0x%p) enter task:0x%p\n", mutex, &uThisTask() );
#endif // __U_DEBUG_H__
#if defined( __solaris__ )
	// Solaris has a magic value at byte 6 of its pthread locks. Check if it moves.
	_STATIC_ASSERT_( offsetof( _pthread_mutex, __pthread_mutex_flags.__pthread_mutex_magic ) == 6 );
	if ( mutex->__pthread_mutex_flags.__pthread_mutex_magic == MUTEX_MAGIC ) pthread_mutex_init( mutex, NULL );
#elif defined(__linux__)
	// SKULLDUGGERY: Linux distributions may use different definitions for pthread_mutex_t, but they should
	// always be layout compatible, and hence the "kind" field should always be in a predictable location.
	// Unfortunately, there is no portable way to check this.
	struct assumed_mutex {
	    int __m_reserved;               /* Reserved for future use */
	    int __m_count;                  /* Depth of recursive locking */
	    void *__m_owner;                /* Owner thread (if recursive or errcheck) */
	    int __m_kind;                   /* Mutex kind: fast, recursive or errcheck */
	};
	if ( ( 0 < ((assumed_mutex*)mutex)->__m_kind && ((assumed_mutex*)mutex)->__m_kind < 10 ) ||
             ( *(unsigned long*)&((assumed_mutex*)mutex)->__m_kind != 0 && ((assumed_mutex*)mutex)->__m_owner == 0 ) ) {
	    pthread_mutex_init( mutex, NULL );
	} // if
#endif
	PthreadLock::get< uOwnerLock >( mutex )->acquire();
	return 0;
    } // pthread_mutex_lock

    int pthread_mutex_trylock( pthread_mutex_t *mutex ) __THROW {
      if ( THREAD_GETMEM( disableInt) ) return 0; // TEMPORARY: profiler allocating memory from the kernel issue
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_mutex_trylock(mutex:0x%p) enter task:0x%p\n", mutex, &uThisTask() );
#endif // __U_DEBUG_H__
#if defined( __solaris__ )
	if ( mutex->__pthread_mutex_flags.__pthread_mutex_magic == MUTEX_MAGIC ) pthread_mutex_init( mutex, NULL );
#endif // __solaris__
	return PthreadLock::get< uOwnerLock >( mutex )->tryacquire() ? 0 : EBUSY;
    } // pthread_mutex_trylock


    int pthread_mutex_unlock( pthread_mutex_t *mutex ) __THROW {
      if ( THREAD_GETMEM( disableInt) ) return 0; // TEMPORARY: profiler allocating memory from the kernel issue
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_mutex_unlock(mutex:0x%p) enter task:0x%p\n", mutex, &uThisTask() );
#endif // __U_DEBUG_H__
	PthreadLock::get< uOwnerLock >( mutex )->release();
	return 0;
    } // pthread_mutex_unlock


    int pthread_mutexattr_init( pthread_mutexattr_t *attr ) __THROW {
	return 0;
    } // pthread_mutexattr_init

    int pthread_mutexattr_destroy( pthread_mutexattr_t *attr ) __THROW {
	return 0;
    } // pthread_mutexattr_destroy

    int pthread_mutexattr_setpshared( pthread_mutexattr_t *attr, int pshared ) __THROW {
	return 0;
    } // pthread_mutexattr_setpshared

    int pthread_mutexattr_getpshared( const pthread_mutexattr_t *attr, int *pshared ) __THROW {
	return 0;
    } // pthread_mutexattr_getpshared

    int pthread_mutexattr_setprotocol( pthread_mutexattr_t *attr, int protocol ) __THROW {
	return 0;
    } // pthread_mutexattr_setprotocol

#if defined( __freebsd__ )
    int pthread_mutexattr_getprotocol( pthread_mutexattr_t *attr, int *protocol ) __THROW {
#else
    int pthread_mutexattr_getprotocol( const pthread_mutexattr_t *attr, int *protocol ) __THROW {
#endif // __freebsd__
	return 0;
    } // pthread_mutexattr_getprotocol

    int pthread_mutexattr_setprioceiling( pthread_mutexattr_t *attr, int prioceiling ) __THROW {
	return 0;
    } // pthread_mutexattr_setprioceiling

#if defined( __freebsd__ )
    int pthread_mutexattr_getprioceiling( pthread_mutexattr_t *attr, int *ceiling ) __THROW {
#else
    int pthread_mutexattr_getprioceiling( const pthread_mutexattr_t *attr, int *ceiling ) __THROW {
#endif // __freebsd__
	return 0;
    } // pthread_mutexattr_getprioceiling

    int pthread_mutex_setprioceiling( pthread_mutex_t *mutex, int prioceiling, int *old_ceiling ) __THROW {
	return 0;
    } // pthread_mutex_setprioceiling

#if defined( __freebsd__ )
    int pthread_mutex_getprioceiling( pthread_mutex_t *mutex, int *ceiling ) __THROW {
#else
    int pthread_mutex_getprioceiling( const pthread_mutex_t *mutex, int *ceiling ) __THROW {
#endif // __freebsd__
	return 0;
    } // pthread_mutex_getprioceiling


    //######################### Condition #########################


    int pthread_cond_init( pthread_cond_t *cond, const pthread_condattr_t *attr ) __THROW {
	PthreadLock::init< uCondLock >( cond );
	return 0;
    } // pthread_cond_init

    int pthread_cond_destroy( pthread_cond_t *cond ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_cond_destroy(cond:0x%p) task:0x%p\n", cond, &uThisTask() );
#endif // __U_DEBUG_H__
	PthreadLock::destroy< uCondLock >( cond );
	return 0;
    } // pthread_cond_destroy

    int pthread_cond_wait( pthread_cond_t *cond, pthread_mutex_t *mutex ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_cond_wait(cond:0x%p) mutex:0x%p enter task:0x%p\n", cond, mutex, &uThisTask() );
#endif // __U_DEBUG_H__
	pthread_testcancel();				// pthread_cond_wait is a cancellation point
	PthreadLock::get< uCondLock >( cond )->wait( *PthreadLock::get< uOwnerLock >( mutex ) );
	return 0;
    } // pthread_cond_wait

    int pthread_cond_timedwait( pthread_cond_t *cond, pthread_mutex_t *mutex, const struct timespec *abstime ) {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_cond_timedwait(cond:0x%p) mutex:0x%p enter task:0x%p\n", cond, mutex, &uThisTask() );
#endif // __U_DEBUG_H__
	pthread_testcancel();				// pthread_cond_timedwait is a cancellation point
	return PthreadLock::get< uCondLock >( cond )->timedwait( *PthreadLock::get< uOwnerLock >( mutex ), uTime( abstime->tv_sec, abstime->tv_nsec ) ) ? 0 : ETIMEDOUT;
    } // pthread_cond_timedwait

    int pthread_cond_signal( pthread_cond_t *cond ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_cond_signal(cond:0x%p) enter task:0x%p\n", cond, &uThisTask() );
#endif // __U_DEBUG_H__
	PthreadLock::get< uCondLock >( cond )->signal();
	return 0;
    } // pthread_cond_signal

    int pthread_cond_broadcast( pthread_cond_t *cond ) __THROW {
#ifdef __U_DEBUG_H__
	uDebugPrt( "pthread_cond_broadcast(cond:0x%p) task:0x%p\n", cond, &uThisTask() );
#endif // __U_DEBUG_H__
	PthreadLock::get< uCondLock >( cond )->broadcast();
	return 0;
    } // pthread_cond_broadcast


    int pthread_condattr_init( pthread_condattr_t *attr ) __THROW {
	// storage for pthread_condattr_t must be >= int
	return 0;
    } // pthread_condattr_init

    int pthread_condattr_destroy( pthread_condattr_t *attr ) __THROW {
	return 0;
    } // pthread_condattr_destroy

    int pthread_condattr_setpshared( pthread_condattr_t *attr, int pshared ) __THROW {
	*((int *)attr) = pshared;
	return 0;
    } // pthread_condattr_setpshared

    int pthread_condattr_getpshared( const pthread_condattr_t *attr, int *pshared ) __THROW {
	*pshared = *((int *)attr);
	return 0;
    } // pthread_condattr_getpshared
} // extern "C"


#if defined( __solaris__ )
#include <thread.h>

// Called by the Solaris system routines, and by the Solaris/pthread thread
// routines.

extern "C" {
    // Creation
    int _thr_create( void *stackaddr, size_t stacksize, void *(*start_func)( void* ), void *arg, long flags, thread_t *new_thread_id ) {
	pthread_attr_t attr;
	pthread_attr_init( &attr );
	pthread_attr_setstacksize( &attr, stacksize );
	pthread_attr_setstackaddr( &attr, stackaddr );
	pthread_attr_setdetachstate( &attr, flags | THR_DETACHED ? PTHREAD_CREATE_DETACHED : PTHREAD_CREATE_JOINABLE );
	pthread_attr_setscope( &attr, flags | THR_BOUND ? PTHREAD_SCOPE_SYSTEM : PTHREAD_SCOPE_PROCESS );
	int code = pthread_create( (pthread_t *)new_thread_id, &attr, start_func, arg );
	pthread_attr_destroy( &attr );
	if ( code != 0 ) {
	    errno = code;
	    code = -1;
	} // if
	return code;
    } // thr_create


    // Mutex
    int _mutex_init( mutex_t *mutex, int type, void * arg ) {
	return pthread_mutex_init( (pthread_mutex_t *)mutex, NULL );
    } // mutex_init

    int _mutex_destroy( mutex_t *mutex ) {
	return pthread_mutex_destroy( (pthread_mutex_t *)mutex );
    } // mutex_destroy

    int _mutex_lock( mutex_t *mutex ) {
	return pthread_mutex_lock( (pthread_mutex_t *)mutex );
    } // mutex_lock

    int _mutex_trylock( mutex_t *mutex ) {
	return pthread_mutex_trylock( (pthread_mutex_t *)mutex );
    } // mutex_trylock

    int _mutex_unlock( mutex_t *mutex ) {
	return pthread_mutex_unlock( (pthread_mutex_t *)mutex );
    } // mutex_unlock

    // Condition Variable
    int cond_init( cond_t *cond, int type, void *arg ) {
	pthread_condattr_t attr;
	pthread_condattr_init( &attr );
	pthread_condattr_setpshared( &attr, type == USYNC_PROCESS ? PTHREAD_PROCESS_SHARED : PTHREAD_PROCESS_PRIVATE );
	pthread_cond_init( (pthread_cond_t *)cond, &attr );
	pthread_condattr_destroy( &attr );
	return 0;
    } // cond_init

    int cond_destroy( cond_t *cond ) {
	return pthread_cond_destroy( (pthread_cond_t *)cond );
    } // cond_destroy

    int cond_wait( cond_t *cond, mutex_t *mutex ) {
	return pthread_cond_wait( (pthread_cond_t *)cond, (pthread_mutex_t *)mutex );
    } // cond_wait

#if defined( __U_TLS__ )
    // TEMPORARY: prototype changed in Solaris 10 release
    int cond_timedwait( cond_t *cond, mutex_t *mutex, const timestruc_t *abstime ) {
#else
    int cond_timedwait( cond_t *cond, mutex_t *mutex, timestruc_t *abstime ) {
#endif // __U_TLS__
	return pthread_cond_timedwait( (pthread_cond_t *)cond, (pthread_mutex_t *)mutex, abstime );
    } // cond_timedwait

    int cond_signal( cond_t *cond ) {
	return pthread_cond_signal( (pthread_cond_t *)cond );
    } // cond_signal

    int cond_broadcast( cond_t *cond ) {
	return pthread_cond_broadcast( (pthread_cond_t *)cond );
    } // cond_broadcast
} // extern "C"
#endif // __solaris__


#if defined( __linux__ )

extern "C" {

// XOPEN

    //######################### Mutex #########################

    int pthread_mutex_timedlock( pthread_mutex_t *__restrict __mutex, __const struct timespec *__restrict __abstime ) __THROW {
	uAbort( "pthread_mutex_timedlock" );
    } // pthread_mutex_timedlock

    //######################### Condition #########################

    int pthread_condattr_getclock( __const pthread_condattr_t * __restrict __attr, __clockid_t *__restrict __clock_id ) __THROW {
	uAbort( "pthread_condattr_getclock" );
    } // pthread_condattr_getclock

    int pthread_condattr_setclock( pthread_condattr_t *__attr, __clockid_t __clock_id ) __THROW {
	uAbort( "pthread_condattr_setclock" );
    } // pthread_condattr_setclock

    //######################### Spinlock #########################

    int pthread_spin_init( pthread_spinlock_t *__lock, int __pshared ) __THROW {
	uAbort( "pthread_spin_init" );
    } // pthread_spin_init

    int pthread_spin_destroy( pthread_spinlock_t *__lock ) __THROW {
	uAbort( "pthread_spin_destroy" );
    } // pthread_spin_destroy

    int pthread_spin_lock( pthread_spinlock_t *__lock ) __THROW {
	uAbort( "pthread_spin_lock" );
    } // pthread_spin_lock

    int pthread_spin_trylock( pthread_spinlock_t *__lock ) __THROW {
	uAbort( "pthread_spin_trylock" );
    } // pthread_spin_trylock

    int pthread_spin_unlock( pthread_spinlock_t *__lock ) __THROW {
	uAbort( "pthread_spin_unlock" );
    } // pthread_spin_unlock

    //######################### Barrier #########################

    int pthread_barrier_init( pthread_barrier_t *__restrict __barrier, __const pthread_barrierattr_t *__restrict __attr, unsigned int __count ) __THROW {
	uAbort( "pthread_barrier_init" );
    } // pthread_barrier_init

    int pthread_barrier_destroy( pthread_barrier_t *__barrier ) __THROW {
	uAbort( "pthread_barrier_destroy" );
    } // pthread_barrier_destroy

    int pthread_barrier_wait( pthread_barrier_t *__barrier ) __THROW {
	uAbort( "pthread_barrier_wait" );
    } // pthread_barrier_wait

    int pthread_barrierattr_init( pthread_barrierattr_t *__attr ) __THROW {
	uAbort( "pthread_barrierattr_init" );
    } // pthread_barrierattr_init

    int pthread_barrierattr_destroy( pthread_barrierattr_t *__attr ) __THROW {
	uAbort( "pthread_barrierattr_destroy" );
    } // pthread_barrierattr_destroy

    int pthread_barrierattr_getpshared( __const pthread_barrierattr_t * __restrict __attr, int *__restrict __pshared ) __THROW {
	uAbort( "pthread_barrierattr_getpshared" );
    } // pthread_barrierattr_getpshared

    int pthread_barrierattr_setpshared( pthread_barrierattr_t *__attr, int __pshared ) __THROW {
	uAbort( "pthread_barrierattr_setpshared" );
    } // pthread_barrierattr_setpshared

    //######################### Clock #########################

    int pthread_getcpuclockid( pthread_t __thread_id, __clockid_t *__clock_id ) __THROW {
	uAbort( "pthread_getcpuclockid" );
    } // pthread_getcpuclockid

    // pthread_atfork()

// UNIX98

    //######################### Mutex #########################

    int pthread_mutexattr_gettype( __const pthread_mutexattr_t *__restrict __attr, int *__restrict __kind ) __THROW {
	uAbort( "pthread_mutexattr_gettype" );
    } // pthread_mutexattr_gettype

    int pthread_mutexattr_settype( pthread_mutexattr_t *__attr, int __kind ) __THROW {
	uAbort( "pthread_mutexattr_settype" );
    } // pthread_mutexattr_settype

    //######################### Read/Write #########################

    int pthread_rwlock_init( pthread_rwlock_t *__restrict __rwlock, __const pthread_rwlockattr_t *__restrict __attr ) __THROW {
	uAbort( "pthread_rwlock_init" );
    } // pthread_rwlock_init

    int pthread_rwlock_destroy( pthread_rwlock_t *__rwlock ) __THROW {
	uAbort( "pthread_rwlock_destroy" );
    } // pthread_rwlock_destroy

    int pthread_rwlock_rdlock( pthread_rwlock_t *__rwlock ) __THROW {
	uAbort( "pthread_rwlock_rdlock" );
    } // pthread_rwlock_rdlock

    int pthread_rwlock_tryrdlock( pthread_rwlock_t *__rwlock ) __THROW {
	uAbort( "pthread_rwlock_tryrdlock" );
    } // pthread_rwlock_tryrdlock

    int pthread_rwlock_wrlock( pthread_rwlock_t *__rwlock ) __THROW {
	uAbort( "pthread_rwlock_wrlock" );
    } // pthread_rwlock_wrlock

    int pthread_rwlock_trywrlock( pthread_rwlock_t *__rwlock ) __THROW {
	uAbort( "pthread_rwlock_trywrlock" );
    } // pthread_rwlock_trywrlock

    int pthread_rwlock_unlock( pthread_rwlock_t *__rwlock ) __THROW {
	uAbort( "pthread_rwlock_unlock" );
    } // pthread_rwlock_unlock

    int pthread_rwlockattr_init( pthread_rwlockattr_t *__attr ) __THROW {
	uAbort( "pthread_rwlockattr_init" );
    } // pthread_rwlockattr_init

    int pthread_rwlockattr_destroy( pthread_rwlockattr_t *__attr ) __THROW {
	uAbort( "pthread_rwlockattr_destroy" );
    } // pthread_rwlockattr_destroy

    int pthread_rwlockattr_getpshared( __const pthread_rwlockattr_t * __restrict __attr, int *__restrict __pshared ) __THROW {
	uAbort( "pthread_rwlockattr_getpshared" );
    } // pthread_rwlockattr_getpshared

    int pthread_rwlockattr_setpshared( pthread_rwlockattr_t *__attr, int __pshared ) __THROW {
	uAbort( "pthread_rwlockattr_setpshared" );
    } // pthread_rwlockattr_setpshared

    int pthread_rwlockattr_getkind_np( __const pthread_rwlockattr_t *__attr, int *__pref ) __THROW {
	uAbort( "pthread_rwlockattr_getkind_np" );
    } // pthread_rwlockattr_getkind_np

    int pthread_rwlockattr_setkind_np( pthread_rwlockattr_t *__attr, int __pref ) __THROW {
	uAbort( "pthread_rwlockattr_setkind_np" );
    } // pthread_rwlockattr_setkind_np

// UNIX98 + XOPEN

    int pthread_rwlock_timedrdlock( pthread_rwlock_t *__restrict __rwlock, __const struct timespec *__restrict __abstime ) __THROW {
	uAbort( "pthread_rwlock_timedrdlock" );
    } // pthread_rwlock_timedrdlock

    int pthread_rwlock_timedwrlock( pthread_rwlock_t *__restrict __rwlock, __const struct timespec *__restrict __abstime ) __THROW {
	uAbort( "pthread_rwlock_timedwrlock" );
    } // pthread_rwlock_timedwrlock

// GNU

    //######################### Parallelism #########################

    int pthread_setaffinity_np( pthread_t __th, size_t __cpusetsize, __const cpu_set_t *__cpuset ) __THROW {
	uAbort( "pthread_setaffinity_np" );
    } // pthread_setaffinity_np

    int pthread_getaffinity_np( pthread_t __th, size_t __cpusetsize, cpu_set_t *__cpuset ) __THROW {
	uAbort( "pthread_getaffinity_np" );
    } // pthread_getaffinity_np

    int pthread_attr_setaffinity_np( pthread_attr_t *__attr, size_t __cpusetsize, __const cpu_set_t *__cpuset ) __THROW {
	uAbort( "pthread_attr_setaffinity_np" );
    } // pthread_attr_setaffinity_np

    int pthread_attr_getaffinity_np( __const pthread_attr_t *__attr, size_t __cpusetsize, cpu_set_t *__cpuset ) __THROW {
	uAbort( "pthread_attr_getaffinity_np" );
    } // pthread_attr_getaffinity_np

    //######################### Cancellation #########################

// pthread_cleanup_push_defer_np()  GNU extension
// pthread_cleanup_pop_restore_np() GNU extension

    void _pthread_cleanup_push_defer( struct _pthread_cleanup_buffer *__buffer, void( *__routine )( void * ), void *__arg ) __THROW {
	uAbort( "_pthread_cleanup_push_defer" );
    } // _pthread_cleanup_push_defer

    void _pthread_cleanup_pop_restore( struct _pthread_cleanup_buffer *__buffer, int __execute ) __THROW {
	uAbort( "_pthread_cleanup_pop_restore" );
    } // _pthread_cleanup_pop_restore
} // extern "C"

#endif // __linux__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
