//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.0, Copyright (C) Peter A. Buhr 1994
// 
// uProcessor.cc -- 
// 
// Author           : Peter A. Buhr
// Created On       : Mon Mar 14 17:39:15 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Sep 13 21:48:34 2006
// Update Count     : 1777
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#define __U_PROFILE__
#define __U_PROFILEABLE_ONLY__

#include <uC++.h>
#include <uProfiler.h>
#include <uProcessor.h>
//#include <uDebug.h>

#include <cstring>					// strerror
#include <cstdio>					// fprintf
#include <cerrno>
#include <unistd.h>					// getpid
#include <inttypes.h>					// uintptr_t

#if defined( __solaris__ )
#if defined( __U_PTHREAD__ )
#include <sys/syscall.h>				// SYS_lwp_exit
#else
#include <sys/lwp.h>					// lwp_exit
#endif // __U_PTHREAD__
#endif // __solaris__

#if defined( __U_PTHREAD__ )
#include <limits.h>					// PTHREAD_STACK_MIN
#endif // __U_PTHREAD__

#if defined( __linux__ )
#include <sys/wait.h>					// waitpid
#if defined( __U_TLS__ )
#include <asm/unistd.h>					// __NR_exit
#else
#include <sched.h>
#if ! defined( CLONE_PARENT )				// TEMPORARY: problems in include files for Linux 2.4
#define CLONE_PARENT	0x00008000			// set if we want to have the same parent as the cloner
#endif // ! CLONE_PARENT

#if defined( __ia64__ )
extern "C" int __clone2( int (*__fn) (void *__arg), void *__child_stack_base, size_t __child_stack_size, int __flags, void *__arg );
#endif

#endif
#endif // __linux__

#if defined( __irix__ ) || defined( __freebsd__ )
#include <sys/wait.h>					// waitpid
#endif // __irix__

#if defined( __freebsd__ ) && defined( __U_MULTI__ )
#include <machine/sysarch.h>				// i386_set_ldt
#endif // __freebsd__ && __U_MULTI__

#if ! defined( __U_MULTI__ ) || defined( __U_ONETIMER__ )
uEventList *uProcessor::events = NULL;
#endif // ! __U_MULTI__ || __U_ONETIMER__

#if ! defined( __U_MULTI__ )
uEventNode *uProcessor::contextEvent = NULL;
uCxtSwtchHndlr *uProcessor::contextSwitchHandler = NULL;
uProfileProcessorSampler *uProcessor::profileProcessorSamplerInstance = NULL;
#endif // __U_MULTI__

#ifdef __U_DEBUG__
static const int uMinPreemption = 1000;			// 1 second (milliseconds)
#endif // __U_DEBUG__


//######################### uProcessorTask #########################


void uProcessorTask::main() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorTask &)0x%p.main, starting\n", this );
#endif // __U_DEBUG_H__
    assert( THREAD_GETMEM( disableInt) && THREAD_GETMEM( disableIntCnt) > 0 );

    // Race between creation of KT and start of child. Since the child needs
    // this information first, set it here.

    processor.pid =					// set LWP pid for processor
#if defined( __U_PTHREAD__ )
	uRealPthread::pthread_self();
#elif defined( __solaris__ ) && defined( __U_MULTI__ )
	_lwp_self();
#else
	getpid();
#endif

    processor.processorClock = &activeProcessorKernel->kernelClock;

    // Although the signal handlers are inherited by each child process, the
    // alarm setting is not.

    processor.setContextSwitchEvent( processor.getPreemption() );

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif // __U_LOCALDEBUGGER_H__

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->createKernelThread( processor, uThisCluster() );
#endif // __U_LOCALDEBUGGER_H__

#if defined( __U_MULTI__ )
    // uniprocessor calls done in uProfilerBoot
    if ( uProfiler::uProfiler_builtinRegisterProcessor ) {
	(*uProfiler::uProfiler_builtinRegisterProcessor)( uProfiler::profilerInstance, bound );
    } // if
#endif // __U_MULTI_H__

    for ( ;; ) {
	assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );
	_Accept( ~uProcessorTask ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, ~uProcessorTask\n", this );
#endif // __U_DEBUG_H__
	    break;
	} else _Accept( setPreemption ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, setPreemption( %d )\n", this, preemption );
#endif // __U_DEBUG_H__
	    processor.setContextSwitchEvent( preemption ); // use it to set the alarm for this processor
	    processor.preemption = preemption;
	} else _Accept( setCluster ) {
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, setCluster\n", this );
#endif // __U_DEBUG_H__

#if __U_LOCALDEBUGGER_H__
	    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif // __U_LOCALDEBUGGER_H__

	    // Remove the processor from the list of processor that live on
	    // this cluster, and add it to the list of processors that live on
	    // the new cluster. The move has to be done by the processor itself
	    // when it is in a stable state. As well, the processor's task has
	    // to be moved to the new cluster but it does not have to be
	    // migrated there since it is a bound task.

	    // change processor's notion of which cluster it is executing on
	    uCluster &prevCluster = processor.getCluster();

	    if ( uProfiler::uProfiler_registerProcessorMigrate ) { // task registered for profiling ?              
		(*uProfiler::uProfiler_registerProcessorMigrate)( uProfiler::profilerInstance, processor, prevCluster, *cluster );
	    } // if

	    prevCluster.processorRemove( processor );
	    processor.currCluster = cluster;
	    THREAD_SETMEM( activeCluster, cluster );
	    cluster->processorAdd( processor );
	    currCluster = cluster;			// change task's notion of which cluster it is executing on

#if __U_LOCALDEBUGGER_H__
	    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->migrateKernelThread( processor, *cluster );
#endif // __U_LOCALDEBUGGER_H__

	    result.signalBlock();
#if defined( __U_MULTI__ )
#if defined( __irix__ ) || defined( __freebsd__ )
	} else _Accept( procCreate ) {
	    assert( &processor == uKernelModule::systemProcessor );
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, procCreate pid:%d\n", this, pid );
#endif // __U_DEBUG_H__
#if defined( __irix__ )
	    pid = sprocsp( __U_START_KT_NAME__, PR_SADDR | PR_SFDS | PR_SUMASK, createProc, (char *)(createProc->processorKer.base), createProc->processorKer.size );
#elif defined( __freebsd__ )
	    pid = rfork_thread( RFPROC | RFMEM | RFSIGSHARE, createProc->processorKer.base, __U_START_KT_NAME__, createProc );
#endif // __irix__
	    if ( pid == -1 ) {
		uAbort( "(uProcessorTask &)0x%p.main, procCreate : internal error, error(%d) %s.", this, errno, strerror( errno ) );
	    } // if
	    result.signalBlock();
#endif
#if defined( __irix__ ) || defined( __linux__ ) || defined( __freebsd__ )
	} else _Accept( procWait ) {
	    assert( &processor == uKernelModule::systemProcessor );
	    int code, status = 0;

	    for ( ;; ) {
		 code = ::waitpid( pid, &status, 0 );	// wait for the specific child and its exit status
	      if ( code != -1 ) break;
	      if ( errno != EINTR ) break;		// timer interrupt ?
	    } // for

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorTask &)0x%p.main, procWait pid:%ld, pid:%d, status:%d\n", this, (long)pid, code, status );
#endif // __U_DEBUG_H__

	    // procWait and uSigChldHandler race to wait on the pid for normal
	    // termination so the loser ignores the ECHILD.
	    if ( code == -1 && errno != ECHILD ) {
		uAbort( "(uProcessorTask &)0x%p.main, procWait : internal error, error(%d) %s.", this, errno, strerror( errno ) );
	    } // if
	    if ( code > 0 && WIFSIGNALED( status ) ) {	// process died as the result of some signal ?
		uKernelModule::coreDumped = true;	// assume that the child dumped core
		uAbort( "Child process %ld died.", (long)pid );
	    } // if
	    result.signalBlock();
#endif
	} else _Accept( procExit ) {
	    exit( retCode );
#endif // __U_MULTI__
	} // _Accept
    } // for

#if defined( __U_MULTI__ )
    uThisProcessor().setContextSwitchEvent( 0 );	// clear the alarm on this processor

    // uniprocessor calls done in uProfilerBoot
    if ( uProfiler::uProfiler_builtinDeregisterProcessor ) {
	(*uProfiler::uProfiler_builtinDeregisterProcessor)( uProfiler::profilerInstance, bound );
    } // if
#endif // __U_MULTI_H__

#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->destroyKernelThread( processor );
#endif // __U_LOCALDEBUGGER_H__

    processor.terminated = true;
} // uProcessorTask::main


void uProcessorTask::setPreemption( unsigned int ms ) {
    preemption = ms;
} // uProcessorTask::setPreemption


void uProcessorTask::setCluster( uCluster &cluster ) {
    uProcessorTask::cluster = &cluster;			// copy arguments
    result.wait();					// wait for result
} // uProcessorTask::setCluster


#ifdef __U_MULTI__
#if defined( __irix__ ) || defined( __freebsd__ )
uPid_t uProcessorTask::procCreate( uProcessor *processor ) {
    createProc = processor;				// copy arguments
    result.wait();					// wait for result
    return pid;
} // uProcessorTask::procCreate
#endif


#if defined( __irix__ ) || defined( __linux__ ) || defined( __freebsd__ )
void uProcessorTask::procWait( uPid_t pid ) {
    uProcessorTask::pid = pid;				// copy arguments
    result.wait();					// wait for result
} // uProcessorTask::procWait
#endif


void uProcessorTask::procExit( int retcode ) {
    uProcessorTask::retCode = retcode;			// copy arguments
    result.wait();					// wait for result
    // CONTROL NEVER REACHES HERE!
#ifdef __U_DEBUG__
    assert( false );
#endif // __U_DEBUG__
} // uProcessorTask::procExit
#endif // __U_MULTI__


uProcessorTask::uProcessorTask( uCluster &cluster, uProcessor &processor ) : uBaseTask( cluster, processor ), processor( processor ) {
} // uProcessorTask::uProcessorTask


uProcessorTask::~uProcessorTask() {
#ifdef __U_MULTI__
    // do not wait for systemProcessor KT as it must return to the shell
  if ( &processor == uKernelModule::systemProcessor ) return;

    uPid_t pid = processor.getPid();			// pid of underlying KT
#if defined( __U_PTHREAD__ ) || defined( __solaris__ )
    int code;
    for ( ;; ) {
#if defined( __U_PTHREAD__ )
	code = uRealPthread::pthread_join( pid, NULL );	// wait for termination of KT
#else
	code = _lwp_wait( pid, NULL );			// wait for termination of KT
#endif // __U_PTHREAD__ || __solaris__
      if ( code == 0 ) break;
      if ( code != EINTR ) break;			// timer interrupt ?
    } // for
    if ( code != 0 ) {
	uAbort( "(uProcessor &)0x%p.~uProcessor() : internal error, wait failed for kernel thread %ld, error(%d) %s.",
		this, (long int)pid, code, strerror( code ) );
    } // if

#elif defined( __linux__ )
    // Linux clone can only wait for a child not a sibling KT.  However, by
    // judicious use of CLONE_PARENT all KTs can be made to have the SKT as
    // their parent.  Therefore, only the SKT can perform the waitpid by
    // calling its processor task to guarantee the operation is performed on
    // the SKT.

    uKernelModule::systemProcessor->procWait( pid );

#elif defined( __irix__ ) || defined( __freebsd__ )
    uKernelModule::systemProcessor->procWait( pid );
#else
    #error uC++ : internal error, unsupported architecture
#endif

#endif // __U_MULTI__
} // uProcessorTask::~uProcessorTask


//######################### uProcessorKernel #########################


__U_START_KT_TYPE__ {
#if defined(__U_MULTI__)
    volatile uKernelModule::uKernelModuleData *km;

#if defined( __irix__ )
    // the user part of the PRDA holds the kernel module
    km = (uKernelModule::uKernelModuleData *)&(PRDA->usr_prda);
    km->ctor();

#elif defined( __linux__ ) && defined( __i386__ )
    /*
		+------------------+
		|                  |
		|  struct pthread  |
		|                  |
		+------------------+ <== gs:0 (thread pointer)
		|                  |
		|       TLS        | <== uKernelModuleBoot in here somewhere
		|                  |
		+------------------+ <== must be aligned to tls_align
		|                  |
		| stack continues  |
		.                  .
		.                  .
		.                  .
    */
#if defined( __U_TLS__ )
    asm ("movl %%gs:0, %0\n\t"
         "leal _ZN13uKernelModule17uKernelModuleBootE@ntpoff(%0),%0" : "=r" (km) );
    km->ctor();
#else
    uKernelModule::uKernelModuleData theKernelModule;
    km = &theKernelModule;
    km->ldtValue = ((uProcessor *)p)->ldtValue;
    km->threadPointer = (unsigned long)km;
    km->ctor();
#endif // __U_TLS__

#elif defined( __linux__ ) && defined( __x86_64__ )
    /*
		+------------------+
		|                  |
		|       TLS        | <== uKernelModuleBoot in here somewhere
		|                  |
		+------------------+ <== fs:0 (thread pointer)
		|                  |
		|  struct pthread  |
		|                  |
		+------------------+
		|                  |
		| stack continues  |
		.                  .
		.                  .
		.                  .
    */
    asm ("movq %%fs:0, %0\n\t"
         "leaq _ZN13uKernelModule17uKernelModuleBootE@tpoff(%0),%0" : "=r" (km) );
    km->ctor();

#elif defined( __linux__ ) && defined( __ia64__ )
#if defined( __U_TLS__ )
    /*
		+------------------+
		|                  |
		|       TLS        | <== uKernelModuleBoot in here somewhere
		|                  |
		+------------------+ <== must be aligned to tls_align
		|     16 bytes     |
		+------------------+ <== r13 (thread pointer)
		|                  |
		|  struct pthread  |
		|                  |
		+------------------+
		|                  |
		| stack continues  |
		.                  .
		.                  .
		.                  .
    */
#ifdef __INTEL_COMPILER
    km = &uKernelModule::uKernelModuleBoot;
#else
    // TEMPORARY: we should be able to say
    //    km = &uKernelModule::uKernelModuleBoot;
    // but a gcc bug (http://gcc.gnu.org/bugzilla/show_bug.cgi?id=19450) prevents this;
    // hence the following:
    asm volatile ("addl %0 = @ltoff(@tprel(_ZN13uKernelModule17uKernelModuleBootE#)), gp;;\n"
                  "ld8 %0 = [%0];;\n"
                  "add %0 = %0, r13;;\n" : "=r" (km) );
#endif // __INTEL_COMPILER
    km->ctor();
#else
    uKernelModule::uKernelModuleData theKernelModule;
    km = &theKernelModule;
    km->ctor();
#endif // __U_TLS__

#elif defined( __sparc__ ) && defined( __U_TLS__ )
    asm volatile ("sethi %%tle_hix22(_ZN13uKernelModule17uKernelModuleBootE),%0\n\t"
		  "xor %0, %%tle_lox10(_ZN13uKernelModule17uKernelModuleBootE), %0\n\t"
		  "add %%g7, %0, %0" : "=r" (km) );
    km->ctor();
#else
    // allocate a kernel module at the bottom of the thread's stack
    uKernelModule::uKernelModuleData theKernelModule;
    km = &theKernelModule;
    km->ctor();
#endif

    // NO DEBUG PRINTS BEFORE THE THREAD REFERENCE IS SET IN CTOR.

#ifdef __U_DEBUG_H__
    uDebugPrt( "startthread, child started\n" );
#endif // __U_DEBUG_H__

    uProcessor &processor = *(uProcessor *)p;
    uProcessorKernel *pk = &processor.processorKer;

    km->processorKernelStorage = pk;

    // initialize thread members

    THREAD_SETMEM( activeProcessor, &processor );
    uCluster *currCluster = THREAD_GETMEM( activeProcessor )->currCluster;
    THREAD_SETMEM( activeCluster, currCluster );
    
    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) == 1 );

    THREAD_GETMEM( This )->disableInterrupts();
    uMachContext::invokeCoroutine( *activeProcessorKernel );
#endif // __U_MULTI__

#if defined( __linux__ ) || defined( __freebsd__ )
    // this line is never reached, but linux allows the start-thread function to
    // return a value, and gcc warns otherwise.
    return 0;
#endif // linux
} // __U_START_KT_TYPE__


inline void uProcessorKernel::taskIsBlocking() {
    uBaseTask &task = uThisTask();			// optimization
    if ( task.getState() != uBaseTask::Terminate ) {
	task.setState( uBaseTask::Blocked );
    } // if
} // uProcessorKernel::taskIsBlocking


void uProcessorKernel::schedule() {
    assert( ! uThisTask().readyRef.listed() );
    assert( ! THREAD_GETMEM( disableIntSpin ) );

    taskIsBlocking();

    kind = 0;
    contextSw();					// not resume because entering kernel
} // uProcessorKernel::schedule


void uProcessorKernel::schedule( uSpinLock *lock ) {
    assert( ! uThisTask().readyRef.listed() );
    assert( THREAD_GETMEM( disableIntSpinCnt ) == 1 );

    taskIsBlocking();

    kind = 1;
    prevLock = lock;
    contextSw();					// not resume because entering kernel
} // uProcessorKernel::schedule


void uProcessorKernel::schedule( uBaseTask *task ) {
    // SKULLDUGGERY: uBootTask is on ready queue for first entry into the kernel.
    assert( &uThisTask() != (uBaseTask *)uKernelModule::bootTask ? ! uThisTask().readyRef.listed() : true );
    assert( ! THREAD_GETMEM( disableIntSpin ) );

    if ( task != &uThisTask() ) {
	taskIsBlocking();
    } // if

    kind = 2;
    nextTask = task;
    contextSw();					// not resume because entering kernel
} // uProcessorKernel::schedule


void uProcessorKernel::schedule( uSpinLock *lock, uBaseTask *task ) {
    assert( ! uThisTask().readyRef.listed() );
    assert( THREAD_GETMEM( disableIntSpinCnt ) == 1 );

    taskIsBlocking();

    kind = 3;
    prevLock = lock;
    nextTask = task;
    contextSw();					// not resume because entering kernel
} // uProcessorKernel::schedule


void uProcessorKernel::onBehalfOfUser() {
    switch( kind ) {
      case 0:
	break;
      case 1:
	prevLock->release();
	break;
      case 2:
	nextTask->wake();
	break;
      case 3:
	prevLock->release();
	nextTask->wake();
	break;
      default:
	uAbort( "(uProcessorKernel &)0x%p.onBehalfOfUser : internal error, schedule kind:%d.", this, kind );
	break;
    } // switch
} // uProcessorKernel::onBehalfOfUser


void uProcessorKernel::setTimer( uDuration dur ) {
  if ( dur < 0 ) return;				// if duration is negative, it's invalid

    // For now, write only code for non-posix timer. When posix timer is
    // available use timer_create and timer_settimer.

    timeval conv = dur;
    itimerval it;
    it.it_value = conv;					// fill in the value to the next expiry
    it.it_interval.tv_sec = 0;				// not periodic
    it.it_interval.tv_XSEC = 0;
#ifdef __U_DEBUG_H__
    char buffer[256];
    uDebugPrtBuf( buffer, "uProcessorKernel::setTimer, it.it_value.tv_sec:%ld, it.it_value.tv_ssec:%ld\n", it.it_value.tv_sec, it.it_value.tv_usec );
#endif // __U_DEBUG_H__
    setitimer( ITIMER_REAL, &it, NULL );		// set the alarm clock to go off
} // uProcessorKernel::setTimer


void uProcessorKernel::setTimer( uTime time ) {
  if ( time == 0 ) return;				// if time is zero, it's invalid

    // Since this is private, and only uSysEventList is a friend, the assumption
    // is made that the time parameter is always in real-time (not virtual
    // time)

#if defined( REALTIME_POSIX )
    timespec curr;
    if ( clocktype < 0 ) type = CLOCK_REALTIME;
    clock_gettime( type, &curr );
#else
    timeval curr;
    GETTIMEOFDAY( &curr );
#endif
    uTime currtime( curr.tv_sec, curr.tv_usec * 1000 );	// convert to nanoseconds

    uDuration dur = time - currtime;
    if ( dur <= 0 ) {					// if duration is negative (it has already past)
	// fill in the value to the next expiry by setting alarm to soonest
	// time an alarm may come
	setTimer( uDuration( 0, TIMEGRAN / 1000000L ) );
    } else {
	setTimer( dur );
    } // if
} // uProcessorKernel::setTimer


#ifndef __U_MULTI__
void uProcessorKernel::nextProcessor( uProcessorDL *&currProc ) {
    // Get next processor to execute.

    unsigned int uPrevPreemption = uThisProcessor().getPreemption(); // remember previous preemption value
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.nextProcessor, from processor 0x%p on cluster %.256s (0x%p) with time slice %d\n",
	       this, &uThisProcessor(), uThisProcessor().currCluster->getName(), uThisProcessor().currCluster, uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__
    uProcessorDL *original = currProc;
    do {						// ignore deleted processors
	currProc = uKernelModule::globalProcessors->succ( currProc );
	if ( currProc == NULL ) {			// make list appear circular
	    currProc = uKernelModule::globalProcessors->head(); // restart at beginning of list
	} // if
    } while ( currProc != original && currProc->processor().terminated );
    THREAD_SETMEM( activeProcessor, &(currProc->processor() ) );
    uCluster *currCluster = THREAD_GETMEM( activeProcessor )->currCluster;
    THREAD_SETMEM( activeCluster, currCluster );
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.nextProcessor, to processor 0x%p on cluster %.256s (0x%p) with time slice %d\n",
	       this, &uThisProcessor(), uThisProcessor().currCluster->getName(), uThisProcessor().currCluster, uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__

    // The time slice must be reset or some programs will not work.

    if ( uThisProcessor().getPreemption() != uPrevPreemption ) {
	uThisProcessor().setContextSwitchEvent( uThisProcessor().getPreemption() );
    } // if
} // nextProcessor
#endif // __U_MULTI__


void uProcessorKernel::main() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.main, child is born\n", this );
#endif // __U_DEBUG_H__

#if ! defined( __U_MULTI__ )
    // SKULLDUGGERY: The system processor in not on the global list until the
    // processor task runs, so explicitly set the current processor to the
    // system processor.

    uProcessorDL *currProc = &(uKernelModule::systemProcessor->globalRef);
    uProcessorDL *cycleStart = NULL;
    bool &okToSelect = uCluster::NBIO->okToSelect;
    uBaseTask *&IOPoller = uCluster::NBIO->IOPoller;
#endif // ! __U_MULTI__

    for ( unsigned int spin = 0;; ) {
	// Advance the spin counter now to detect if a task is executed.

	spin += 1;

	if ( ! uThisProcessor().external.empty() ) {	// check processor specific ready queue
	    // Only this processor removes from this ready queue so no other
	    // processor can remove this task after it has been seen.

	    THREAD_SETMEM( activeTask, uThisProcessor().currTask = &(uThisProcessor().external.dropHead()->task()));

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(1) bef: task %.256s (0x%p) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:%.256s (0x%p) on processor:0x%p, %d,%d,%d,%d,%d\n",
		       this, uThisProcessor().currTask->currCoroutine->name, uThisProcessor().currTask->currCoroutine,
		       uThisProcessor().currTask->limit, uThisProcessor().currTask->base, uThisProcessor().currTask->stackPointer(),
		       uThisProcessor().currCluster->getName(), uThisProcessor().currCluster, &uThisProcessor(),
		       THREAD_GETMEM( disableInt ),
		       THREAD_GETMEM( disableIntCnt ),
		       THREAD_GETMEM( disableIntSpin ),
		       THREAD_GETMEM( disableIntSpinCnt ),
		       THREAD_GETMEM( inKernelRF )
		);
#endif // __U_DEBUG_H__

	    // SKULLDUGGERY: The processor task is part of the kernel, and
	    // therefore, must execute as uninterruptible code. By incrementing
	    // the interrupt counter here, the decrement when the processor
	    // task is scheduled leaves the processor task's execution in the
	    // kernel with regard to interrupts. This assumes the disableInt
	    // flag is set already.

	    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );
	    THREAD_GETMEM( This )->disableInterrupts();

#if defined( __U_MULTI__ ) && defined( __U_SWAPCONTEXT__ )
#if defined( __i386__ ) && ! defined( __U_PTHREAD__ )
	    ((ucontext_t *)(uThisProcessor().currTask->currCoroutine->storage))->uc_mcontext.gregs[REG_GS] = THREAD_GETMEM( ldtValue );    
#elif defined( __ia64__ )
	    ((ucontext_t *)(uThisProcessor().currTask->currCoroutine->storage))->uc_mcontext.sc_gr[13] = (unsigned long)THREAD_GETMEM( threadPointer );
#elif defined( __sparc__ )
	    ((ucontext_t *)(uThisProcessor().currTask->currCoroutine->storage))->uc_mcontext.gregs[REG_G7] = (int)(THREAD_GETMEM( This ) ); 
#endif
#ifdef __U_ONETIMER__
	    if ( &uThisProcessor() == uKernelModule::systemProcessor ) {
		sigdelset( (sigset_t *)&(((ucontext_t *)(uThisProcessor().currTask->currCoroutine->storage))->uc_sigmask), SIGALRM );
	    } else {
		sigaddset( (sigset_t *)&(((ucontext_t *)(uThisProcessor().currTask->currCoroutine->storage))->uc_sigmask), SIGALRM );
	    } // if
#endif // __U_ONETIMER__
#endif // __U_MULTI__ && __U_SWAPCONTEXT__
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ ) && defined( __ia64__ )
	    if ( &uThisProcessor() == uKernelModule::systemProcessor ) {
		sigdelset( &((uContext_t *)(uThisProcessor().currTask->currCoroutine->storage))->sigMask, SIGALRM );
	    } else {
		sigaddset( &((uContext_t *)(uThisProcessor().currTask->currCoroutine->storage))->sigMask, SIGALRM );
	    } // if
#endif // __U_MULTI__ && __U_ONETIMER__ && __ia64__

	    uSwitch( storage, uThisProcessor().currTask->currCoroutine->storage );
	    THREAD_GETMEM( This )->enableInterrupts();
	    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(1) aft: task %.256s (0x%p) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:%.256s (0x%p) on processor:0x%p, %d,%d,%d,%d,%d\n",
		       this, uThisProcessor().currTask->currCoroutine->name, uThisProcessor().currTask->currCoroutine,
		       uThisProcessor().currTask->limit, uThisProcessor().currTask->base, uThisProcessor().currTask->stackPointer(),
		       uThisProcessor().currCluster->getName(), uThisProcessor().currCluster, &uThisProcessor(),
		       THREAD_GETMEM( disableInt ),
		       THREAD_GETMEM( disableIntCnt ),
		       THREAD_GETMEM( disableIntSpin ),
		       THREAD_GETMEM( disableIntSpinCnt ),
		       THREAD_GETMEM( inKernelRF )
		);
#endif // __U_DEBUG_H__

	    spin = 0;					// set number of spins back to zero
	    onBehalfOfUser();				// execute code on scheduler stack on behalf of user

	    if ( uThisProcessor().terminated ) {
#ifdef __U_MULTI__
		if ( &uThisProcessor() != uKernelModule::systemProcessor ) break;
		// If control reaches here, the boot task must be the only task
		// on the system-cluster ready-queue, and it must be restarted
		// to finish the close down.
#else
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uProcessorKernel &)0x%p.main termination, currProc:0x%p, uThisProcessor:0x%p\n",
			   this, &(currProc->processor()), &uThisProcessor() );
#endif // __U_DEBUG_H__
		// In the uniprocessor case, only terminate the processor kernel
		// when the processor task for the system processor is deleted;
		// otherwise the program stops when the first processor is deleted.

		if ( &uThisProcessor() != uKernelModule::systemProcessor ) {
		    // Get next processor to execute because the current one
		    // just terminated.

		    nextProcessor( currProc );
		} // if
#endif // __U_MULTI__
	    } // if
	} // if

	uBaseTask *uReadyTask = &(uThisProcessor().currCluster->readyTaskTryRemove());
	if ( uReadyTask != NULL ) {			// ready queue not empty, schedule that task
	    assert( ! uReadyTask->readyRef.listed() );
	    THREAD_SETMEM( activeTask, uThisProcessor().currTask = uReadyTask );

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(2) bef: task %.256s (0x%p) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:%.256s (0x%p) on processor:0x%p, %d,%d,%d,%d,%d\n",
		       this, uReadyTask->currCoroutine->name, uReadyTask->currCoroutine,
		       uReadyTask->currCoroutine->limit, uReadyTask->currCoroutine->base, uReadyTask->currCoroutine->stackPointer(),
		       uThisProcessor().currCluster->getName(), uThisProcessor().currCluster, &uThisProcessor(),
		       THREAD_GETMEM( disableInt ),
		       THREAD_GETMEM( disableIntCnt ),
		       THREAD_GETMEM( disableIntSpin ),
		       THREAD_GETMEM( disableIntSpinCnt ),
		       THREAD_GETMEM( inKernelRF )
		);
#endif // __U_DEBUG_H__

	    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );

#if defined( __U_MULTI__ ) && defined( __U_SWAPCONTEXT__ )
#if defined( __i386__ ) && ! defined( __U_PTHREAD__ )
	    ((ucontext_t *)(uReadyTask->currCoroutine->storage))->uc_mcontext.gregs[REG_GS] = THREAD_GETMEM( ldtValue );    
#elif defined( __ia64__ )
	    ((ucontext_t *)(uReadyTask->currCoroutine->storage))->uc_mcontext.sc_gr[13] = THREAD_GETMEM( threadPointer );
#elif defined( __sparc__ )
	    ((ucontext_t *)(uReadyTask->currCoroutine->storage))->uc_mcontext.gregs[REG_G7] = (int)(THREAD_GETMEM( This ) ); 
#endif
#ifdef __U_ONETIMER__
	    if ( &uThisProcessor() == uKernelModule::systemProcessor ) {
		sigdelset( (sigset_t *)&(((ucontext_t *)(uReadyTask->currCoroutine->storage))->uc_sigmask), SIGALRM );
	    } else {
		sigaddset( (sigset_t *)&(((ucontext_t *)(uReadyTask->currCoroutine->storage))->uc_sigmask), SIGALRM );
	    } // if
#endif // __U_ONETIMER__
#endif // __U_MULTI__ && __U_SWAPCONTEXT__
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ ) && defined( __ia64__ )
	    if ( &uThisProcessor() == uKernelModule::systemProcessor ) {
		sigdelset( &((uContext_t *)(uReadyTask->currCoroutine->storage))->sigMask, SIGALRM );
	    } else {
		sigaddset( &((uContext_t *)(uReadyTask->currCoroutine->storage))->sigMask, SIGALRM );
	    } // if
#endif // __U_MULTI__ && __U_ONETIMER__ && __ia64__

	    uSwitch( storage, uReadyTask->currCoroutine->storage );
	    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );

	    // SKULLDUGGERY: During spinning, the activeTask should not point
	    // to the last task executed. Otherwise, the profiler incorrectly
	    // displays the last task executed during spinning. So activeTask
	    // is set to the uProcessorTask for this processor.
	    THREAD_SETMEM( activeTask, uThisProcessor().currTask = uThisProcessor().procTask );

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, scheduling(2) aft: task %.256s (0x%p) (limit:0x%p,base:0x%p,stack:0x%p) from cluster:%.256s (0x%p) on processor:0x%p, %d,%d,%d,%d,%d\n",
		       this, uReadyTask->currCoroutine->name, uReadyTask->currCoroutine,
		       uReadyTask->limit, uReadyTask->base, uReadyTask->stackPointer(),
		       uThisProcessor().currCluster->getName(), uThisProcessor().currCluster, &uThisProcessor(),
		       THREAD_GETMEM( disableInt ),
		       THREAD_GETMEM( disableIntCnt ),
		       THREAD_GETMEM( disableIntSpin ),
		       THREAD_GETMEM( disableIntSpinCnt ),
		       THREAD_GETMEM( inKernelRF )
		);
#endif // __U_DEBUG_H__

#ifdef __U_MULTI__
	    spin = 0;					// set number of spins back to zero
#else
	    // Poller task does not count as an executed task, if its last
	    // execution found no I/O and this processor's ready queue is
	    // empty. Check before calling onBehalfOfUser, because IOPoller
	    // may put itself back on the ready queue, which makes the ready
	    // queue appear non-empty.

	    if ( uReadyTask != IOPoller || uCluster::NBIO->descriptors != 0 || ! uThisProcessor().currCluster->readyTasksEmpty() ) {
		spin = 0;				// set number of spins back to zero
	    } // if
#endif // __U_MULTI__
	    onBehalfOfUser();				// execute code on scheduler stack on behalf of user
	} // if

#ifdef __U_MULTI__
	if ( spin > uThisProcessor().getSpin() ) {
	    // SKULLDUGGERY: During pausing, the processor's currTask should
	    // not point to the last task executed. Otherwise, the visualizer
	    // incorrectly displays the last task executed during pausing. So
	    // currTask is set to the uProcessorTask for this processor.

#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, on processor 0x%p nothing to do\n", this, &uThisProcessor() );
#endif // __U_DEBUG_H__
	    THREAD_SETMEM( activeTask, uThisProcessor().currTask = uThisProcessor().procTask );
	    uThisProcessor().currCluster->processorPause(); // put processor to sleep
	    spin = 0;					// set number of spins back to zero
	} // if
#else
	// A cycle starts when a processor executes no tasks. If the cycle
	// completes and no task has executed, deadlock has occurred unless
	// there are pending I/O tasks. If there are pending I/O tasks, the I/O
	// poller task pauses the UNIX process at the "select".

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uProcessorKernel &)0x%p.main, cycleStart:0x%p, currProc:%.256s (0x%p), spin:%d, uReadyTask:0x%p, IOPoller:0x%p\n",
		  this, cycleStart, currProc->processor().currCluster->getName(), currProc, spin, uReadyTask, IOPoller );
#endif // __U_DEBUG_H__

	if ( cycleStart == currProc && spin != 0 ) {
#if __U_LOCALDEBUGGER_H__
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uProcessorKernel &)0x%p.main, uLocalDebuggerInstance:0x%p, IOPoller: %.256s (0x%p), dispatcher:0x%p, debugger_blocked_tasks:%d, uPendingIO.head:0x%p, uPendingIO.tail:0x%p\n",
		       this, uLocalDebugger::uLocalDebuggerInstance,
		       IOPoller != NULL ? IOPoller->getName() : "no I/O poller task", IOPoller,
		       uLocalDebugger::uLocalDebuggerInstance != NULL ? uLocalDebugger::uLocalDebuggerInstance->dispatcher : NULL,
		       uLocalDebugger::uLocalDebuggerInstance != NULL ? uLocalDebugger::uLocalDebuggerInstance->debugger_blocked_tasks : 0,
		       uCluster::NBIO->uPendingIO.head(), uCluster::NBIO->uPendingIO.tail() );
#endif // __U_DEBUG_H__
#endif // __U_LOCALDEBUGGER_H__
	    if ( IOPoller != NULL			// I/O poller task ?
#if __U_LOCALDEBUGGER_H__
		&& (
		    uLocalDebugger::uLocalDebuggerInstance == NULL || // local debugger initialized ?
		    IOPoller != (uBaseTask *)uLocalDebugger::uLocalDebuggerInstance->dispatcher || // I/O poller the local debugger reader ?
		    uLocalDebugger::uLocalDebuggerInstance->debugger_blocked_tasks != 0 || // any tasks debugger blocked ?
		    uCluster::NBIO->uPendingIO.head() != uCluster::NBIO->uPendingIO.tail() // any other tasks waiting for I/O ?
		    )
#endif // __U_LOCALDEBUGGER_H__
		) {
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uProcessorKernel &)0x%p.main, poller blocking\n", this );
#endif // __U_DEBUG_H__

		okToSelect = true;			// tell poller it is ok to call UNIX select, reset in uPollIO
	    } else if ( uThisProcessor().events->userEventPresent() ) { // tasks sleeping, except system task  ?
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uProcessorKernel &)0x%p.main, sleeping with pending events\n", this );
#endif // __U_DEBUG_H__
		uThisProcessor().currCluster->processorPause(); // put processor to sleep
	    } else {
		fprintf( stderr, "Clusters and tasks present at deadlock:\n" );
		uSeqIter<uClusterDL> ci;
		uClusterDL *cr;
		for ( ci.over( *uKernelModule::globalClusters ); ci >> cr; ) {
		    uCluster *cluster = &cr->cluster();
		    fprintf( stderr, "%.256s (0x%p)\n", cluster->getName(), cluster );

		    fprintf( stderr, "\ttasks:\n" );
		    uBaseTaskDL *bt;
		    for ( uSeqIter<uBaseTaskDL> iter( cluster->tasksOnCluster ); iter >> bt; ) {
			uBaseTask *task = &bt->task();
			fprintf( stderr, "\t\t %.256s (0x%p)\n", task->getName(), task );
		    } // for
		} // for
		uAbort( "No ready or pending tasks.\n"
			"Possible cause is tasks are in a synchronization or mutual exclusion deadlock." );
	    } // if
	} // if

	if ( spin == 0 ) {				// task executed ?
	    cycleStart = currProc;			// mark new start for cycle
	} // if

	// Get next processor to execute.

	nextProcessor( currProc );
#endif // __U_MULTI__
    } // for

    // ACCESS NO KERNEL DATA STRUCTURES AFTER THIS POINT BECAUSE THEY MAY NO
    // LONGER EXIST.

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.main, exiting\n", this );
#endif // __U_DEBUG_H__

    uThisCluster().makeProcessorActive();

#if defined( __freebsd__ ) && defined( __U_MULTI__ )
    // return the ldt index to the pool of available ones for re-use
    i386_set_ldt( THREAD_GETMEM( ldtValue ) >> 3, NULL, 1 );
#endif // __freebsd__ && __U_MULTI__

#if defined( __U_MULTI__ )
#if defined( __linux__ ) && defined( __U_TLS__ )
    syscall( __NR_exit, 0 );
#elif defined( __solaris__ ) && defined( __U_TLS__ )
    syscall( SYS_lwp_exit, 0 );
#elif defined( __U_PTHREAD__ )
    uRealPthread::pthread_exit( NULL );
#elif defined( __solaris__ )
    _lwp_exit();					// exit without calling the global destructors.
#else
    _exit( 0 );						// exit without calling the global destructors.
#endif // __solaris__
#endif // __U_MULTI__
} // uProcessorKernel::main


uProcessorKernel::uProcessorKernel()
#ifdef __U_PTHREAD__
	: uBaseCoroutine( PTHREAD_STACK_MIN )
#endif // __U_PTHREAD__
{
} // uProcessorKernel::uProcessorKernel

uProcessorKernel::~uProcessorKernel() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessorKernel &)0x%p.~uProcessorKernel, exiting\n", this );
#endif // __U_DEBUG_H__
} // uProcessorKernel::~uProcessorKernel


void *uProcessorKernel::operator new( size_t, void *storage ) {
    return storage;
} // uProcessorKernel::operator new

void *uProcessorKernel::operator new( size_t size ) {
    return ::operator new( size );
} // uProcessorKernel::operator new


//######################### uProcessor #########################


void *uProcessor::operator new( size_t, void *storage ) {
    return storage;
} // uProcessor::operator new

void *uProcessor::operator new( size_t size ) {
    return ::operator new( size );
} // uProcessor::operator new


void uProcessor::createProcessor( uCluster &cluster, bool detached, int ms, int spin ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessor &)0x%p.createProcessor, on cluster %.256s (0x%p)\n", this, cluster.getName(), &cluster );
#endif // __U_DEBUG_H__

#ifdef __U_DEBUG__
    if ( ms == 0 ) {					// 0 => infinity, reset to minimum preemption second for local debugger
	ms = uMinPreemption;				// approximate infinity
    } else if ( ms < 0 ) {				// special case for testing, not for public consumption
	ms = 0;
    } // if
#ifdef __U_MULTI__
    debugIgnore = false;
#else
    debugIgnore = true;
#endif // __U_MULTI__
#endif // __U_DEBUG__

    currTask = NULL;
    currCluster = &cluster;
    uProcessor::detached = detached;
    preemption = ms;
    uProcessor::spin = spin;

#ifdef __U_MULTI__
#ifdef __U_ONETIMER__
    procEvents = new uProcEventList( *this );
#else
    events = new uSysEventList;
#endif // __U_ONETIMER__
    contextSwitchHandler = new uCxtSwtchHndlr;
    contextEvent = new uEventNode( *contextSwitchHandler, this );
#endif // __U_MULTI__
    terminated = false;
    currCluster->processorAdd( *this );

    uKernelModule::globalProcessorLock->acquire();	// add processor to global processor list.
    uKernelModule::globalProcessors->addTail( &(globalRef) );
    uKernelModule::globalProcessorLock->release();

    procTask = new uProcessorTask( cluster, *this );
    
#if defined( __linux__ ) && defined( __i386__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
    ldtValue = allocLDT();
#endif // __i386__ && __U_MULTI__ && ! __U_PTHREAD__
} // uProcessor::createProcessor


uProcessor::uProcessor( uCluster &cluster, double ) : idleRef( *this ), processorRef( *this ), globalRef( *this ) {
    createProcessor( cluster, false, uDefaultPreemption(), uDefaultSpin() );
} // uProcessor::uProcessor


uProcessor::uProcessor( unsigned int ms, unsigned int spin ) : idleRef( *this ), processorRef( *this ), globalRef( *this ) {
    createProcessor( uThisCluster(), false, ms, spin );
#if defined( __U_MULTI__ )
    // Register the processor before creating the processor task because the
    // processor task uses the profiler sampler. Uniprocessor calls done in
    // uProfilerBoot.
    if ( uProfiler::uProfiler_registerProcessor ) {
	(*uProfiler::uProfiler_registerProcessor)( uProfiler::profilerInstance, *this );
    } // if
#endif // __U_MULTI_H__
    uThisProcessor().fork( this );			// processor executing this declaration forks the UNIX process
} // uProcessor::uProcessor


uProcessor::uProcessor( bool detached, unsigned int ms, unsigned int spin ) : idleRef( *this ), processorRef( *this ), globalRef( *this ) {
    createProcessor( uThisCluster(), detached, ms, spin );
    
#if defined( __U_MULTI__ )
    // see above
    if ( uProfiler::uProfiler_registerProcessor ) {
	(*uProfiler::uProfiler_registerProcessor)( uProfiler::profilerInstance, *this );
    } // if
#endif // __U_MULTI_H__
    uThisProcessor().fork( this );			// processor executing this declaration forks the UNIX process
} // uProcessor::uProcessor


uProcessor::uProcessor( uCluster &clus, unsigned int ms, unsigned int spin ) : idleRef( *this ), processorRef( *this ), globalRef( *this ) {
    createProcessor( clus, false, ms, spin );
#if defined( __U_MULTI__ )
    // see above
    if ( uProfiler::uProfiler_registerProcessor ) {
	(*uProfiler::uProfiler_registerProcessor)( uProfiler::profilerInstance, *this );
    } // if
#endif // __U_MULTI_H__
    uThisProcessor().fork( this );			// processor executing this declaration forks the UNIX process
} // uProcessor::uProcessor


uProcessor::uProcessor( uCluster &clus, bool detached, unsigned int ms, unsigned int spin ) : idleRef( *this ), processorRef( *this ), globalRef( *this ) {
    createProcessor( clus, detached, ms, spin );
#if defined( __U_MULTI__ )
    // see above
    if ( uProfiler::uProfiler_registerProcessor ) {
	(*uProfiler::uProfiler_registerProcessor)( uProfiler::profilerInstance, *this );
    } // if
#endif // __U_MULTI_H__
    uThisProcessor().fork( this );			// processor executing this declaration forks the UNIX process
} // uProcessor::uProcessor


uProcessor::~uProcessor() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uProcessor &)0x%p.~uProcessor\n", this );
#endif // __U_DEBUG_H__

    delete procTask;

#if defined( __U_MULTI__ )
    // Deregister the processor after deleting the processor task because the
    // processor task uses the profiler sampler.
    if ( uProfiler::uProfiler_deregisterProcessor ) {
	(*uProfiler::uProfiler_deregisterProcessor)( uProfiler::profilerInstance, *this );
    } // if
#endif // __U_MULTI_H__

    // Remove processor from global processor list. It is removed here because
    // the next action after this is the termination of the UNIX process in
    // uProcessorKernel. Therefore, even if the application is aborted and the
    // process is not on the list of UNIX processes for this application, the
    // current UNIX process will terminate itself.  It cannot be removed in
    // uProcessKernel because the uProcessor storage may be freed already by
    // another processor.

    uKernelModule::globalProcessorLock->acquire();	// remove processorto global processor list.
    uKernelModule::globalProcessors->remove( &(globalRef) );
    uKernelModule::globalProcessorLock->release();

    currCluster->processorRemove( *this );
#ifdef __U_MULTI__
    delete contextEvent;
    delete contextSwitchHandler;
#ifdef __U_ONETIMER__
    delete procEvents;
#else
    delete events;
#endif // __U_ONETIMER__
#endif // __U_MULTI__
    
#if defined( __linux__ ) && defined( __i386__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
    freeLDT( ldtValue );
#endif // __linux__ && __i386__ && __U_MULTI__ && ! __U_PTHREAD__
} // uProcessor::~uProcessor


void uProcessor::fork( uProcessor *processor ) {
#ifdef __U_MULTI__
    uKernelModule::initialization = false;

#if defined( __U_PTHREAD__ )

    int ret;
    pthread_attr_t attr;
    // SIGALRM must only be caught by the system processor
    sigset_t mask, old_mask;
    sigemptyset( &mask );
    sigaddset( &mask, SIGALRM );
    sigprocmask( SIG_BLOCK, &mask, &old_mask );
    assert( &uThisProcessor() == uKernelModule::systemProcessor || sigismember( &old_mask, SIGALRM ) );

    ret = uRealPthread::pthread_attr_init( &attr );
    if ( ret ) {
	uAbort( "(uProcessorKernel &)0x%p.fork() : internal error, pthread_attr_init failed, error(%d) %s.", this, ret, strerror( ret ) );
    } // if
    assert( processor->processorKer.size == PTHREAD_STACK_MIN );
    ret = uRealPthread::pthread_attr_setstack( &attr, processor->processorKer.limit, processor->processorKer.size );
    if ( ret ) {
	uAbort( "(uProcessorKernel &)0x%p.fork() : internal error, pthread_attr_setstack failed, error(%d) %s.", this, ret, strerror( ret ) );
    } // if
    ret = uRealPthread::pthread_create( &processor->pid, &attr, __U_START_KT_NAME__, processor );
    if ( ret ) {
	uAbort( "(uProcessorKernel &)0x%p.fork() : internal error, pthread_create failed, error(%d) %s.", this, ret, strerror( ret ) );
    } // if

    sigprocmask( SIG_SETMASK, &old_mask, NULL );

#elif defined( __linux__ )
    int flag = CLONE_VM | CLONE_FS | CLONE_FILES | SIGCHLD;
    // KTs created by the initial (system) kernel-thread (SKT) already have the
    // SKT as their parent. A KT created as a child of these sub-KTs is marked
    // with CLONE_PARENT so its parent is that of its creator, i.e., the
    // SKT. Hence, the SKT has the shell KT as its parent and all KTs created
    // directly or indirectly from the SKT have the SKT as their parent.
    if ( &uThisProcessor() != uKernelModule::systemProcessor ) {
	flag |= CLONE_PARENT;
    } // if

#if defined( __i386__ ) 
    processor->pid = clone( __U_START_KT_NAME__, (char *)(processor->processorKer.base), flag, processor );
#elif defined( __ia64__ )
    processor->pid = __clone2( __U_START_KT_NAME__, (char *)(processor->processorKer.limit), processor->processorKer.size, flag, processor );
#else
    #error uC++ : internal error, unsupported architecture
#endif
    if ( processor->pid == -1 ) {
	uAbort( "(uProcessorKernel &)0x%p.fork() : internal error, create failed for kernel thread, error(%d) %s.", this, errno, strerror( errno ) );
    } //if

#elif defined( __solaris__ )
    ucontext *ucp;
#if defined( __U_SWAPCONTEXT__ )
    ucp = (ucontext *)(processor->processorKer.storage);
    // _lwp_makecontext overwrites the existing context except for the signal mask
    // so the signal mask obtained from getcontext during startHere is used
#else
    ucontext uc;
    ucp = &uc;
    // make sure the new lwp has an empty signal mask at start
    sigemptyset( &uc.uc_sigmask );
#endif

#ifdef __U_ONETIMER__
    // SIGALRM must only be caught by the system processor
    sigaddset( &uc.uc_sigmask, SIGALRM );
#endif // __U_ONETIMER__

    _lwp_makecontext( ucp, __U_START_KT_NAME__, processor, NULL, (char *)(processor->processorKer.limit), processor->processorKer.size );
    int code = _lwp_create( ucp, 0, &processor->pid );
    if ( code != 0 ) {
	uAbort( "(uProcessorKernel &)0x%p.fork() : internal error, create failed for kernel thread, error(%d) %s.", this, code, strerror( code ) );
    } // if
#elif defined( __irix__ ) || defined( __freebsd__ )
    processor->pid = uKernelModule::systemProcessor->procCreate( processor );
#else
    #error uC++ : internal error, unsupported architecture
#endif

    uKernelModule::initialization = true;

#else

    processor->pid = pid;

#endif // __U_MULTI__
} // uProcessor::fork


void uProcessor::setContextSwitchEvent( uDuration duration ) {
    assert( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) == 1 );

    if ( ! contextEvent->listed() && duration != 0 ) { // first context switch event ?
	contextEvent->timerT = activeProcessorKernel->kernelClock.getTime() + duration;
	contextEvent->timerD = duration;
	contextEvent->add();
    } else if ( duration > 0  && contextEvent->timerD != duration ) { // if event is different from previous ? change it
	contextEvent->remove();
	contextEvent->timerT = activeProcessorKernel->kernelClock.getTime() + duration;
	contextEvent->timerD = duration;
	contextEvent->add();
    } else if ( duration == 0 && contextEvent->timerT != 0 ) { // zero duration and current CS is nonzero ?
	contextEvent->remove();
	contextEvent->timerT = 0;   
	contextEvent->timerD = 0;
    } // if
}; // uProcessor::setContextSwitchEvent


void uProcessor::setContextSwitchEvent( int msecs ) {
    setContextSwitchEvent( uDuration( msecs / 1000L, msecs % 1000L * ( TIMEGRAN / 1000L ) ) ); // convert msecs to uDuration type
} // uProcessor::setContextSwitchEvent


#ifdef __U_MULTI__
#if defined( __irix__ ) || defined( __freebsd__ )
uPid_t uProcessor::procCreate( uProcessor *processor ) {
    return procTask->procCreate( processor );
} // uProcessor::procCreate
#endif


#if defined( __irix__ ) || ( defined( __linux__ ) && ! defined( __U_PTHREAD__ ) ) || defined( __freebsd__ )
void uProcessor::procWait( uPid_t pid ) {
    procTask->procWait( pid );
} // uProcessor::procWait
#endif


void uProcessor::procExit( int retcode ) {
    procTask->procExit( retcode );
} // uProcessor::procExit
#endif // __U_MULTI__


uClock &uProcessor::getClock() const {
    return *processorClock;
} // uProcessor::getClock


uPid_t uProcessor::getPid() const {
    return pid;
} // uProcessor::getPid


uCluster &uProcessor::getCluster() const {
    return *currCluster;
} // uProcessor::getCluster


uCluster &uProcessor::setCluster( uCluster &cluster ) {
  if ( &cluster == &(uThisProcessor().getCluster()) ) return cluster; // trivial case

    uCluster &prev = cluster;
    procTask->setCluster( cluster );			// operation must be done by the processor itself
    return prev;
} // uProcessor::setCluster


uBaseTask &uProcessor::getTask() const {
    return *currTask;
} // uProcessor::getTask


bool uProcessor::getDetach() const {
    return detached;
} // uProcessor::getTask


unsigned int uProcessor::setPreemption( unsigned int ms ) {
    int temp = ms;					// make writable
#ifdef __U_DEBUG__
    if ( ms == 0 ) {					// 0 => infinity, reset to minimum preemption second for local debugger
	temp = uMinPreemption;				// approximate infinity
    } else if ( ms < 0 ) {				// special case for testing, not for public consumption
	temp = 0;
    } // if
#endif // __U_DEBUG__
    int prev = preemption;
    procTask->setPreemption( temp );

    // Asynchronous with regard to other processors, synchronous with regard to
    // this processor.

    if ( &uThisProcessor() == this ) {
	uThisTask().yield();
    } // if
    return prev;
} // uProcessor::setPreemption


unsigned int uProcessor::getPreemption() const {
    return preemption;
} // uProcessor::getPreemption


unsigned int uProcessor::setSpin( unsigned int spin ) {
    int prev = spin;
    uProcessor::spin = spin;
    return prev;
} // uProcessor::setSpin


unsigned int uProcessor::getSpin() const {
    return spin;
} // uProcessor::getSpin


bool uProcessor::idle() const {
    return idleRef.listed();
} // uProcessor::idle


#if defined( __linux__ ) && defined( __i386__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
// elements of the bitset are "true" if the corresponding ldt index
// is available for use; initialize all elements to true initially
uBitSet< U_MAX_LDT > uProcessor::LDTFreeSet;

uOwnerLock uProcessor::LDTFreeSetLock;
bool uProcessor::LDTFreeSetInit = false;


int uProcessor::allocLDT() {
    if ( uKernelModule::initialization ) LDTFreeSetLock.acquire();

    if ( ! LDTFreeSetInit ) {
        LDTFreeSet.setAll();
        LDTFreeSetInit = true;
    } // if
    
    int idx = LDTFreeSet.findFirstSet();
    if ( idx == -1 ) {
        uAbort( "uProcessor::allocLDT() : internal error, LDT space exhausted." );
    } // if
    LDTFreeSet.clr( idx );

    if ( uKernelModule::initialization ) LDTFreeSetLock.release();

#ifdef __U_DEBUG_H__
    if( uKernelModule::initialization ) uDebugPrt( buffer, "uProcessor::allocLDT allocated new ldt entry %d (index %d)\n", idx * 8 + 7, idx );
#endif // __U_DEBUG_H__

    return idx * 8 + 7;
} // uProcessor::allocLDT


void uProcessor::freeLDT( int oldLDT ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( buffer, "uProcessor::freeLDT adding ldt entry %d (index %d) to free set\n", oldLDT, ( oldLDT - 7 ) / 8 );
#endif // __U_DEBUG_H__

    LDTFreeSetLock.acquire();
    LDTFreeSet.set( ( oldLDT - 7 ) / 8 );
    LDTFreeSetLock.release();
} // uProcessor::freeLDT
#endif // __linux__ && __i386__ && __U_MULTI__ && ! __U_PTHREAD__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
