//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.0, Copyright (C) Peter A. Buhr 1994
// 
// uC++.h -- 
// 
// Author           : Peter A. Buhr
// Created On       : Fri Dec 17 22:04:27 1993
// Last Modified By : Peter A. Buhr
// Last Modified On : Mon Sep 25 17:06:55 2006
// Update Count     : 4651
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 

// ***************************************************************************
// WARNING: If a .cc file is added to the kernel that does not define
// __U_KERNEL__ BEFORE including uC++.h, the profiler will likely fail to
// register processors created using the uDefaultProcessors mechanism.
// Furthermore, no explicit error is reported in this case -- the problem can
// only be detected by observing erroneous results reported by certain metrics
// that are sensitive to the registration of processors.
// ***************************************************************************


#ifndef __U_CPLUSPLUS_H__
#define __U_CPLUSPLUS_H__

#pragma __U_NOT_USER_CODE__

#if defined( __U_MULTI__ ) && defined( __U_TLS__ )
#define __U_THREAD__ __thread
#define __U_PTHREAD__
#ifndef __U_ONETIMER__
#define __U_ONETIMER__
#endif // ! __U_ONETIMER__
#else
#define __U_THREAD__
#endif // __U_MULTI__ && __U_TLS__

#if defined( __solaris__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
#include <sys/lwp.h>
#endif // __solaris__ && __U_MULTI__ && ! __U_PTHREAD__

/* The GNU Libc defines C library functions with throw () when compiled under
   C++, to enable optimizations.  When uC++ overrides these functions, it must
   provide identical exception specifications.  */
#if defined( __linux__ )
#define __THROW throw ()

/* Certain library functions have had __THROW removed from their prototypes
   to support the NPTL implementation of pthread cancellation. To compile with
   a pre-NPTL version of the header files use

   #define __OLD_THROW throw ()
*/
#define __OLD_THROW
#else
#define __OLD_THROW
#define __THROW
#endif // __linux__

/* ___errno (Solaris) and __errno_location (Linux) provide libc with the
   address of the task-specific errno variable.  Both functions should take no
   arguments; however, a gcc bug (Bug c++/11390) prevents functions defined in
   system headers from being interpreted properly when they are redeclared in
   user code, such as redeclaring as "friend"s.  So some creativity is
   required.

   Irix has a similar function called __oserror, but it is not called by the
   standard library if overridden in uC++; in this case, the system ensures
   that the errnos of different kernel threads stay independent, so we can
   simply save the per-task errno on each context switch. Solaris works in
   this way when pthreads are used. */
#if defined( __solaris__ )
#if ! defined( _REENTRANT )
#define _REENTRANT
#endif
#if ! defined( __U_TLS__ )
#if __GNUC__ > 3 || __GNUC_MINOR__ > 3
#define __U_ERRNO_FUNC__ ___errno()
#else
// TEMPORARY: bug fixed in gcc 3.4.0
// http://gcc.gnu.org/bugzilla/show_bug.cgi?id=11390
#define __U_ERRNO_FUNC__ ___errno(...)
#endif
#endif // ! __U_TLS__
#include <cerrno>

#elif defined( __linux__ )
#if ! defined( _LIBC_REENTRANT )
#define _LIBC_REENTRANT
#endif
#include <cerrno>
#if ( __GLIBC__ <= 2 ) && ( __GLIBC_MINOR__ < 3 )
#define __U_ERRNO_FUNC__ __errno_location( void )
#endif // ! glibc 2.3

#elif defined( __irix__ )
#if ! defined( _SGI_MP_SOURCE )
#define _SGI_MP_SOURCE
#endif
#include <cerrno>
#include <ulocks.h>

#elif defined( __freebsd__ )
#include <cerrno>
#define __U_ERRNO_FUNC__ __error()

#else
    #error uC++ : internal error, unsupported architecture
#endif

#ifdef __U_ERRNO_FUNC__
extern "C" int *__U_ERRNO_FUNC__ __THROW __attribute__(( used ));
#endif // __U_ERRNO_FUNC__

#if defined( __i386__ )
// #define __U_SWAPCONTEXT__				// use builtin swapcontext
#elif defined( __x86_64__ )
// no usable swapcontext on this platform
#elif defined( __ia64__ )
#define __U_SWAPCONTEXT__				// use builtin swapcontext
#elif defined( __sparc__ )
// #define __U_SWAPCONTEXT__				// use builtin swapcontext
#elif defined( __mips__ )
// #define __U_SWAPCONTEXT__				// use builtin swapcontext
#else
    #error uC++ : internal error, unsupported architecture
#endif

#include <uStaticAssert.h>				// access: _STATIC_ASSERT_
#include <assert.h>
//#include <uDebug.h>

#include <uStack.h>
#include <uQueue.h>
#include <uSequence.h>
#include <uBitSet.h>
#include <uDefault.h>

#include "uKernelThreads.h"
#include "uAtomic.h"

#include <cstddef>					// ptrdiff_t
#include <cstdlib>					// malloc, calloc, realloc, free
#if defined( __linux__ )
#include <malloc.h>					// memalign
#endif // __linux__
#if defined( __freebsd__ )
extern "C" void *memalign( size_t alignment, size_t size );
#endif // __freebsd__
#include <csignal>					// signal, etc.
#include <ucontext.h>					// ucontext_t
#include <inttypes.h>					// uintptr_t

#include <exception>
#include <iosfwd>					// std::filebuf
#include <pthread.h>					// PTHREAD_CANCEL_*


#if defined( __U_PTHREAD__ )
typedef pthread_t uPid_t;
#elif defined( __solaris__ ) && defined( __U_MULTI__ )
typedef lwpid_t uPid_t;
#else
typedef pid_t uPid_t;
#endif


#define __U_MAXENTRYBITS__ 128				// N-1 mutex members, including destructor, and timeout
#define __Monitor _Mutex class				// short form for monitor
#define _Monitor _Mutex class
#define uMonitor _Mutex class				// deprecated
#define __Cormonitor _Mutex _Coroutine			// short form for coroutine monitor
#define _Cormonitor _Mutex _Coroutine
#define uCorMonitor _Mutex _Coroutine			// deprecated

class uKernelBoot;					// forward declaration
_Task uBootTask;					// forward declaration
_Task uSystemTask;					// forward declaration
class uMachContext;					// forward declaration
class uBaseCoroutine;					// forward declaration
class uBaseTask;					// forward declaration
class uSpinLock;					// forward declaration
class uLock;						// forward declaration
class uOwnerLock;					// forward declaration
class uCondLock;					// forward declaration
class uSemaphore;					// forward declaration
class uProcessor;					// forward declaration
class uDefaultScheduler;				// forward declaration
class uCluster;						// forward declaration
class uNBIO;						// forward declaration
_Coroutine uProcessorKernel;				// forward declaration
_Task uProcessorTask;					// forward declaration
class uHeapManager;					// forward declaration
class uSerial;						// forward declaration
class uSerialMember;					// forward declaration
class uEventList;					// forward declaration
_Task uMain;						// forward declaration

extern uBaseCoroutine &uThisCoroutine();		// forward declaration
extern uBaseTask &uThisTask();				// forward declaration
extern uProcessor &uThisProcessor();			// forward declaration
extern uCluster &uThisCluster();			// forward declaration


#ifdef __U_PTHREAD__
#include <pthread.h>

class uRealPthread {
    friend class uProcessor;				// access: pthread_create, pthread_attr_*
    friend _Task uProcessorTask;			// access: pthread_join
    friend class uProcessorKernel;			// access: pthread_exit
    friend class uCluster;				// access: pthread_kill
    friend class uKernelBoot;				// access: startup

    static void startup();

    static typeof( ::pthread_create ) *pthread_create;
    static typeof( ::pthread_attr_init ) *pthread_attr_init;
    static typeof( ::pthread_attr_setstack ) *pthread_attr_setstack;
    static typeof( ::pthread_kill ) *pthread_kill;
    static typeof( ::pthread_join ) *pthread_join;
  public:
    static typeof( ::pthread_self ) *pthread_self;
};
#endif // __U_PTHREAD__


//######################### Profiling ########################


_Task uProfiler;					// forward declaration
class uProfilerBoot;
class uProfileTaskSampler;
class uProfileClusterSampler;
class uProfileProcessorSampler;

extern "C" void __cyg_profile_func_enter( void *pcCurrentFunction, void *pcCallingFunction );
extern "C" void __cyg_profile_func_exit( void *pcCurrentFunction, void *pcCallingFunction );


//######################### uAbort/exit #########################


extern void uAbort( const char *fmt = "", ... ) __attribute__(( format (printf, 1, 2), noreturn ));
extern void uExit( int code ) __attribute__(( deprecated, noreturn ));


//######################### Signal Handling #########################


// define parameter types for signal handlers

#define __U_SIGCXT__ ucontext_t *
#define __U_SIGPARMS__ int sig, siginfo_t *sfp, __U_SIGCXT__ cxt
#define __U_SIGTYPE__ int, siginfo_t *, __U_SIGCXT__

class uSigHandlerModule {
    friend class uKernelBoot;				// access: uSigHandlerModule
    friend _Task uLocalDebugger;			// access: signal
    friend _Task uProfiler;				// access: signal, signalContextPC

    static void signal( int sig, void (*handler)(__U_SIGPARMS__), int flags = 0 );
    static void *signalContextPC( __U_SIGCXT__ cxt );
    static void *functionAddress( void (*function)() );
    static void sigChldHandler( __U_SIGPARMS__ );
    static void sigTermHandler( __U_SIGPARMS__ );
    static void sigAlrmHandler( __U_SIGPARMS__ );
    static void sigSegvBusHandler( __U_SIGPARMS__ );
    static void sigIllHandler( __U_SIGPARMS__ );
    static void sigFpeHandler( __U_SIGPARMS__ );

    uSigHandlerModule( uSigHandlerModule & );		// no copy
    uSigHandlerModule &operator=( uSigHandlerModule & ); // no assignment

    uSigHandlerModule();
  public:
}; // uSigHandlerModule


//######################### uProcessor #########################


class uProcessorDL : public uSeqable {
    uProcessor &_processor;
  public:
    uProcessorDL( uProcessor &_processor ) : _processor( _processor ) {}
    uProcessor &processor() const { return _processor; }
}; // uProcessorDL

typedef uSequence<uProcessorDL> uProcessorSeq;


//######################### uCluster #########################


class uClusterDL : public uSeqable {
    uCluster &_cluster;
  public:
    uClusterDL( uCluster &_cluster ) : _cluster( _cluster ) {}
    uCluster &cluster() const { return _cluster; }
}; // uClusterDL

typedef uSequence<uClusterDL> uClusterSeq;


//######################### uBaseTask #########################


class uBaseTaskDL : public uSeqable {
    uBaseTask &_task;
  public:
    uBaseTaskDL( uBaseTask &_task ) : _task( _task ) {}
    uBaseTask &task() const { return _task; }
}; // uBaseTaskDL

typedef uSequence<uBaseTaskDL> uBaseTaskSeq;


//######################### uKernelModule #########################


extern "C" {						// TEMPORARY: profiler allocating memory from the kernel issue
    int pthread_mutex_lock( pthread_mutex_t *mutex ) __THROW;
    int pthread_mutex_trylock( pthread_mutex_t *mutex ) __THROW;
    int pthread_mutex_unlock( pthread_mutex_t *mutex ) __THROW;
} // extern "C"


#define __U_START_KT_NAME__ uKernelModule::startThread
#if defined( __U_PTHREAD__ )
#define __U_START_KT_TYPEGEN__( name ) void *name( void *p )
#elif defined( __linux__) || defined( __freebsd__ )
#define __U_START_KT_TYPEGEN__( name ) int name( void *p )
#elif defined( __solaris__ )
#define __U_START_KT_TYPEGEN__( name ) void name( void *p )
#elif defined( __irix__ )
#define __U_START_KT_TYPEGEN__( name ) void name( void *p, size_t size )
#else
    #error uC++ : internal error, unsupported architecture
#endif
#define __U_START_KT_TYPE__ __U_START_KT_TYPEGEN__( __U_START_KT_NAME__ )
#define __U_START_KT_TYPE_UNQUAL__ __U_START_KT_TYPEGEN__( startThread )


class uKernelModule {
    friend void uAbort( const char *fmt, ... );		// access: globalAbort, coreDumped
    friend void exit( int retcode ) __THROW;		// access: globalAbort
    friend class uSigHandlerModule;			// access: uKernelModuleBoot, globalAbort, coreDumped, rollForward
    friend class uMachContext;				// access: everything
    friend class uBaseCoroutine;			// access: uKernelModuleBoot
    friend class uBaseTask;				// access: uKernelModuleBoot
    friend class uSerial;				// access: uKernelModuleBoot
    friend class uSerialDestructor;			// access: uKernelModuleBoot
    friend class uSpinLock;				// access: uKernelModuleBoot
    friend class uOwnerLock;				// access: uKernelModuleBoot, initialization
    friend class uCondLock;				// access: uKernelModuleBoot
    friend class uSemaphore;				// access: uKernelModuleBoot
    friend class uContext;				// access: uKernelModuleBoot
    friend class uCoroutineMain;			// access: uKernelModuleBoot
    friend _Coroutine uProcessorKernel;			// access: uKernelModuleBoot, globalProcessors, globalClusters, systemProcessor
    friend class uProcessor;				// access: everything
    friend uBaseTask &uThisTask();			// access: uKernelModuleBoot
    friend uProcessor &uThisProcessor();		// access: uKernelModuleBoot
    friend uCluster &uThisCluster();			// access: uKernelModuleBoot
    friend _Task uProcessorTask;			// access: uKernelModuleBoot
    friend class uCluster;				// access: uKernelModuleBoot, globalClusters, globalClusterLock, rollForward
    friend _Task uBootTask;				// access: uKernelModuleBoot, systemCluster
    friend _Task uSystemTask;				// access: systemCluster
    friend _Task uMain;					// access: bootTask
    friend class uKernelBoot;				// access: everything
    friend class uInitProcessorsBoot;			// access: numProcessors, userProcessors
    friend class uHeapManager;				// access: bootTaskStorage, kernelModuleInitialized, startup
    friend class uNBIO;					// access: uKernelModuleBoot
    friend class uRealPthread;				// access: interposeSymbol

    // real-time

    friend class uSysEventList;				// access: uKernelModuleBoot
    friend class uProcEventList;			// access: processorKernelStorage
    friend class uEventList;                            // access: uKernelModuleBoot
    friend class uEventListPop;				// access: uKernelModuleBoot

    // debugging

    friend _Task uLocalDebugger;			// access: uKernelModuleBoot, bootTask, globalClusters, systemProcessor, systemCluster
    friend class uLocalDebuggerHandler;			// access: uKernelModuleBoot

    // profiling

    friend int pthread_mutex_lock( pthread_mutex_t *mutex ) __THROW; // TEMPORARY: profiler allocating memory from the kernel issue
    friend int pthread_mutex_trylock( pthread_mutex_t *mutex ) __THROW; // TEMPORARY: profiler allocating memory from the kernel issue
    friend int pthread_mutex_unlock( pthread_mutex_t *mutex ) __THROW; // TEMPORARY: profiler allocating memory from the kernel issue
    friend void __cyg_profile_func_enter( void *pcCurrentFunction, void *pcCallingFunction ); // access: uKernelModuleBoot
    friend void __cyg_profile_func_exit( void *pcCurrentFunction, void *pcCallingFunction ); // access: uKernelModuleBoot
    friend class uProfiler;				// access: uKernelModuleBoot
    friend class uProfilerBoot;				// access: uKernelModuleBoot, bootTask, systemCluster, userProcessor
    friend class HWCounters;				// access: uKernelModuleBoot
    template<typename Elem, unsigned int BlockSize, typename Admin> friend struct uFixedListArray; // access: uKernelModuleBoot

    friend class uKernelSampler;			// access: globalClusters
    friend class uClusterSampler;			// access: globalClusters


    struct uKernelModuleData {
#if defined( __i386__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
	unsigned int ldtValue;
#endif // __i386__

	volatile uKernelModuleData *This;

#	define __U_MAX_PAGE_SIZE__ (16 * 1024)
#	define activeProcessorKernel (THREAD_GETMEM( processorKernelStorage ))
	uProcessor *activeProcessor;			// current active processor
	// The next two private variables shadow the corresponding fields in the
	// processor data structure. They are an optimization so that routines
	// uThisCluster and uThisTask do not have to be atomic routines, and as a
	// consequence can be inlined. The problem is the multiple activeProcessor
	// variables (one per UNIX process). A task executing on one processor can
	// be time sliced after loading the address of the active processor into a
	// register, rescheduled on another processor and restarted, but now the
	// previously loaded processor pointer is incorrect. By updating these two
	// shadow variables whenever the corresponding processor field is changed
	// (and this occurs atomically in the kernel), the appropriate data
	// structure (cluster or task) can be accessed with a single load
	// instruction, which is atomic.
	uCluster  *activeCluster;			// current active cluster for processor
	uBaseTask *activeTask;				// current active task for processor

	bool disableInt;				// task in kernel: no time slice interrupts
	unsigned int disableIntCnt;

	bool disableIntSpin;				// task in spin lock; no time slice interrupts
	unsigned int disableIntSpinCnt;

	unsigned int inKernelRF;			// in kernel roll forward flag

	uProcessorKernel *processorKernelStorage;	// system-cluster processor kernel

	// The thread pointer value needs to be accessible so that it can be
	// properly restored on context switches.  On a non-tls system the thread
	// pointer points directly at the kernel module, i.e. tp == This.  On a
	// tls system the system places the kernel module, so tp != This.
	unsigned long threadPointer;


	void disableInterrupts() volatile {
	    THREAD_SETMEM( disableInt, true );
	    int old = THREAD_GETMEM( disableIntCnt );
	    THREAD_SETMEM( disableIntCnt, old + 1 );
	} // uKernelModuleData::disableInterrupts


	void enableInterrupts() volatile {
#ifdef __U_DEBUG__
	    assert( disableInt && disableIntCnt > 0 );
#endif // __U_DEBUG__

	    disableIntCnt -= 1;				// decrement number of disablings
	    if ( disableIntCnt == 0 ) {
		// Enable interrupts first, otherwise a window exists in which a
		// signal could occur but a rollforward is not done.
		disableInt = false;			// enable interrupts
		if ( THREAD_GETMEM( inKernelRF ) && ! THREAD_GETMEM( disableIntSpin ) ) { // SIGALRM undelivered and not in kernel?
		    rollForward();			// deliver SIGALRM
		} // if
	    } // if
#ifdef __U_DEBUG__
	    assert( ! THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) == 0 || THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 );
#endif // __U_DEBUG__
	} // KernelModule::enableInterrupts


	void disableIntSpinLock() volatile {
#ifdef __U_DEBUG__
	    assert( ! THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) == 0 || THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__

	    THREAD_SETMEM( disableIntSpin, true );
	    int old = THREAD_GETMEM( disableIntSpinCnt ); // processor independent increment
	    THREAD_SETMEM( disableIntSpinCnt, old + 1 );

#ifdef __U_DEBUG__
	    assert( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__
	} // uKernelModuleData::disableIntSpinLock


	void enableIntSpinLock() volatile {
#ifdef __U_DEBUG__
	    assert( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__

	    disableIntSpinCnt -= 1;			// decrement number of disablings
	    if ( disableIntSpinCnt == 0 ) {
		disableIntSpin = false;			// enable interrupts

		if ( THREAD_GETMEM( inKernelRF ) && ! THREAD_GETMEM( disableInt ) ) { // SIGALRM undelivered and not in kernel?
		    rollForward();			// deliver SIGALRM
		} // if
	    } // if

#ifdef __U_DEBUG__
	    assert( ! THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) == 0 || THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__
	} // uKernelModuleData::enableIntSpinLock


	void enableIntSpinLockNoRF() volatile {
#ifdef __U_DEBUG__
	    assert( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__

	    disableIntSpinCnt -= 1;			// decrement number of disablings
	    if ( disableIntSpinCnt == 0 ) {
		disableIntSpin = false;			// enable interrupts
	    } // if

#ifdef __U_DEBUG__
	    assert( ! THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) == 0 || THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__
	} // uKernelModuleData::enableIntSpinLock

	void ctor() volatile;				// POD constructor
    }; // uKernelModuleData


    // shared, initialized in uC++.cc

    static bool kernelModuleInitialized;
    static volatile __U_THREAD__ uKernelModuleData uKernelModuleBoot;
    static bool initialization;				// initialization/finalization done
    static unsigned int attaching;			// flag to signal the local kernel to start attaching.
    static bool coreDumped;				// ensure only one core file
#ifndef __U_MULTI__
    static bool deadlock;				// deadlock detected in kernel
#endif // ! __U_MULTI__
    static bool globalAbort;				// indicate aborting processor
    static bool globalSpinAbort;			// indicate aborting processor to spin locks
    static uSpinLock *globalAbortLock;			// only one aborting processors
    static uSpinLock *globalProcessorLock;		// mutual exclusion for global processor operations
    static uProcessorSeq *globalProcessors;		// global list of processors
    static uSpinLock *globalClusterLock;		// mutual exclusion for global cluster operations
    static uClusterSeq *globalClusters;			// global list of cluster
    static uDefaultScheduler *systemScheduler;		// pointer to systen scheduler for system cluster
    static uCluster *systemCluster;			// pointer to system cluster
    static uProcessor *systemProcessor;			// pointer to system processor
    static uBootTask *bootTask;				// pointer to boot task for global constructors/destructors
    static uProcessor **userProcessors;			// pointer to user processors
    static unsigned int numProcessors;			// number of user processors
    static char systemProcessorStorage[];
    static char systemClusterStorage[];
    static char bootTaskStorage[];

    static std::filebuf *cerrFilebuf, *clogFilebuf, *coutFilebuf, *cinFilebuf;

    static void rollForward( bool inKernel = false );
    static void processEvents( uEventList *events, bool inKernel = false );
    static __U_START_KT_TYPE_UNQUAL__;

    static void abortExit();
    static void startup();				// init boot KM
    static void *interposeSymbol( const char *symbolName, const char *version = NULL );
  public:
    static uSystemTask *systemTask;			// pointer to system task for global constructors/destructors
    static uCluster *userCluster;			// pointer to user cluster

    static bool afterMain;
    static bool inExit;
    static int retCode;

#if defined( __ia64__ ) && ! defined( __U_TLS__ )
    // Last machine word of this structure must be reserved for the "in atomic
    // offset load/store" flag (see uKernelThread.h).
    unsigned long IA64OffsetStoreFlag;
#endif // __ia64__
}; // uKernelModule


//######################### uSpinLock #########################


class uSpinLock {					// non-yielding spinlock
    friend class uKernelBoot;				// access: new
    friend class uEventList;				// access: acquireNoRF, releaseNoRF

    unsigned int value;

    uSpinLock( uSpinLock & );				// no copy
    uSpinLock &operator=( uSpinLock & );		// no assignment

    void *operator new( size_t, void *storage );
    void *operator new( size_t size );

    void acquireNoRF();
    void releaseNoRF();
  public:
    uSpinLock();
    void acquire();
    bool tryacquire();
    void release();
}; // uSpinLock


// RAII mutual-exclusion lock.  Useful for mutual exclusion in free routines.
// Handles exception termination and multiple block exit or return.

class uCSpinLock {
    uSpinLock &spinLock;

    uCSpinLock( uCSpinLock & );				// no copy
    uCSpinLock &operator=( uCSpinLock & );		// no assignment
  public:
    uCSpinLock( uSpinLock &spinLock );
    ~uCSpinLock();
}; // uCSpinLock


//######################### uLock #########################

class uLock {						// yielding spinlock
    unsigned int value;
    uSpinLock spinLock;

    uLock( uLock & );					// no copy
    uLock &operator=( uLock & );			// no assignment
  public:
    uLock();
    uLock( unsigned int val );
    void acquire();
    bool tryacquire();
    void release();
}; // uLock


//######################### Abnormal Event Handling #########################


#include <uEHM.h>


//######################### Real-Time #########################


#include <uCalendar.h>


class uSignalHandler : public uColable {
  protected:
    uBaseTask *This;
    virtual ~uSignalHandler() {}
  public:
    virtual void uHandler() = 0;
}; // uSignalHandler


#include <uAlarm.h>


class uCxtSwtchHndlr : public uSignalHandler {
    friend class uKernelBoot;				// access: constructor
    friend class uProcessor;				// access: constructor

    void uHandler();
}; // uCxtSwtchHndlr


//######################### uOwnerLock #########################


#if defined( __irix__ )
extern "C" int _usinitsema( usema_t *sem, int val );
#endif // __irix__

class uOwnerLock {
    friend class uCondLock;				// access: _add, _release
#if defined( __irix__ )
    friend int _usinitsema( usema_t *sem, int val );	// new
#endif // __irix__

    // These data fields must be initialized to zero. Therefore, this lock can
    // be used in the same storage area as a pthread_mutex_t, if
    // sizeof(pthread_mutex_t) >= sizeof(uOwnerLock).

    unsigned int count;					// number of recursive entries; no overflow checking
    // Solaris has a magic value in its pthread locks, so place the spin lock in
    // that position as it cannot take on the magic value (see library/pthread.cc).
    uSpinLock spin;
    uBaseTask *_owner;					// owner with respect to recursive entry
    uQueue<uBaseTaskDL> waiting;

    uOwnerLock( uOwnerLock & );				// no copy
    uOwnerLock &operator=( uOwnerLock & );		// no assignment

    void _add( uBaseTask &task );			// helper routines for uCondLock
    void _release();
  public:
    uOwnerLock();
#ifdef __U_DEBUG__
    ~uOwnerLock();
#endif // __U_DEBUG__
    unsigned int times() const;
    uBaseTask *owner() const;
    void acquire();
    bool tryacquire();
    void release();

    void *operator new( size_t, void *storage );
    void *operator new( size_t size );
}; // uOwnerLock


//######################### uCondLock #########################


class uCondLockTimedwaitHndlr;				// forward declaration

class uCondLock {
    // These data fields must be initialized to zero. Therefore, this lock can
    // be used in the same storage area as a pthread_cond_t, if
    // sizeof(pthread_cond_t) >= sizeof(uCondLock).

    friend class uCondLockTimedwaitHndlr;		// access: waitTimeout

    uSpinLock spin;
    uQueue<uBaseTaskDL> waiting;

    uCondLock( uCondLock & );				// no copy
    uCondLock &operator=( uCondLock & );		// no assignment
    void waitTimeout( uBaseTask &task, uCondLockTimedwaitHndlr &h ); // timeout
  public:
    uCondLock();
#ifdef __U_DEBUG__
    ~uCondLock();
#endif // __U_DEBUG__
    bool empty() const;
    void wait( uOwnerLock &lock );
    bool timedwait( uOwnerLock &lock, uDuration duration );
    bool timedwait( uOwnerLock &lock, uTime time );
    void signal();
    void broadcast();

    void *operator new( size_t, void *storage );
    void *operator new( size_t size );
}; // uCondLock


class uCondLockTimedwaitHndlr : public uSignalHandler {	// real-time
    friend class uCondLock;				// access: uWakeupHndlr
    
    uCondLock &condlock;
    bool timedout;

    uCondLockTimedwaitHndlr( uBaseTask &task, uCondLock &condlock );
    uCondLockTimedwaitHndlr( uCondLock &condlock );
    void uHandler();
}; // uCondLockTimedwaitHndlr


//######################### uContext #########################


class uContext : public uSeqable {
    void *key;
  public:
    uContext();
    uContext( void *key );
    virtual ~uContext();

    // These two routines cannot be abstract (i.e., = 0) because there is a
    // race condition during initialization of a derived class when the base
    // class constructor is invoked. A context switch can occur immediately
    // after the base instance has put itself on the task context list but
    // before the virtual function vector is updated for the derived class.
    // Hence, the save or restore routine of the base class may be called.
    // This situtation is not a problem because either the task has not
    // performed any operations that involve the new context or the task is
    // removing the context and not performing anymore operations using it.

    virtual void save();
    virtual void restore();
}; // uContext

typedef uSequence<uContext> uContextSeq;


//######################### uFloatingPointContext #########################


// Provide floating point context switch support for coroutines and tasks.  It
// is built with the more general context class in the same way that a user can
// extend the amount of context that is saved and restored with a coroutine or
// task.

#if defined( __i386__ )
// saved by caller
#elif defined( __x86_64__ )
// saved by caller
#elif defined( __ia64__ )
#if ! defined( __U_SWAPCONTEXT__ )
#define __U_FLOATINGPOINTDATASIZE__ 40			// 20 16-byte registers
#endif // ! __U_SWAPCONTEXT__
#elif defined( __sparc__ )
// saved by caller
#elif defined( __mips__ )
#ifdef _ABIN32
#define __U_FLOATINGPOINTDATASIZE__ 6			// doubles
#elif _ABI64
#define __U_FLOATINGPOINTDATASIZE__ 8			// doubles
#else
    #error uC++ : internal error, unsupported architecture
#endif
#else
    #error uC++ : internal error, unsupported architecture
#endif


// Some architectures store the floating point registers with the integers
// registers during a basic context switch, so there is no need for a data area
// to storage the floating point registers.

#ifdef __U_FLOATINGPOINTDATASIZE__
class uFloatingPointContext : public uContext {
    static int uniqueKey;
    double floatingPointData[__U_FLOATINGPOINTDATASIZE__] __attribute__(( aligned(16) ));
  public:
    uFloatingPointContext();
#else
class uFloatingPointContext {
  public:
#endif // __U_FLOATINGPOINTDATASIZE__
    void save();					// save and restore the floating point context
    void restore();
} __attribute__(( unused )); // uFloatingPointContext


//######################### uMachContext #########################


#if defined( __U_SWAPCONTEXT__ )
#define uSwitch( from, to ) swapcontext( (ucontext_t *)(from), (ucontext_t *)(to) )
#define __U_CONTEXT_T__ ucontext_t
#else
#define __U_CONTEXT_T__ uContext_t
extern "C" void uSwitch( void *from, void *to ) asm ("uSwitch"); // assembler routine that performs the context switch
#endif


// Define a class that contains the machine dependent context and routines that
// initialize and switch between contexts.

class uMachContext {
    friend class uContext;				// access: extras, additionalContexts
    friend class uProcessorTask;			// access: size, base, limit
    friend class uBaseCoroutine;			// access: storage
    friend class uCoroutineConstructor;			// access: startHere
    friend class uTaskConstructor;			// access: startHere
    friend _Coroutine uProcessorKernel;			// access: storage
    friend class uProcessor;				// access: storage
    friend class uKernelBoot;				// access: storage
#ifdef __U_ERRNO_FUNC__
    friend int *__U_ERRNO_FUNC__ __THROW;		// access: uErrno
#endif // __U_ERRNO_FUNC__
    friend __U_START_KT_TYPE__;				// acesss: invokeCoroutine

    struct uContext_t {
	void *SP;
#if ! defined( __ia64__ )
	void *FP;
	void *PC;
#else
	void *BSP;					// register backing store pointer
	sigset_t sigMask;
#endif
    };

    unsigned int size;					// size of stack
    void *storage;					// pointer to stack
    void *base;						// base of stack
    void *limit;					// stack grows towards stack limit
    int uErrno;						// copy of global UNIX variable errno

    union {
	long int allExtras;				// allow access to all extra flags
	struct {					// put all extra flags in this structure
	    unsigned int usercxts : 1;			// user defined contexts
	} is;
    } extras;						// indicates extra work during the context switch

    void createContext( unsigned int stacksize );	// used by all constructors

    // This routine returns the magic word that is placed at both ends of an
    // allocated stack.  Later, those locations can be checked to see if the
    // magic word is still intact.

    unsigned int magic();

    void startHere( void (*uInvoke)( uMachContext & ) );

    uMachContext( uMachContext & );			// no copy
    uMachContext &operator=( uMachContext & );		// no assignment
  protected:
    static void invokeCoroutine( uBaseCoroutine &This ) __attribute__(( noreturn ));
    static void invokeTask( uBaseTask &This ) __attribute__(( noreturn ));
    static void cleanup( uBaseTask &This ) __attribute__(( noreturn ));

    uContextSeq additionalContexts;			// list of additional contexts for this execution state

    void extraSave();
    void extraRestore();

    void save() {
#if ! defined( __U_ERRNO_FUNC__ )
	uErrno = errno;
#endif // ! __U_ERRNO_FUNC__
	// Any extra work that must occur on this side of a context switch is
	// performed here.
	if ( extras.allExtras ) {
	    extraSave();
	} // if

#ifdef __U_DEBUG__
	verify();
#endif // __U_DEBUG__
    } // uMachContext::save

    void restore() {
#ifdef __U_DEBUG__
	verify();
#endif // __U_DEBUG__

	// Any extra work that must occur on this side of a context switch is
	// performed here.
	if ( extras.allExtras ) {
	    extraRestore();
	} // if
#if ! defined( __U_ERRNO_FUNC__ )
	errno = uErrno;
#endif // ! __U_ERRNO_FUNC__
    } // uMachContext::restore

    virtual void main() = 0;				// starting routine for coroutine or task
  public:
    uMachContext( unsigned int stacksize );
    virtual ~uMachContext();

    void *stackPointer() const;
#if defined( __ia64__ )
    void *registerStackPointer() const;
#endif // __ia64__
    unsigned int stackSize() const;
    ptrdiff_t stackFree() const;
    ptrdiff_t stackUsed() const;
    void verify();

    // These members should be private but cannot be because they are
    // referenced from user code.

    static void *rtnAdr( void (*rtn)() );		// access: see profiler
#if defined( __ia64__ )
    static void *gpAdr( void (*rtn)() );
#endif // __ia64__
}; // uMachContext


//######################### uBaseCoroutine #########################


extern "C" void pthread_exit( void *status );

class uBaseCoroutine : public uMachContext {
    friend class uBaseTask;				// access: serial, profileTaskSamplerInstance
    friend class uTaskDestructor;			// access: profileTaskSamplerInstance
    friend class uKernelBoot;				// access: last
    friend class uMachContext;				// access: notHalted, main, suspend, setState, corStarter, corFinish
    friend _Task uBootTask;				// access: notHalted
    friend class uCoroutineConstructor;			// access: name, serial
    friend class uCoroutineDestructor;			// access: UnwindStack
    friend class uTaskConstructor;			// access: name, serial
    friend _Coroutine uProcessorKernel;			// access: contextSw

    // cancellation

    friend _Task Pthread;				// access: Cleanup, PthreadCleanup, UnwindStack
    friend _Task uMain;					// access: Cleanup

    // exception handling

    friend class uEHM;					// access: resumedObj, topResumedType, DEStack, handlerStackTop, handlerStackVisualTop, unexpectedRtn
    friend class uEHM::ResumeWorkHorseInit;		// access: resumedObj, topResumedType, handlerStackVisualTop
    friend class uEHM::uResumptionHandlers;		// access: handlerStackTop, handlerStackVisualTop
    friend class uEHM::uDeliverEStack;			// access: DEStack
    friend std::unexpected_handler std::set_unexpected( std::unexpected_handler func ) throw(); // access: unexpectedRtn
    friend void uEHM::unexpected();			// access: unexpected

    // profiler

    friend class uProfilerBoot;                         // access: serial
    friend class uProfileTaskSampler;			// access: profileTaskSamplerInstance
  public:
    enum State { Halt, Active, Inactive };
    enum CancellationState { CancelEnabled = PTHREAD_CANCEL_ENABLE, CancelDisabled = PTHREAD_CANCEL_DISABLE };
    enum CancellationType { CancelPoll = PTHREAD_CANCEL_DEFERRED, CancelImplicit = PTHREAD_CANCEL_ASYNCHRONOUS };
  private:
    const char *name;					// textual name for coroutine/task, initialized by uC++ generated code
    bool notHalted;					// indicate if execuation state is not halted
    State state;					// current execution status for coroutine
    uBaseCoroutine *_starter;				// first coroutine to resume this one
    uSerial *serial;					// original serial instance for cormonitor/task (versus currently used instance)

    uBaseCoroutine *last;				// last coroutine to resume this one
    uBaseTask *currSerialOwner;				// task accessing monitors from this coroutine
    unsigned int currSerialCount;			// counter to determine when to unset uCurrSerialOwner

    // cancellation

    _ThrowEvent UnwindStack;
    bool _cancelled;					// cancellation flag
    bool _cancelInProgress;				// cancellation in progress flag
    CancellationState cancelState;			// enabled/disabled
    CancellationType cancelType;			// deferred/asynchronous

#if defined( __irix__ )    
    typedef __pthread_cncl_hdlr *Cleanup;
#else    
    struct PthreadCleanup : uColable {
	void (* routine )(void *);
	void * args;

	void *operator new( size_t, void *storage ) {
	    return storage;
	} // Pthread_cleanup::operator new
    }; // PthreadCleanup

    typedef uStack<PthreadCleanup> Cleanup;
#endif
    void unwindStack();    

    // exception handling

    uEHM::uResumptionHandlers *handlerStackTop, *handlerStackVisualTop;
    uEHM::uDualClass *resumedObj;			// the object that is currently being handled during resumption
    const std::type_info *topResumedType;		// the top of the currently handled resumption stack (unchanged during stack unwind through EH)
    uEHM::uDeliverEStack *DEStack;			// manage exception enable/disable
    std::unexpected_handler unexpectedRtn;		// per coroutine handling unexpected action
    bool unexpected;					// indicate if unexpected error occurs

    // profiling

    mutable uProfileTaskSampler *profileTaskSamplerInstance; // pointer to related profiling object

    void createCoroutine();

    void setState( State s ) {
	state = s;
    } // uBaseCoroutine::setState

    void contextSw();					// switch between a task and the kernel
    void contextSw2();					// switch between two coroutine contexts

    void corStarter() {					// remembers who started a coroutine
	_starter = last;
    } // uBaseCoroutine::corStarter
    void corFinish() __attribute__(( noreturn ));

    uBaseCoroutine( uBaseCoroutine & );			// no copy
    uBaseCoroutine &operator=( uBaseCoroutine & );	// no assignment
  protected:
    // Duplicate "main" (see uMachContext) to get better error message about
    // missing "main" member for coroutine.
    virtual void main() = 0;				// starting routine for coroutine
    // Only allow direct access, i.e., preclude indirect access C.resume()/C.suspend()
    void resume();
    void suspend();
  public:
    uBaseCoroutine();
    uBaseCoroutine( unsigned int stacksize );

    const char *setName( const char *name );
    const char *getName() const;

    State getState() const {
	return notHalted ? state : Halt;
    } // uBaseCoroutine::getState

    uBaseCoroutine &starter() const {			// starter coroutine => did first resume
	return *_starter;
    } // uBaseCoroutine::starter

    uBaseCoroutine &resumer() const {			// last resumer coroutine
	return *last;
    } // uBaseCoroutine::resumer

    // cancellation

    void cancel() { _cancelled = true; }
    bool cancelled() { return _cancelled; }
    bool cancelInProgress() { return _cancelInProgress; }

    // exception handling

    _DualEvent Failure;					// general failure
    _DualEvent UnHandledException;

    // These members should be private but cannot be because they are
    // referenced from user code.

    void setCancelState( CancellationState state );
    CancellationState getCancelState() { return cancelState; }
    void setCancelType( CancellationType type );
    CancellationType getCancelType() { return cancelType; }

    template<uBaseCoroutine::CancellationState newState> class Cancel { // RAII helper to enable/disable cancellation
	uBaseCoroutine::CancellationState prev;
      public:
	Cancel() {
	    uBaseCoroutine &coroutine = uThisCoroutine();
	    prev = coroutine.getCancelState();
	    coroutine.setCancelState( newState );
	} // Cancel::Cancel

	~Cancel() {
	    uThisCoroutine().setCancelState( prev );
	} // Cancel::~Cancel
    }; // Cancel

    uEHM::AsyncEMsgBuffer asyncEBuf;			// list of pending nonlocal exceptions
}; // uBaseCoroutine


//######################### Abnormal Event Handling/Cancellation #########################


// Include the kernel generated exception definitions.

#include <uKernelEvent.h>


class uEnableCancel : public uBaseCoroutine::Cancel<uBaseCoroutine::CancelEnabled> {
  public:
    uEnableCancel() {
	uEHM::poll();
    } // uEnableCancel::uEnableCancel
}; // uEnableCancel

class uDisableCancel : public uBaseCoroutine::Cancel<uBaseCoroutine::CancelDisabled> {
}; // uDisableCancel


//######################### uBaseCoroutine (cont) #########################


_DualEvent uBaseCoroutine::Failure : public uKernelFailure {
  public:
    Failure( const char *const msg = "" );
    virtual ~Failure();
    virtual void defaultTerminate() const;
}; // uBaseCoroutine::Failure


_DualEvent uBaseCoroutine::UnHandledException : public uBaseCoroutine::Failure {
    friend void uMachContext::invokeCoroutine( uBaseCoroutine &This ); // access: multiple

    bool multiple;					// multiple exceptions ?
    uBaseCoroutine &origFailedCor;
    char origFailedCorName[uEHMMaxName + sizeof("...")];
  public:
    UnHandledException( const char *const msg = "" );
    virtual ~UnHandledException();
    const uBaseCoroutine &origSource() const;
    const char *origName() const;
    virtual void defaultTerminate() const;
}; // uBaseCoroutine::UnHandledException


_ThrowEvent uBaseCoroutine::UnwindStack {
    friend class uMachContext;				// access: exec_dtor
    friend _Task Pthread;				// access: exec_dtor

    bool exec_dtor;
  public:
    UnwindStack( bool = false );
    ~UnwindStack();			
}; // uBaseCoroutine::UnwindStack


//######################### Real-Time (cont) #########################


class uBaseScheduleFriend {
  protected:
    virtual ~uBaseScheduleFriend() {}
    uBaseTask &getInheritTask( uBaseTask &task ) const;
    int getActivePriority( uBaseTask &task ) const;
    int getActivePriorityValue( uBaseTask &task ) const;
    int setActivePriority( uBaseTask &task1, int priority );
    int setActivePriority( uBaseTask &task1, uBaseTask &task2 );
    int getBasePriority( uBaseTask &task ) const;
    int setBasePriority( uBaseTask &task, int priority );
    int getActiveQueueValue( uBaseTask &task ) const;
    int setActiveQueue( uBaseTask &task1, int priority );
    int getBaseQueue( uBaseTask &task ) const;
    int setBaseQueue( uBaseTask &task, int priority );
    bool isEntryBlocked( uBaseTask &task ) const;
    bool checkHookConditions( uBaseTask &task1, uBaseTask &task2 ) const;
}; // uBaseScheduleFriend


template<class Node> class uBaseSchedule : protected uBaseScheduleFriend {
  public:
    virtual bool empty() const = 0;
    virtual void add( Node *node ) = 0;
    virtual Node *drop() = 0;
    virtual bool checkPriority( Node &owner, Node &calling ) = 0;
    virtual void resetPriority( Node &owner, Node &calling ) = 0;
    virtual void addInitialize( uBaseTaskSeq &taskList ) = 0;
    virtual void removeInitialize( uBaseTaskSeq &taskList ) = 0;
    virtual void rescheduleTask( uBaseTaskDL *taskNode, uBaseTaskSeq &taskList ) = 0;
}; // uBaseSchedule


class uBasePrioritySeq : public uBaseScheduleFriend {
    friend class uSerial;
  protected:
    uBaseTaskSeq list;
    bool executeHooks;
  public:
    uBasePrioritySeq();
    virtual bool empty() const;
    virtual uBaseTaskDL *head() const;
    virtual int add( uBaseTaskDL *node, uBaseTask *owner );
    virtual uBaseTaskDL *drop();
    virtual void remove( uBaseTaskDL *node );
    virtual void onAcquire( uBaseTask &owner );
    virtual void onRelease( uBaseTask &oldOwner );
    int reposition( uBaseTask &task, uSerial &sserial );
}; // uBasePrioritySeq


class uBasePriorityQueue : public uBasePrioritySeq {
    uQueue<uBaseTaskDL> list;
  public:
    virtual bool empty() const;
    virtual uBaseTaskDL *head() const;
    virtual int add( uBaseTaskDL *node, uBaseTask *owner ); 
    virtual uBaseTaskDL *drop();
    virtual void remove( uBaseTaskDL *node ); 
    virtual void onAcquire( uBaseTask &owner );
    virtual void onRelease( uBaseTask &oldOwner );
}; // uBasePriorityQueue


class uRepositionEntry {
    uBaseTask &blocked, &calling;
    uSerial &bSerial, &cSerial;
  public:
    uRepositionEntry( uBaseTask &blocked, uBaseTask &calling );
    int uReposition( bool relCallingLock );
}; // uRepositionEntry


class uDefaultScheduler : public uBaseSchedule<uBaseTaskDL> {
    uBaseTaskSeq list;					// list of tasks awaiting execution
  public:
    bool empty() const;
    void add( uBaseTaskDL *taskNode );
    uBaseTaskDL *drop();
    bool checkPriority( uBaseTaskDL &owner, uBaseTaskDL &calling );
    void resetPriority( uBaseTaskDL &owner, uBaseTaskDL &calling );
    void addInitialize( uBaseTaskSeq &taskList );
    void removeInitialize( uBaseTaskSeq &taskList );
    void rescheduleTask( uBaseTaskDL *taskNode, uBaseTaskSeq &taskList );
}; // uDefaultScheduler


//######################### Translator Generated Definitions #########################


class uBasePIQ {
  protected:
    virtual ~uBasePIQ() {}
  public:
    virtual int getHighestPriority() = 0;
}; // uBasePIQ


#include <uPIHeap.h>


enum uAction { uNo, uYes };				// used to indicate the most derived constructor/destructor of an object


class uCoroutineConstructor {				// placed in the constructor of a coroutine
  public:
    uCoroutineConstructor( uAction f, uSerial &serial, uBaseCoroutine &coroutine, const char *name );
} __attribute__(( unused )); // uCoroutineConstructor


class uCoroutineDestructor {				// placed in the destructor of a coroutine
    uAction f;
    uBaseCoroutine &coroutine;
  public:
    uCoroutineDestructor( uAction f, uBaseCoroutine &coroutine );
    ~uCoroutineDestructor();
}; // uCoroutineDestructor


class uCoroutineMain {					// placed in the main member of a coroutine
  private:
    uBaseCoroutine &coroutine;
  public:
    uCoroutineMain( uBaseCoroutine &coroutine );
    ~uCoroutineMain();					// needed by the debugger
}; // uCoroutineMain


class uSerial {
    friend class uCondition;				// access: acceptSignalled, leave2
    friend class uSerialConstructor;			// access: mr, prevSerial, leave
    friend class uSerialDestructor;			// access: prevSerial, acceptSignalled, lastAcceptor, mutexOwner, leave, ~uSerialDestructor, enterDestructor
    friend class uTaskConstructor;			// access: acceptSignalled
    friend class uSerialMember;				// access: lastAcceptor, alive, enter, leave
    friend _Task uBootTask;				// access: acceptSignalled
    friend class uMachContext;				// access: leave2
    friend class uBasePrioritySeq;			// access: mutexOwnerlock TEMPORARY
    friend class uRepositionEntry;			// access: lock, entryList TEMPORARY
    friend class uBaseScheduleFriend;			// access: checkHookConditions
    friend class uTimeoutHndlr;				// access: enterTimeout

    // profiling

    friend class uProfileTaskSampler;			// access: profileTaskSamplerInstance

    uSpinLock lock;					// provide mutual exclusion while examining serial state
    uBaseTask *mutexOwner;				// active thread in the mutex object
    uBitSet< __U_MAXENTRYBITS__ > mask;			// entry mask of accepted mutex members and timeout
    unsigned int *mutexMaskPosn;			// location to place mask position in accept statement
    uBasePrioritySeq &entryList;			// tasks waiting to enter mutex object
#if 1							// normally use stack; use queue for experiments
    uStack<uBaseTaskDL> acceptSignalled;		// tasks suspended within the mutex object
#else
    uQueue<uBaseTaskDL> acceptSignalled;		// tasks suspended within the mutex object
#endif
    uBaseTask *constructorTask;				// identity of task creating mutex object
    uBaseTask *destructorTask;				// identity of task calling mutex object's destructor
    uSerial *prevSerial;				// task's previous serial (see uSerialMember, recursive entry during constructor)
    unsigned int mr;					// mutex recursion counter for multiple serial-object entry
    enum uDestructorState { NoDestructor, DestrCalled, DestrScheduled }; // identify the state of the destructor
    uDestructorState destructorStatus;			// has the destructor been called ? 
    bool alive;						// serial destroyed ?
    bool acceptMask;					// entry mask set by uAcceptReturn or uAcceptWait
    bool acceptLocked;					// flag indicating if mutex lock has been acquired for the accept statement

    // real-time

    uEventNode uTimeoutEvent;				// event node for event list
    uEventList *events;					// event list when event added

    // exception handling

    uBaseTask *lastAcceptor;				// acceptor of current entry for communication between acceptor and caller

    // profiling

    mutable uProfileTaskSampler *profileSerialSamplerInstance; // pointer to related profiling object

    void resetDestructorStatus();			// allow destructor to be called
    void enter( unsigned int &mr, uBasePrioritySeq &ml, int mp );
    void enterDestructor( unsigned int &mr, uBasePrioritySeq &ml, int mp );
    void enterTimeout();
    void leave( unsigned int mr );
    void leave2();
    void removeTimeout();
    bool checkHookConditions( uBaseTask *task );	// check conditions for executing hooks

    uSerial( uSerial & );				// no copy
    uSerial &operator=( uSerial & );			// no assignment
  public:
    // These members should be private but cannot be because they are
    // referenced from user code.

    uSerial( uBasePrioritySeq &entryList );
    ~uSerial();

    // calls generated by translator in application code
    void acceptStart( unsigned int &mutexMaskPosn );
    bool acceptTry( uBasePrioritySeq &ml, int mp );
    void acceptTry();
    bool acceptTry2( uBasePrioritySeq &ml, int mp );
    bool acceptTestMask();
    void acceptSetMask();
    void acceptElse();
    void acceptPause();
    void acceptPause( uDuration duration );
    void acceptPause( uTime time );
    void acceptEnd();

    class uProtectAcceptStmt {
	uSerial &serial;
      public:
	unsigned int mutexMaskPosn;			// bit position (0-127) in the entry mask for active mutex member

	uProtectAcceptStmt( uSerial &serial ) : serial( serial ) {
	    serial.acceptStart( mutexMaskPosn );
	} // uSerial::uProtectAcceptStmt::uProtectAcceptStmt

	uProtectAcceptStmt( uSerial &serial, bool ) : serial( serial ) {
	    serial.removeTimeout();
	    serial.acceptStart( mutexMaskPosn );
	} // uSerial::uProtectAcceptStmt::uProtectAcceptStmt

	~uProtectAcceptStmt() {
	    serial.acceptEnd();
	} // uSerial::uProtectAcceptStmt::~uProtectAcceptStmt
    }; // uSerial::uProtectAcceptStmt

    // exception handling

    _DualEvent Failure;					// raised when entry encounters an exceptional event
    _DualEvent EntryFailure;				// derived from Failure
    _DualEvent RendezvousFailure;			// derived from Failure as accepted statement may be hidden inside any mutex entry
}; // uSerial


class uTimeoutHndlr : public uSignalHandler {		// real-time
    friend class uSerial;				// access: uWakeupHndlr
    
    uSerial &serial;

    uTimeoutHndlr( uBaseTask &task, uSerial &serial );
    uTimeoutHndlr( uSerial &serial );
    void uHandler();
}; // uTimeoutHndlr


_DualEvent uSerial::Failure : public uKernelFailure {	// general event for failure in a mutex member routine
    // The field serial is to identify the mutex object involved, which can be
    // a uMutexCoroutine or uTask.

    const uSerial *const serial;
  public:
    Failure( const uSerial *const serial, const char *const msg = "" );
    Failure( const char *const msg = "" );
    virtual ~Failure();
    const uSerial *serialId() const;
    virtual void defaultTerminate() const;
}; // uSerial::Failure


_DualEvent uSerial::EntryFailure : public uSerial::Failure {
    // this is raised at an execution when it is in either the entry queue or
    // the suspended stack of a mutex object
  public:
    EntryFailure( const uSerial *const serial, const char *const msg = "" );
    EntryFailure( const char *const msg = "" );
    virtual ~EntryFailure();
    virtual void defaultTerminate() const;
}; // uSerial::EntryFailure


_DualEvent uSerial::RendezvousFailure : public uSerial::Failure {
    // event when failing during a rendezvous

    // This exception should be raised at the acceptor concurrently. Rendezvous
    // failure is a entry failure because an accepted statement can be hidden
    // inside a mutex object entry.

    const uBaseCoroutine *const _caller;
  public:
    RendezvousFailure( const uSerial *const serial, const char *const msg = "" );
    virtual ~RendezvousFailure();
    const uBaseCoroutine *caller() const;
    virtual void defaultTerminate() const;
}; // uSerial::RendezvousFailure


class uTaskConstructor {				// placed in the constructor of a task
    uAction f;
    uSerial &serial;
    uBaseTask &task;
  public:
    uTaskConstructor( uAction f, uSerial &serial, uBaseTask &task, uBasePIQ &piq, const char *n, bool profile );
    ~uTaskConstructor();
} __attribute__(( unused )); // uTaskConstructor


class uTaskDestructor {					// placed in the destructor of a task
    friend class uTaskConstructor;			// access: cleanup

    uAction f;
    uBaseTask &task;

    static void cleanup( uBaseTask &task );
  public:
    uTaskDestructor( uAction f, uBaseTask &task ) : f( f ), task( task ) {
    } // uTaskDestructor::uTaskDestructor

    ~uTaskDestructor();
}; // uTaskDestructor


class uTaskMain {					// placed in the main member of a task
    uBaseTask &task;
  public:
    uTaskMain( uBaseTask &task );
    ~uTaskMain();
}; // uTaskMain


class uSerialConstructor {				// placed in the constructor of a mutex class
    uAction f;
    uSerial &serial;
  public:
    uSerialConstructor( uAction f, uSerial &serial );
    uSerialConstructor( uAction f, uSerial &serial, const char *n );
    ~uSerialConstructor();
}; // uSerialConstructor


class uSerialDestructor {				// placed in the destructor of a mutex class
    uAction f;
    uSerial *prevSerial;				// task's previous serial
    unsigned int mr;					// mutex recursion counter for multiple serial-object entry
    unsigned int nlevel;				// nesting level counter for accessed serial-objects
  public:
    uSerialDestructor( uAction f, uSerial &serial, uBasePrioritySeq &ml, int mp );
    ~uSerialDestructor();
}; // uSerialDestructor


class uSerialMember {					// placed in the mutex member of a mutex class
    uSerial *prevSerial;				// task's previous serial
    unsigned int mr;					// mutex recursion counter for multiple serial-object entry
    unsigned int nlevel;				// nesting level counter for accessed serial-objects

    // exception handling

    friend class uSerial;				// access: caller, acceptor
    friend class uCondition;				// access: caller

    uBaseTask *acceptor;				// acceptor of the entry invocation, NULL => no acceptor
    bool acceptorSuspended;				// true only when the acceptor, if there is one, remains blocked
    bool noUserOverride;				// true when acceptor has not been called

    void finalize( uBaseTask &task );
  public:
    uSerialMember( uSerial &serial, uBasePrioritySeq &ml, int mp );
    ~uSerialMember();
}; // uSerialMember


//######################### uBaseTask (cont) #########################


class uBaseTask : public uBaseCoroutine {
    friend class uMachContext;				// access: currCoroutine, profileActive, setState, main
    friend class uBaseCoroutine;			// access: currCoroutine, profileActive, setState
    friend uBaseCoroutine &uThisCoroutine();		// access: currCoroutine
    friend class uOwnerLock;				// access: entryRef, profileActive, wake
    friend class uCondLock;				// access: entryRef, ownerLock, profileActive, wake
    friend class uSpinLock;                             // access: profileActive
    friend class uSemaphore;				// access: entryRef, wake
    friend class uSerial;				// access: everything
    friend class uSerialConstructor;			// access: profileActive, setSerial
    friend class uSerialDestructor;			// access: mutexRef, profileActive, mutexRecursion, setState
    friend class uCondition;				// access: currCoroutine, mutexRef, info, profileActive
    friend class uCoroutineConstructor;			// access: profileActive
    friend class uCoroutineDestructor;			// access: profileActive
    friend class uTaskConstructor;			// access: currCluster, mutexRef, profileActive, setSerial
    friend class uTaskDestructor;			// access: currCluster, profileActive
    friend class uTaskMain;				// access: recursion, profileActive 
    friend _Coroutine uProcessorKernel;			// access: currCoroutine, setState, wake
    friend _Task uProcessorTask;			// access: currCluster, uBaseTask
    friend class uCluster;				// access: currCluster, readyRef, clusterRef, bound
    friend _Task uBootTask;				// access: wake
    friend class uCoroutineMain;			// access: profileActive
    friend class uHeapManager;				// access: currCoroutine, inheritTask, profileActive
    friend void *malloc( size_t size ) __THROW;		// access: profileActive
    friend void *memalign( size_t alignment, size_t size ) __THROW; // access: profileActive
    friend class uEventList;				// access: profileActive

    // exception handling

    friend void uEHM::terminateHandler();		// access: terminateRtn
    friend std::terminate_handler std::set_terminate( std::terminate_handler func ) throw(); // access: terminateRtn

    // debugging

    friend class uLocalDebuggerHandler;			// access: taskDebugMask, processBP
    friend _Task uLocalDebugger;			// access: bound, taskDebugMask, debugPCandSRR
    friend class uSigHandlerModule;			// access: debugPCandSRR

    // profiling

    friend _Task uProfiler;				// access: currCoroutine
    friend void __cyg_profile_func_enter( void *pcCurrentFunction, void *pcCallingFunction );
    friend void __cyg_profile_func_exit( void *pcCurrentFunction, void *pcCallingFunction );
    friend class uExecutionMonitor;			// access: profileActive
    friend _Task uMain;					// access: profileActive
  public:
    enum State { Start, Ready, Running, Blocked, Terminate };
  private:
    void createTask( uCluster &cluster );
    uBaseTask( uCluster &cluster, uProcessor &processor ); // only used by uProcessorTask
    void setState( State state );
    void wake();

    uBaseTask( uBaseTask & );				// no copy
    uBaseTask &operator=( uBaseTask & );		// no assignment

    // debugging

    char taskDebugMask[8];				// 64 bit breakpoint mask for task (used only by debugger)
    void *debugPCandSRR;				// PC of break point; address of return message for IPC SRR
    bool processBP;					// true if task is in the middle of processing a breakpoint

    State state;					// current state of task
    unsigned int recursion;				// allow recursive entry of main member
    unsigned int mutexRecursion;			// number of recursive calls while holding mutex
    uCluster *currCluster;				// cluster task is executing on
    uBaseCoroutine *currCoroutine;			// coroutine being executed by tasks thread
    long int info;					// condition information stored with blocked task

    uBaseTaskDL clusterRef;				// double link field: list of tasks on cluster
    uBaseTaskDL readyRef;				// double link field: ready queue
    uBaseTaskDL entryRef;				// double link field: general entry deque (all waiting tasks)
    uBaseTaskDL mutexRef;				// double link field: mutex member, suspend stack, condition variable
    uProcessor &bound;					// processor to which this task is bound, if applicable
    uBasePrioritySeq *calledEntryMem;			// pointer to called mutex queue
    uOwnerLock *ownerLock;				// pointer to owner lock used for signalling conditions

    // profiling

    bool profileActive;					// indicates if this context is supposed to be profiled
    void profileActivate( uBaseTask &task );

    // exception handling

    uSerialMember *acceptedCall;                       // pointer to the last mutex entry accepted by this thread
    std::terminate_handler terminateRtn;		// per task handling termination action
  protected:
    // real-time

    friend class uSerialMember;				// access: setSerial, currCoroutine, profileActive, acceptedCall
    friend class uBaseScheduleFriend;			// access: entryRef, getInheritTask, setActivePriority, setBasePriority, setActiveQueue, setBaseQueue, uIsEntryBlocked
    friend class uWakeupHndlr;                          // access: wake
    friend class uBasePrioritySeq;			// access: entryRef, mutexRef, calledEntryMem
    friend class uRepositionEntry;			// access: entryList TEMPORARY

    // Duplicate "main" (see uMachContext) to get better error message about
    // missing "main" member for task.
    virtual void main() = 0;				// starting routine for task

    int priority;
    int activePriority;
    uBaseTask *inheritTask;
    int queueIndex;
    int activeQueueIndex;
    uSerial *currSerial;				// current serial task is using (not original serial)

    unsigned int currSerialLevel;			// counter for checking non-nested entry/exit from multiple accessed mutex objects

    uBaseTask &getInheritTask() {
	return *inheritTask;
    } // uBaseTask::getInheritTask

    int setActivePriority( int priority ) {
	int temp = activePriority;
	activePriority = priority;
	return temp;
    } // uBaseTask::setActivePriority

    int setActivePriority( uBaseTask &task ) {
	int temp = activePriority;
	inheritTask = &task;
	activePriority = inheritTask->getActivePriority();
	return temp;
    } // uBaseTask::setActivePriority

    int setBasePriority( int priority ) {
	int temp = priority;
	uBaseTask::priority = priority;
	return temp;
    } // uBaseTask::setBasePriority

    int setActiveQueue( int q ) {
	int temp = activeQueueIndex;
	// uInheritTask = &t;  is this needed or should this just be called from setActivePriority ??
	activeQueueIndex = q;
	return temp;
    } // uBaseTask::setActiveQueue

    int setBaseQueue( int q ) {
	int temp = queueIndex;
	queueIndex = q;
	return temp;
    } // uBaseTask::setBaseQueue

    uSerial &setSerial( uSerial &serial ) {
	uSerial *temp = currSerial;
	currSerial = &serial;
	return *temp;
    } // uBaseTask::setSerial
  public:
    uBaseTask() : clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( uThisCluster() );
    } // uBaseTask::uBaseTask

    uBaseTask( unsigned int stacksize ) : uBaseCoroutine ( stacksize ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( uThisCluster() );
    } // uBaseTask::uBaseTask

    uBaseTask( uCluster &cluster );

    uBaseTask( uCluster &cluster, unsigned int stacksize ) : uBaseCoroutine( stacksize ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( cluster );
    } // uBaseTask::uBaseTask

    ~uBaseTask() {
    } // uBaseTask::~uBaseTask

    void yield() {
	uYieldNoPoll();
	uEHM::poll();
    } // uBaseTask::yield
    void yield( unsigned int times );

    uCluster &migrate( uCluster &cluster );

    uCluster &getCluster() const {
	return *currCluster;
    } // uBaseTask::getCluster

    uBaseCoroutine &getCoroutine() const {
	return *currCoroutine;
    } // uBaseTask::getCoroutine

    State getState() const {
	return state;
    } // uBaseTask::getState

    int getActivePriority() const {
	// special case for base of active priority stack
	return this == inheritTask ? priority : activePriority;
    } // uBaseTask::getActivePriority

    int getActivePriorityValue() const {		// TEMPORARY: replace previous member?
	 return activePriority;
    } // uBaseTask::getActivePriorityValue

    int getBasePriority() const {
	return priority;
    } // uBaseTask::getBasePriority

    int getActiveQueueValue() const {			// TEMPORARY: rename
	return activeQueueIndex;
    } // uBaseTask::getActiveQueueValue

    int getBaseQueue() const {
	return queueIndex;
    } // uBaseTask::getBaseQueue

    uSerial &getSerial() const {
	return *currSerial;
    } // uBaseTask::getSerial

    // profiling

    void profileActivate();
    void profileInactivate();
    void printCallStack() const;

    // These members should be private but cannot be because they are
    // referenced from user code.

    uBasePIQ *uPIQ;					// TEMPORARY
    void *pthreadData;					// pointer to pthread specific data

    void uYieldNoPoll();
    void uYieldYield( unsigned int times );		// inserted by translator for -yield
    void uYieldInvoluntary();                           // pre-allocates metric memory before yielding

    void uSleep( uDuration duration );
    void uSleep( uTime time );
}; // uBaseTask


class uWakeupHndlr : public uSignalHandler {		// real-time
    friend class uBaseTask;				// access: uWakeupHndlr

    uWakeupHndlr( uBaseTask &task ) {
	This = &task;
    } // uWakeupHndlr::uWakeupHndlr

    void uHandler() {
	This->wake();
    } // uWakeupHndlr::uHandler
}; // uWakeupHndlr


inline uBaseTask &uThisTask() {
    return *THREAD_GETMEM( activeTask );
} // uThisTask


inline uBaseCoroutine &uThisCoroutine() {
    return *uThisTask().currCoroutine;
} // uThisCoroutine


#if defined( __U_ERRNO_FUNC__ )
// ___errno on solaris, __errno_location on linux, __oserror on irix, __error on freebsd
extern "C" inline int *__U_ERRNO_FUNC__ __THROW {
    return &uThisCoroutine().uErrno;
} // __U_ERRNO_FUNC__
#endif // __U_ERRNO_FUNC__


//######################### uCondition #########################


class uCondition {
    uQueue<uBaseTaskDL> condQueue;			// queue of blocked tasks
    uSerial *owner;					// mutex object owning condition, only set in wait

    uCondition( uCondition & );				// no copy
    uCondition &operator=( uCondition & );		// no assignment
  public:
    uCondition();
    ~uCondition();

    void wait();					// wait on condition
    void wait( long int info );				// wait on condition with information
    void signal();					// signal condition
    void signalBlock();					// signal condition
    bool empty() const;					// test for tasks on condition
    long int front() const;				// return task information

    // exception handling

    _DualEvent WaitingFailure;				// condition queue deleted before restarted from waiting

    // These members should be private but cannot be because they are
    // referenced from user code.
}; // uCondition


_DualEvent uCondition::WaitingFailure : public uSerial::Failure {
    const uCondition &cond;
  public:
    WaitingFailure( const uCondition &cond, const char *const msg = "" );
    virtual ~WaitingFailure();
    const uCondition &conditionId() const;
    virtual void defaultTerminate() const;
}; // uCondition::WaitingFailure 


//######################### uProcessorKernel #########################


#define uSCHEDULE(parm...) \
    THREAD_GETMEM( This )->disableInterrupts(); \
    activeProcessorKernel->schedule( parm ); \
    THREAD_GETMEM( This )->enableInterrupts(); \
    /* Are uC++ kernel memory allocation hooks active? */ \
    if ( uThisTask().profileActive && uProfiler::uProfiler_postallocateMetricMemory ) { \
	(*uProfiler::uProfiler_postallocateMetricMemory)( uProfiler::profilerInstance, uThisTask() ); \
    }


_Coroutine uProcessorKernel {
    friend class uKernelBoot;				// access: new, uProcessorKernel, ~uProcessorKernel
    friend class uSerial;				// access: schedule, kernelClock
    friend class uSerialDestructor;			// access: schedule
    friend class uOwnerLock;				// access: schedule
    friend class uCondLock;				// access: schedule, kernelClock
    friend class uSemaphore;				// access: schedule
    friend class uBaseTask;				// access: schedule, kernelClock
    friend _Task uProcessorTask;			// access: terminated, kernelClock
    friend class uProcessor;				// access: uProcessorKernel
    friend class uNBIO;					// access: kernelClock

    // real-time

    friend class uEventList;				// access: schedule
    friend class uEventListPop;				// access: kernelClock
    friend class uSysEventList;				// access: setTimer
    friend class uProcEventList;			// access: kernelClock

    unsigned int kind;					// specific kind of schedule operation
    uSpinLock *prevLock;				// comunication
    uBaseTask *nextTask;				// task to be wakened

    void taskIsBlocking();
    void schedule();
    void schedule( uSpinLock *lock );
    void schedule( uBaseTask *task );
    void schedule( uSpinLock *lock, uBaseTask *task );
    void onBehalfOfUser();
    void setTimer( uDuration time );
    void setTimer( uTime time );
#ifndef __U_MULTI__
    void nextProcessor( uProcessorDL *&currProc );
#endif // __U_MULTI__
    void main();

    uProcessorKernel();
    ~uProcessorKernel();

    void *operator new( size_t, void *storage );
    void *operator new( size_t size );

    uClock kernelClock;
  public:
}; // uProcessorKernel


//######################### uProcessor (cont) #########################


class uProcessor {
    friend class uKernelBoot;				// access: new, uProcessor, events, contextEvent, contextSwitchHandler, setContextSwitchEvent
    friend class uKernelModule;				// access: events
    friend class uCluster;				// access: pid, idleRef, external, processorRef, setContextSwitchEvent
    friend _Task uBootTask;				// access: currTask
    friend _Coroutine uProcessorKernel;			// access: events, currTask, currCluster, procTask, external, globalRef, setContextSwitchEvent
    friend _Task uProcessorTask;			// access: pid, processorClock, preemption, currCluster, setContextSwitchEvent
    friend class uNBIO;					// access: setContextSwitchEvent
    friend class uEventList;				// access: events, contextSwitchHandler
    friend class uProcEventList;			// access: events
    friend class uEventNode;                            // access: events
    friend class uEventListPop;                         // access: contextSwitchHandler
    friend void exit( int retcode ) __THROW;		// access: uProcExit
    friend __U_START_KT_TYPE__;				// access: everything
#if defined( __i386__ ) || defined( __ia64__ ) && ! defined( __old_perfmon__ )
    friend class HWCounters;				// access: uPerfctrContext (i386) or uPerfmon_fd (ia64)
#endif

    // debugging

    friend _Task uLocalDebugger;			// access: debugIgnore
    friend _Task uLocalDebuggerReader;			// access: debugIgnore

    bool debugIgnore;					// ignore processor migration

    // profiling
    
    friend class uProfileProcessorSampler;		// access: profileProcessorSamplerInstance

#if ! defined( __U_MULTI__ ) || defined( __U_ONETIMER__ )
    static
#endif // ! __U_MULTI__ || __U_ONETIMER__
    uEventList *events;					// list of events for all processors (ONETIMER or !MULTI) or this processor (MULTI & !ONETIMER)
#if defined( __U_ONETIMER__ ) && defined( __U_MULTI__ )
    uProcEventList *procEvents;				// list of events specific to this processor
#endif // __U_ONETIMER__ && defined( __U_MULTI__ )
#if ! defined( __U_MULTI__ )
    static						// shared info on uniprocessor
#endif // ! __U_MULTI__
    uEventNode *contextEvent;				// context-switch node for event list
#if ! defined( __U_MULTI__ )
    static						// shared info on uniprocessor
#endif // ! __U_MULTI__
    uCxtSwtchHndlr *contextSwitchHandler;		// special time slice handler

#ifdef __U_MULTI__
    uProcessorKernel processorKer;			// need a uProcessorKernel
#endif // __U_MULTI__

#if defined( __i386__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
    unsigned int ldtValue;
    
#   define U_MAX_LDT 8192
    static uBitSet< U_MAX_LDT > LDTFreeSet;
    static uOwnerLock LDTFreeSetLock;
    static bool LDTFreeSetInit;
    static int allocLDT();
    static void freeLDT( int oldLDT );
#endif // __i386__ &&__U_MULTI__ && ! __U_PTHREAD__

    // profiling

#ifndef __U_MULTI__
    static
#else
    mutable
#endif // ! __U_MULTI__
    uProfileProcessorSampler *profileProcessorSamplerInstance; // pointer to related profiling object

    uProcessor( uProcessor & );				// no copy
    uProcessor &operator=( uProcessor &);		// no assignment

    void *operator new( size_t, void *storage );
  protected:
    uClock *processorClock;				// clock bound to processor

    uPid_t pid;

    unsigned int preemption;
    unsigned int spin;

    uProcessorTask *procTask;				// handle processor specific requests
    uBaseTaskSeq external;				// ready queue for processor task

    uBaseTask *currTask;				// task executing on processor
    uCluster *currCluster;				// cluster processor currently associated with

    bool detached;					// processor detached ?
    bool terminated;                                    // processor being deleted ?

    uProcessorDL idleRef;				// double link field: list of idle processors
    uProcessorDL processorRef;				// double link field: list of processors on a cluster
    uProcessorDL globalRef;				// double link field: list of all processors

    void createProcessor( uCluster &cluster, bool detached, int ms, int spin );
    void fork( uProcessor *processor );
    void setContextSwitchEvent( int msecs );		// set the real-time timer
    void setContextSwitchEvent( uDuration duration );	// set the real-time timer
#ifdef __U_MULTI__
#if defined( __irix__ ) || defined( __freebsd__ )
    pid_t procCreate( uProcessor *processor );
#endif
#if defined( __irix__ ) ||  ( defined( __linux__ ) && ! defined( __U_PTHREAD__ ) ) || defined( __freebsd__ )
    void procWait( pid_t pid );
#endif
    void procExit( int retcode ) __attribute__(( noreturn ));
#endif // __U_MULTI__

    uProcessor( uCluster &cluster, double );		// used solely during kernel boot
  public:
    void *operator new( size_t size );			// must be public for normal dynamic "new"

    uProcessor( unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    uProcessor( bool detached, unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    uProcessor( uCluster &cluster, unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    uProcessor( uCluster &cluster, bool detached, unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    ~uProcessor();

    uClock &getClock() const;
    uPid_t getPid() const;
    uCluster &setCluster( uCluster &cluster );
    uCluster &getCluster() const;
    uBaseTask &getTask() const;
    bool getDetach() const;
    unsigned int setPreemption( unsigned int ms );
    unsigned int getPreemption() const;
    unsigned int setSpin( unsigned int spin );
    unsigned int getSpin() const;
    bool idle() const;
} __attribute__(( unused )); // uProcessor


inline uProcessor &uThisProcessor() {
    return *THREAD_GETMEM( activeProcessor );
} // uThisProcessor


//######################### uCluster (cont) #########################


#if defined( __solaris__ ) || defined( __irix__ )
#include <sys/select.h>					// fd_set
#else
#include <sys/types.h>					// fd_set
#endif


class uCluster {
    // The cluster cannot be a monitor because the processor kernel is only a
    // coroutine. If the kernel called a mutex routine in the cluster and could
    // not get in, the previous task executed by the kernel is put on the mutex
    // entry queue and the kernel is restarted. When the kernel restarts, it
    // now enters the critical section when it should be scheduling a new
    // task. Therefore explicit locks must be used for these queues.

    friend class uBaseTask;				// access: makeTaskReady, taskAdd, taskRemove
    friend class uNBIO;					// access: makeProcessorIdle, makeProcessorActive
    friend class uKernelBoot;				// access: new, NBIO, taskAdd, taskRemove
    friend _Coroutine uProcessorKernel;			// access: NBIO, readyTaskTryRemove, readyTasksEmpty, tasksOnCluster, makeProcessorActive, processorPause
    friend _Task uProcessorTask;			// access: processorAdd, processorRemove
    friend class uTaskConstructor;			// access: taskAdd
    friend class uTaskDestructor;			// access: taskRemove
    friend class uProcessor;				// access: processorAdd, processorRemove
    friend class uRealTimeBaseTask;			// access: taskReschedule
    friend class uPeriodicBaseTask;			// access: taskReschedule
    friend class uSporadicBaseTask;			// access: taskReschedule
    friend class uFileAccess;				// access: closeFD
    friend class uSocket;				// access: closeFD
    friend class uSocketAccept;				// access: closeFD

    // debugging

    friend _Task uLocalDebugger;			// access: debugIgnore
    friend _Task uLocalDebuggerReader;			// access: debugIgnore

    bool debugIgnore;					// ignore cluster migration

    // profiling

    friend class uProfileClusterSampler;		// access: profileClusterSamplerInstance

    // real-time

    friend class uEventList;				// access: wakeProcessor
    friend class uProcWakeupHndlr;			// access: wakeProcessor

    uClusterDL globalRef;				// double link field: list of all clusters

    uCluster( uCluster & );				// no copy
    uCluster &operator=( uCluster & );			// no assignment

    void *operator new( size_t, void *storage );
    void *operator new( size_t size );
  protected:
    const char *name;					// textual name for cluster, default value
    uSpinLock readyIdleTaskLock;			// protect readyTasks, idleProcessors and tasksOnCluster
    uBaseSchedule<uBaseTaskDL> *readyTasks;		// list of tasks awaiting execution by processors on this cluster
    bool defaultReadyTasks;				// indicates if the cluster allocated the ready task
    uProcessorSeq idleProcessors;			// list of idle processors associated with this cluster
    uBaseTaskSeq tasksOnCluster;			// list of tasks on this cluster
    uSpinLock processorsOnClusterLock;
    uProcessorSeq processorsOnCluster;			// list of processors associated with this cluster
    unsigned int stackSize;

    // Make a pointer to allow static declaration for uniprocessor.
#ifndef __U_MULTI__
    static						// shared info on uniprocessor
#endif // ! __U_MULTI__
    uNBIO *NBIO;					// non-blocking I/O facilities

    // profiling

    mutable uProfileClusterSampler *profileClusterSamplerInstance; // pointer to related profiling object

    void createCluster( unsigned int stacksize, const char *name );
    static void wakeProcessor( uPid_t pid );
    void processorPause();
    void makeProcessorIdle( uProcessor &processor );
    void makeProcessorActive( uProcessor &processor );
    void makeProcessorActive();
    bool readyTasksEmpty();
    void makeTaskReady( uBaseTask &readyTask );
    uBaseTask &readyTaskTryRemove();
    void taskAdd( uBaseTask &task );
    void taskRemove( uBaseTask &task );
    virtual void processorAdd( uProcessor &processor );
    void processorRemove( uProcessor &processor );
    void taskReschedule( uBaseTask &task );
    void closeFD( int fd );
  public:
    uCluster( unsigned int stacksize = uDefaultStackSize(), const char *name = "*unnamed*" );
    uCluster( const char *name );
    uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, unsigned int stacksize = uDefaultStackSize(), const char *name = "*unnamed*" );
    uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, const char *name = "*unnamed*" );
    virtual ~uCluster();

    const char *setName( const char *name );
    const char *getName() const;
    unsigned int setStackSize( unsigned int stacksize );
    unsigned int getStackSize() const;

    void taskResetPriority( uBaseTask &owner, uBaseTask &calling );
    void taskSetPriority( uBaseTask &owner, uBaseTask &calling );

    static const int readSelect;
    static const int writeSelect;
    static const int exceptSelect;

    int select( int fd, int rwe, timeval *timeout = NULL );
    int select( fd_set *rfd, fd_set *wfd, fd_set *efd, timeval *timeout = NULL );
    int select( int nfds, fd_set *rfd, fd_set *wfd, fd_set *efd, timeval *timeout = NULL );

    const uBaseTaskSeq &getTasksOnCluster() {
	return tasksOnCluster;
    } // uCluster::getTasksOnCluster

    const uProcessorSeq &getProcessorsOnCluster() {
	return processorsOnCluster;
    } // uCluster::getProcessorsOnCluster
}; // uCluster


inline uCluster &uThisCluster() {
    return *THREAD_GETMEM( activeCluster );
} // uThisCluster


//######################### uNBIO #########################


class uNBIO {						// monitor (private mutex member)
    friend class uCluster;				// access: closeFD
    friend _Coroutine uProcessorKernel;			// access: okToSelect, IOPoller
    friend class uSelectTimeoutHndlr;			// access: NBIOnode

    struct NBIOnode : public uSeqable {
	int nfds;					// return value
	uBaseTask *pendingTask;
	enum { singleFd, multipleFds } fdType;
	bool timedout;
	union {
	    struct {					// used if waiting for only one fd
		int fd;
		int *uRWE;
	    } sfd;
	    struct {					// used if waiting for multiple fds
		fd_mask uNFDs;
		fd_set *uRFDs;
		fd_set *uWFDs;
		fd_set *uEFDs;
	    } mfd;
	} smfd;
	uCondition pending;
    }; // NBIOnode

    fd_set uRFDs, uWFDs, uEFDs;				// master copy of all waiting I/O for all tasks
    fd_set lrfds, lwfds, lefds;				// temporary copies for select call

    int maxFD;						// highest FD used in select mask
    bool okToSelect;					// uniprocessor flag indicating blocking select
    int descriptors;					// declared here so uniprocessor kernel can check if I/O occurred
    uBaseTask *IOPoller;				// pointer to current IO poller task, or 0
    uSequence<NBIOnode> pendingIO;			// list of waiting tasks

    void checkIO( uDuration delay );
    bool checkPoller( NBIOnode &node );
    void setMaxFD();
    _Mutex void closeFD( int fd );			// called by uCluster
    _Mutex bool pollIO( NBIOnode &node );
    _Mutex bool initSfd( NBIOnode &node );
    _Mutex bool initSfd( NBIOnode &node, uEventNode &uTimeoutEvent );
    _Mutex bool initMfds( fd_mask nfds, NBIOnode &node );
    _Mutex bool initMfds( fd_mask nfds, NBIOnode &node, uEventNode &uTimeoutEvent );
  public:
    uNBIO();
    ~uNBIO();
    int select( int fd, int &rwe, timeval *timeout = NULL );
    int select( fd_mask nfds, fd_set *rfds, fd_set *wfds, fd_set *efds, timeval *timeout = NULL );
}; // uNBIO


class uSelectTimeoutHndlr : public uSignalHandler {	// real-time
    friend class uNBIO;
  
    uNBIO::NBIOnode &node;

    uSelectTimeoutHndlr( uBaseTask &task, uNBIO::NBIOnode &node ) : node( node ) {
	This = &task;
    } // uSelectTimeoutHndlr::uSelectTimeoutHndlr

    void uHandler() {
	node.timedout = true;
    } // uSelectTimeoutHndlr::uHandle
}; // uSelectTimeoutHndlr


//######################### uMain #########################

_Task uMain {
    friend _Task Pthread;				// access: cleanup_handlers
    friend void pthread_exit( void *status );		// access: joinval

    // The arguments that are passed to main are passed to the first user task
    // and stored as member variables.

    int argc;
    char **argv;

    // A reference to a variable that holds the return code that the uMain task
    // returns to the OS.

    int &uRetCode;

    // make uMain have pthread cancellation properties
    uBaseCoroutine::Cleanup cleanup_handlers;
    void * joinval;					// only used as a placeholder to make pthread code uniform (cannot yet join with uMain)

    // Main routine for the first user task, declared here, defined by user.

    void main();
  public:
    uMain( uCluster &cluster, int size, int argc, char *argv[], int &retcode );
    ~uMain();
}; // uMain


//######################### Kernel Boot #########################


class uKernelBoot {
    static int count;

    static void startup();
    static void finishup();
  public:
    uKernelBoot() {
	count += 1;
	if ( count == 1 ) {
	    startup();
	} // if
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uKernelBoot &)0x%p.uKernelBoot\n", this );
#endif // __U_DEBUG_H__
    } // uKernelBoot::uKernelBoot

    ~uKernelBoot() {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uKernelBoot &)0x%p.~uKernelBoot\n", this );
#endif // __U_DEBUG_H__
	if ( count == 1 ) {
	    finishup();
	} // if
	count -= 1;
    } // uKernelBoot::uKernelBoot
}; // uKernelBoot


class uInitProcessorsBoot {
    static int count;

    static void startup();
    static void finishup();
  public:
    uInitProcessorsBoot() {
	count += 1;
	if ( count == 1 ) {
	    startup();
	} // if
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uInitProcessorsBoot &)0x%p.uInitProcessorsBoot\n", this );
#endif // __U_DEBUG_H__
    } // uInitProcessorsBoot::uInitProcessorsBoot

    ~uInitProcessorsBoot() {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uInitProcessorsBoot &)0x%p.~uInitProcessorsBoot\n", this );
#endif // __U_DEBUG_H__
	if ( count == 1 ) {
	    finishup();
	} // if
	count -= 1;
    } // uInitProcessorsBoot::~uInitProcessorsBoot
}; // uInitProcessorsBoot


// debugging

#ifdef __U_DEBUG__
#include <uLocalDebugger.h>
#endif // __U_DEBUG__


// Create an instance in each translation unit, but only the first instance to
// execute performs the system bootstrap. Do not include instances in the
// kernel modules.

#ifndef __U_KERNEL__
#include <ios>
static std::ios_base::Init __ioinit;			// ensure streams are initialized before startup
static uKernelBoot uBootKernel;

#if __U_LOCALDEBUGGER_H__
static uLocalDebuggerBoot uBootLocalDebugger;
#endif // __U_LOCALDEBUGGER_H__

#if defined( __U_PROFILE__ )
#ifndef __U_PROFILEABLE_ONLY__
#include <uProfilerBoot.h>
static uProfilerBoot uBootProfiler;

#endif // __U_PROFILEABLE_ONLY__
#endif // __U_PROFILE__

static uInitProcessorsBoot uBootProcessorsInit;

#endif // __U_KERNEL__


#pragma __U_USER_CODE__


#endif // __U_CPLUSPLUS_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
