//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.4.0, Copyright (C) Peter A. Buhr 1994
// 
// uC++.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Fri Dec 17 22:10:52 1993
// Last Modified By : Peter A. Buhr
// Last Modified On : Sun Sep 24 09:23:35 2006
// Update Count     : 2305
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>
#include <uProfiler.h>
#include <uHeapLmmm.h>
#include <uBootTask.h>
#include <uSystemTask.h>
#include <uFilebuf.h>
//#include <uDebug.h>

#include <iostream>
#include <exception>
#include <dlfcn.h>
#include <cstdio>
#include <unistd.h>					// _exit

#if defined( __linux__ ) && defined( __i386__ ) && ! defined( __U_PTHREAD__ )
#include <asm/unistd.h>					// for _syscall3, __NR_modify_ldt

#ifndef _syscall3				        // fedora change to account for moved definition of _syscall3
#include <linux/unistd.h>
#endif

#include <asm/ldt.h>					// for struct modify_ldt_ldt_s
#include <linux/version.h>				// for KERNEL_VERSION

// modify_ldt has no prototype in the headers
// this is copied from the man page
_syscall3( int, modify_ldt, int, func, void *, ptr, unsigned long, bytecount );

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,40)
#define modify_ldt_ldt_s user_desc			// name change
#endif

#endif // __linux__ && __i386__ && ! __U_PTHREAD__

#if defined( __freebsd__ ) && defined( __i386__ )
#include <machine/segments.h>				// segment_descriptor
#include <machine/sysarch.h>				// i386_set_ldt
#endif // __freebsd__ && __i386__

#if defined( __irix__ )
#include <ulocks.h>					// for usconfig
#include <sys/wait.h>					// for waitpid
#endif // __irix__

#if defined( __ia64__ )
#include <ia64intrin.h>					// __sync_lock_release
#endif // __ia64__

bool            uKernelModule::kernelModuleInitialized	= false;
bool            uKernelModule::initialization		= false;
bool		uKernelModule::coreDumped		= false;
#ifndef __U_MULTI__
bool		uKernelModule::deadlock			= false;
#endif // ! __U_MULTI__
bool		uKernelModule::globalAbort		= false;
bool		uKernelModule::globalSpinAbort		= false;
uSpinLock	*uKernelModule::globalAbortLock	= NULL;
uSpinLock	*uKernelModule::globalProcessorLock	= NULL;
uSpinLock	*uKernelModule::globalClusterLock	= NULL;
uDefaultScheduler *uKernelModule::systemScheduler	= NULL;
uCluster	*uKernelModule::systemCluster		= NULL;
uProcessor	*uKernelModule::systemProcessor	= NULL;
uBootTask	*uKernelModule::bootTask		= (uBootTask *)&bootTaskStorage;
uSystemTask	*uKernelModule::systemTask		= NULL;
uProcessor	**uKernelModule::userProcessors		= NULL;
unsigned int	uKernelModule::numProcessors		= 0;
uCluster	*uKernelModule::userCluster		= NULL;

unsigned int    uKernelModule::attaching		= 0; // debugging

char		uKernelModule::systemProcessorStorage[sizeof(uProcessor)] __attribute__(( aligned (16) )) = {0};
char		uKernelModule::systemClusterStorage[sizeof(uCluster)] __attribute__(( aligned (16) )) = {0};
char		uKernelModule::bootTaskStorage[sizeof(uBootTask)] __attribute__(( aligned (16) )) = {0};

std::filebuf	*uKernelModule::cerrFilebuf = NULL, *uKernelModule::clogFilebuf = NULL, *uKernelModule::coutFilebuf = NULL, *uKernelModule::cinFilebuf = NULL;


// Fake uKernelModule used before uKernelBoot::startup.
volatile __U_THREAD__ uKernelModule::uKernelModuleData uKernelModule::uKernelModuleBoot;

uProcessorSeq	*uKernelModule::globalProcessors	= NULL;
uClusterSeq	*uKernelModule::globalClusters		= NULL;

bool		uKernelModule::afterMain		= false;
bool		uKernelModule::inExit			= false;
int		uKernelModule::retCode			= 0;

#ifdef __U_FLOATINGPOINTDATASIZE__
int		uFloatingPointContext::uniqueKey	= 0;
#endif // __U_FLOATINGPOINTDATASIZE__

#define __U_TIMEOUTPOSN__ 0				// bit 0 is reserved for timeout
#define __U_DESTRUCTORPOSN__ 1				// bit 1 is reserved for destructor

#ifndef __U_MULTI__
uNBIO		*uCluster::NBIO				= NULL;
#endif // ! __U_MULTI__

int		uKernelBoot::count			= 0;
int		uInitProcessorsBoot::count		= 0;


extern "C" void _pthread_deletespecific( void * );	// see pthread simulation
extern "C" void _pthread_delete_kernel_threads();
extern "C" void _pthread_pid_destroy( void );
extern "C" void _pthread_shutdown();


//######################### main #########################


// Declare a variable in shared memory to obtain a return code from the user
// program.  Unless modified, the return code is zero.

int uRetCode = 0;


// The main routine that gets the first task started with the OS supplied
// arguments, waits for its completion, and returns the result code to the OS.
// Define in this translation unit so it cannot be replaced by a user.

#if defined( __irix__ )
extern "C" void __do_global_dtors();
extern "C" void _exithandle();
extern "C" void _cleanup();
#endif // __irix__

int main( int argc, char *argv[] ) {
    {
	uMain uUserMain( *uKernelModule::userCluster, uMainStackSize(), argc, argv, uRetCode );
    }

    uKernelModule::afterMain = true;

#if defined( __irix__ )
    // SKULLDUGGERY: irix does not run global destructors until all sprocs
    // have terminated.  But uC++ depends on destructors to terminate sprocs.
    // So, invoke the destructor sequence by hand here, and then _exit to
    // prevent the destructors from being done twice.  Because _exit stops
    // execution, all of the regular cleanups must done.

    _exithandle();					// atexit functions
    _cleanup();						// close files
    __do_global_dtors();				// global destructors
    _exit( uRetCode );
#endif // __irix__

    // Return the program return code to the operating system.

    return uRetCode;
} // main


//######################### uSpinLock #########################


void *uSpinLock::operator new( size_t, void *storage ) {
    return storage;
} // uSpinLock::operator new

void *uSpinLock::operator new( size_t size ) {
    return ::operator new( size );
} // uSpinLock::operator new


uSpinLock::uSpinLock() {
    value = 0;						// unlock
} // uSpinLock::uSpinLock


void uSpinLock::acquire() {
    // No race condition exists for accessing disableIntSpin in the
    // multiprocessor case because this variable is private to each UNIX
    // process. Also, the spin lock must be acquired after adjusting
    // disableIntSpin because the time slicing must see the attempt to access
    // the lock first to prevent live-lock on the same processor.  For example,
    // one task acquires the ready queue lock, a time slice occurs, and it does
    // not appear that the current task is in the kernel because
    // disableIntSpin is not set so the signal handler tries to yield.
    // However, the ready queue lock is held so the yield live-locks. There is
    // a similar situation on releasing the lock.

#ifdef __U_DEBUG__
#ifndef __U_MULTI__
    if ( value != 0 ) {					// locked ?
	uAbort( "(uSpinLock &)0x%p.acquire() : internal error, attempt to multiply acquire spin lock by same task.", this );
    } // if
#endif // __U_MULTI__
#endif // __U_DEBUG__

    THREAD_GETMEM( This )->disableIntSpinLock();

#ifdef __U_MULTI__
    if ( uAtomic( &value ) != 0 ) {			// test and set, uAtomic returns 0 or non-zero

#if 0
        THREAD_GETMEM( This )->disableInterrupts();
	if ( uThisTask().profileActive && uProfiler::uProfiler_builtinRegisterTaskStartSpin ) {
	    (*uProfiler::uProfiler_builtinRegisterTaskStartSpin)( uProfiler::profilerInstance, uThisTask() );
	} // if
	THREAD_GETMEM( This )->enableInterrupts();
#endif

	int spin = 1;
	for ( ;; ) {					// poll for lock
	    THREAD_GETMEM( This )->enableIntSpinLock();
	    for ( int i = 0; i < spin; i += 1 ) {	// exponential spin
		if ( uKernelModule::globalSpinAbort ) {
		    _exit( -1 );			// close down in progress, shutdown immediately!
		} // if
	    } // for
	    spin += spin;				// powers of 2
	    if ( spin > 65536 ) spin = 1;		// prevent overflow
	    THREAD_GETMEM( This )->disableIntSpinLock();
	    if ( uAtomic( &value ) == 0 ) break;	// test and set, uAtomic returns 0 or non-zero
	} // for

#if 0
        THREAD_GETMEM( This )->disableInterrupts();
	if ( uThisTask().profileActive && uProfiler::uProfiler_builtinRegisterTaskStopSpin ) {
	    (*uProfiler::uProfiler_builtinRegisterTaskStopSpin)( uProfiler::profilerInstance, uThisTask() );
	} // if
	THREAD_GETMEM( This )->enableInterrupts();
#endif
    } // if
#if defined( __sparc__ )
    asm( "membar #LoadLoad" );				// flush the cache
#endif // __sparc__

#else
    value = 1;						// lock
#endif // __U_MULTI__
} // uSpinLock::acquire

// Same as acquire, except it calls uKernelModule::enableIntSpinLockNoRF()
void uSpinLock::acquireNoRF() {
    // No race condition exists for accessing disableIntSpin in the
    // multiprocessor case because this variable is private to each UNIX
    // process. Also, the spin lock must be acquired after adjusting
    // disableIntSpin because the time slicing must see the attempt to access
    // the lock first to prevent live-lock on the same processor.  For example,
    // one task acquires the ready queue lock, a time slice occurs, and it does
    // not appear that the current task is in the kernel because
    // disableIntSpin is not set so the signal handler tries to yield.
    // However, the ready queue lock is held so the yield live-locks. There is
    // a similar situation on releasing the lock.

#ifdef __U_DEBUG__
#ifndef __U_MULTI__
    if ( value != 0 ) {					// locked ?
	uAbort( "(uSpinLock &)0x%p.acquireNoRF() : internal error, attempt to multiply acquire spin lock by same task.", this );
    } // if
#endif // __U_MULTI__
#endif // __U_DEBUG__

    THREAD_GETMEM( This )->disableIntSpinLock();

#ifdef __U_MULTI__
    if ( uAtomic( &value ) != 0 ) {			// test and set, uAtomic returns 0 or non-zero

#if 0
	THREAD_GETMEM( This )->disableInterrupts();
	if ( uThisTask().profileActive && uProfiler::uProfiler_builtinRegisterTaskStartSpin ) {
	    (*uProfiler::uProfiler_builtinRegisterTaskStartSpin)( uProfiler::profilerInstance, uThisTask() );
	} // if
	THREAD_GETMEM( This )->enableInterrupts();
#endif

	int spin = 1;
	for ( ;; ) {					// poll for lock
	    THREAD_GETMEM( This )->enableIntSpinLockNoRF();
	    for ( int i = 0; i < spin; i += 1 ) {	// exponential spin
		if ( uKernelModule::globalSpinAbort ) {
		    _exit( -1 );			// close down in progress, shutdown immediately!
		} // if
	    } // for
	    spin += spin;				// powers of 2
	    if ( spin > 65536 ) spin = 1;		// prevent overflow
	    THREAD_GETMEM( This )->disableIntSpinLock();
	    if ( uAtomic( &value ) == 0 ) break;	// test and set, uAtomic returns 0 or non-zero
	} // for

#if 0
	THREAD_GETMEM( This )->disableInterrupts();
	if ( uThisTask().profileActive && uProfiler::uProfiler_builtinRegisterTaskStopSpin ) {
	    (*uProfiler::uProfiler_builtinRegisterTaskStopSpin)( uProfiler::profilerInstance, uThisTask() );
	} // if
	THREAD_GETMEM( This )->enableInterrupts();
#endif
    } // if
#if defined( __sparc__ )
    asm( "membar #LoadLoad" );				// flush the cache
#endif // __sparc__

#else
    value = 1;						// lock
#endif // __U_MULTI__
} // uSpinLock::acquireNoRF

bool uSpinLock::tryacquire() {
#ifdef __U_DEBUG__
#ifndef __U_MULTI__
    if ( value != 0 ) {					// locked ?
	uAbort( "(uSpinLock &)0x%p.tryacquire() : internal error, attempt to multiply acquire spin lock by same task.", this );
    } // if
#endif // __U_MULTI__
#endif // __U_DEBUG__

    THREAD_GETMEM( This )->disableIntSpinLock();

#ifdef __U_MULTI__
    if ( uAtomic( &value ) == 0 ) {			// get the lock ?
#if defined( __sparc__ )
	asm( "membar #LoadLoad" );			// flush the cache
#endif // __sparc__
	return true;
    } else {
	THREAD_GETMEM( This )->enableIntSpinLock();
	return false;
    } // if
#else
    value = 1;						// lock
    return true;
#endif // __U_MULTI__
} // uSpinLock::tryacquire

void uSpinLock::release() {
#if defined( __sparc__ )
    asm( "membar #StoreStore | #LoadStore" );		// flush the cache
#endif // __sparc__
    assert( value != 0 );
#if defined( __ia64__ )
    __sync_lock_release( &value );
#else
    value = 0;						// unlock
#endif // __ia64__
    THREAD_GETMEM( This )->enableIntSpinLock();
} // uSpinLock::release

// Same as release, except it calls uKernelModule::enableIntSpinLockNoRF()
void uSpinLock::releaseNoRF() {
#if defined( __sparc__ )
    asm( "membar #StoreStore | #LoadStore" );		// flush the cache
#endif // __sparc__
    assert( value != 0 );
#if defined( __ia64__ )
    __sync_lock_release( &value );
#else
    value = 0;						// unlock
#endif // __ia64__
    THREAD_GETMEM( This )->enableIntSpinLockNoRF();
} // uSpinLock::releaseNoRF


uCSpinLock::uCSpinLock( uSpinLock &spinLock ) : spinLock( spinLock ) {
    spinLock.acquire();
} // uCSpinLock::uCSpinLock

uCSpinLock::~uCSpinLock() {
    spinLock.release();
} // uCSpinLock::~uCSpinLock


//######################### uLock #########################


uLock::uLock() {
    value = 1;
} // uLock::uLock

uLock::uLock( unsigned int val ) {
#ifdef __U_DEBUG__
    if ( val > 1 ) {
	uAbort( "Attempt to initialize uLock 0x%p to %d that exceeds range 0-1.", this, val );
    } // if
#endif // __U_DEBUG__
    value = val;
} // uLock::uLock

void uLock::acquire() {
    for ( ;; ) {
	spinLock.acquire();
      if ( value == 1 ) break;
	spinLock.release();
	uThisTask().yield();
    } // for
    value = 0;
    spinLock.release();
} // uLock::acquire

bool uLock::tryacquire() {
    spinLock.acquire();
    if ( value == 1 ) {
	value = 0;
	spinLock.release();
	return true;
    } else {
	spinLock.release();
	return false;
    } // if
} // uLock::tryacquire

void uLock::release() {
    value = 1;
} // uLock::release


//######################### uOwnerLock #########################


void *uOwnerLock::operator new( size_t, void *storage ) {
    return storage;
} // uOwnerLock::operator new

void *uOwnerLock::operator new( size_t size ) {
    return ::operator new( size );
} // uOwnerLock::operator new

void uOwnerLock::_add( uBaseTask &task ) {		// used by uCondLock::signal
    spin.acquire();
    if ( _owner != NULL ) {				// lock in use ?
	waiting.addTail( &(task.entryRef) );		// move task to owner lock list
    } else {
	_owner = &task;					// become owner
	count = 1;
	task.wake();					// restart new owner
    } // if
    spin.release();
} // uOwnerLock::_add

void uOwnerLock::_release() {				// used by uCondLock::wait
    spin.acquire();
    if ( ! waiting.empty() ) {				// waiting tasks ?
	_owner = &(waiting.dropHead()->task());		// remove task at head of waiting list and make new owner
	count = 1;
	_owner->wake();					// restart new owner
    } else {
	_owner = NULL;					// release, no owner
	count = 0;
    } // if
    spin.release();
} // uOwnerLock::_release

uOwnerLock::uOwnerLock() {
    _owner = NULL;					// no one owns the lock
    count = 0;						// so count is zero
} // uOwnerLock::uOwnerLock

#ifdef __U_DEBUG__
uOwnerLock::~uOwnerLock() {
    spin.acquire();
    if ( ! waiting.empty() ) {
	uBaseTask *task = &(waiting.head()->task());	// waiting list could change as soon as spin lock released
	spin.release();
	uAbort( "Attempt to delete owner lock with task %.256s (0x%p) still on it.", task->getName(), task );
    } // if
    spin.release();
} // uOwnerLock::uOwnerLock
#endif // __U_DEBUG__

unsigned int uOwnerLock::times() const {
    return count;
} // uOwnerLock::times

uBaseTask *uOwnerLock::owner() const {
    return _owner;
} // uOwnerLock::times

void uOwnerLock::acquire() {
    uBaseTask &task = uThisTask();			// optimization

    if ( uKernelModule::initialization ) assert( ! THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) == 0 );
    spin.acquire();
    if ( _owner != &task ) {				// don't own lock yet
	if ( _owner != NULL ) {				// but if lock in use
	    waiting.addTail( &(task.entryRef) );	// suspend current task
	    uSCHEDULE( &spin );				// atomically release owner spin lock and block
	    // _owner and count set in release
	    return;
	} else {
	    _owner = &task;				// become owner
	    count = 1;
	} // if
    } else {
	count += 1;					// remember how often
    } // if
    spin.release();
} // uOwnerLock::acquire

bool uOwnerLock::tryacquire() {
    uBaseTask &task = uThisTask();			// optimization

    if ( uKernelModule::initialization ) assert( ! THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) == 0 );
    spin.acquire();
    if ( _owner != &task ) {				// don't own lock yet
	if ( _owner != NULL ) {				// but if lock in use
	    spin.release();
	    return false;				// don't wait for the lock
	} // if
	_owner = &task;					// become owner
	count = 1;
    } else {
	count += 1;					// remember how often
    } // if
    spin.release();
    return true;
} // uOwnerLock::tryacquire

void uOwnerLock::release() {
    if ( uKernelModule::initialization ) assert( ! THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) == 0 );
    spin.acquire();
#ifdef __U_DEBUG__
    if ( _owner == NULL ) {
	spin.release();
	uAbort( "Attempt to release owner lock that is not locked." );
    } // if
    if ( _owner == (uBaseTask *)-1 ) {
	spin.release();
	uAbort( "Attempt to release owner lock that is in an invalid state. Possible cause is the lock has been freed." );
    } // if
    if ( _owner != &uThisTask() ) {
	uBaseTask *task = _owner;			// owner could change as soon as spin lock released
	spin.release();
	uAbort( "Attempt to release owner lock that is currently owned by task %.256s (0x%p).", task->getName(), task );
    } // if
#endif // __U_DEBUG__
    count -= 1;						// release the lock
    if ( count == 0 ) {					// if this is the last
	if ( ! waiting.empty() ) {			// waiting tasks ?
	    _owner = &(waiting.dropHead()->task());	// remove task at head of waiting list and make new owner
	    count = 1;
	    _owner->wake();				// restart new owner
	} else {
	    _owner = NULL;				// release, no owner
	} // if
    } // if
    spin.release();
} // uOwnerLock::release


//######################### uCondLock #########################


void *uCondLock::operator new( size_t, void *storage ) {
    return storage;
} // uCondLock::operator new

void *uCondLock::operator new( size_t size ) {
    return ::operator new( size );
} // uCondLock::operator new

uCondLock::uCondLock() {
} // uCondLock::uCondLock

#ifdef __U_DEBUG__
uCondLock::~uCondLock() {
    spin.acquire();
    if ( ! waiting.empty() ) {
	uBaseTask *task = &(waiting.head()->task());
	spin.release();
	uAbort( "Attempt to delete condition lock with task %.256s (0x%p) still blocked on it.", task->getName(), task );
    } // if
    spin.release();
} // uCondLock::uCondLock
#endif // __U_DEBUG__

bool uCondLock::empty() const {
    return waiting.empty();
} // uCondLock::empty

void uCondLock::wait( uOwnerLock &lock ) {
    uBaseTask &task = uThisTask();			// optimization
#ifdef __U_DEBUG__
    uBaseTask *owner = lock.owner();			// owner could change
    if ( owner != &task ) {
	uAbort( "Attempt by waiting task %.256s (0x%p) to release condition lock currently owned by task %.256s (0x%p).",
		task.getName(), &task, owner->getName(), owner );
    } // if
#endif // __U_DEBUG__
    task.ownerLock = &lock;				// task remembers this lock before blocking for use in signal
    unsigned int prevcnt = lock.count;			// remember this lock's recursive count before blocking
    spin.acquire();
    waiting.addTail( &(task.entryRef) );		// queue current task
    // Must add to the condition queue first before releasing the owner lock
    // because testing for empty condition can occur immediately after the
    // owner lock is released.
    lock._release();					// release owner lock
    uSCHEDULE( &spin );					// atomically release condition spin lock and block
    // spin released by schedule, owner lock is acquired when task restarts
    assert( &task == lock.owner() );
    lock.count = prevcnt;				// reestablish lock's recursive count after blocking
} // uCondLock::wait

bool uCondLock::timedwait( uOwnerLock &lock, uDuration duration ) {
    return timedwait( lock, activeProcessorKernel->kernelClock.getTime() + duration );
} // uCondLock::timedwait

bool uCondLock::timedwait( uOwnerLock &lock, uTime time ) {
    uBaseTask &task = uThisTask();			// optimization
#ifdef __U_DEBUG__
    uBaseTask *owner = lock.owner();			// owner could change
    if ( owner != &task ) {
	uAbort( "Attempt by waiting task %.256s (0x%p) to release condition lock currently owned by task %.256s (0x%p).",
		task.getName(), &task, owner->getName(), owner );
    } // if
#endif // __U_DEBUG__
    task.ownerLock = &lock;				// task remembers this lock before blocking for use in signal
    unsigned int prevcnt = lock.count;			// remember this lock's recursive count before blocking
    spin.acquire();

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCondLock &)0x%p.timedwait, task:0x%p\n", this, &task );
#endif // __U_DEBUG_H__

    uCondLockTimedwaitHndlr handler( task, *this );	// handler to wake up blocking task

    uEventNode timeoutEvent( task, handler, time, 0 );
    timeoutEvent.executeLocked = true;

    timeoutEvent.add();

    waiting.addTail( &(task.entryRef) );		// queue current task
    // Must add to the condition queue first before releasing the owner lock
    // because testing for empty condition can occur immediately after the
    // owner lock is released.
    lock._release();					// release owner lock
    uSCHEDULE( &spin );					// atomically release owner spin lock and block
    // spin released by schedule, owner lock is acquired when task restarts
    assert( &task == lock.owner() );
    lock.count = prevcnt;				// reestablish lock's recursive count after blocking

    timeoutEvent.remove();

    return ! handler.timedout;
} // uCondLock::timedwait

void uCondLock::waitTimeout( uBaseTask &task, uCondLockTimedwaitHndlr &h ) {
    // This uCondLock member is called from the kernel, and therefore, cannot
    // block, but it can spin.

    spin.acquire();
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCondLock &)0x%p.waitTimeout, task:0x%p\n", this, &task );
#endif // __U_DEBUG_H__
    if ( task.entryRef.listed() ) {			// is task on queue
	waiting.remove( &(task.entryRef) );		// remove task at head of waiting list
	h.timedout = true;
	spin.release();
	task.ownerLock->_add( task );			// restart it or chain to its owner lock
    } else {
	spin.release();
    } // if
} // uCondLock::waitTimeout

void uCondLock::signal() {
    spin.acquire();
    if ( waiting.empty() ) {				// signal on empty condition is no-op
	spin.release();
	return;
    } // if
    uBaseTask &task = waiting.dropHead()->task();	// remove task at head of waiting list
    spin.release();
    task.ownerLock->_add( task );			// restart it or chain to its owner lock
} // uCondLock::signal

void uCondLock::broadcast() {
    // It is impossible to chain the entire waiting list to the associated
    // owner lock because each wait can be on a different owner lock. Hence,
    // each task has to be individually processed to move it onto the correct
    // owner lock.

    uQueue<uBaseTaskDL> temp;
    spin.acquire();
    uQueue<uBaseTaskDL>::transfer( temp, waiting );
    spin.release();
    while ( ! temp.empty() ) {
	uBaseTask &task = temp.dropHead()->task();	// remove task at head of waiting list
	task.ownerLock->_add( task );			// restart it or chain to its owner lock
    } // while
} // uCondLock::broadcast


//######################### Real-Time #########################


void uCxtSwtchHndlr::uHandler() {
    // Do not use yield here because it polls for async events. Async events
    // cannot be delivered because there is a signal handler stack frame on the
    // current stack, and it is unclear what the semantics are for abnormally
    // terminating that frame.

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uCxtSwtchHndlr &)0x%p.uHandler yield task:0x%p\n", this, &uThisTask() );
#endif // __U_DEBUG_H__
    uThisTask().uYieldInvoluntary();
} // uCxtSwtchHndlr::uHandler


uTimeoutHndlr::uTimeoutHndlr( uBaseTask &task, uSerial &serial ) : serial( serial ) {
    This = &task;
} // uTimeoutHndlr::uTimeoutHndlr

uTimeoutHndlr::uTimeoutHndlr( uSerial &serial ) : serial( serial ) {
    This = NULL;
} // uTimeoutHndlr::uTimeoutHndlr

void uTimeoutHndlr::uHandler() {
    serial.enterTimeout();
} // uTimeoutHndlr::uHandler


uCondLockTimedwaitHndlr::uCondLockTimedwaitHndlr( uBaseTask &task, uCondLock &condlock ) : condlock( condlock ) {
    This = &task;
    timedout = false;
} // uCondLockTimedwaitHndlr::uCondLockTimedwaitHndlr

uCondLockTimedwaitHndlr::uCondLockTimedwaitHndlr( uCondLock &condlock ) : condlock( condlock ) {
    This = NULL;
    timedout = false;
} // uCondLockTimedwaitHndlr::uCondLockTimedwaitHndlr

void uCondLockTimedwaitHndlr::uHandler() {
    condlock.waitTimeout( *This, *this );
} // uCondLockTimedwaitHndlr::uHandler


uBaseTask &uBaseScheduleFriend::getInheritTask( uBaseTask &task ) const {
    return task.getInheritTask();
} // uBaseScheduleFriend::getInheritTask

int uBaseScheduleFriend::getActivePriority( uBaseTask &task ) const {
    // special case for base of active priority stack
    return task.getActivePriority();
} // uBaseScheduleFriend::getActivePriority

int uBaseScheduleFriend::getActivePriorityValue( uBaseTask &task ) const {
    return task.getActivePriorityValue();
} // uBaseScheduleFriend::getActivePriorityValue

int uBaseScheduleFriend::setActivePriority( uBaseTask &task1, int priority ) {
    return task1.setActivePriority( priority );
} // uBaseScheduleFriend::setActivePriority

int uBaseScheduleFriend::setActivePriority( uBaseTask &task1, uBaseTask &task2 ) {
    return task1.setActivePriority( task2 );
} // uBaseScheduleFriend::setActivePriority

int uBaseScheduleFriend::getBasePriority( uBaseTask &task ) const {
    return task.getBasePriority();
} // uBaseScheduleFriend::getBasePriority

int uBaseScheduleFriend::setBasePriority( uBaseTask &task, int priority ) {
    return task.setBasePriority( priority );
} // uBaseScheduleFriend::setBasePriority

int uBaseScheduleFriend::getActiveQueueValue( uBaseTask &task ) const {
    return task.getActiveQueueValue();
} // uBaseScheduleFriend::getActiveQueueValue

int uBaseScheduleFriend::setActiveQueue( uBaseTask &task1, int q ) {
    return task1.setActiveQueue( q );
} // uBaseScheduleFriend::setActiveQueue

int uBaseScheduleFriend::getBaseQueue( uBaseTask &task ) const {
    return task.getBaseQueue();
} // uBaseScheduleFriend::getBaseQueue

int uBaseScheduleFriend::setBaseQueue( uBaseTask &task, int q ) {
    return task.setBaseQueue( q );
} // uBaseScheduleFriend::setBaseQueue

bool uBaseScheduleFriend::isEntryBlocked( uBaseTask &task ) const {
    return task.entryRef.listed();
} // uBaseScheduleFriend::isEntryBlocked

bool uBaseScheduleFriend::checkHookConditions( uBaseTask &task1, uBaseTask &task2 ) const {
    return task2.getSerial().checkHookConditions( &task1 );
} // uBaseScheduleFriend::checkHookConditions


//######################### uBasePrioritySeq #########################


uBasePrioritySeq::uBasePrioritySeq() {
    executeHooks = false;
} // uBasePrioritySeq::uBasePrioritySeq

bool uBasePrioritySeq::empty() const {
    return list.empty();
} // uBasePrioritySeq::empty

uBaseTaskDL *uBasePrioritySeq::head() const {
    return list.head();
} // uBasePrioritySeq::head

int uBasePrioritySeq::add( uBaseTaskDL *node, uBaseTask *uOwner ) {
    list.addTail( node );
    return 0;
} // uBasePrioritySeq::add

uBaseTaskDL *uBasePrioritySeq::drop() {
    return list.dropHead();
} // uBasePrioritySeq::drop

void uBasePrioritySeq::remove( uBaseTaskDL *node ) {
    list.remove( node );
} // uBasePrioritySeq::remove

void uBasePrioritySeq::onAcquire( uBaseTask &uOwner ) {
} // uBasePrioritySeq::onAcquire

void uBasePrioritySeq::onRelease( uBaseTask &uOldOwner ) {
} // uBasePrioritySeq::onRelease

int uBasePrioritySeq::reposition( uBaseTask &task, uSerial &serial ) {
    remove( &(task.entryRef) );				// remove from entry queue
    task.calledEntryMem->remove( &(task.mutexRef) );	// remove from mutex queue
    
    // Call cluster routine to adjust ready queue and active priority as owner
    // is not on entry queue, it can be updated based on its uPIQ.

    uThisCluster().taskSetPriority( task, task );
    
    task.calledEntryMem->add( &(task.mutexRef), serial.mutexOwner ); // add to mutex queue
    return add( &(task.entryRef), serial.mutexOwner );	// add to entry queue, automatically does transitivity
} // uBasePrioritySeq::reposition


//######################### uBasePriorityQueue #########################


bool uBasePriorityQueue::empty() const {
    return list.empty();
} // uBasePriorityQueue::empty

uBaseTaskDL *uBasePriorityQueue::head() const {
    return list.head();
} // uBasePriorityQueue::head

int uBasePriorityQueue::add( uBaseTaskDL *node, uBaseTask *uOwner ) {
    list.add( node );
    return 0;						// dummy value
} // uBasePriorityQueue::add

uBaseTaskDL *uBasePriorityQueue::drop() {
    return list.drop();
} // uBasePriorityQueue::drop

void uBasePriorityQueue::remove( uBaseTaskDL *node ) {
    // Only used with default FIFO case, so node to remove is at the front of
    // the list.
    list.drop();
} // uBasePriorityQueue::remove

void uBasePriorityQueue::onAcquire( uBaseTask &uOwner ) {
} // uBasePriorityQueue::onAcquire

void uBasePriorityQueue::onRelease( uBaseTask &uOldOwner ) {
} // uBasePriorityQueue::onRelease



//######################### uRepositionEntry #########################


uRepositionEntry::uRepositionEntry( uBaseTask &blocked, uBaseTask &calling ) :
	blocked( blocked ), calling( calling ), bSerial( blocked.getSerial() ), cSerial( calling.getSerial() ) {
} // uRepositionEntry::uRepositionEntry

int uRepositionEntry::uReposition( bool relCallingLock ) {
    int relPrevLock = 0;
    
    bSerial.lock.acquire();
    
    // If owner's current mutex object changes, then owner fixes its own active
    // priority. Recheck if inheritance is necessary as only owner can lower
    // its priority => updated.

    if ( &bSerial != &blocked.getSerial() || ! blocked.entryRef.listed() ||
	 blocked.uPIQ->getHighestPriority() >= blocked.getActivePriorityValue() ) {
	// As owner restarted, the end of the blocking chain has been reached.
	bSerial.lock.release();
	return relPrevLock;
    } // if

    if ( relCallingLock ) {
	// release the old lock as correct current lock is acquired
	cSerial.lock.release();
	relPrevLock = 1;
    } // if

    if ( bSerial.entryList.reposition( blocked, bSerial ) == 0 ) {
	// only last call does not release lock, so reacquire first entry lock
	if ( relCallingLock == true ) uThisTask().getSerial().lock.acquire();
	bSerial.lock.release();
    } // if

    // The return value is based on the release of cSerial.lock not
    // bSerial.lock.  The return value from bSerial is processed in the if
    // statement above, so it does not need to be propagated.

    return relPrevLock;
} // uRepositionEntry::uReposition


//######################### uDefaultScheduler #########################


bool uDefaultScheduler::empty() const {
    return list.empty();
} // uDefaultScheduler::empty

void uDefaultScheduler::add( uBaseTaskDL *taskNode ) {
    list.addTail( taskNode );
} // uDefaultScheduler::add

uBaseTaskDL *uDefaultScheduler::drop() {
    return list.dropHead();
} // uDefaultScheduler::drop

bool uDefaultScheduler::checkPriority( uBaseTaskDL &, uBaseTaskDL & ) { return false; }

void uDefaultScheduler::resetPriority( uBaseTaskDL &, uBaseTaskDL & ) {}

void uDefaultScheduler::addInitialize( uBaseTaskSeq & ) {};

void uDefaultScheduler::removeInitialize( uBaseTaskSeq & ) {};

void uDefaultScheduler::rescheduleTask( uBaseTaskDL *, uBaseTaskSeq & ) {};


//######################### uKernelModule #########################


void *uKernelModule::interposeSymbol( const char *symbolName, const char *version ) {
    static void *library;
    const char *error;
    void *originalFunc;
    if ( library == NULL ) {
#if defined( RTLD_NEXT )
	library = RTLD_NEXT;
#else
	// missing RTLD_NEXT => must hard-code library name, assuming libstdc++
	library = dlopen( "libstdc++.so", RTLD_LAZY );
	error = dlerror();
	if ( error != NULL ) {
	    fprintf( stderr, "uKernelModule::interposeSymbol : internal error, %s\n", error );
	    _exit( -1 );
	} // if
#endif // RTLD_NEXT
    } // if
#if defined( __linux__ )
    if( version == NULL ) {
#endif // __linux__
	originalFunc = dlsym( library, symbolName );
#if defined( __linux__ )
    } else {
	originalFunc = dlvsym( library, symbolName, version );
    } // if
#endif // __linux__
    error = dlerror();
    if ( error != NULL ) {
	fprintf( stderr, "uKernelModule::interposeSymbol : internal error, %s\n", error );
	_exit( -1 );
    } // if
    return originalFunc;
} // uKernelModule::interposeSymbol


void uKernelModule::startup() {
    uKernelModule::kernelModuleInitialized = true;
    volatile uKernelModule::uKernelModuleData *km;

#if defined( __irix__ ) && defined( __U_MULTI__ )
    // the user part of the PRDA holds the kernel module
    km = (uKernelModule::uKernelModuleData *)&(PRDA->usr_prda);
#elif defined( __ia64__ ) && defined( __linux__ ) && defined( __U_MULTI__ )
#if defined( __U_TLS__ ) && ! defined( __INTEL_COMPILER )
    asm volatile ("addl %0 = @ltoff(@tprel(_ZN13uKernelModule17uKernelModuleBootE#)), gp;;\n"
                  "ld8 %0 = [%0];;\n"
                  "add %0 = %0, r13;;\n" : "=r" (km) );
#else
    km = &uKernelModule::uKernelModuleBoot;
#endif // __U_TLS__
#elif defined( __i386__ ) && defined( __linux__ ) && defined( __U_MULTI__ )
#if defined( __U_TLS__ )
    asm volatile ("movl %%gs:0,%0\n"
		  "leal _ZN13uKernelModule17uKernelModuleBootE@ntpoff(%0),%0"
		  : "=r" (km) );
#else
    km = &uKernelModule::uKernelModuleBoot;
    km->threadPointer = (unsigned long)km;
    km->ldtValue = uProcessor::allocLDT();
#endif // __U_TLS__
#elif defined( __x86_64__ ) && defined( __linux__ ) && defined( __U_MULTI__ )
    asm volatile ("movq %%fs:0,%0\n"
		  "leaq _ZN13uKernelModule17uKernelModuleBootE@tpoff(%0),%0"
		  : "=r" (km) );
#elif defined( __sparc__ ) && defined( __U_TLS__ ) && defined( __U_MULTI__ )
    asm volatile ("sethi %%tle_hix22(_ZN13uKernelModule17uKernelModuleBootE),%0\n\t"
		  "xor %0, %%tle_lox10(_ZN13uKernelModule17uKernelModuleBootE), %0\n\t"
		  "add %%g7, %0, %0" : "=r" (km) );
#elif defined( __i386__ ) && defined( __freebsd__ )
    km = &uKernelModule::uKernelModuleBoot;
    km->threadPointer = (unsigned long)km;
#else
    // use statically allocated kernel module
    km = &uKernelModule::uKernelModuleBoot;
#endif

    km->ctor();
} // uKernelModule::startup


void uKernelModule::uKernelModuleData::ctor() volatile {
    This = this;
    kernelModuleInitialized = true;

    activeProcessor = (uProcessor *)&uKernelModule::systemProcessorStorage;
    activeCluster = (uCluster *)&uKernelModule::systemClusterStorage;
    activeTask = (uBaseTask *)&bootTaskStorage;

    disableInt = true;
    disableIntCnt = 1;

    disableIntSpin = false;
    disableIntSpinCnt = 0;

    inKernelRF = 0;

#if defined( __U_MULTI__ )
    // set private memory pointer
#if defined( __linux__ )
#if defined( __i386__ )
#if ! defined( __U_TLS__ )
    int ldtIndex = ( ldtValue - 7 ) / 8;
    struct modify_ldt_ldt_s ldtEntry =
    {
    	ldtIndex,
    	(unsigned long int) threadPointer,
    	0xfffff /* 4GB in pages */,
    	1, 0, 0, 1, 0, 1 //, 0
    };
    if ( modify_ldt( 1, &ldtEntry, sizeof(ldtEntry) ) != 0 ) {
	uAbort( "(uKernelModule &)0x%p.ctor() : internal error, modify_ldt.", this );
    } // if
    asm volatile ( "movw %w0, %%gs" : : "q" (ldtValue) );
#endif // ! __U_TLS__

#elif defined( __ia64__ )
#if defined( __U_TLS__ )
#ifdef __INTEL_COMPILER
    threadPointer = __getReg( _IA64_REG_TP );
#else
    register volatile uKernelModule *thread_self asm( "r13" );
    threadPointer = (unsigned long)thread_self;
#endif // __INTEL_COMPILER
#else
    threadPointer = (unsigned long)&IA64OffsetStoreFlag;
    IA64OffsetStoreFlag = 0;
    /* Register r13 (tp) is reserved by the ABI as "thread pointer". */
    asm volatile ("mov r13=%0" : : "r" (&IA64OffsetStoreFlag));
#endif // __U_TLS__

#elif defined( __x86_64__ )
    // assume TLS -- nothing to do

#else
    #error uC++ : internal error, unsupported architecture
#endif

#elif defined( __freebsd__ )
#if defined( __i386__ )
    struct segment_descriptor sd = {
	sizeof( uKernelModuleData ) & 0xffff,	/* segment extent (lsb) */
	(unsigned)this & 0xffffff,		/* segment base address (lsb)  */
	SDT_MEMRW,				/* segment type */
	3,					/* segment descriptor priority level */
	1,					/* segment descriptor present */
	sizeof( uKernelModuleData ) >> 16,	/* segment extent (msb) */
	0,					/* unused */
	1,					/* default 32 vs 16 bit size */
	0,					/* limit granularity (byte/page units)*/
	(unsigned)this >> 24			/* segment base address  (msb) */
    };
    int ldtIndex = i386_set_ldt( LDT_AUTO_ALLOC, (union descriptor*)&sd, 1 );
    if ( ldtIndex == -1 ) {
	uAbort( "(uKernelModule &)0x%p.ctor() : internal error, i386_set_ldt: (%d) %s.", this, errno, strerror( errno ) );
    } // if
    ldtValue = (ldtIndex << 3) | 7;
    asm ( "mov %0, %%gs" : : "rm" (ldtValue) );
#else
    #error uC++ : internal error, unsupported architecture
#endif // __i386__

#elif defined( __solaris__ )
#if ! defined( __U_TLS__ )
    _lwp_setprivate( (void *)this );
#endif // __U_TLS__

#elif defined( __irix__ )

#else
    #error uC++ : internal error, unsupported architecture
#endif
#endif // __U_MULTI__
} // uKernelModule::ctor


void uKernelModule::processEvents( uEventList *events, bool inKernel ) {
#ifdef __U_DEBUG_H__
    char buffer[256];
    int debugCnt = 0;
#endif // __U_DEBUG_H__
    uEventNode *event;

    for ( uEventListPop iter( *events, inKernel ); iter >> event; ) {
#ifdef __U_DEBUG_H__
	debugCnt += 1;
	uDebugPrtBuf( buffer, "processEvents, pop #%d\n", debugCnt );
#endif // __U_DEBUG_H__
    } // for
} // uKernelModule::processEvents


void uKernelModule::rollForward( bool inKernel ) {
#ifdef __U_DEBUG_H__
    char buffer[256];
    uDebugPrtBuf( buffer, "rollForward( %d ), disableInt:%d, disableIntCnt:%d, disableIntSpin:%d, disableIntSpinCnt:%d, inKernelRF:%d\n",
		  inKernel, THREAD_GETMEM(disableInt), THREAD_GETMEM(disableIntCnt), THREAD_GETMEM(disableIntSpin), THREAD_GETMEM(disableIntSpinCnt), THREAD_GETMEM(inKernelRF) );
#endif // __U_DEBUG_H__

#if defined( __U_ONETIMER__ ) && defined( __U_MULTI__ )
    if ( &uThisProcessor() == uKernelModule::systemProcessor ) {
	processEvents( uThisProcessor().events, true );
    } // if
    processEvents( uThisProcessor().procEvents, inKernel );
#else
    processEvents( uThisProcessor().events, inKernel );
#endif // __U_ONETIMER__ && __U_MULTI__

#ifdef __U_DEBUG_H__
    uDebugPrtBuf( buffer, "rollForward, leaving, inKernelRF:%d\n", THREAD_GETMEM( inKernelRF) );
#endif // __U_DEBUG_H__
} // rollForward


//######################### Translator Generated Definitions #########################


uCoroutineConstructor::uCoroutineConstructor( uAction f, uSerial &serial, uBaseCoroutine &coroutine, const char *name ) {
    if ( f == uYes ) {
	coroutine.startHere( (void (*)( uMachContext & ))uMachContext::invokeCoroutine );
	coroutine.name = name;
	coroutine.serial = &serial;			// set cormonitor's serial instance

	if ( uThisTask().profileActive && uProfiler::uProfiler_registerCoroutine && // profiling & coroutine registered for profiling ?
	     dynamic_cast<uProcessorKernel *>(&coroutine) == NULL ) { // and not kernel coroutine
	    (*uProfiler::uProfiler_registerCoroutine)( uProfiler::profilerInstance, coroutine, serial );
	} // if
    } // if
} // uCoroutineConstructor::uCoroutineConstructor


uCoroutineDestructor::uCoroutineDestructor( uAction f, uBaseCoroutine &coroutine ) : f( f ), coroutine( coroutine ) {
} // uCoroutineDestructor::uCoroutineDestructor

uCoroutineDestructor::~uCoroutineDestructor() {
    if ( f == uYes ) {
	if ( uThisTask().profileActive && uProfiler::uProfiler_deregisterCoroutine ) { // profiling this coroutine & coroutine registered for profiling ? 
	    (*uProfiler::uProfiler_deregisterCoroutine)( uProfiler::profilerInstance, coroutine );
	} // if
    } // if

    // Clean up the stack of a non-terminated coroutine (i.e., run its
    // destructors); a terminated coroutine's stack is already cleaned
    // up. Ignore the uProcessorKernel coroutine because it has a special
    // shutdown sequence.
    if ( coroutine.getState() != uBaseCoroutine::Halt	// coroutine not halted
	 && &coroutine.resumer() != NULL		// and its main is started
	 && dynamic_cast<uProcessorKernel *>(&coroutine) == NULL ) { // but not the processor Kernel
	// Mark for cancellation, then resume the coroutine to trigger a call
	// to uPoll on the backside of its suspend(). uPoll detects the
	// cancellation and calls unwind_stack, which throws exception
	// UnwindStack to unwinding the stack. UnwindStack is ultimately caught
	// inside uMachContext::uInvokeCoroutine.
	coroutine.cancel();
	coroutine.resume();
    } // if
} // uCoroutineDestructor::~uCoroutineDestructor


uCoroutineMain::uCoroutineMain( uBaseCoroutine &coroutine ) : coroutine( coroutine ) {
    // also appears in uBaseCoroutine::uContextSw2
    if ( uThisTask().profileActive && uProfiler::uProfiler_registerCoroutineUnblock ) {
	if ( THREAD_GETMEM( disableInt ) ) return;	// ignore profiling for kernel coroutine
	(*uProfiler::uProfiler_registerCoroutineUnblock)( uProfiler::profilerInstance, uThisTask() );
    } // if
} // uCoroutineMain::uCoroutineMain

uCoroutineMain::~uCoroutineMain( ) {			// necessary for KDB
} // uCoroutineMain::uCoroutineMain


uSerial::uSerial( uBasePrioritySeq &entryList ) : entryList( entryList ) {
    mask.clrAll();					// mutex members start closed
    mutexOwner = &uThisTask();				// set the current mutex owner to the creating task

    // Make creating task the owner of the mutex.
    prevSerial = &mutexOwner->getSerial();		// save previous serial
    mutexOwner->setSerial( *this );			// set new serial
    mr = mutexOwner->mutexRecursion;			// save previous recursive count
    mutexOwner->mutexRecursion = 0;			// reset recursive count

    acceptMask = false;
    mutexMaskPosn = NULL;

    destructorTask = NULL;  
    destructorStatus = NoDestructor;
    constructorTask = mutexOwner;

    // real-time

    uTimeoutEvent.executeLocked = true;
    events = NULL;

    // exception handling

    lastAcceptor = NULL;
    alive = true;

    // profiling

    profileSerialSamplerInstance = NULL;
} // uSerial::uSerial

uSerial::~uSerial() {
    alive = false;					// no more entry calls can be accepted
    uBaseTask &task = uThisTask();			// optimization
    task.setSerial( *prevSerial );			// reset previous serial

    for ( ;; ) {
	uBaseTaskDL *p = acceptSignalled.drop();
      if ( p == NULL ) break;
	mutexOwner = &(p->task());
	_Throw EntryFailure( this, "blocked on acceptor/signalled stack" ) _At *(mutexOwner->currCoroutine);
	acceptSignalled.add( &(task.mutexRef) );	// suspend current task on top of accept/signalled stack
	uSCHEDULE( mutexOwner );
    } // for

    if ( ! entryList.empty() ) {			// no need to acquire the lock if the queue is empty
	for ( ;; ) {
	    lock.acquire();
	    uBaseTaskDL *p = entryList.drop();
	  if ( p == NULL ) break;
	    mutexOwner = &(p->task());
	    mutexOwner->calledEntryMem->remove( &(mutexOwner->mutexRef) );
	    _Throw EntryFailure( this, "blocked on entry queue" ) _At *(mutexOwner->currCoroutine);
	    acceptSignalled.add( &(task.mutexRef) );	// suspend current task on top of accept/signalled stack
	    uSCHEDULE( &lock, mutexOwner );
	} // for
	lock.release();
    } // if
} // uSerial::~uSerial

void uSerial::resetDestructorStatus() {
    destructorStatus = NoDestructor;
    destructorTask = NULL;
} // uSerial::rresetDestructorStatus

void uSerial::enter( unsigned int &mr, uBasePrioritySeq &ml, int mp ) {
    uBaseTask &task = uThisTask();			// optimization
    lock.acquire();

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerial &)0x%p.enter, mask:0x%x,0x%x,0x%x,0x%x, owner:0x%p, ml:0x%p, mp:%d\n",
	       this, mask[0], mask[1], mask[2], mask[3], mutexOwner, &ml, mp );
#endif // __U_DEBUG_H__
    if ( mask.isSet( mp ) ) {				// member acceptable ?
	mask.clrAll();					// clear the mask
	mr = task.mutexRecursion;			// save previous recursive count
	task.mutexRecursion = 0;			// reset recursive count
	mutexOwner = &task;				// set the current mutex owner
	if ( entryList.executeHooks ) {
	    // always execute hook as calling task cannot be constructor or destructor
	    entryList.onAcquire( *mutexOwner );	// perform any priority inheritance
	} // if
	lock.release();
    } else if ( mutexOwner == &task ) {			// already hold mutex ?
	task.mutexRecursion += 1;			// another recursive call at the mutex object level
	lock.release();
    } else {						// otherwise block the calling task
	ml.add( &(task.mutexRef), mutexOwner );		// add to end of mutex queue
	task.calledEntryMem = &ml;			// remember which mutex member called
	entryList.add( &(task.entryRef), mutexOwner );	// add mutex object to end of entry queue
	uSCHEDULE( &lock );				// find someone else to execute; release lock on kernel stack
	mr = task.mutexRecursion;			// save previous recursive count
	task.mutexRecursion = 0;			// reset recursive count
	_Enable <uSerial::Failure>;			// implicit poll
    } // if
    if ( mutexMaskPosn != NULL ) *mutexMaskPosn = mp;	// set active mutex member
} // uSerial::enter

// enter routine for destructor, does not poll
void uSerial::enterDestructor( unsigned int &mr, uBasePrioritySeq &ml, int mp ) {
    uBaseTask &task = uThisTask();			// optimization

    if ( destructorStatus != NoDestructor ) {		// only one task is allowed to call destructor
	uAbort( "Attempt by task %.256s (0x%p) to call the destructor for uSerial 0x%p, but this destructor was already called by task %.256s (0x%p).\n"
		"Possible cause is multiple tasks simultaneously deleting a mutex object.",
		task.getName(), &task, this, destructorTask->getName(), destructorTask );
    } // if

    lock.acquire();

    destructorStatus = DestrCalled;
    destructorTask = &task;

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerial &)0x%p.enterDestructor, mask:0x%x,0x%x,0x%x,0x%x, owner:0x%p, ml:0x%p, mp:%d\n",
	       this, mask[0], mask[1], mask[2], mask[3], mutexOwner, &ml, mp );
#endif // __U_DEBUG_H__
    if ( mask.isSet( mp ) ) {				// member acceptable ?
	mask.clrAll();					// clear the mask
	mr = task.mutexRecursion;			// save previous recursive count
	task.mutexRecursion = 0;			// reset recursive count
	mutexOwner = &task;				// set the current mutex owner
	destructorStatus = DestrScheduled;
	// hook is not executed for destructor
	lock.release();
    } else if ( mutexOwner == &task ) {			// already hold mutex ?
	uAbort( "Attempt by task %.256s (0x%p) to call the destructor for uSerial 0x%p, but this task has outstanding nested calls to this mutex object.\n"
		"Possible cause is deleting a mutex object with outstanding nested calls to one of its members.",
		task.getName(), &task, this );
    } else {						// otherwise block the calling task
	task.calledEntryMem = &ml;			// remember which mutex member was called
	uSCHEDULE( &lock );				// find someone else to execute; release lock on kernel stack
	mr = task.mutexRecursion;			// save previous recursive count
	task.mutexRecursion = 0;			// reset recursive count
    } // if
    if ( mutexMaskPosn != NULL ) *mutexMaskPosn = mp;	// set active mutex member
} // uSerial::enterDestructor

void uSerial::enterTimeout() {
    // This monitor member is called from the kernel, and therefore, cannot
    // block, but it can spin.

    lock.acquire();

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerial &)0x%p.enterTimeout, mask:0x%x,0x%x,0x%x,0x%x, owner:0x%p, mutexMaskPosn:0x%p\n",
	       this, mask[0], mask[1], mask[2], mask[3], mutexOwner, mutexMaskPosn );
#endif // __U_DEBUG_H__
    if ( mask.isSet( __U_TIMEOUTPOSN__ ) ) {		// timeout member acceptable ?  0 => timeout mask bit
	mask.clrAll();					// clear the mask
	*mutexMaskPosn = 0;				// set timeout mutex member  0 => timeout mask bit
	mutexOwner = &(acceptSignalled.drop()->task()); // next task to gain control of the mutex object
	
	// priority-inheritance, bump up priority of mutexowner from head
	// of prioritized entry queue (NOT leaving task), because suspended
	// stack is not prioritized.
	
	if ( entryList.executeHooks && checkHookConditions( mutexOwner ) ) {
	    entryList.onAcquire( *mutexOwner );
	} // if

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSerial &)0x%p.enterTimeout, waking task %.256s (0x%p) \n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
	mutexOwner->wake();				// wake up next task to use this mutex object
    } // if

    lock.release();
} // uSerial::enterTimeout

// leave and leave2 do not poll for concurrent exceptions because they are
// called in some critical destructors.  Throwing an exception out of these
// destructors causes problems.

void uSerial::leave( unsigned int mr ) {		// used when a task is leaving a mutex and has not queued itself before calling
    uBaseTask &task = uThisTask();			// optimization

    if ( task.mutexRecursion != 0 ) {			// already hold mutex ?
	if ( acceptMask ) {
	    // lock is acquired and mask set by accept statement
	    acceptMask = false;
	    lock.release();
	} // if
	task.mutexRecursion -= 1;
    } else {
	if ( acceptMask ) {
	    // lock is acquired and mask set by accept statement
	    acceptMask = false;
	    mutexOwner = NULL;				// reset no task in mutex object
	    if ( entryList.executeHooks && checkHookConditions( &task )  ) {
		entryList.onRelease( task );
	    } // if
	    if ( &task == destructorTask ) resetDestructorStatus();
	    lock.release();
	} else if ( acceptSignalled.empty() ) {		// no tasks waiting re-entry to mutex object ?
	    lock.acquire();
	    if ( destructorStatus != DestrCalled ) {
		if ( entryList.empty() ) {		// no tasks waiting entry to mutex object ?
		    mask.setAll();			// accept all members
		    mask.clr( 0 );			// except timeout
		    mutexOwner = NULL;			// reset no task in mutex object
		    if ( entryList.executeHooks && checkHookConditions( &task )  ) {
			entryList.onRelease( task );
		    } // if
		    if ( &task == destructorTask ) resetDestructorStatus();
		    lock.release();
		} else {				// tasks wating entry to mutex object
		    mutexOwner = &(entryList.drop()->task()); // next task to gain control of the mutex object
		    mutexOwner->calledEntryMem->remove( &(mutexOwner->mutexRef) ); // also remove task from mutex queue
		    if ( entryList.executeHooks ) {
			if ( checkHookConditions( &task ) ) entryList.onRelease( task );
			if ( checkHookConditions( mutexOwner ) ) entryList.onAcquire( *mutexOwner );
		    } // if
		    if ( &task == destructorTask ) resetDestructorStatus();
		    lock.release();
#ifdef __U_DEBUG_H__
		    uDebugPrt( "(uSerial &)0x%p.leave, waking task %.256s (0x%p)\n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
		    mutexOwner->wake();			// wake up next task to use this mutex object
		} // if
	    } else {
		mutexOwner = destructorTask;
		destructorStatus = DestrScheduled;
		if ( entryList.executeHooks ) {
		    if ( checkHookConditions( &task ) ) entryList.onRelease( task );
		    // do not call acquire the hook for the destructor
		} // if
		lock.release();
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uSerial &)0x%p.leave, waking task %.256s (0x%p)\n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
		mutexOwner->wake();			// wake up next task to use this mutex object
	    } // if
	} else {
	    // priority-inheritance, bump up priority of mutexowner from head
	    // of prioritized entry queue (NOT leaving task), because suspended
	    // stack is not prioritized.

	    if ( entryList.executeHooks ) {
		lock.acquire();				// acquire entry lock to prevent inversion during transfer 
		mutexOwner = &(acceptSignalled.drop()->task()); // next task to gain control of the mutex object
		if ( checkHookConditions( &task ) ) entryList.onRelease( task );
		if ( checkHookConditions( mutexOwner ) ) entryList.onAcquire( *mutexOwner );
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uSerial &)0x%p.leave, waking task %.256s (0x%p)\n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
		mutexOwner->wake();			// wake up next task to use this mutex object
		if ( &task == destructorTask ) resetDestructorStatus();
		lock.release();
	    } else {
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uSerial &)0x%p.leave, waking task %.256s (0x%p)\n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
		mutexOwner = &(acceptSignalled.drop()->task()); // next task to gain control of the mutex object
		if ( &task == destructorTask ) {
		    lock.acquire();
		    resetDestructorStatus();
		    lock.release();
		} // if
		mutexOwner->wake();			// wake up next task to use this mutex object
	    } // if
	} // if
	task.mutexRecursion = mr;			// restore previous recursive count
    } // if

#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerial &)0x%p.leave, mask:0x%x,0x%x,0x%x,0x%x, owner:0x%p\n", this, mask[0], mask[1], mask[2], mask[3], mutexOwner );
#endif // __U_DEBUG_H__
} // uSerial::leave

void uSerial::leave2() {				// used when a task is leaving a mutex and has queued itself before calling
    uBaseTask &task = uThisTask();			// optimization

    if ( acceptMask ) {
	// lock is acquired and mask set by accept statement
	acceptMask = false;
	mutexOwner = NULL;				// reset no task in mutex object
	if ( entryList.executeHooks && checkHookConditions( &task )  ) {
	    entryList.onRelease( task );
	} // if
	if ( &task == destructorTask ) resetDestructorStatus();
	uSCHEDULE( &lock );				// find someone else to execute; release lock on kernel stack
    } else if ( acceptSignalled.empty() ) {		// no tasks waiting re-entry to mutex object ?
	lock.acquire();
        if ( destructorStatus != DestrCalled ) {
	    if ( entryList.empty() ) {			// no tasks waiting entry to mutex object ?
		mask.setAll();				// accept all members
		mask.clr( 0 );				// except timeout
		mutexOwner = NULL;
		if ( entryList.executeHooks && checkHookConditions( &task ) ) {
		    entryList.onRelease( task );
		} // if
		uSCHEDULE( &lock );			// find someone else to execute; release lock on kernel stack
	    } else {
		mutexOwner = &(entryList.drop()->task()); // next task to gain control of the mutex object
		mutexOwner->calledEntryMem->remove( &(mutexOwner->mutexRef) ); // also remove task from mutex queue
		if ( entryList.executeHooks ) {
		    if ( checkHookConditions( &task ) ) entryList.onRelease( task );
		    if ( checkHookConditions( mutexOwner ) ) entryList.onAcquire( *mutexOwner );
		} // if
#ifdef __U_DEBUG_H__
		uDebugPrt( "(uSerial &)0x%p.leave2, waking task %.256s (0x%p)\n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
		uSCHEDULE( &lock, mutexOwner );		// find someone else to execute; release lock and wake on kernel stack
	    } // if
	} else {
	   mutexOwner = destructorTask;
	   destructorStatus = DestrScheduled;
	   if ( entryList.executeHooks ) {
	       if ( checkHookConditions( &task ) ) entryList.onRelease( task );
	       // do not call acquire the hook for the destructor
	   } // if
#ifdef __U_DEBUG_H__
	    uDebugPrt( "(uSerial &)0x%p.leave2, waking task %.256s (0x%p)\n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
	    uSCHEDULE( &lock, mutexOwner );		// find someone else to execute; release lock and wake on kernel stack
	} // if
    } else {
	// priority-inheritance, bump up priority of mutexowner from head of
	// prioritized entry queue (NOT leaving task), because suspended stack
	// is not prioritized.

#ifdef __U_DEBUG_H__
	uDebugPrt( "(uSerial &)0x%p.leave2, waking task %.256s (0x%p)\n", this, mutexOwner->getName(), mutexOwner );
#endif // __U_DEBUG_H__
	if ( entryList.executeHooks ) {
	    lock.acquire();
	    mutexOwner = &(acceptSignalled.drop()->task()); // next task to gain control of the mutex object
	    if ( checkHookConditions( &task ) ) entryList.onRelease( task );
	    if ( checkHookConditions( mutexOwner ) ) entryList.onAcquire( *mutexOwner );
	    uSCHEDULE( &lock, mutexOwner );		// find someone else to execute; release lock and wake on kernel stack
	} else {
	    mutexOwner = &(acceptSignalled.drop()->task()); // next task to gain control of the mutex object
	    uSCHEDULE( mutexOwner );			// find someone else to execute; wake on kernel stack
	} // if
    } // if
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerial &)0x%p.leave2, mask:0x%x,0x%x,0x%x,0x%x, owner:0x%p\n",
	       this, mask[0], mask[1], mask[2], mask[3], mutexOwner );
#endif // __U_DEBUG_H__
} // uSerial::leave2

void uSerial::removeTimeout() {
    if ( events != NULL ) {
	uTimeoutEvent.remove();
    } // if
} // uSerial::removeTimeout

bool uSerial::checkHookConditions( uBaseTask *task ) {
    return task != constructorTask && task != destructorTask;
} // uSerial::checkHookConditions


// The field uSerial::lastAcceptor is set in acceptTry and acceptPause and
// reset to NULL in uSerialMember::uSerialMember, so an exception can be thrown
// when all the guards in the accept statement fail.  This ensures that
// lastAcceptor is set only when a task rendezouvs with another task.

void uSerial::acceptStart( unsigned int &mutexMaskPosn ) {
    uBaseTask &task = uThisTask();			// optimization
#ifdef __U_DEBUG__
    if ( &task != mutexOwner ) {			// must have mutex lock to wait
	uAbort( "Attempt to accept in a mutex object not locked by this task.\n"
		"Possible cause is accepting in a nomutex member routine." );
    } // if
#endif // __U_DEBUG__

    if ( task.profileActive && uProfiler::uProfiler_registerAcceptStart ) { // task registered for profiling ?
	(*uProfiler::uProfiler_registerAcceptStart)( uProfiler::profilerInstance, *this, task );
    } // if

    uSerial::mutexMaskPosn = &mutexMaskPosn;
    acceptLocked = false;
} // uSerial::acceptStart

bool uSerial::acceptTry( uBasePrioritySeq &ml, int mp ) {
    if ( ! acceptLocked ) {				// lock is acquired on demand
	lock.acquire();
	mask.clrAll();
	acceptLocked = true;
    } // if
    if ( mp == __U_DESTRUCTORPOSN__ ) {			// ? destructor accepted
	// Handles the case where destructor has not been called or the
	// destructor has been scheduled.  If the destructor has been
	// scheduled, there is potential for synchronization deadlock when only
	// the destructor is accepted.
	if ( destructorStatus != DestrCalled ) {
	    mask.set( mp );				// add this mutex member to the mask
	    return false;				// the accept failed
	} else {
	    uBaseTask &task = uThisTask();		// optimization
	    mutexOwner = destructorTask;		// next task to use this mutex object
	    destructorStatus = DestrScheduled;		// change status of destructor to scheduled
	    lastAcceptor = &task;			// saving the acceptor thread of a rendezvous
	    mask.clrAll();				// clear the mask
	    acceptSignalled.add( &(task.mutexRef) );	// suspend current task on top of accept/signalled stack
	    if ( entryList.executeHooks ) {
		// no check for destructor because it cannot accept itself
		if ( checkHookConditions( &task ) ) entryList.onRelease( task );  
		// do not call the acquire hook for the destructor
	    } // if
	    uSCHEDULE( &lock, mutexOwner );		// find someone else to execute; release lock and wake on kernel stack
	    if ( task.acceptedCall ) {			// accepted entry is suspended if true
		task.acceptedCall->acceptorSuspended = false; // acceptor resumes
		task.acceptedCall = NULL;
	    } // if
	    _Enable <uSerial::RendezvousFailure>;	// implicit poll
	    return true;
	} // if
    } else {
	if ( ml.empty() ) {
	    mask.set( mp );				// add this mutex member to the mask
	    return false;				// the accept failed
	} else {
	    uBaseTask &task = uThisTask();		// optimization
	    mutexOwner = &(ml.drop()->task());		// next task to use this mutex object
	    lastAcceptor = &task;			// saving the acceptor thread of a rendezvous
	    entryList.remove( &(mutexOwner->entryRef) ); // also remove task from entry queue
	    mask.clrAll();				// clear the mask
	    acceptSignalled.add( &(task.mutexRef) );	// suspend current task on top of accept/signalled stack
	    if ( entryList.executeHooks ) {
		if ( checkHookConditions( &task ) ) entryList.onRelease( task );  
		if ( checkHookConditions( mutexOwner ) ) entryList.onAcquire( *mutexOwner );
	    } // if
	    uSCHEDULE( &lock, mutexOwner );		// find someone else to execute; release lock and wake on kernel stack
	    if ( task.acceptedCall ) {			// accepted entry is suspended if true
		task.acceptedCall->acceptorSuspended = false; // acceptor resumes
		task.acceptedCall = NULL;
	    } // if
	    _Enable <uSerial::RendezvousFailure>;	// implicit poll
	    return true;
	} // if
    } // if
} // uSerial::acceptTry

void uSerial::acceptTry() {
    if ( ! acceptLocked ) {				// lock is acquired on demand
	lock.acquire();
	mask.clrAll();
	acceptLocked = true;
    } // if
    mask.set( 0 );					// add this mutex member to the mask, 0 => timeout mask bit
} // uSerial::acceptTry

bool uSerial::acceptTry2( uBasePrioritySeq &ml, int mp ) {
    if ( ! acceptLocked ) {				// lock is acquired on demand
	lock.acquire();
	mask.clrAll();
	acceptLocked = true;
    } // if
    if ( mp == __U_DESTRUCTORPOSN__ ) {			// ? destructor accepted
	// Handles the case where destructor has not been called or the
	// destructor has been scheduled.  If the destructor has been
	// scheduled, there is potential for synchronization deadlock when only
	// the destructor is accepted.
	if ( destructorStatus != DestrCalled ) {
	    mask.set( mp );				// add this mutex member to the mask
	    return false;				// the accept failed
	} else {
	    uBaseTask *acceptedTask = destructorTask;	// next task to use this mutex object
	    destructorStatus = DestrScheduled;		// change status of destructor to scheduled
	    mask.clrAll();				// clear the mask
	    lock.release();
	    acceptSignalled.add( &(acceptedTask->mutexRef) ); // move accepted task on top of accept/signalled stack
	    return true;
	} // if
    } else {
	if ( ml.empty() ) {
	    mask.set( mp );				// add this mutex member to the mask
	    return false;				// the accept failed
	} else {
	    uBaseTask *acceptedTask = &(ml.drop()->task()); // next task to use this mutex object
	    entryList.remove( &(acceptedTask->entryRef) ); // also remove task from entry queue
	    mask.clrAll();				// clear the mask
	    lock.release();
            acceptSignalled.add( &(acceptedTask->mutexRef) ); // move accepted task on top of accept/signalled stack
	    return true;
	} // if
    } // if
} // uSerial::acceptTry2

bool uSerial::acceptTestMask() {
    return mask.isAllClr();
} // uSerial::acceptTestMask

void uSerial::acceptSetMask() {
    // The lock acquired at the start of the accept statement cannot be
    // released here, otherwise, it is necessary to recheck the mutex queues
    // before exit. As a consequence, all destructors between here and
    // ~uSerialMember (which executes leave) are executed with the mutex lock
    // closed, preventing tasks from queuing on this mutex object.
    acceptMask = true;
} // uSerial::acceptSetMask

void uSerial::acceptElse() {
    if ( acceptLocked ) {
	mask.clrAll();
	lock.release();
    } // if
} // uSerial::acceptElse

void uSerial::acceptPause() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerial &)0x%p.acceptPause, mask:0x%x,0x%x,0x%x,0x%x, owner:0x%p\n",
	       this, mask[0], mask[1], mask[2], mask[3], mutexOwner );
#endif // __U_DEBUG_H__
    // lock is acquired at beginning of accept statement
    uBaseTask &task = uThisTask();			// optimization
    lastAcceptor = &task;				// saving the acceptor thread of a rendezvous
    acceptSignalled.add( &(task.mutexRef) );		// suspend current task on top of accept/signalled stack
    mutexOwner = NULL;
    if ( entryList.executeHooks && checkHookConditions( &task ) ) {
	entryList.onRelease( task );
    } // if
    uSCHEDULE( &lock );					// find someone else to execute; release lock on kernel stack
    if ( task.acceptedCall ) {				// accepted entry is suspended if true
        task.acceptedCall->acceptorSuspended = false;	// acceptor resumes
	task.acceptedCall = NULL;
    } // if
    _Enable <uSerial::Failure>;				// implicit poll
} // uSerial::acceptPause

void uSerial::acceptPause( uDuration duration ) {
    acceptPause( activeProcessorKernel->kernelClock.getTime() + duration );
} // uSerial::acceptPause

void uSerial::acceptPause( uTime time ) {
    uBaseTask &task = uThisTask();			// optimization
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerial &)0x%p.acceptPause, mask:0x%x,0x%x,0x%x,0x%x, owner:0x%p\n",
	       this, mask[0], mask[1], mask[2], mask[3], mutexOwner );
#endif // __U_DEBUG_H__
    uTimeoutHndlr handler( task, *this );		// handler to wake up blocking task

    uTimeoutEvent.executeLocked = true;
    uTimeoutEvent.timerT = time;
    uTimeoutEvent.who = &task;
    uTimeoutEvent.SigHandler = &handler;

    uTimeoutEvent.add();

    // lock is acquired at beginning of accept statement
    uBaseTask &uCallingTask = task;			// optimization
    lastAcceptor = &uCallingTask;			// saving the acceptor thread of a rendezvous
    acceptSignalled.add( &(uCallingTask.mutexRef) );	// suspend current task on top of accept/signalled stack

    mutexOwner = NULL;
    if ( entryList.executeHooks && checkHookConditions( &uCallingTask ) ) {
	entryList.onRelease( uCallingTask );
    } // if
    uSCHEDULE( &lock );					// find someone else to execute; release lock on kernel stack

    uTimeoutEvent.remove();

    if ( uCallingTask.acceptedCall ) {			// accepted entry is suspended if true
        uCallingTask.acceptedCall->acceptorSuspended = false; // acceptor resumes
	uCallingTask.acceptedCall = NULL;
    } // if
    _Enable <uSerial::Failure>;				// implicit poll
} // uSerial::acceptPause

void uSerial::acceptEnd() {
    mutexMaskPosn = NULL;

    if ( uThisTask().profileActive && uProfiler::uProfiler_registerAcceptEnd ) { // task registered for profiling ?              
	(*uProfiler::uProfiler_registerAcceptEnd)( uProfiler::profilerInstance, *this, uThisTask() );
    } // if
} // uSerial::acceptEnd


uSerial::Failure::Failure( const uSerial *const serial, const char *const msg ) : uKernelFailure( msg ), serial( serial ) {}

uSerial::Failure::Failure( const char *const msg ) : uKernelFailure( msg ), serial( NULL ) {}

uSerial::Failure::~Failure() {}

const uSerial *uSerial::Failure::serialId() const { return serial; }

void uSerial::Failure::defaultTerminate() const {
    uAbort( "(uSerial &)0x%p : %.256s.", serialId(), message() );
} // uSerial::Failure::defaultTerminate


uSerial::EntryFailure::EntryFailure( const uSerial *const serial, const char *const msg ) : uSerial::Failure( serial, msg ) {}

uSerial::EntryFailure::EntryFailure( const char *const msg ) : uSerial::Failure( msg ) {}

uSerial::EntryFailure::~EntryFailure() {}

void uSerial::EntryFailure::defaultTerminate() const {
    uAbort( "(uSerial &)0x%p : Entry failure while executing mutex destructor, task %.256s (0x%p) found %.256s.",
	    serialId(), sourceName(), &source(), message() );
} // uSerial::EntryFailure::defaultTerminate


uSerial::RendezvousFailure::RendezvousFailure( const uSerial *const serial, const char *const msg ) : uSerial::Failure( serial, msg ), _caller( &uThisCoroutine() ) {}

uSerial::RendezvousFailure::~RendezvousFailure() {}

const uBaseCoroutine *uSerial::RendezvousFailure::caller() const { return _caller; }

void uSerial::RendezvousFailure::defaultTerminate() const {
    uAbort( "(uSerial &)0x%p : Rendezvous failure in %.256s from task %.256s (0x%p) to mutex member of task %.256s (0x%p).",
	    serialId(), message(), sourceName(), &source(), uThisTask().getName(), &uThisTask() );
} // uSerial::RendezvousFailure::defaultTerminate


uTaskConstructor::uTaskConstructor( uAction f, uSerial &serial, uBaseTask &task, uBasePIQ &piq, const char *n, bool profile ) : f( f ), serial( serial ), task( task ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uTaskConstructor &)0x%p.uTaskConstructor, f:%d, s:0x%p, t:0x%p, piq:0x%p, n:%s, profile:%d\n", this, f, &serial, &task, &piq, n, profile );
#endif // __U_DEBUG_H__
    if ( f == uYes ) {
	task.startHere( (void (*)( uMachContext & ))uMachContext::invokeTask );
	task.name = n;
	task.serial = &serial;				// set task's serial instance
	task.setSerial( serial );
	task.profileActive = profile;
	task.uPIQ = &piq;

	if ( task.profileActive && uProfiler::uProfiler_registerTask ) { // profiling this task & task registered for profiling ? 
	    (*uProfiler::uProfiler_registerTask)( uProfiler::profilerInstance, task, serial, uThisTask() );
	} // if

	serial.acceptSignalled.add( &(task.mutexRef) );

#if __U_LOCALDEBUGGER_H__
	if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif // __U_LOCALDEBUGGER_H__

	task.currCluster->taskAdd( task );		// add task to the list of tasks on this cluster
    } // if
} // uTaskConstructor::uTaskConstructor


uTaskConstructor::~uTaskConstructor() {
    if ( f == uYes && std::uncaught_exception() ) {
	// An exception was thrown during task construction. It is necessary to
	// clean up constructor side-effects.

	// Since no task could have been accepted, the acceptor/signaller stack
	// should only contain this task.  It must be removed here so that it
	// is not scheduled by ~uSerial.
	serial.acceptSignalled.drop();

	uTaskDestructor::cleanup( task );
    } // if
} // uTaskConstructor::~uTaskConstructor


uTaskDestructor::~uTaskDestructor() {
    if ( f == uYes ) {
#ifdef __U_DEBUG__
	if ( task.uBaseCoroutine::getState() != uBaseCoroutine::Halt ) {
	    uAbort( "Attempt to delete task %.256s (0x%p) that is not halted.\n"
		    "Possible cause is task blocked on a condition queue.",
		    task.getName(), &task );
	} // if
#endif // __U_DEBUG__

	cleanup( task );
    } // if
} // uTaskDestructor::uTaskDestructor


void uTaskDestructor::cleanup( uBaseTask &task ) {
#if __U_LOCALDEBUGGER_H__
    if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->checkPoint();
#endif // __U_LOCALDEBUGGER_H__

    task.profileActive = false;
    if ( task.profileTaskSamplerInstance && uProfiler::uProfiler_deregisterTask ) { // task registered for profiling ?
	(*uProfiler::uProfiler_deregisterTask)( uProfiler::profilerInstance, task );
    } // if

    task.currCluster->taskRemove( task );		// remove the task from the list of tasks that live on this cluster.
} // uTaskDestructor::cleanup


uTaskMain::uTaskMain( uBaseTask &task ) : task( task ) {
    // SKULLDUGGERY: To allow "main" to be treated as a normal member routine,
    // a counter is used to allow recursive entry.

    task.recursion += 1;
    if ( task.recursion == 1 ) {			// first call ?
	if ( task.profileActive && uProfiler::uProfiler_registerTaskStartExecution ) { 
	    (*uProfiler::uProfiler_registerTaskStartExecution)( uProfiler::profilerInstance, task ); 
	} // if
#if __U_LOCALDEBUGGER_H__
	// Registering a task with the global debugger must occur in this
	// routine for the register set to be correct.

	if ( uLocalDebugger::uLocalDebuggerActive ) {
	    uLocalDebugger::uLocalDebuggerInstance->createULThread();
	} // if
#endif // __U_LOCALDEBUGGER_H__
    } // if
} // uTaskMain::uTaskMain


uTaskMain::~uTaskMain() {
    task.recursion -= 1;
    if ( task.recursion == 0 ) {
	if ( task.profileActive && uProfiler::uProfiler_registerTaskEndExecution ) {
	    (*uProfiler::uProfiler_registerTaskEndExecution)( uProfiler::profilerInstance, task ); 
	} // if   
#if __U_LOCALDEBUGGER_H__
	if ( uLocalDebugger::uLocalDebuggerActive ) uLocalDebugger::uLocalDebuggerInstance->destroyULThread();
#endif // __U_LOCALDEBUGGER_H__
    } // if
} // uTaskMain::~uTaskMain


uSerialConstructor::uSerialConstructor( uAction f, uSerial &serial ) : f( f ), serial( serial ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerialConstructor &)0x%p.uSerialConstructor, f:%d, s:0x%p\n", this, f, &serial );
#endif // __U_DEBUG_H__
} // uSerialConstructor::uSerialConstructor

uSerialConstructor::uSerialConstructor( uAction f, uSerial &serial, const char *n ) : f( f ), serial( serial ) {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerialConstructor &)0x%p.uSerialConstructor, f:%d, s:0x%p, n:%s\n", this, f, &serial, n );
#endif // __U_DEBUG_H__
    if ( f == uYes ) {
	if ( uThisTask().profileActive && uProfiler::uProfiler_registerMonitor ) { // task registered for profiling ?
	    (*uProfiler::uProfiler_registerMonitor)( uProfiler::profilerInstance, serial, n, uThisTask() );
	} // if
    } // if
} // uSerialConstructor::uSerialConstructor

uSerialConstructor::~uSerialConstructor() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "(uSerialConstructor &)0x%p.~uSerialConstructor\n", this );
#endif // __U_DEBUG_H__
    if ( f == uYes && ! std::uncaught_exception() ) {
	uBaseTask &task = uThisTask();			// optimization

	task.setSerial( *serial.prevSerial );		// reset previous serial
	serial.leave( serial.mr );
    } // if
} // uSerialConstructor::~uSerialConstructor


uSerialDestructor::uSerialDestructor( uAction f, uSerial &serial, uBasePrioritySeq &ml, int mp ) : f( f ) {
    if ( f == uYes ) {
	uBaseTask &task = uThisTask();			// optimization
#ifdef __U_DEBUG__
	nlevel = task.currSerialLevel += 1;
#endif // __U_DEBUG__
	serial.prevSerial = &task.getSerial();		// save previous serial
	task.setSerial( serial );			// set new serial

	serial.enterDestructor( mr, ml, mp );
	if ( ! serial.acceptSignalled.empty() ) {
	    serial.lastAcceptor = NULL;
	    uBaseTask &uCallingTask = task;		// optimization
	    serial.mutexOwner = &(serial.acceptSignalled.drop()->task());
	    serial.acceptSignalled.add( &(uCallingTask.mutexRef) ); // suspend terminating task on top of accept/signalled stack
	    uSCHEDULE( serial.mutexOwner );		// find someone else to execute; wake on kernel stack
	} // if
    } // if
} // uSerialDestructor::uSerialDestructor

uSerialDestructor::~uSerialDestructor() {
    if ( f == uYes ) {
	uBaseTask &task = uThisTask();			// optimization
	uSerial &serial = task.getSerial();		// get current serial
	// Useful for dynamic allocation if an exception is thrown in the
	// destructor so the object can continue to be used and deleted again.
#ifdef __U_DEBUG__
	if ( nlevel != task.currSerialLevel ) {
	    uAbort( "Attempt to perform a non-nested entry and exit from multiple accessed mutex objects." );
	} // if
	task.currSerialLevel -= 1;
#endif // __U_DEBUG__
	if ( std::uncaught_exception() ) {
	    serial.leave( mr );
	} else {
	    if ( task.profileActive && uProfiler::uProfiler_deregisterMonitor ) { // task registered for profiling ?
		(*uProfiler::uProfiler_deregisterMonitor)( uProfiler::profilerInstance, serial, task );
	    } // if

	    task.mutexRecursion = mr;			// restore previous recursive count
	} // if
    } // if
} // uSerialDestructor::~uSerialDestructor


void uSerialMember::finalize( uBaseTask &task ) {
#ifdef __U_DEBUG__
    if ( nlevel != task.currSerialLevel ) {
	uAbort( "Attempt to perform a non-nested entry and exit from multiple accessed mutex objects." );
    } // if
    task.currSerialLevel -= 1;
#endif // __U_DEBUG__
    task.setSerial( *prevSerial );			// reset previous serial
} // uSerialMember::finalize

uSerialMember::uSerialMember( uSerial &serial, uBasePrioritySeq &ml, int mp ) {
    uBaseTask &task = uThisTask();			// optimization

    // There is a race condition between setting and testing this flag.
    // However, it is the best that can be expected because the mutex storage
    // is being deleted.

    if ( ! serial.alive ) {				// check against improper memory management
	_Throw uSerial::EntryFailure( &serial, "mutex object has been destroyed" );
    } // if

    if ( task.profileActive && uProfiler::uProfiler_registerMutexFunctionEntryTry ) { // task registered for profiling ?
	(*uProfiler::uProfiler_registerMutexFunctionEntryTry)( uProfiler::profilerInstance, serial, task );
    } // if

    try {
	// Polling in enter happens after properly setting values of mr and
	// therefore, in the catch clause, it can be used to retore the mr
	// value in the uSerial object.

	prevSerial = &task.getSerial();			// save previous serial
	task.setSerial( serial );			// set new serial
#ifdef __U_DEBUG__
	nlevel = task.currSerialLevel += 1;
#endif // __U_DEBUG__
	serial.enter( mr, ml, mp );
        acceptor = serial.lastAcceptor;
	acceptorSuspended = acceptor != NULL;
	if ( acceptorSuspended ) {
	    acceptor->acceptedCall = this;
	} // if
	serial.lastAcceptor = NULL;			// avoid messing up subsequent mutex method invocation
    } catch( ... ) {					// UNDO EFFECTS OF FAILED SERIAL ENTRY
	finalize( task );
        if ( serial.lastAcceptor ) {
	    serial.lastAcceptor->acceptedCall = NULL;	// the rendezvous did not materialize
	    _Resume uSerial::RendezvousFailure( &serial, "accepted call" ) _At *serial.lastAcceptor->currCoroutine; // acceptor is not initialized
	    serial.lastAcceptor = NULL;
	} // if
	serial.leave( mr );				// look at ~uSerialMember()
	_Throw;
    } // try

    noUserOverride = true;

    if ( task.profileActive && uProfiler::uProfiler_registerMutexFunctionEntryDone ) { // task registered for profiling ?
	(*uProfiler::uProfiler_registerMutexFunctionEntryDone )( uProfiler::profilerInstance, serial, task );
    } // if
} // uSerialMember::uSerialMember

uSerialMember::~uSerialMember() {
    uBaseTask &task = uThisTask();			// optimization
    uSerial &serial = task.getSerial();			// get current serial

    finalize( task );
    if ( acceptor ) {
	acceptor->acceptedCall = NULL;			// accepted mutex member terminates
	// raise a concurrent exception at the acceptor
	if ( std::uncaught_exception() && noUserOverride && serial.alive && acceptorSuspended ) {
	    // Return the acceptor only when the acceptor remains suspended and the
	    // mutex object has yet to be destroyed, otherwise return a NULL reference.
	    // Side-effect: prevent the kernel from resuming (_Resume) a concurrent
	    // exception at the suspended acceptor if the rendezvous terminates with an
	    // exception.

	    noUserOverride = false;
	    _Resume uSerial::RendezvousFailure( &serial, "accepted call" ) _At *(( serial.alive && acceptorSuspended ) ? acceptor->currCoroutine : NULL);
	} // if
    } // if

    if ( task.profileActive && uProfiler::uProfiler_registerMutexFunctionExit ) { // task registered for profiling ?
	(*uProfiler::uProfiler_registerMutexFunctionExit)( uProfiler::profilerInstance, serial, task );
    } // if

    serial.leave( mr );
} // uSerialMember::~uSerialMember


uCondition::uCondition() : owner( NULL ) {
} // uCondition::uCondition

uCondition::~uCondition() {
    // A uCondition object must be destroyed before its owner.  Concurrent
    // execution of the destructor for a uCondition and its owner is
    // unacceptable.  The flag owner->alive tells if a mutex object is
    // destroyed or not but it cannot protect against concurrent execution.  As
    // long as uCondition objects are declared inside its owner mutex object,
    // the proper order of destruction is guaranteed.

    if ( ! condQueue.empty() ) {
	// wake each task blocked on the condition with an async event
	for ( ;; ) {
	    uBaseTaskDL *p = condQueue.head();		// get the task blocked at the start of the condition
	  if ( p == NULL ) break;			// list empty ?
	    uEHM::uDeliverEStack dummy( false );	// block all async exceptions in destructor
	    uBaseTask &task = p->task();
	    _Throw WaitingFailure( *this, "found blocked task on condition variable during deletion" ) _At *(task.currCoroutine); // throw async event at blocked task
	    signalBlock();				// restart (signal) the blocked task
	} // for
    } // if
} // uCondition::~uCondition


#ifdef __U_DEBUG__
#define uConditionMsg( operation ) \
    "Attempt to " operation " a condition variable for a mutex object not locked by this task.\n" \
    "Possible cause is accessing the condition variable outside of a mutex member for the mutex object owning the variable."
#endif // __U_DEBUG__


void uCondition::wait() {				// wait on a condition
    uBaseTask &task = uThisTask();			// optimization
    uSerial &serial = task.getSerial();

#ifdef __U_DEBUG__
    if ( owner != NULL && &task != owner->mutexOwner ) { // must have mutex lock to wait
	uAbort( uConditionMsg( "wait on" ) );
    } // if
#endif // __U_DEBUG__
    if ( owner != &serial ) {				// only owner can use condition
	if ( owner == NULL ) {				// owner exist ?
	    owner = &serial;				// set condition owner
	} // if
    } // if

    if ( task.profileActive && uProfiler::uProfiler_registerWait ) { // task registered for profiling ?
	(*uProfiler::uProfiler_registerWait)( uProfiler::profilerInstance, *this, task, serial );
    } // if

    condQueue.add( &(task.mutexRef) );			// add to end of condition queue

    serial.leave2();					// release mutex and let it schedule another task
    _Enable <uSerial::Failure>;				// implicit poll

    if ( task.profileActive && uProfiler::uProfiler_registerReady ) { // task registered for profiling ?
	(*uProfiler::uProfiler_registerReady)( uProfiler::profilerInstance, *this, task, serial );
    } // if
} // uCondition::wait

void uCondition::wait( long int info ) {		// wait on a condition with information
    uThisTask().info = info;				// store the information with this task
    wait();						// wait on this condition
} // uCondition::wait


#ifdef __U_DEBUG__
#define uSignalCheck() \
    /* must have mutex lock to signal */ \
    if ( owner != NULL && &task != owner->mutexOwner ) uAbort( uConditionMsg( "signal" ) );
#endif // __U_DEBUG__


void uCondition::signal() {				// signal a condition
    if ( ! condQueue.empty() ) {
	uBaseTask &task = uThisTask();			// optimization
	uSerial &serial = task.getSerial();
#ifdef __U_DEBUG__
	uSignalCheck();
#endif // __U_DEBUG__
	if ( task.profileActive && uProfiler::uProfiler_registerSignal ) { // task registered for profiling ?
	    (*uProfiler::uProfiler_registerSignal)( uProfiler::profilerInstance, *this, task, serial );
	} // if

	serial.acceptSignalled.add( condQueue.drop() );	// move signalled task on top of accept/signalled stack
    } // if
} // uCondition::uS

void uCondition::signalBlock() {			// signal a condition
    if ( ! condQueue.empty() ) {
	uBaseTask &task = uThisTask();			// optimization
	uSerial &serial = task.getSerial();
#ifdef __U_DEBUG__
	uSignalCheck();
#endif // __U_DEBUG__
	if ( task.profileActive && uProfiler::uProfiler_registerSignal ) { // task registered for profiling ?
	    (*uProfiler::uProfiler_registerSignal)( uProfiler::profilerInstance, *this, task, serial );
	} // if
	if ( task.profileActive && uProfiler::uProfiler_registerWait ) { // task registered for profiling ?
	    (*uProfiler::uProfiler_registerWait)( uProfiler::profilerInstance, *this, task, serial );
	} // if

	serial.acceptSignalled.add( &(task.mutexRef) ); // suspend signaller task on accept/signalled stack
	serial.acceptSignalled.addHead( condQueue.drop() ); // move signalled task on head of accept/signalled stack
	serial.leave2();				// release mutex and let it schedule the signalled task

	if ( task.profileActive && uProfiler::uProfiler_registerReady ) { // task registered for profiling ?
	    (*uProfiler::uProfiler_registerReady)( uProfiler::profilerInstance, *this, task, serial );
	} // if
    } // if
} // uCondition::signalBlock


bool uCondition::empty() const {			// test for tasks on a condition
    return condQueue.empty();				// check if the condition queue is empty
} // uCondition::empty

long int uCondition::front() const {			// return task information
#ifdef __U_DEBUG__
    if ( condQueue.empty() ) {				// condition queue must not be empty
	uAbort( "Attempt to access user data on an empty condition.\n"
		"Possible cause is not checking if the condition is empty before reading stored data." );
    } // if
#endif // __U_DEBUG__
    return condQueue.head()->task().info;		// return condition information stored with blocked task
} // uCondition::front


uCondition::WaitingFailure::WaitingFailure( const uCondition &cond, const char *const msg ) : uSerial::Failure( cond.owner, msg ), cond( cond ) {}

uCondition::WaitingFailure::~WaitingFailure() {}

const uCondition &uCondition::WaitingFailure::conditionId() const { return cond; }

void uCondition::WaitingFailure::defaultTerminate() const {
    uAbort( "(uCondition &)0x%p : Waiting failure as task %.256s (0x%p) found blocked task %.256s (0x%p) on condition variable during deletion.",
	    &conditionId(), sourceName(), &source(), uThisTask().getName(), &uThisTask() );
} // uCondition::WaitingFailure::defaultTerminate


//######################### uMain #########################


uMain::uMain( uCluster &cluster, int size, int argc, char *argv[], int &retcode ) :
	uBaseTask( cluster, size ), argc( argc ), argv( argv ), uRetCode( retcode ) {
    // task uMain is always profiled when the profiler is active
    profileActivate( *uKernelModule::bootTask );	// make boot task the parent

#if defined( __irix__ )    
    cleanup_handlers = NULL;
#endif
} // uMain::uMain


uMain::~uMain() {
    _pthread_delete_kernel_threads();
} // uMain::~uMain


//######################### uRealPthread #########################


#ifdef __U_PTHREAD__

typeof( ::pthread_create ) *uRealPthread::pthread_create;
typeof( ::pthread_attr_init ) *uRealPthread::pthread_attr_init;
typeof( ::pthread_attr_setstack ) *uRealPthread::pthread_attr_setstack;
typeof( ::pthread_self ) *uRealPthread::pthread_self;
typeof( ::pthread_kill ) *uRealPthread::pthread_kill;
typeof( ::pthread_join ) *uRealPthread::pthread_join;

void uRealPthread::startup() {
    char *version = NULL;
#if defined( __linux__ ) && defined( __i386__ )
    version = "GLIBC_2.1";
#endif // __linux__&& __i386__
    pthread_create = (typeof(pthread_create))uKernelModule::interposeSymbol( "pthread_create", version );
    pthread_attr_init = (typeof(pthread_attr_init))uKernelModule::interposeSymbol( "pthread_attr_init", version );
    pthread_attr_setstack = (typeof(pthread_attr_setstack))uKernelModule::interposeSymbol( "pthread_attr_setstack" );
    pthread_self = (typeof(pthread_self))uKernelModule::interposeSymbol( "pthread_self" );
    pthread_kill = (typeof(pthread_kill))uKernelModule::interposeSymbol( "pthread_kill" );
    pthread_join = (typeof(pthread_join))uKernelModule::interposeSymbol( "pthread_join" );
} // uRealPthread::startup

#endif // __U_PTHREAD__


//######################### Kernel Boot #########################


#if defined( __irix__ ) && ! defined( __U_MULTI__ )
void uNullFunc( void *, size_t ) {
    // this function exists only to give the dummy sproc somewhere to go
    _exit( 0 );                                         // avoid cleanups
} // uNullFunc
#endif // __irix__ && ! __U_MULTI__


void uKernelBoot::startup() {
    if ( ! uKernelModule::kernelModuleInitialized ) {
	uKernelModule::startup();
    } // if

#ifdef __U_PTHREAD__
    // must come before any call to uDebugPrt
    uRealPthread::startup();
#endif // __U_PTHREAD__

#ifdef __U_DEBUG__
    // Force initialization of exception system, which results in dynamic
    // allocations that are not subsequently freed.

    std::uncaught_exception();

    // Storage allocated before the start of uC++ is normally not freed until
    // after uC++ completes (if at all). Hence, this storage is not considered
    // when calculating unfreed storage when the heap's destructor is called in
    // finishup.

    if ( uHeapManager::heapManagerInstance == NULL ) {
	uHeapManager::boot();
    } // if
    uHeapManager::preAlloc = uHeapManager::heapManagerInstance->checkFree();
#endif // __U_DEBUG__

    // create kernel locks

    uKernelModule::globalAbortLock = new uSpinLock;
    uKernelModule::globalProcessorLock = new uSpinLock;
    uKernelModule::globalClusterLock = new uSpinLock;

#ifdef __U_DEBUG_H__
    uDebugPrt( "uKernelBoot::startup1, disableInt:%d, disableIntCnt:%d\n",
	       THREAD_GETMEM( disableInt ), THREAD_GETMEM( disableIntCnt ) );
#endif // __U_DEBUG_H__

    // initialize kernel signal handlers

    uSigHandlerModule();

    // create global lists

    uKernelModule::globalProcessors = new uProcessorSeq;
    uKernelModule::globalClusters = new uClusterSeq;

    // SKULLDUGGERY: Initialize the global pointers with the appropriate memory
    // locations and then "new" the cluster and processor. Because the global
    // pointers are initialized, all the references through them in the cluster
    // and processor constructors work out. (HA!)

    uKernelModule::systemProcessor = (uProcessor *)&uKernelModule::systemProcessorStorage;
    uKernelModule::systemCluster = (uCluster *)&uKernelModule::systemClusterStorage;
    uKernelModule::systemScheduler = new uDefaultScheduler;

#if defined( __irix__ )
#ifdef __U_MULTI__
    // Irix has a default limit of 8 KTs (sprocs) in the default share group.
    // This must be changed *before* the first call to sproc or sprocsp.  There
    // can be at most 10000 in a share group.
    usconfig( CONF_INITUSERS, 256 );
#else // ! __U_MULTI__
    // SKULLDUGGERY: libc only provides locking around standard library functions
    // after sproc has been called.  Since there would otherwise be no calls to
    // sproc in a uniprocessor program, make a dummy call here.  uNullFunc returns
    // immediately, so the sproc created here has a short life.

    // Asking sproc to allocate a stack seems to be an expensive operation.
    #define U_DUMMY_STACK_SIZE 4000
    static char dummy_stack[ U_DUMMY_STACK_SIZE ];
    pid_t pid = sprocsp( uNullFunc, PR_SADDR | PR_SFDS | PR_SUMASK, NULL, dummy_stack + U_DUMMY_STACK_SIZE - 8, U_DUMMY_STACK_SIZE );
    if ( pid == -1 ) {
	uAbort( "uKernelBoot::startup, sprocsp : internal error, error(%d) %s.", errno, strerror( errno ) );
    } // if

    // uProcWait and uSigChldHandler race to wait on the pid for normal
    // termination so the loser ignores the ECHILD.
    int code = waitpid( pid, NULL, 0 );
    if ( code == -1 && errno != ECHILD ) {
	uAbort( "uKernelBoot::startup, waitpid : internal error, error(%d) %s.", errno, strerror( errno ) );
    } // if
#endif // ! __U_MULTI__
#endif // __irix__

    // create system cluster: it is at a fixed address so storing the result is unnecessary.

    new((void *)&uKernelModule::systemClusterStorage) uCluster( *uKernelModule::systemScheduler, uDefaultStackSize(), "systemCluster" );
    new((void *)&uKernelModule::systemProcessorStorage) uProcessor( *uKernelModule::systemCluster, 1.0 );

#if ! defined( __U_MULTI__ ) || defined( __U_ONETIMER__ )
    uProcessor::events = new uSysEventList;
#endif // ! __U_MULTI__ || __U_ONETIMER__

#ifndef __U_MULTI__
    uProcessor::contextSwitchHandler = new uCxtSwtchHndlr;
    uProcessor::contextEvent = new uEventNode( *uProcessor::contextSwitchHandler );
    uCluster::NBIO = new uNBIO;
#endif // ! __U_MULTI__

    // create processor kernel

    uProcessorKernel *pk = new uProcessorKernel;
    THREAD_SETMEM( processorKernelStorage, pk );

    // set thread register in the new context

#if defined( __U_MULTI__ ) && defined( __U_SWAPCONTEXT__ )
#if defined( __linux__ ) && defined( __i386__ )
#if ! defined( __U_PTHREAD__ )
    ((ucontext_t *)THREAD_GETMEM(processorKernelStorage)->storage)->uc_mcontext.gregs[REG_GS] = THREAD_GETMEM( ldtValue );
#endif // ! __U_PTHREAD__
#elif defined( __linux__ ) && defined( __x86_64__ )
#if ! defined( __U_PTHREAD__ )
    #error uC++ : internal error, unsupported architecture
#endif // ! __U_PTHREAD__
#elif defined( __linux__ ) && defined( __ia64__ )
    ((ucontext_t *)THREAD_GETMEM(processorKernelStorage)->storage)->uc_mcontext.sc_gr[13] = (unsigned long)THREAD_GETMEM( threadPointer );
#elif defined( __solaris__ ) && defined( __sparc__ ) && ! defined( __U_PTHREAD__ )
    ((ucontext_t *)THREAD_GETMEM(processorKernelStorage)->storage)->uc_mcontext.gregs[REG_G7] = (int)(THREAD_GETMEM( This ) );
#elif defined( __irix__ ) && defined( __mips__ )
    // No thread register => nothing needs to be set in the mcontext
#elif defined( __freebsd__ ) && defined( __i386__ )
    ((ucontext_t *)THREAD_GETMEM(processorKernelStorage)->storage)->uc_mcontext.mc_gs = THREAD_GETMEM( ldtValue );
#else
    #error uC++ : internal error, unsupported architecture
#endif
#ifdef __U_ONETIMER__
    sigdelset( (sigset_t *)&(((ucontext_t *)THREAD_GETMEM(processorKernelStorage)->storage)->uc_sigmask), SIGALRM );
#endif // __U_ONETIMER__
#endif // __U_MULTI__ && __U_SWAPCONTEXT__
#if defined( __U_MULTI__ ) && defined( __U_ONETIMER__ ) && defined( __ia64__ )
    sigdelset( &((uMachContext::uContext_t *)(THREAD_GETMEM(processorKernelStorage)->storage))->sigMask, SIGALRM );
#endif // __U_MULTI__ && __U_ONETIMER__ && __ia64__

    // start boot task, which executes the global constructors and destructors

    uKernelModule::bootTask = new((void *)&uKernelModule::bootTaskStorage) uBootTask();

    // SKULLDUGGERY: Set the processor's last resumer to the boot task so it
    // returns to it when the processor coroutine terminates. This has to be
    // done explicitly because the kernel coroutine is never resumed only
    // context switched to. Therefore, uLast is never set by resume, and
    // subsequently copied to uStart.

    activeProcessorKernel->last = uKernelModule::bootTask;

    // SKULLDUGGERY: Force a context switch to the system processor to set the
    // boot task's context to the current UNIX context. Hence, the boot task
    // does not begin through uInvoke, like all other tasks. It also starts the
    // system processor's companion task so that it is ready to receive
    // processor specific requests. The trick here is that uBootTask is on the
    // ready queue when this call is made. Normally, a task is not on a ready
    // queue when it is running. As result, there has to be a special check in
    // schedule to not assert that this task is NOT on a ready queue when it
    // starts the context switch.

    uThisTask().uYieldNoPoll();

    // create system task

    uKernelModule::systemTask = new uSystemTask();


    // THE SYSTEM IS NOW COMPLETELY RUNNING


    // Obtain the addresses of the original set_terminate and set_unexpected
    // using the dynamic loader. Use the original versions to initialize the
    // hidden variables holding the terminate and unexpected routines. All
    // other references to set_terminate and set_unexpected refer to the uC++
    // ones.

    std::terminate_handler (*orig_set_terminate)( std::terminate_handler );
    orig_set_terminate = (std::terminate_handler (*)(std::terminate_handler))uKernelModule::interposeSymbol( "_ZSt13set_terminatePFvvE" );
    orig_set_terminate( uEHM::terminateHandler );

    std::unexpected_handler (*orig_set_unexpected)( std::unexpected_handler );
    orig_set_unexpected = (std::unexpected_handler (*)(std::unexpected_handler))uKernelModule::interposeSymbol( "_ZSt14set_unexpectedPFvvE" );
    orig_set_unexpected( uEHM::unexpectedHandler );

    // create user cluster

    uKernelModule::userCluster = new uCluster( "userCluster" );

    // create user processors

    uKernelModule::numProcessors = uDefaultProcessors();
#ifdef __U_DEBUG__
    if ( uKernelModule::numProcessors == 0 ) {
	uAbort( "uDefaultProcessors must return a value 1 or greater" );
    } // if
#endif // __U_DEBUG__

    uKernelModule::userProcessors = new uProcessor*[ uKernelModule::numProcessors ];
    uKernelModule::userProcessors[0] = new uProcessor( *uKernelModule::userCluster );

    // uOwnerLock has a runtime check testing if locking is attempted from
    // inside the kernel. This check only applies once the system becomes
    // concurrent. During the previous boot-strapping code, some locks may be
    // invoked (and hence a runtime check would occur) but the system is not
    // concurrent. Hence, these locks are always open and no blocking can
    // occur. This flag enables uOwnerLock checking after this point.

    uKernelModule::initialization = true;

#ifdef __U_DEBUG_H__
    uDebugPrt( "uKernelBoot::startup2, disableInt:%d, disableIntCnt:%d, uPreemption:%d\n",
	       THREAD_GETMEM( disableInt ), THREAD_GETMEM( disableIntCnt ), uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__

    uKernelModule::bootTask->migrate( *uKernelModule::userCluster );

    // reset filebuf for default streams

    uKernelModule::cerrFilebuf = new std::filebuf( 2, 0 ); // unbufferred
    std::cerr.rdbuf( uKernelModule::cerrFilebuf );
    uKernelModule::clogFilebuf = new std::filebuf( 2 );
    std::clog.rdbuf( uKernelModule::clogFilebuf );
    uKernelModule::coutFilebuf = new std::filebuf( 1 );
    std::cout.rdbuf( uKernelModule::coutFilebuf );
    uKernelModule::cinFilebuf  = new std::filebuf( 0 );
    std::cin.rdbuf( uKernelModule::cinFilebuf );

#ifdef __U_DEBUG_H__
    uDebugPrt( "uKernelBoot::startup3, disableInt:%d, disableIntCnt:%d, uPreemption:%d\n",
	       THREAD_GETMEM( disableInt ), THREAD_GETMEM( disableIntCnt ), uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__
} // uKernelBoot::startup


void uKernelBoot::finishup() {
#ifdef __U_DEBUG_H__
    uDebugPrt( "uKernelBoot::finishup1, disableInt:%d, disableIntCnt:%d, uPreemption:%d\n",
	       THREAD_GETMEM( disableInt ), THREAD_GETMEM( disableIntCnt ), uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__

    // Flush standard output streams as required by 27.4.2.1.6

    delete uKernelModule::cinFilebuf;
    std::cout.flush();
    std::cout.rdbuf( NULL );
    delete uKernelModule::coutFilebuf;
    std::clog.flush();
    std::clog.rdbuf( NULL );
    delete uKernelModule::clogFilebuf;
    std::cerr.flush();
    std::cerr.rdbuf( NULL );
    delete uKernelModule::cerrFilebuf;

    _pthread_shutdown();

    if ( ! uKernelModule::afterMain ) _exit( uKernelModule::retCode );

    uKernelModule::bootTask->migrate( *uKernelModule::systemCluster );

#ifdef __U_DEBUG_H__
    uDebugPrt( "uKernelBoot::finishup2, disableInt:%d, disableIntCnt:%d, uPreemption:%d\n",
	       THREAD_GETMEM( disableInt ), THREAD_GETMEM( disableIntCnt ), uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__

    delete uKernelModule::userProcessors[0];
    delete [] uKernelModule::userProcessors;
    delete uKernelModule::userCluster;

    delete uKernelModule::systemTask;

    // Turn off uOwnerLock checking.

    uKernelModule::initialization = false;

    THREAD_GETMEM( This )->disableInterrupts();
    uThisProcessor().setContextSwitchEvent( 0 );	// clear the alarm on this processor
    THREAD_GETMEM( This )->enableInterrupts();

#ifndef __U_MULTI__
    delete uCluster::NBIO;
    delete uProcessor::contextEvent;
    delete uProcessor::contextSwitchHandler;
#endif // ! __U_MULTI__

#if ! defined( __U_MULTI__ ) || defined( __U_ONETIMER__ )
    delete uProcessor::events;
#endif //  ! __U_MULTI__ || __U_ONETIMER_

    // SKULLDUGGERY: The termination order for the boot task is different from
    // the starting order. This results from the fact that the boot task must
    // have a processor before it can start. However, the system processor must
    // have the thread from the boot task to terminate. Deleting the cluster
    // first requires that the boot task be removed first from the list of
    // tasks on the cluster or the cluster complains about unfinished
    // tasks. The boot task must be added again before its deletion because it
    // removes itself from the list. Also, when the boot task is being deleted
    // it is using the *already* deleted system cluster, which works only
    // because the system cluster storage is not dynamically allocated so the
    // storage is not scrubbed; therefore, it still has the necessary state
    // values to allow the boot task to be deleted. As well, the ready queue
    // has to be allocated spearately from the system cluster so it can be
    // deleted *after* the boot task is deleted because the boot task access
    // the ready queue during its deletion.

    // remove the boot task so the cluster does not complain
    uKernelModule::systemCluster->taskRemove( *(uBaseTask *)uKernelModule::bootTask );

    // remove system processor, processor task and cluster
    uKernelModule::systemProcessor->uProcessor::~uProcessor();
    uKernelModule::systemCluster->uCluster::~uCluster();

    // remove processor kernal coroutine with execution still pending
    delete THREAD_GETMEM( processorKernelStorage );

    // add the boot task back so it can remove itself from the list
    uKernelModule::systemCluster->taskAdd( *(uBaseTask *)uKernelModule::bootTask );
    ((uBootTask *)uKernelModule::bootTask)->uBootTask::~uBootTask();

    // Clean up storage associated with the boot task by pthread-like
    // thread-specific data (see ~uTaskMain). Must occur *after* all calls that
    // might call std::uncaught_exception, otherwise the exception
    // data-structures are created again for the task. (Note: ~uSerialX members
    // call std::uncaught_exception).

    if ( ((uBootTask *)uKernelModule::bootTask)->pthreadData != NULL ) {
	_pthread_deletespecific( ((uBootTask *)uKernelModule::bootTask)->pthreadData );
    } // if

    // Clean up storage associated with pthread pid table (if any).

    _pthread_pid_destroy();

    // no tasks on the ready queue so it can be deleted
    delete uKernelModule::systemScheduler;

#ifdef __U_DEBUG_H__
    uDebugPrt( "uKernelBoot::finishup3, disableInt:%d, disableIntCnt:%d, uPreemption:%d\n",
	       THREAD_GETMEM( disableInt ), THREAD_GETMEM( disableIntCnt ), uThisProcessor().getPreemption() );
#endif // __U_DEBUG_H__

    delete uKernelModule::globalClusters;
    delete uKernelModule::globalProcessors;

    delete uKernelModule::globalClusterLock;
    delete uKernelModule::globalProcessorLock;
    delete uKernelModule::globalAbortLock;

    // Explicitly invoking the destructor does not close down the heap because
    // it might still be used before the application terminates. The heap's
    // destructor does check for unreleased storage at this point. (The
    // constructor for the heap is called on the first call to malloc.)

    uHeapManager::heapManagerInstance->uHeapManager::~uHeapManager();
} // uKernelBoot::finishup


void uInitProcessorsBoot::startup() {
    for ( unsigned int i = 1; i < uKernelModule::numProcessors; i += 1 ) {
	uKernelModule::userProcessors[i] = new uProcessor( *uKernelModule::userCluster );
    } // for
} // uInitProcessorsBoot::startup


void uInitProcessorsBoot::finishup() {
    for ( unsigned int i = 1; i < uKernelModule::numProcessors; i += 1 ) {
	delete uKernelModule::userProcessors[i];
    } // for
} // uInitProcessorsBoot::finishup


// Local Variables: //
// compile-command: "gmake install" //
// End: //
