//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr and Richard C. Bilson 2006
// 
// Future.h -- 
// 
// Author           : Peter A. Buhr and Richard C. Bilson
// Created On       : Wed Aug 30 22:34:05 2006
// Last Modified By : Peter A. Buhr
// Last Modified On : Wed Sep 19 13:38:48 2007
// Update Count     : 458
// 
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 

#ifndef __U_FUTURE_H__
#define __U_FUTURE_H__


//############################## uBaseFuture ##############################


namespace UPP {
    template<typename T> _Monitor uBaseFuture {
	T result;					// future result
      public:
	_Event Cancellation {};				// raised if future cancelled

	// These members should be private but cannot be because they are
	// referenced from user code.

	bool addAccept( UPP::BaseFutureDL *acceptState ) {
	    if ( available() ) return false;
	    acceptClients.addTail( acceptState );
	    return true;
	} // uBaseFuture::addAccept

	void removeAccept( UPP::BaseFutureDL *acceptState ) {
	    acceptClients.remove( acceptState );
	} // uBaseFuture::removeAccept
      protected:
	uCondition delay;				// clients waiting for future result
	uSequence<UPP::BaseFutureDL> acceptClients;	// clients waiting for future result in selection
	uBaseEvent *cause;				// synchronous exception raised during future computation
	bool available_, cancelled_;			// future status

	void makeAvailable() {
	    available_ = true;
	    while ( ! delay.empty() ) delay.signal();	// unblock waiting clients ?
	    if ( ! acceptClients.empty() ) {		// select-blocked clients ?
		UPP::BaseFutureDL *bt;			// unblock select-blocked clients
		for ( uSeqIter<UPP::BaseFutureDL> iter( acceptClients ); iter >> bt; ) {
		    bt->signal();
		} // for
	    } // if
	} // uBaseFuture::makeAvailable

	void check() {
	    if ( cancelled() ) _Throw Cancellation();
	    if ( cause != NULL ) cause->reraise();
	} // uBaseFuture::check
      public:
	uBaseFuture() : cause( NULL ), available_( false ), cancelled_( false ) {}

	_Nomutex bool available() { return available_; } // future result available ?
	_Nomutex bool cancelled() { return cancelled_; } // future result cancelled ?

	// used by client

	T operator()() {				// access result, possibly having to wait
	    check();					// cancelled or exception ?
	    if ( ! available() ) {
		delay.wait();
		check();				// cancelled or exception ?
	    } // if
	    return result;
	} // uBaseFuture::operator()()

	_Nomutex operator T() {				// cheap access of result after waiting
	    check();					// cancelled or exception ?
#ifdef __U_DEBUG__
	    if ( ! available() ) {
		uAbort( "Attempt to access future result 0x%p without first performing a blocking access operation.", this );
	    } // if
#endif // __U_DEBUG__
	    return result;
	} // uBaseFuture::operator T()

	// used by server

	void delivery( T res ) {			// make result available in the future
	    if ( available() ) {
		if ( cancelled() ) return;
#ifdef __U_DEBUG__
		else {
		    uAbort( "Attempt to deliver a second result to future 0x%p before current result retrieved.", this );
		} // if
#endif // __U_DEBUG__
	    } // if
	    result = res;
	    makeAvailable();
	} // uBaseFuture::delivery

	void reset() {					// mark future as empty (for reuse)
	    available_ = cancelled_ = false;		// reset for next value
	    delete cause;
	    cause = NULL;
	} // uBaseFuture::reset

	void exception( uBaseEvent *ex ) {		// make exception available in the future
#ifdef __U_DEBUG__
	    if ( cause != NULL ) {
		char name[uEHMMaxName];
		uAbort( "Attempt to deliver a second exception 0x%p of type %s to future 0x%p.\n"
			"Current delivered exception is 0x%p.",
			ex, uEHM::getCurrentEventName( ex->getRaiseKind(), name, uEHMMaxName ), this, cause );
	    } // if
#endif // __U_DEBUG__
	    cause = ex;
	    makeAvailable();
	} // uBaseFuture::exception
    }; // uBaseFuture
} // UPP


//############################## Future_ESM ##############################


// Caller is responsible for storage management by preallocating the future and
// passing it as an argument to the asynchronous call.  Cannot be copied.

template<typename T, typename ServerData> _Monitor Future_ESM : public UPP::uBaseFuture<T> {
    using UPP::uBaseFuture<T>::cancelled_;

    _Mutex bool checkCancel() {
      if ( cancelled() ) return false;			// only cancel once
	cancelled_ = true;
	return ! available();
    } // Future_ESM::checkCancel
    
    _Mutex void compCancelled() {
	if ( ! available() ) {				// must recheck
	    makeAvailable();				// unblock waiting clients ?
	} // if
    } // Future_ESM::compCancelled

    _Mutex void compNotCancelled() {
	if ( ! available() ) {				// must recheck
	    delay.wait();				// wait for value
	} // if
    } // Future_ESM::compNotCancelled
  public:
    using UPP::uBaseFuture<T>::available;
    using UPP::uBaseFuture<T>::reset;
    using UPP::uBaseFuture<T>::makeAvailable;
    using UPP::uBaseFuture<T>::check;
    using UPP::uBaseFuture<T>::delay;
    using UPP::uBaseFuture<T>::cancelled;

    Future_ESM() {}
    ~Future_ESM() { reset(); }

    // used by client

    _Nomutex void cancel() {				// cancel future result
	// To prevent deadlock, call the server without holding future mutex,
	// because server may attempt to deliver a future value. (awkward code)
	if ( checkCancel() ) {				// need to contact server ?
	    if ( serverData.cancel() ) {		// synchronously contact server
		compCancelled();			// server computation cancelled, unblock waiting clients
	    } else {
		compNotCancelled();			// server computation not cancelled, wait for value
	    } // if
	} // if
    } // Future_ESM::cancel

    // used by server

    ServerData serverData;				// information needed by server
}; // Future_ESM


//############################## Future_ISM ##############################


// Future is responsible for storage management by using reference counts.  Can
// be copied.

template<typename T> class Future_ISM {
    public:
    struct ServerData {
	virtual ~ServerData() {}
	virtual bool cancel() = 0;
    };
  private:
    _Monitor Impl : public UPP::uBaseFuture<T> {	// mutual exclusion implementation
	using UPP::uBaseFuture<T>::cancelled_;
	using UPP::uBaseFuture<T>::cause;

	unsigned int refCnt;				// number of references to future
	ServerData *serverData;
      public:
	using UPP::uBaseFuture<T>::available;
	using UPP::uBaseFuture<T>::reset;
	using UPP::uBaseFuture<T>::makeAvailable;
	using UPP::uBaseFuture<T>::check;
	using UPP::uBaseFuture<T>::delay;
	using UPP::uBaseFuture<T>::cancelled;

	Impl() : refCnt( 1 ), serverData( NULL ) {}
	Impl( ServerData *serverData_ ) : refCnt( 1 ), serverData( serverData_ ) {}

	~Impl() {
	    delete serverData;
	} // Impl::~Impl

	void incRef() {
	    refCnt += 1;
	} // Impl::incRef

	bool decRef() {
	    refCnt -= 1;
	  if ( refCnt != 0 ) return false;
	    delete cause;
	    return true;
	} // Impl::decRef

	void cancel() {					// cancel future result
	  if ( cancelled() ) return;			// only cancel once
	    cancelled_ = true;
	    if ( ! available() ) {
		if ( serverData != NULL ) serverData->cancel();
		makeAvailable(); 			// unblock waiting clients ?
	    } // if
	} // Impl::cancel
    }; // Impl

    Impl *impl;						// storage for implementation
  public:
    Future_ISM() : impl( new Impl ) {}
    Future_ISM( ServerData *serverData ) : impl( new Impl( serverData ) ) {}

    ~Future_ISM() {
	if ( impl->decRef() ) delete impl;
    } // Future_ISM::~Future_ISM

    Future_ISM( const Future_ISM<T> &rhs ) {
	impl = rhs.impl;				// point at new impl
	impl->incRef();					//   and increment reference count
    } // Future_ISM::Future_ISM

    Future_ISM<T> &operator=( const Future_ISM<T> &rhs ) {
      if ( rhs.impl == impl ) return *this;
	if ( impl->decRef() ) delete impl;		// no references => delte current impl
	impl = rhs.impl;				// point at new impl
	impl->incRef();					//   and increment reference count
	return *this;
    } // Future_ISM::operator=

    // used by client

    typedef typename UPP::uBaseFuture<T>::Cancellation Cancellation; // raised if future cancelled

    bool available() { return impl->available(); }	// future result available ?
    bool cancelled() { return impl->cancelled(); }	// future result cancelled ?

    // used by client

    T operator()() {					// access result, possibly having to wait
	return (*impl)();
    } // Future_ISM::operator()()

    operator T() {					// cheap access of result after waiting
	return (T)(*impl);
    } // Future_ISM::operator T()

    void cancel() {					// cancel future result
	impl->cancel();
    } // Future_ISM::cancel

    bool addAccept( UPP::BaseFutureDL *acceptState ) {
	return impl->addAccept( acceptState );
    } // Future_ISM::addAccept

    void removeAccept( UPP::BaseFutureDL *acceptState ) {
	return impl->removeAccept( acceptState );
    } // Future_ISM::removeAccept

    bool equals( const Future_ISM<T> &other ) {		// referential equality
	return impl == other.impl;
    } // Future_ISM::equals

    // used by server

    void delivery( T result ) {				// make result available in the future
	impl->delivery( result );
    } // Future_ISM::delivery

    void exception( uBaseEvent *cause ) {		// make exception available in the future
	impl->exception( cause );
    } // Future_ISM::exception

    void reset() {					// mark future as empty (for reuse)
	impl->reset();
    } // Future_ISM::reset
}; // Future_ISM


template< typename Result, typename Other >
UPP::BinarySelector< UPP::OrCondition, Future_ISM< Result >, UPP::UnarySelector< Other, int >, int > operator||( const Future_ISM< Result > &s1, const Other &s2 ) {
    return UPP::BinarySelector< UPP::OrCondition, Future_ISM< Result >, UPP::UnarySelector< Other, int >, int >( s1, UPP::UnarySelector< Other, int >( s2 ) );
} // operator||

template< typename Result, typename Other >
UPP::BinarySelector< UPP::AndCondition, Future_ISM< Result >, UPP::UnarySelector< Other, int >, int > operator&&( const Future_ISM< Result > &s1, const Other &s2 ) {
    return UPP::BinarySelector< UPP::AndCondition, Future_ISM< Result >, UPP::UnarySelector< Other, int >, int >( s1, UPP::UnarySelector< Other, int >( s2 ) );
} // operator&&

// operators specific to Future_ESM

template< typename Result, typename ServerData, typename Other >
UPP::BinarySelector< UPP::OrCondition, UPP::UnarySelector< Future_ESM< Result, ServerData >, int >, UPP::UnarySelector< Other, int >, int > operator||( const Future_ESM< Result, ServerData > &s1, const Other &s2 ) {
    return UPP::BinarySelector< UPP::OrCondition,UPP::UnarySelector< Future_ESM< Result, ServerData >, int >, UPP::UnarySelector< Other, int >, int >( UPP::UnarySelector< Future_ESM< Result, ServerData >, int >( s1 ), UPP::UnarySelector< Other, int >( s2 ) );
} // operator||

template< typename Result, typename ServerData, typename Other >
UPP::BinarySelector< UPP::AndCondition, UPP::UnarySelector< Future_ESM< Result, ServerData >, int >, UPP::UnarySelector< Other, int >, int > operator&&( const Future_ESM< Result, ServerData > &s1, const Other &s2 ) {
    return UPP::BinarySelector< UPP::AndCondition, UPP::UnarySelector< Future_ESM< Result, ServerData >, int >, UPP::UnarySelector< Other, int >, int >( UPP::UnarySelector< Future_ESM< Result, ServerData >, int >( s1 ), UPP::UnarySelector< Other, int >( s2 ) );
} // operator&&


#endif // __U_FUTURE_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
