//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 1994
// 
// uFile.h -- Nonblocking UNIX I/O library
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 16:38:54 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Sep 18 12:17:14 2007
// Update Count     : 141
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#ifndef __U_FILE_H__
#define __U_FILE_H__


#include <uIOcntl.h>


#pragma __U_NOT_USER_CODE__


#include <fcntl.h>					// open, mode flags
#include <sys/stat.h>					// stat


//######################### uFile #########################


class uFile {
    friend class uFileWrapper;
    friend class uFileAccess;

    char *name;
    int accessCnt;

    _Mutex void access() {
	accessCnt += 1;
    } // uFile::access

    _Mutex void unaccess() {
	accessCnt -= 1;
    } // uFile::unaccess

    void createFile( const char *name ) {
	uFile::name = new char[strlen( name ) + 1];
	strcpy( uFile::name, name );
	accessCnt = 0;
    } // uFile::createFile
  public:
    _Event Failure : public uIOFailure {
	const uFile &f;
	char name[uEHMMaxName];
      protected:
	Failure( const uFile &f, const char *const msg );
      public:
	const uFile &file() const;
	const char *getName() const;
	virtual void defaultTerminate() const;
    }; // uFile::Failure

    _Event TerminateFailure : public Failure {
	const int accessCnt;
      public:
	TerminateFailure( const uFile &f, const int accessCnt, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uFile::TerminateFailure

    _Event StatusFailure : public Failure {
	const struct stat &buf;
      public:
	StatusFailure( const uFile &f, const struct stat &buf, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uFile::StatusFailure


    uFile( const char *name ) {
	createFile( name );
    } // uFile::uFile

    virtual ~uFile();

    const char *getName() const;
    void status( struct stat &buf );
}; // uFile


//######################### uFileIO #########################


class uFileIO {						// monitor
  protected:
    uIOaccess &access;

    virtual void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;
    virtual void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;
    virtual void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) = 0;

    uFileIO( uIOaccess &acc ) : access( acc ) {
    } // uFileIO::uFileIO

    virtual ~uFileIO() {
    } // uFileIO::~uFileIO
  public:
    int read( char *buf, int len, uDuration *timeout = NULL );
    int readv( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );
    _Mutex int write( const char *buf, int len, uDuration *timeout = NULL );
    int writev( const struct iovec *iov, int iovcnt, uDuration *timeout = NULL );

    int fd() {
	return access.fd;
    } // uFileIO::fd
}; // uFileIO


//######################### uFileAccess #########################


class uFileAccess : public uFileIO {			// monitor
    template< class char_t, class traits > friend class std::basic_filebuf; // access: constructor

    uFile &ufile;
    uIOaccess access;

    uFileAccess( int fd, uFile &f ) : uFileIO( access ), ufile( f ) {
	access.fd = fd;
	access.poll.setStatus( uPoll::PollOnDemand );
	ufile.access();
    } // uFileAccess::uFileAccess
  protected:
    void readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op );
    void writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op );
  public:
    _Event Failure : public uFile::Failure {
	const uFileAccess &fa;
	int fd;
      protected:
	Failure( const uFileAccess &fa, const char *const msg );
      public:
	const uFileAccess &fileAccess() const;
	int fileDescriptor() const;
	virtual void defaultTerminate() const;
    }; // uFileAccess::Failure
    friend _Event Failure;

    _Event OpenFailure : public Failure {
	const int flags;
	const int mode;
      public:
	OpenFailure( uFileAccess &fa, int flags, int mode, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uFileAccess::OpenFailure

    _Event CloseFailure : public Failure {
      public:
	CloseFailure( uFileAccess &fa, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uFileAccess::CloseFailure

    _Event SeekFailure : public Failure {
	const off_t offset;
	const int whence;
      public:
	SeekFailure( const uFileAccess &fa, const off_t offset, const int whence, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uFileAccess::SeekFailure

    _Event SyncFailure : public Failure {
      public:
	SyncFailure( const uFileAccess &fa, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uFileAccess::SyncFailure

    _Event ReadFailure : public Failure {
      protected:
	const char *buf;
	const int len;
	const uDuration *timeout;
      public:
	ReadFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
	virtual void defaultTerminate() const;
    }; // uFileAccess::ReadFailure

    _Event ReadTimeout : public ReadFailure {
      public:
	ReadTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    }; // uFileAccess::ReadTimeout

    _Event WriteFailure : public Failure {
      protected:
	const char *buf;
	const int len;
	const uDuration *timeout;
      public:
	WriteFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
	virtual void defaultResume() const;		// handle special case when errno == EIO
	virtual void defaultTerminate() const;
    }; // uFileAccess::WriteFailure

    _Event WriteTimeout : public WriteFailure {
      public:
	WriteTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg );
    }; // uFileAccess::WriteTimeout


    uFileAccess( uFile &f, int flags, int mode = 0644 );
    _Mutex virtual ~uFileAccess();

    off_t lseek( off_t offset, int whence );
    int fsync();
}; // uFileAccess


#pragma __U_USER_CODE__

#endif // __U_FILE_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
