//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 1994
// 
// uFile.cc -- 
// 
// Author           : Peter Buhr
// Created On       : Tue Mar 29 16:42:36 1994
// Last Modified By : Peter A. Buhr
// Last Modified On : Thu Sep  6 11:33:52 2007
// Update Count     : 316
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 


#define __U_KERNEL__
#include <uC++.h>
#include <uFile.h>

//#include <uDebug.h>

#include <cstring>					// strerror
#include <unistd.h>					// read, write, close, etc.
#include <sys/uio.h>					// readv, writev


//######################### uFile #########################


uFile::~uFile() {
    TerminateFailure temp( *this, accessCnt, "terminating access with outstanding accessor(s)" );
    if ( name != 0 ) delete name;			// safe to delete name as the name is copied in the exception object
    if ( accessCnt != 0 ) {
	if ( ! std::uncaught_exception() ) _Throw temp;
    } // if
} // uFile::~uFile


const char *uFile::getName() const {
    return
#ifdef __U_DEBUG__
	( name == NULL || name == (const char *)-1 ) ? "*unknown*" : // storage might be scrubbed
#endif // __U_DEBUG__
	name;
} // uFile::getName


void uFile::status( struct stat &buf ) {
    int retcode;

    for ( ;; ) {
	retcode = ::stat( name, &buf );
      if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( retcode == -1 ) {
	_Resume uFile::StatusFailure( *this, buf, "could not obtain statistical information for file" );
    } // if
} // uFile::status


//######################### uFileIO #########################


int uFileIO::read( char *buf, int len, uDuration *timeout ) {
    int rlen;

    struct Read : public uIOClosure {
	char *buf;
	int len;

	int action( int rwe ) {
#ifdef KNOT
	    read_syscalls += 1;
#endif // KNOT
	    return ::read( access.fd, buf, len );
	}
	Read( uIOaccess &access, int &rlen, char *buf, int len ) : uIOClosure( access, rlen ), buf( buf ), len( len ) {}
    } readClosure( access, rlen, buf, len );

    readClosure.wrapper( 0 );
    if ( rlen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    readFailure( buf, len, timeout, "read" );
	} // if
	int mask = uCluster::readSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( readClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( readClosure, mask, &t ) == 0 ) { // timeout ?
		readTimeout( buf, len, timeout, "read" );
	    } // if
	} // if
    } // if

    return rlen;
} // uFileIO::read


int uFileIO::readv( const struct iovec *iov, int iovcnt, uDuration *timeout ) {
    int rlen;

    struct Readv : public uIOClosure {
	const struct iovec *iov;
	int iovcnt;

	int action( int rwe ) { return ::readv( access.fd, iov, iovcnt ); }
	Readv( uIOaccess &access, int &rlen, const struct iovec *iov, int iovcnt ) : uIOClosure( access, rlen ), iov( iov ), iovcnt( iovcnt ) {}
    } readvClosure( access, rlen, iov, iovcnt );

    readvClosure.wrapper( 0 );
    if ( rlen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    readFailure( (const char *)iov, iovcnt, timeout, "readv" );
	} // if
	int mask = uCluster::readSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( readvClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( readvClosure, mask, &t ) == 0 ) { // timeout ?
		readTimeout( (const char *)iov, iovcnt, timeout, "readv" );
	    } // if
	} // if
    } // if

    return rlen;
} // uFileIO::readv


int uFileIO::write( const char *buf, int len, uDuration *timeout ) {
    int count, wlen;

    struct Write : public uIOClosure {
	const char *buf;
	int len;

	int action( int rwe ) {
#ifdef KNOT
	    write_syscalls += 1;
#endif // KNOT
	    return ::write( access.fd, buf, len );
	}
	Write( uIOaccess &access, int &wlen ) : uIOClosure( access, wlen ) {}
    } writeClosure( access, wlen );

    for ( count = 0; count < len; count += wlen ) {	// ensure all data is written
	writeClosure.buf = buf + count;
	writeClosure.len = len - count;
	writeClosure.wrapper( 0 );
	if ( wlen == -1 ) {
	    if ( errno != U_EWOULDBLOCK ) {
		// EIO means the write is to stdout but the shell has terminated (I
		// think). Normally, people want this to work as if stdout is
		// magically redirected to /dev/null, instead of aborting the
		// program.
      if ( errno == EIO ) goto fini;
		writeFailure( buf, len, timeout, "write" );
	    } // if
	    int mask = uCluster::writeSelect;
	    if ( timeout == NULL ) {
		uThisCluster().select( writeClosure, mask );
	    } else {
		timeval t = *timeout;			// convert to timeval for select
		if ( uThisCluster().select( writeClosure, mask, &t ) == 0 ) { // timeout ?
		    writeTimeout( buf, len, timeout, "write" );
		} // if
	    } // if
	} // if
    } // for
  fini:

    return len;						// always return the specified length
} // uFileIO::write


int uFileIO::writev( const struct iovec *iov, int iovcnt, uDuration *timeout ) {
    int wlen;

    struct Writev : public uIOClosure {
	const struct iovec *iov;
	int iovcnt;

	int action( int rwe ) { return ::writev( access.fd, iov, iovcnt ); }
	Writev( uIOaccess &access, int &wlen, const struct iovec *iov, int iovcnt ) : uIOClosure( access, wlen ), iov( iov ), iovcnt( iovcnt ) {}
    } writevClosure( access, wlen, iov, iovcnt );

    writevClosure.wrapper( 0 );
    if ( wlen == -1 ) {
	if ( errno != U_EWOULDBLOCK ) {
	    // EIO means the write is to stdout but the shell has terminated (I
	    // think). Normally, people want this to work as if stdout is magically
	    // redirected to /dev/null, instead of aborting the program.
      if ( errno == EIO ) goto fini;
	    writeFailure( (const char *)iov, iovcnt, timeout, "writev" );
	} // if
	int mask = uCluster::writeSelect;
	if ( timeout == NULL ) {
	    uThisCluster().select( writevClosure, mask );
	} else {
	    timeval t = *timeout;			// convert to timeval for select
	    if ( uThisCluster().select( writevClosure, mask, &t ) == 0 ) { // timeout ?
		writeTimeout( (const char *)iov, iovcnt, timeout, "writev" );
	    } // if
	} // if
    } // if
  fini:

    return wlen;
} // uFileIO::writev


//######################### uFileAccess #########################


void uFileAccess::readFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "file " ), op ), " fails" );
    _Throw uFileAccess::ReadFailure( *this, buf, len, timeout, msg );
} // uFileAccess::readFailure


void uFileAccess::readTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during file " ), op );
    _Throw uFileAccess::ReadTimeout( *this, buf, len, timeout, msg );
} // uFileAccess::readTimeout


void uFileAccess::writeFailure( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcat( strcpy( msg, "file " ), op ), " fails" );
    _Throw uFileAccess::WriteFailure( *this, buf, len, timeout, msg );
} // uFileAccess::writeFailure


void uFileAccess::writeTimeout( const char *buf, const int len, const uDuration *timeout, const char *const op ) {
    char msg[32];
    strcat( strcpy( msg, "timeout during file " ), op );
    _Throw uFileAccess::WriteTimeout( *this, buf, len, timeout, msg );
} // uFileAccess::writeTimeout


uFileAccess::uFileAccess( uFile &f, int flags, int mode ) : uFileIO( access ), ufile( f ) {
    for ( ;; ) {
	access.fd = ::open( ufile.name, flags, mode );
      if ( access.fd != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( access.fd == -1 ) {
        _Throw uFileAccess::OpenFailure( *this, flags, mode, "unable to access file" );
    } // if
    access.poll.computeStatus( access.fd );
    if ( access.poll.getStatus() == uPoll::AlwaysPoll ) access.poll.setPollFlag( access.fd );
    ufile.access();
} // uFileAccess::uFileAccess


uFileAccess::~uFileAccess() {
    ufile.unaccess();
    if ( access.poll.getStatus() == uPoll::AlwaysPoll ) access.poll.clearPollFlag( access.fd );
    if ( access.fd >= 3 ) {				// don't close the standard file descriptors
	int retcode;

	uThisCluster().closeFD( access.fd );
	for ( ;; ) {
	    retcode = ::close( access.fd );
	  if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
	} // for
	if ( retcode == -1 ) {
	    if ( ! std::uncaught_exception() ) _Resume uFileAccess::CloseFailure( *this, "unable to terminate access to file" );
	} // if
    } // if
} // uFileAccess::~uFileAccess


off_t uFileAccess::lseek( off_t offset, int whence ) {
    off_t retcode;

    for ( ;; ) {
	retcode = ::lseek( access.fd, offset, whence );
      if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( retcode == -1 ) {
        _Throw uFileAccess::SeekFailure( *this, offset, whence, "could not seek file" );
    } // if
    return retcode;
} // uFileAccess::lseek


int uFileAccess::fsync() {
    int retcode;

    for ( ;; ) {
	retcode = ::fsync( access.fd );
      if ( retcode != -1 || errno != EINTR ) break;	// timer interrupt ?
    } // for
    if ( retcode == -1 ) {
        _Throw uFileAccess::SyncFailure( *this, "could not fsync file" );
    } // if
    return retcode;
} // uFileAccess::fsync


//######################### uFile (cont) #########################


uFile::Failure::Failure( const uFile &f, const char *const msg ) : uIOFailure( msg ), f( f ) {
    // file name is copied because its storage can be freed and scrubbed before handler starts
    uEHM::strncpy( name, f.getName(), uEHMMaxName );
} // uFile::Failure::Failure

const uFile &uFile::Failure::file() const { return f; }

const char *uFile::Failure::getName() const { return name; }

void uFile::Failure::defaultTerminate() const {
    uAbort( "(uFile &)0x%p, %.256s \"%.256s\".", &file(), message(), getName() );
} // uFile::Failure::defaultTerminate


uFile::TerminateFailure::TerminateFailure( const uFile &f, const int accessCnt, const char *const msg ) :
	uFile::Failure( f, msg ), accessCnt( accessCnt ) {}

void uFile::TerminateFailure::defaultTerminate() const {
    uAbort( "(uFile &)0x%p.~uFile(), %.256s, %d accessor(s) outstanding.", &file(), message(), accessCnt );
} // uFile::TerminateFailure::defaultTerminate


uFile::StatusFailure::StatusFailure( const uFile &f, const struct stat &buf, const char *const msg ) : uFile::Failure( f, msg ), buf( buf ) {}

void uFile::StatusFailure::defaultTerminate() const {
    uAbort( "(uFile &)0x%p.status( buf:0x%p ), %.256s \"%.256s\".\nError(%d) : %s.",
	    &file(), &buf, message(), getName(), errNo(), strerror( errNo() ) );
} // uFile::StatusFailure::defaultTerminate


//######################### uFileAccess (cont) #########################


uFileAccess::Failure::Failure( const uFileAccess &fa, const char *const msg ) : uFile::Failure( fa.ufile, msg ), fa( fa ) {
    fd = fa.access.fd;
} // uFileAccess::Failure::Failure

const uFileAccess &uFileAccess::Failure::fileAccess() const {
    return fa;
} // uFileAccess::Failure::fileAccess

int uFileAccess::Failure::fileDescriptor() const {
    return fd;
} // uFileAccess::Failure::fileDescriptor

void uFileAccess::Failure::defaultTerminate() const {
    uAbort( "(uFileAccess &)0x%p( file:0x%p ), %.256s file \"%.256s\".",
	    &fileAccess(), &file(), message(), getName() );
} // uFileAccess::Failure::defaultTerminate


uFileAccess::OpenFailure::OpenFailure( uFileAccess &fa, int flags, int mode, const char *const msg ) :
	uFileAccess::Failure( fa, msg ), flags( flags ), mode( mode ) {}

void uFileAccess::OpenFailure::defaultTerminate() const {
    uAbort( "(uFileAccess &)0x%p.uFileAccess( file:0x%p, flags:0x%x, mode:0x%x ), %.256s \"%.256s\".\nError(%d) : %s.",
	    &fileAccess(), &file(), flags, mode, message(), getName(), errNo(), strerror( errNo() ) );
} // uFile::OpenFailure::defaultTerminate


uFileAccess::CloseFailure::CloseFailure( uFileAccess &fa, const char *const msg ) : uFileAccess::Failure( fa, msg ) {}

void uFileAccess::CloseFailure::defaultTerminate() const {
    uAbort( "(uFileAccess &)0x%p.~uFileAccess(), %.256s \"%.256s\".\nError(%d) : %s.",
	    &fileAccess(), message(), getName(), errNo(), strerror( errNo() ) );
} // uFile::CloseFailure::defaultTerminate


uFileAccess::SeekFailure::SeekFailure( const uFileAccess &fa, const off_t offset, const int whence, const char *const msg ) :
	uFileAccess::Failure( fa, msg ), offset( offset ), whence( whence ) {}

void uFileAccess::SeekFailure::defaultTerminate() const {
    uAbort( "(uFile &)0x%p.lseek( offset:%ld, whence:%d ), %.256s file \"%.256s\".\nError(%d) : %s.",
	    &file(), (long int)offset, whence, message(), getName(), errNo(), strerror( errNo() ) );
} // uFileAccess::SeekFailure::defaultTerminate


uFileAccess::SyncFailure::SyncFailure( const uFileAccess &fa, const char *const msg ) : uFileAccess::Failure( fa, msg ) {}

void uFileAccess::SyncFailure::defaultTerminate() const {
    uAbort( "(uFileAccess &)0x%p.fsync(), %.256s \"%.256s\".\nError(%d) : %s.",
	    &file(), message(), getName(), errNo(), strerror( errNo() ) );
} // uFileAccess::SyncFailure::defaultTerminate


void uFileAccess::WriteFailure::defaultResume() const {
    if ( errNo() != EIO ) {
	_Throw *this;
    } // if
} // uFileAccess::WriteFailure::defaultResume


uFileAccess::ReadFailure::ReadFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg ) :
	uFileAccess::Failure( fa, msg ), buf( buf ), len( len ), timeout( timeout ) {}

void uFileAccess::ReadFailure::defaultTerminate() const {
    uAbort( "(uFileAccess &)0x%p.read( buf:0x%p, len:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &fileAccess(), buf, len, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uFileAccess::ReadFailure::defaultTerminate

uFileAccess::ReadTimeout::ReadTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg ) :
	uFileAccess::ReadFailure( fa, buf, len, timeout, msg ) {}


uFileAccess::WriteFailure::WriteFailure( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg ) :
	uFileAccess::Failure( fa, msg ), buf( buf ), len( len ), timeout( timeout ) {}

void uFileAccess::WriteFailure::defaultTerminate() const {
    uAbort( "(uFileAccess &)0x%p.write( buf:0x%p, len:%d, timeout:0x%p ) : %.256s for file descriptor %d.\nError(%d) : %s.",
	    &fileAccess(), buf, len, timeout, message(), fileDescriptor(), errNo(), strerror( errNo() ) );
} // uFileAccess::WriteFailure::defaultTerminate

uFileAccess::WriteTimeout::WriteTimeout( const uFileAccess &fa, const char *buf, const int len, const uDuration *timeout, const char *const msg ) :
	uFileAccess::WriteFailure( fa, buf, len, timeout, msg ) {}


// Local Variables: //
// compile-command: "gmake install" //
// End: //
