//                              -*- Mode: C++ -*- 
// 
// uC++ Version 5.5.0, Copyright (C) Peter A. Buhr 1994
// 
// uC++.h -- 
// 
// Author           : Peter A. Buhr
// Created On       : Fri Dec 17 22:04:27 1993
// Last Modified By : Peter A. Buhr
// Last Modified On : Tue Sep 18 12:15:55 2007
// Update Count     : 4880
//
// This  library is free  software; you  can redistribute  it and/or  modify it
// under the terms of the GNU Lesser General Public License as published by the
// Free Software  Foundation; either  version 2.1 of  the License, or  (at your
// option) any later version.
// 
// This library is distributed in the  hope that it will be useful, but WITHOUT
// ANY  WARRANTY;  without even  the  implied  warranty  of MERCHANTABILITY  or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
// for more details.
// 
// You should  have received a  copy of the  GNU Lesser General  Public License
// along  with this library.
// 

// ***************************************************************************
// WARNING: If a .cc file is added to the kernel that does not define
// __U_KERNEL__ BEFORE including uC++.h, the profiler will likely fail to
// register processors created using the uDefaultProcessors mechanism.
// Furthermore, no explicit error is reported in this case -- the problem can
// only be detected by observing erroneous results reported by certain metrics
// that are sensitive to the registration of processors.
// ***************************************************************************


#ifndef __U_CPLUSPLUS_H__
#define __U_CPLUSPLUS_H__

#pragma __U_NOT_USER_CODE__

#if defined( __U_MULTI__ ) && defined( __U_TLS__ )
#define __U_THREAD__ __thread
#define __U_PTHREAD__
#ifndef __U_ONETIMER__
#define __U_ONETIMER__
#endif // ! __U_ONETIMER__
#else
#define __U_THREAD__
#endif // __U_MULTI__ && __U_TLS__

#if defined( __solaris__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
#include <sys/lwp.h>
#endif // __solaris__ && __U_MULTI__ && ! __U_PTHREAD__

/* The GNU Libc defines C library functions with throw () when compiled under
   C++, to enable optimizations.  When uC++ overrides these functions, it must
   provide identical exception specifications.  */
#if defined( __linux__ )
#define __THROW throw ()

/* Certain library functions have had __THROW removed from their prototypes
   to support the NPTL implementation of pthread cancellation. To compile with
   a pre-NPTL version of the header files use

   #define __OLD_THROW throw ()
*/
#define __OLD_THROW
#else
#define __OLD_THROW
#define __THROW
#endif // __linux__

/* ___errno (Solaris) and __errno_location (Linux) provide libc with the
   address of the task-specific errno variable.  Both functions should take no
   arguments; however, a gcc bug (Bug c++/11390) prevents functions defined in
   system headers from being interpreted properly when they are redeclared in
   user code, such as redeclaring as "friend"s.  So some creativity is
   required.

   Irix has a similar function called __oserror, but it is not called by the
   standard library if overridden in uC++; in this case, the system ensures
   that the errnos of different kernel threads stay independent, so we can
   simply save the per-task errno on each context switch. Solaris works in
   this way when pthreads are used. */
#if defined( __solaris__ )
#if ! defined( _REENTRANT )
#define _REENTRANT
#endif
#if ! defined( __U_TLS__ )
#if __GNUC__ > 3 || __GNUC_MINOR__ > 3
#define __U_ERRNO_FUNC__ ___errno()
#else
// TEMPORARY: bug fixed in gcc 3.4.0
// http://gcc.gnu.org/bugzilla/show_bug.cgi?id=11390
#define __U_ERRNO_FUNC__ ___errno(...)
#endif
#endif // ! __U_TLS__
#include <cerrno>

#elif defined( __linux__ )
#if ! defined( _LIBC_REENTRANT )
#define _LIBC_REENTRANT
#endif
#include <cerrno>
#if ( __GLIBC__ <= 2 ) && ( __GLIBC_MINOR__ < 3 )
#define __U_ERRNO_FUNC__ __errno_location( void )
#endif // ! glibc 2.3

#elif defined( __irix__ )
#if ! defined( _SGI_MP_SOURCE )
#define _SGI_MP_SOURCE
#endif
#include <cerrno>
#include <ulocks.h>

#elif defined( __freebsd__ )
#include <cerrno>
#define __U_ERRNO_FUNC__ __error()

#else
    #error uC++ : internal error, unsupported architecture
#endif

#ifdef __U_ERRNO_FUNC__
extern "C" int *__U_ERRNO_FUNC__ __THROW __attribute__(( used ));
#endif // __U_ERRNO_FUNC__

#if defined( __i386__ )
// #define __U_SWAPCONTEXT__				// use builtin swapcontext
#elif defined( __x86_64__ )
// no usable swapcontext on this platform
#elif defined( __ia64__ )
#define __U_SWAPCONTEXT__				// use builtin swapcontext
#elif defined( __sparc__ )
// #define __U_SWAPCONTEXT__				// use builtin swapcontext
#elif defined( __mips__ )
// #define __U_SWAPCONTEXT__				// use builtin swapcontext
#else
    #error uC++ : internal error, unsupported architecture
#endif

#include <uStaticAssert.h>				// access: _STATIC_ASSERT_
#include <assert.h>
//#include <uDebug.h>

#include <uAlign.h>
#include <uStack.h>
#include <uQueue.h>
#include <uSequence.h>
#include <uBitSet.h>
#include <uDefault.h>

#include "uKernelThreads.h"
#include "uAtomic.h"

#include <cstddef>					// ptrdiff_t
#include <cstdlib>					// malloc, calloc, realloc, free
#if defined( __linux__ )
#include <malloc.h>					// memalign
#include <link.h>					// dl_iterate_phdr
#endif // __linux__
#if defined( __freebsd__ )
extern "C" void *memalign( size_t alignment, size_t size );
#endif // __freebsd__
#include <csignal>					// signal, etc.
#include <ucontext.h>					// ucontext_t

#include <exception>
#include <iosfwd>					// std::filebuf
#include <pthread.h>					// PTHREAD_CANCEL_*
#include <unwind-cxx.h>					// struct __cxa_eh_globals


#if defined( __U_PTHREAD__ )
typedef pthread_t uPid_t;
#elif defined( __solaris__ ) && defined( __U_MULTI__ )
typedef lwpid_t uPid_t;
#else
typedef pid_t uPid_t;
#endif


#ifdef KNOT
#include <sys/socket.h>

extern "C" {
    /* Globals used to count actual syscalls */
    extern unsigned int accept_syscalls;
    extern unsigned int poll_syscalls;
    extern unsigned int sendfile_syscalls;
    extern unsigned int read_syscalls;
    extern unsigned int write_syscalls; 
    extern unsigned int epoll_ctl_syscalls;
    extern unsigned int epoll_wait_syscalls;

    extern unsigned int iopoller_advance;
    extern unsigned int iopoller_exchange;
    extern unsigned int iopoller_spin;

    /* Globals used to count syscall errors */
    extern unsigned int poll_eintr;
    extern unsigned int poll_nothing;
    extern unsigned int poll_events;
    extern unsigned int poll_errs;

    /* Caching statistics */
    extern unsigned long long cache_bytes_malloced;
    extern unsigned int cache_bytes_freed;
    extern unsigned int cache_bytes_requested;
    extern unsigned int cache_bytes_served;
    extern unsigned int cache_malloc_fails;
    extern unsigned int cache_evictions;

    /* Kernel statistics */
    extern unsigned int context_switches;

    void print_stats();
} // extern "C"
#endif // KNOT


#define __U_MAXENTRYBITS__ 128				// N-1 mutex members, including destructor, and timeout
#define __Monitor _Mutex class				// short form for monitor
#define _Monitor _Mutex class
#define uMonitor _Mutex class				// deprecated
#define __Cormonitor _Mutex _Coroutine			// short form for coroutine monitor
#define _Cormonitor _Mutex _Coroutine
#define uCorMonitor _Mutex _Coroutine			// deprecated

class uKernelBoot;					// forward declaration
_Task uBootTask;					// forward declaration
_Task uSystemTask;					// forward declaration
class uMachContext;					// forward declaration
class uBaseCoroutine;					// forward declaration
class uBaseTask;					// forward declaration
class uSpinLock;					// forward declaration
class uLock;						// forward declaration
class uOwnerLock;					// forward declaration
class uCondLock;					// forward declaration
class uProcessor;					// forward declaration
class uDefaultScheduler;				// forward declaration
class uCluster;						// forward declaration
_Task uProcessorTask;					// forward declaration
class uHeapManager;					// forward declaration
class uSerial;						// forward declaration
class uSerialMember;					// forward declaration
class uEventList;					// forward declaration
_Task uPthreadable;					// forward declaration
_Task uMain;						// forward declaration
class HWCounters;					// forward declaration
_Task uLocalDebugger;					// forward declaration
class uIOClosure;					// forward declaration

namespace UPP {
    _Task Pthread;					// forward declaration
    _Coroutine uProcessorKernel;			// forward declaration
    class uNBIO;					// forward declaration
} // UPP

extern uBaseCoroutine &uThisCoroutine();		// forward declaration
extern uBaseTask &uThisTask();				// forward declaration
extern uProcessor &uThisProcessor();			// forward declaration
extern uCluster &uThisCluster();			// forward declaration

namespace UPP {
#ifdef __U_PTHREAD__
    class uRealPthread {
	friend class ::uProcessor;			// access: pthread_create, pthread_attr_*
	friend _Task ::uProcessorTask;			// access: pthread_join
	friend class uProcessorKernel;			// access: pthread_exit
	friend class ::uCluster;			// access: pthread_kill
	friend class ::uKernelBoot;			// access: startup
	friend class ::HWCounters;			// access: pthread_kill

	static void startup();

	static typeof( ::pthread_create ) *pthread_create;
	static typeof( ::pthread_attr_init ) *pthread_attr_init;
	static typeof( ::pthread_attr_setstack ) *pthread_attr_setstack;
	static typeof( ::pthread_kill ) *pthread_kill;
	static typeof( ::pthread_join ) *pthread_join;
      public:
	static typeof( ::pthread_self ) *pthread_self;
    }; // uRealPthread
#endif // __U_PTHREAD__
} // UPP


//######################### Profiling ########################


_Task uProfiler;					// forward declaration
class uProfilerBoot;
class uProfileTaskSampler;
class uProfileClusterSampler;
class uProfileProcessorSampler;

extern "C" void __cyg_profile_func_enter( void *pcCurrentFunction, void *pcCallingFunction );
extern "C" void __cyg_profile_func_exit( void *pcCurrentFunction, void *pcCallingFunction );

#if defined( __solaris__ )
#define __U_HW_OVFL_SIG__ SIGEMT                        // hardware-counter overflow signal
#elif defined( __linux__ )
#define __U_HW_OVFL_SIG__ SIGIO
#endif // __solaris__


//######################### uAbort/exit #########################


extern void uAbort( const char *fmt = "", ... ) __attribute__(( format (printf, 1, 2), noreturn ));
extern void uExit( int code ) __attribute__(( deprecated, noreturn ));


//######################### Signal Handling #########################


// define parameter types for signal handlers

#define __U_SIGCXT__ ucontext_t *
#define __U_SIGPARMS__ int sig, siginfo_t *sfp, __U_SIGCXT__ cxt
#define __U_SIGTYPE__ int, siginfo_t *, __U_SIGCXT__

namespace UPP {
    class uSigHandlerModule {
	friend class ::uKernelBoot;			// access: uSigHandlerModule
	friend _Task ::uLocalDebugger;			// access: signal
	friend _Task ::uProfiler;			// access: signal, signalContextPC

	static void signal( int sig, void (*handler)(__U_SIGPARMS__), int flags = 0 );
	static void *signalContextPC( __U_SIGCXT__ cxt );
	static void *functionAddress( void (*function)() );
	static void sigChldHandler( __U_SIGPARMS__ );
	static void sigTermHandler( __U_SIGPARMS__ );
	static void sigAlrmHandler( __U_SIGPARMS__ );
	static void sigSegvBusHandler( __U_SIGPARMS__ );
	static void sigIllHandler( __U_SIGPARMS__ );
	static void sigFpeHandler( __U_SIGPARMS__ );

	uSigHandlerModule( uSigHandlerModule & );	// no copy
	uSigHandlerModule &operator=( uSigHandlerModule & ); // no assignment

	uSigHandlerModule();
      public:
    }; // uSigHandlerModule
} // UPP


//######################### uProcessor #########################


class uProcessorDL : public uSeqable {
    uProcessor &processor_;
  public:
    uProcessorDL( uProcessor &processor_ ) : processor_( processor_ ) {}
    uProcessor &processor() const { return processor_; }
}; // uProcessorDL

typedef uSequence<uProcessorDL> uProcessorSeq;


//######################### uCluster #########################


class uClusterDL : public uSeqable {
    uCluster &cluster_;
  public:
    uClusterDL( uCluster &cluster_ ) : cluster_( cluster_ ) {}
    uCluster &cluster() const { return cluster_; }
}; // uClusterDL

typedef uSequence<uClusterDL> uClusterSeq;


//######################### uBaseTask #########################


class uBaseTaskDL : public uSeqable {
    uBaseTask &task_;
  public:
    uBaseTaskDL( uBaseTask &task_ ) : task_( task_ ) {}
    uBaseTask &task() const { return task_; }
}; // uBaseTaskDL

typedef uSequence<uBaseTaskDL> uBaseTaskSeq;


//######################### uKernelModule #########################


extern "C" {						// TEMPORARY: profiler allocating memory from the kernel issue
    int pthread_mutex_lock( pthread_mutex_t *mutex ) __THROW;
    int pthread_mutex_trylock( pthread_mutex_t *mutex ) __THROW;
    int pthread_mutex_unlock( pthread_mutex_t *mutex ) __THROW;
    void pthread_shutdown_();
} // extern "C"


#define __U_START_KT_NAME__ uKernelModule::startThread
#if defined( __U_PTHREAD__ )
#define __U_START_KT_TYPEGEN__( name ) void *name( void *p )
#elif defined( __linux__) || defined( __freebsd__ )
#define __U_START_KT_TYPEGEN__( name ) int name( void *p )
#elif defined( __solaris__ )
#define __U_START_KT_TYPEGEN__( name ) void name( void *p )
#elif defined( __irix__ )
#define __U_START_KT_TYPEGEN__( name ) void name( void *p, size_t size )
#else
    #error uC++ : internal error, unsupported architecture
#endif
#define __U_START_KT_TYPE__ __U_START_KT_TYPEGEN__( __U_START_KT_NAME__ )
#define __U_START_KT_TYPE_UNQUAL__ __U_START_KT_TYPEGEN__( startThread )


class uKernelModule {
    friend void uAbort( const char *fmt, ... );		// access: globalAbort, coreDumped
    friend void exit( int retcode ) __THROW;		// access: globalAbort
    friend class UPP::uSigHandlerModule;		// access: uKernelModuleBoot, globalAbort, coreDumped, rollForward
    friend class uMachContext;				// access: everything
    friend class uBaseCoroutine;			// access: uKernelModuleBoot
    friend class uBaseTask;				// access: uKernelModuleBoot
    friend class uSerial;				// access: uKernelModuleBoot
    friend class uSpinLock;				// access: uKernelModuleBoot
    friend class uOwnerLock;				// access: uKernelModuleBoot, initialization
    friend class uCondLock;				// access: uKernelModuleBoot
    friend class uContext;				// access: uKernelModuleBoot
    friend class uCoroutineMain;			// access: uKernelModuleBoot
    friend _Coroutine UPP::uProcessorKernel;		// access: uKernelModuleBoot, globalProcessors, globalClusters, systemProcessor
    friend class uProcessor;				// access: everything
    friend uBaseTask &uThisTask();			// access: uKernelModuleBoot
    friend uProcessor &uThisProcessor();		// access: uKernelModuleBoot
    friend uCluster &uThisCluster();			// access: uKernelModuleBoot
    friend _Task uProcessorTask;			// access: uKernelModuleBoot
    friend class uCluster;				// access: uKernelModuleBoot, globalClusters, globalClusterLock, rollForward
    friend _Task uBootTask;				// access: uKernelModuleBoot, systemCluster
    friend _Task uSystemTask;				// access: systemCluster
    friend _Task uMain;					// access: bootTask
    friend class uKernelBoot;				// access: everything
    friend class uInitProcessorsBoot;			// access: numProcessors, userProcessors
    friend class uHeapManager;				// access: bootTaskStorage, kernelModuleInitialized, startup
    friend class UPP::uNBIO;				// access: uKernelModuleBoot
#ifdef __U_PTHREAD__
    friend class UPP::uRealPthread;			// access: interposeSymbol
#endif // __U_PTHREAD__
#if defined( __linux__ )
    friend int dl_iterate_phdr( int (*callback)( dl_phdr_info *, size_t, void * ), void *data ); // access: interposeSymbol
#endif // __linux__

    // real-time

    friend class uSysEventList;				// access: uKernelModuleBoot
#if defined( __U_ONETIMER__ ) && defined( __U_MULTI__ )
    friend class uProcEventList;			// access: processorKernelStorage
#endif // __U_ONETIMER__ && defined( __U_MULTI__ )
    friend class uEventList;                            // access: uKernelModuleBoot
    friend class uEventListPop;				// access: uKernelModuleBoot

    // debugging

    friend _Task uLocalDebugger;			// access: uKernelModuleBoot, bootTask, globalClusters, systemProcessor, systemCluster
    friend class uLocalDebuggerHandler;			// access: uKernelModuleBoot

    // profiling

    friend int pthread_mutex_lock( pthread_mutex_t *mutex ) __THROW; // TEMPORARY: profiler allocating memory from the kernel issue
    friend int pthread_mutex_trylock( pthread_mutex_t *mutex ) __THROW; // TEMPORARY: profiler allocating memory from the kernel issue
    friend int pthread_mutex_unlock( pthread_mutex_t *mutex ) __THROW; // TEMPORARY: profiler allocating memory from the kernel issue
    friend void __cyg_profile_func_enter( void *pcCurrentFunction, void *pcCallingFunction ); // access: uKernelModuleBoot
    friend void __cyg_profile_func_exit( void *pcCurrentFunction, void *pcCallingFunction ); // access: uKernelModuleBoot
    friend class uProfiler;				// access: uKernelModuleBoot
    friend class uProfilerBoot;				// access: uKernelModuleBoot, bootTask, systemCluster, userProcessor
    friend class HWCounters;				// access: uKernelModuleBoot
    friend class CGMonitor;				// access: uKernelModuleBoot
    template<typename Elem, unsigned int BlockSize, typename Admin> friend struct uFixedListArray; // access: uKernelModuleBoot

    friend class uKernelSampler;			// access: globalClusters
    friend class uClusterSampler;			// access: globalClusters
    friend void pthread_shutdown_();			// access: abortExit()

    struct uKernelModuleData {
#if defined( __i386__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
	unsigned int ldtValue;
#endif // __i386__

	volatile uKernelModuleData *This;

#	define __U_MAX_PAGE_SIZE__ (16 * 1024)
#	define activeProcessorKernel (THREAD_GETMEM( processorKernelStorage ))
	uProcessor *activeProcessor;			// current active processor
	// The next two private variables shadow the corresponding fields in the
	// processor data structure. They are an optimization so that routines
	// uThisCluster and uThisTask do not have to be atomic routines, and as a
	// consequence can be inlined. The problem is the multiple activeProcessor
	// variables (one per UNIX process). A task executing on one processor can
	// be time sliced after loading the address of the active processor into a
	// register, rescheduled on another processor and restarted, but now the
	// previously loaded processor pointer is incorrect. By updating these two
	// shadow variables whenever the corresponding processor field is changed
	// (and this occurs atomically in the kernel), the appropriate data
	// structure (cluster or task) can be accessed with a single load
	// instruction, which is atomic.
	uCluster  *activeCluster;			// current active cluster for processor
	uBaseTask *activeTask;				// current active task for processor

	bool disableInt;				// task in kernel: no time slice interrupts
	unsigned int disableIntCnt;

	bool disableIntSpin;				// task in spin lock; no time slice interrupts
	unsigned int disableIntSpinCnt;

	unsigned int inKernelRF;			// in kernel roll forward flag

	UPP::uProcessorKernel *processorKernelStorage;	// system-cluster processor kernel

	// The thread pointer value needs to be accessible so that it can be
	// properly restored on context switches.  On a non-tls system the thread
	// pointer points directly at the kernel module, i.e. tp == This.  On a
	// tls system the system places the kernel module, so tp != This.
	unsigned long threadPointer;

	void disableInterrupts() volatile {
	    THREAD_SETMEM( disableInt, true );
	    int old = THREAD_GETMEM( disableIntCnt );
	    THREAD_SETMEM( disableIntCnt, old + 1 );
	} // uKernelModule::uKernelModuleData::disableInterrupts

	void enableInterrupts() volatile {
#ifdef __U_DEBUG__
	    assert( disableInt && disableIntCnt > 0 );
#endif // __U_DEBUG__

	    disableIntCnt -= 1;				// decrement number of disablings
	    if ( disableIntCnt == 0 ) {
		// Enable interrupts first, otherwise a window exists in which a
		// signal could occur but a rollforward is not done.
		disableInt = false;			// enable interrupts
		if ( THREAD_GETMEM( inKernelRF ) && ! THREAD_GETMEM( disableIntSpin ) ) { // SIGALRM undelivered and not in kernel?
		    rollForward();			// deliver SIGALRM
		} // if
	    } // if
#ifdef __U_DEBUG__
	    assert( ( ! THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) == 0 ) || ( THREAD_GETMEM( disableInt ) && THREAD_GETMEM( disableIntCnt ) > 0 ) );
#endif // __U_DEBUG__
	} // KernelModule::uKernelModuleData::enableInterrupts

	void disableIntSpinLock() volatile {
#ifdef __U_DEBUG__
	    assert( ( ! THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) == 0 ) || ( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 ) );
#endif // __U_DEBUG__

	    THREAD_SETMEM( disableIntSpin, true );
	    int old = THREAD_GETMEM( disableIntSpinCnt ); // processor independent increment
	    THREAD_SETMEM( disableIntSpinCnt, old + 1 );

#ifdef __U_DEBUG__
	    assert( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__
	} // uKernelModule::uKernelModuleData::disableIntSpinLock

	void enableIntSpinLock() volatile {
#ifdef __U_DEBUG__
	    assert( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__

	    disableIntSpinCnt -= 1;			// decrement number of disablings
	    if ( disableIntSpinCnt == 0 ) {
		disableIntSpin = false;			// enable interrupts

		if ( THREAD_GETMEM( inKernelRF ) && ! THREAD_GETMEM( disableInt ) ) { // SIGALRM undelivered and not in kernel?
		    rollForward();			// deliver SIGALRM
		} // if
	    } // if

#ifdef __U_DEBUG__
	    assert( ( ! THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) == 0 ) || ( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 ) );
#endif // __U_DEBUG__
} // uKernelModule::uKernelModuleData::enableIntSpinLock

	void enableIntSpinLockNoRF() volatile {
#ifdef __U_DEBUG__
	    assert( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 );
#endif // __U_DEBUG__

	    disableIntSpinCnt -= 1;			// decrement number of disablings
	    if ( disableIntSpinCnt == 0 ) {
		disableIntSpin = false;			// enable interrupts
	    } // if

#ifdef __U_DEBUG__
	    assert( ( ! THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) == 0 ) || ( THREAD_GETMEM( disableIntSpin ) && THREAD_GETMEM( disableIntSpinCnt ) > 0 ) );
#endif // __U_DEBUG__
	} // uKernelModule::uKernelModuleData::enableIntSpinLock

	void ctor() volatile;				// POD constructor
    }; // uKernelModuleData


    // shared, initialized in uC++.cc

    static bool kernelModuleInitialized;
    static volatile __U_THREAD__ uKernelModuleData uKernelModuleBoot;
    static bool initialization;				// initialization/finalization done
    static unsigned int attaching;			// flag to signal the local kernel to start attaching.
    static bool coreDumped;				// ensure only one core file
#ifndef __U_MULTI__
    static bool deadlock;				// deadlock detected in kernel
#endif // ! __U_MULTI__
    static bool globalAbort;				// indicate aborting processor
    static bool globalSpinAbort;			// indicate aborting processor to spin locks
    static uSpinLock *globalAbortLock;			// only one aborting processors
    static uSpinLock *globalProcessorLock;		// mutual exclusion for global processor operations
    static uProcessorSeq *globalProcessors;		// global list of processors
    static uSpinLock *globalClusterLock;		// mutual exclusion for global cluster operations
    static uClusterSeq *globalClusters;			// global list of cluster
    static uDefaultScheduler *systemScheduler;		// pointer to systen scheduler for system cluster
    static uCluster *systemCluster;			// pointer to system cluster
    static uProcessor *systemProcessor;			// pointer to system processor
    static uBootTask *bootTask;				// pointer to boot task for global constructors/destructors
    static uProcessor **userProcessors;			// pointer to user processors
    static unsigned int numProcessors;			// number of user processors
    static char systemProcessorStorage[];
    static char systemClusterStorage[];
    static char bootTaskStorage[];

    static std::filebuf *cerrFilebuf, *clogFilebuf, *coutFilebuf, *cinFilebuf;

    static void rollForward( bool inKernel = false );
    static void processEvents( uEventList *events, bool inKernel = false );
    static __U_START_KT_TYPE_UNQUAL__;

    static void abortExit();
    static void startup();				// init boot KM
    static void *interposeSymbol( const char *symbolName, const char *version = NULL );
  public:
    static uSystemTask *systemTask;			// pointer to system task for global constructors/destructors
    static uCluster *userCluster;			// pointer to user cluster

    static bool afterMain;
    static bool inExit;
    static int retCode;
#if defined( __ia64__ ) && ! defined( __U_TLS__ )
    // Last machine word of this structure must be reserved for the "in atomic
    // offset load/store" flag (see uKernelThread.h).
    unsigned long IA64OffsetStoreFlag;
#endif // __ia64__
}; // uKernelModule


inline uProcessor &uThisProcessor() {
    return *THREAD_GETMEM( activeProcessor );
} // uThisProcessor


inline uCluster &uThisCluster() {
    return *THREAD_GETMEM( activeCluster );
} // uThisCluster


inline uBaseTask &uThisTask() {
    return *THREAD_GETMEM( activeTask );
} // uThisTask


//######################### uSpinLock #########################


class uSpinLock {					// non-yielding spinlock
    friend class uKernelBoot;				// access: new
    friend class uEventList;				// access: acquireNoRF, releaseNoRF

    unsigned int value;

    uSpinLock( uSpinLock & );				// no copy
    uSpinLock &operator=( uSpinLock & );		// no assignment

    void *operator new( size_t, void *storage ) {
	return storage;
    } // uSpinLock::operator new

    void *operator new( size_t size ) {
	return ::operator new( size );
    } // uSpinLock::operator new

    void acquireNoRF();
    void releaseNoRF();
  public:
    uSpinLock() {
	value = 0;					// unlock
    } // uSpinLock::uSpinLock

    void acquire();
    bool tryacquire();
    void release();
}; // uSpinLock


// RAII mutual-exclusion lock.  Useful for mutual exclusion in free routines.
// Handles exception termination and multiple block exit or return.

class uCSpinLock {
    uSpinLock &spinLock;

    uCSpinLock( uCSpinLock & );				// no copy
    uCSpinLock &operator=( uCSpinLock & );		// no assignment
  public:
    uCSpinLock( uSpinLock &spinLock ) : spinLock( spinLock ) {
	spinLock.acquire();
    } // uCSpinLock::uCSpinLock

    ~uCSpinLock() {
	spinLock.release();
    } // uCSpinLock::~uCSpinLock
}; // uCSpinLock


//######################### uLock #########################

class uLock {						// yielding spinlock
    unsigned int value;
    uSpinLock spinLock;

    uLock( uLock & );					// no copy
    uLock &operator=( uLock & );			// no assignment
  public:
    uLock() {
	value = 1;
    } // uLock::uLock

    uLock( unsigned int val ) {
#ifdef __U_DEBUG__
	if ( val > 1 ) {
	    uAbort( "Attempt to initialize uLock 0x%p to %d that exceeds range 0-1.", this, val );
	} // if
#endif // __U_DEBUG__
	value = val;
    } // uLock::uLock

    void acquire();
    bool tryacquire();

    void release() {
	value = 1;
    } // uLock::release
}; // uLock


//######################### Abnormal Event Handling #########################


#include <uEHM.h>


_Event uKernelFailure {
  protected:
    uKernelFailure( const char *const msg = "" );
  public:
    virtual ~uKernelFailure();
    virtual void defaultTerminate() const;
}; // uKernelFailure


_Event uIOFailure {
    // general IO exception-type
    int uErrno;
  protected:
    uIOFailure( const char *const msg );
  public:
    virtual ~uIOFailure();
    int errNo() const;
}; // uIOFailure


//######################### Real-Time #########################


#include <uCalendar.h>


class uSignalHandler : public uColable {
  protected:
    uBaseTask *This;
    virtual ~uSignalHandler() {}
  public:
    virtual void uHandler() = 0;
}; // uSignalHandler


#include <uAlarm.h>


class uCxtSwtchHndlr : public uSignalHandler {
    friend class uKernelBoot;				// access: constructor
    friend class uProcessor;				// access: constructor

    void uHandler();
}; // uCxtSwtchHndlr


//######################### uOwnerLock #########################


#if defined( __irix__ )
extern "C" int _usinitsema( usema_t *sem, int val );
#endif // __irix__

class uOwnerLock {
    friend class uCondLock;				// access: add_, release_
#if defined( __irix__ )
    friend int _usinitsema( usema_t *sem, int val );	// new
#endif // __irix__

    // These data fields must be initialized to zero. Therefore, this lock can
    // be used in the same storage area as a pthread_mutex_t, if
    // sizeof(pthread_mutex_t) >= sizeof(uOwnerLock).

    unsigned int count;					// number of recursive entries; no overflow checking
    // Solaris has a magic value in its pthread locks, so place the spin lock in
    // that position as it cannot take on the magic value (see library/pthread.cc).
    uSpinLock spin;
    uBaseTask *owner_;					// owner with respect to recursive entry
    uQueue<uBaseTaskDL> waiting;

    uOwnerLock( uOwnerLock & );				// no copy
    uOwnerLock &operator=( uOwnerLock & );		// no assignment

    void add_( uBaseTask &task );			// helper routines for uCondLock
    void release_();
  public:
    uOwnerLock() {
	owner_ = NULL;					// no one owns the lock
	count = 0;					// so count is zero
    } // uOwnerLock::uOwnerLock

#ifdef __U_DEBUG__
    ~uOwnerLock();
#endif // __U_DEBUG__

    unsigned int times() const {
	return count;
    } // uOwnerLock::times

    uBaseTask *owner() const {
	return owner_;
    } // uOwnerLock::times

    void acquire();
    bool tryacquire();
    void release();

    void *operator new( size_t, void *storage ) {
	return storage;
    } // uOwnerLock::operator new

    void *operator new( size_t size ) {
	return ::operator new( size );
    } // uOwnerLock::operator new
}; // uOwnerLock


//######################### uCondLock #########################


class uCondLock {
    struct TimedWaitHandler : public uSignalHandler {	// real-time
	uCondLock &condlock;
	bool timedout;

	TimedWaitHandler( uBaseTask &task, uCondLock &condlock );
	TimedWaitHandler( uCondLock &condlock );
	void uHandler();
    }; // TimedWaitHandler

    // These data fields must be initialized to zero. Therefore, this lock can
    // be used in the same storage area as a pthread_cond_t, if
    // sizeof(pthread_cond_t) >= sizeof(uCondLock).

    uSpinLock spin;
    uQueue<uBaseTaskDL> waiting;

    uCondLock( uCondLock & );				// no copy
    uCondLock &operator=( uCondLock & );		// no assignment
    void waitTimeout( uBaseTask &task, TimedWaitHandler &h ); // timeout
  public:
    uCondLock() {}
#ifdef __U_DEBUG__
    ~uCondLock();
#endif // __U_DEBUG__
    bool empty() const {
	return waiting.empty();
    } // uCondLock::empty

    void wait( uOwnerLock &lock );
    bool wait( uOwnerLock &lock, uDuration duration );
    bool wait( uOwnerLock &lock, uTime time );
    void signal();
    void broadcast();

    void *operator new( size_t, void *storage ) {
	return storage;
    } // uCondLock::operator new

    void *operator new( size_t size ) {
	return ::operator new( size );
    } // uCondLock::operator new
}; // uCondLock


//######################### uSemaphore #########################


namespace UPP {
    class uSemaphore {
	struct TimedWaitHandler : public uSignalHandler { // real-time
	    uSemaphore &semaphore;
	    bool timedout;

	    TimedWaitHandler( uBaseTask &task, uSemaphore &semaphore );
	    TimedWaitHandler( uSemaphore &semaphore );
	    void uHandler();
	}; // TimedWaitHandler

	// These data fields must be initialized to zero. Therefore, this lock
	// can be used in the same storage area as a sem_t, if sizeof(sem_t) >=
	// sizeof(uSemaphore).

	uSpinLock spin;
	int count;
	uQueue<uBaseTaskDL> waiting;

	void waitTimeout( uBaseTask &task, TimedWaitHandler &h );

	uSemaphore( uSemaphore & );			// no copy
	uSemaphore &operator=( uSemaphore & );		// no assignment
      public:
	uSemaphore( unsigned int count = 1 ) : count( count ) {
	} // uSemaphore::uSemaphore

	void P();					// wait on semaphore
	bool P( uDuration duration );			// wait on semaphore or timeout
	bool P( uTime time );				// wait on semaphore or timeout
	void P( uSemaphore &s );			// wait on semaphore and release another
	bool P( uSemaphore &s, uDuration duration );	// wait on semaphore and release another or timeout
	bool P( uSemaphore &s, uTime time );		// wait on semaphore and release another or timeout
	bool TryP();					// conditionally wait on a semaphore
	void V();					// signal semaphore
	void V( unsigned int times );			// signal semaphore

	int counter() const {				// semaphore counter
	    return count;
	} // uSemaphore::counter

	bool empty() const {				// no tasks waiting on semaphore ?
	    return count >= 0;
	} // uSemaphore::empty

	void *operator new( size_t, void *storage ) {
	    return storage;
	} // uSemaphore::operator new

	void *operator new( size_t size ) {
	    return ::operator new( size );
	} // uSemaphore::operator new
    }; // uSemaphore
} // UPP


//######################### uContext #########################


class uContext : public uSeqable {
    void *key;
  public:
    uContext();
    uContext( void *key );
    virtual ~uContext();

    // These two routines cannot be abstract (i.e., = 0) because there is a
    // race condition during initialization of a derived class when the base
    // class constructor is invoked. A context switch can occur immediately
    // after the base instance has put itself on the task context list but
    // before the virtual function vector is updated for the derived class.
    // Hence, the save or restore routine of the base class may be called.
    // This situtation is not a problem because either the task has not
    // performed any operations that involve the new context or the task is
    // removing the context and not performing anymore operations using it.

    virtual void save();
    virtual void restore();
}; // uContext

typedef uSequence<uContext> uContextSeq;


//######################### uFloatingPointContext #########################


// Provide floating point context switch support for coroutines and tasks.  It
// is built with the more general context class in the same way that a user can
// extend the amount of context that is saved and restored with a coroutine or
// task.

#if defined( __i386__ )
// saved by caller
#elif defined( __x86_64__ )
// saved by caller
#elif defined( __ia64__ )
#if ! defined( __U_SWAPCONTEXT__ )
#define __U_FLOATINGPOINTDATASIZE__ 40			// 20 16-byte registers
#endif // ! __U_SWAPCONTEXT__
#elif defined( __sparc__ )
// saved by caller
#elif defined( __mips__ )
#ifdef _ABIN32
#define __U_FLOATINGPOINTDATASIZE__ 6			// doubles
#elif _ABI64
#define __U_FLOATINGPOINTDATASIZE__ 8			// doubles
#else
    #error uC++ : internal error, unsupported architecture
#endif
#else
    #error uC++ : internal error, unsupported architecture
#endif


// Some architectures store the floating point registers with the integers
// registers during a basic context switch, so there is no need for a data area
// to storage the floating point registers.

#ifdef __U_FLOATINGPOINTDATASIZE__
class uFloatingPointContext : public uContext {
    static int uniqueKey;
    double floatingPointData[__U_FLOATINGPOINTDATASIZE__] __attribute__(( aligned(16) ));
  public:
    uFloatingPointContext();
#else
class uFloatingPointContext {
  public:
#endif // __U_FLOATINGPOINTDATASIZE__
    void save();					// save and restore the floating point context
    void restore();
} __attribute__(( unused )); // uFloatingPointContext


//######################### uMachContext #########################


#if defined( __U_SWAPCONTEXT__ )
#define uSwitch( from, to ) swapcontext( (ucontext_t *)(from), (ucontext_t *)(to) )
#define __U_CONTEXT_T__ ucontext_t
#else
#define __U_CONTEXT_T__ uContext_t
extern "C" void uSwitch( void *from, void *to ) asm ("uSwitch"); // assembler routine that performs the context switch
#endif


// Define a class that contains the machine dependent context and routines that
// initialize and switch between contexts.

class uMachContext {
    friend class uContext;				// access: extras, additionalContexts
    friend class uProcessorTask;			// access: size, base, limit
    friend class uBaseCoroutine;			// access: storage
    friend class uCoroutineConstructor;			// access: startHere
    friend class uTaskConstructor;			// access: startHere
    friend _Coroutine UPP::uProcessorKernel;		// access: storage
    friend class uProcessor;				// access: storage
    friend class uKernelBoot;				// access: storage
    friend __U_START_KT_TYPE__;				// acesss: invokeCoroutine

    struct uContext_t {
	void *SP;
#if ! defined( __ia64__ )
	void *FP;
	void *PC;
#else
	void *BSP;					// register backing store pointer
	sigset_t sigMask;
#endif
    };

    unsigned int size;					// size of stack
    void *storage;					// pointer to stack
    void *limit;					// stack grows towards stack limit
    void *base;						// base of stack
    void *context;					// address of uContext_t
    union {
	long int allExtras;				// allow access to all extra flags
	struct {					// put all extra flags in this structure
	    unsigned int usercxts : 1;			// user defined contexts
	} is;
    } extras;						// indicates extra work during the context switch
    bool userStack;					// use specified stack storage ?

    void createContext( void *storage, unsigned int stackSize ); // used by all constructors

    // This routine returns the magic word that is placed at both ends of an
    // allocated stack.  Later, those locations can be checked to see if the
    // magic word is still intact.

    unsigned int magic();

    void startHere( void (*uInvoke)( uMachContext & ) );

    uMachContext( uMachContext & );			// no copy
    uMachContext &operator=( uMachContext & );		// no assignment
  protected:
    static void invokeCoroutine( uBaseCoroutine &This ) __attribute__(( noreturn ));
    static void invokeTask( uBaseTask &This ) __attribute__(( noreturn ));
    static void cleanup( uBaseTask &This ) __attribute__(( noreturn ));

    uContextSeq additionalContexts;			// list of additional contexts for this execution state

    void extraSave();
    void extraRestore();

    void save() {
	// Any extra work that must occur on this side of a context switch is
	// performed here.
	if ( extras.allExtras ) {
	    extraSave();
	} // if

#ifdef __U_DEBUG__
	verify();
#endif // __U_DEBUG__
    } // uMachContext::save

    void restore() {
#ifdef __U_DEBUG__
	verify();
#endif // __U_DEBUG__

	// Any extra work that must occur on this side of a context switch is
	// performed here.
	if ( extras.allExtras ) {
	    extraRestore();
	} // if
    } // uMachContext::restore

    virtual void main() = 0;				// starting routine for coroutine or task
  public:
    uMachContext( unsigned int stackSize ) {
	// stack storage provides a minimum of stackSize memory for the stack plus ancillary storage
	createContext( NULL, uCeiling( stackSize, 16 ) );
    } // uMachContext::uMachContext

    uMachContext( void *storage, unsigned int storageSize ) {
	createContext( storage, uFloor( storageSize, 16 ) - uCeiling( sizeof(__U_CONTEXT_T__), 16 ) - 32 );
    } // uMachContext::uMachContext

    virtual ~uMachContext() {
	if ( ! userStack ) free( storage );
    } // uMachContext::~uMachContext

    void *stackPointer() const;
#if defined( __ia64__ )
    void *registerStackPointer() const;
#endif // __ia64__

    unsigned int stackSize() const {
	return size;
    } // uMachContext::stackSize

    void *stack() const {
	return storage;
    } // uMachContext::stack

    ptrdiff_t stackFree() const;
    ptrdiff_t stackUsed() const;
    void verify();

    // These members should be private but cannot be because they are
    // referenced from user code.

    static void *rtnAdr( void (*rtn)() );		// access: see profiler
#if defined( __ia64__ )
    static void *gpAdr( void (*rtn)() );
#endif // __ia64__
}; // uMachContext


//######################### uBaseCoroutine #########################


extern "C" void pthread_exit( void *status );

class uBaseCoroutine : public uMachContext {
    friend class uBaseTask;				// access: serial, profileTaskSamplerInstance
    friend class uTaskDestructor;			// access: profileTaskSamplerInstance
    friend class uKernelBoot;				// access: last
    friend class uMachContext;				// access: notHalted, main, suspend, setState, corStarter, corFinish
    friend _Task uBootTask;				// access: notHalted
    friend class uCoroutineConstructor;			// access: name, serial
    friend class uCoroutineDestructor;			// access: UnwindStack
    friend class uTaskConstructor;			// access: name, serial
    friend _Coroutine UPP::uProcessorKernel;		// access: contextSw
#ifdef __U_ERRNO_FUNC__
    friend int *__U_ERRNO_FUNC__ __THROW;		// access: uErrno
#endif // __U_ERRNO_FUNC__

    // cancellation

    friend _Task uPthreadable;				// access: Cleanup, PthreadCleanup

    // exception handling

    friend class uEHM;					// access: resumedObj, topResumedType, DEStack, handlerStackTop, handlerStackVisualTop, unexpectedRtn
    friend class uEHM::ResumeWorkHorseInit;		// access: resumedObj, topResumedType, handlerStackVisualTop
    friend class uEHM::uResumptionHandlers;		// access: handlerStackTop, handlerStackVisualTop
    friend class uEHM::uDeliverEStack;			// access: DEStack
    friend std::unexpected_handler std::set_unexpected( std::unexpected_handler func ) throw(); // access: unexpectedRtn
    friend void uEHM::unexpected();			// access: unexpected

    // profiler

    friend class uProfilerBoot;                         // access: serial
    friend class uProfileTaskSampler;			// access: profileTaskSamplerInstance
  public:
    enum State { Halt, Active, Inactive };
    enum CancellationState { CancelEnabled = PTHREAD_CANCEL_ENABLE, CancelDisabled = PTHREAD_CANCEL_DISABLE };
    enum CancellationType { CancelPoll = PTHREAD_CANCEL_DEFERRED, CancelImplicit = PTHREAD_CANCEL_ASYNCHRONOUS };
  private:
    const char *name;					// textual name for coroutine/task, initialized by uC++ generated code
    uBaseCoroutine *starter_;				// first coroutine to resume this one
    uSerial *serial;					// original serial instance for cormonitor/task (versus currently used instance)
    int uErrno;						// copy of global UNIX variable errno
    State state;					// current execution status for coroutine
    bool notHalted;					// indicate if execuation state is not halted

    uBaseCoroutine *last;				// last coroutine to resume this one
    uBaseTask *currSerialOwner;				// task accessing monitors from this coroutine
    unsigned int currSerialCount;			// counter to determine when to unset uCurrSerialOwner

    // cancellation

    _Event UnwindStack;
    bool cancelled_;					// cancellation flag
    bool cancelInProgress_;				// cancellation in progress flag
    CancellationState cancelState;			// enabled/disabled
    CancellationType cancelType;			// deferred/asynchronous

#if defined( __irix__ )    
    typedef __pthread_cncl_hdlr PthreadCleanup;
    typedef PthreadCleanup *Cleanup;
#else    
    struct PthreadCleanup : uColable {
	void (* routine )(void *);
	void * args;
#if defined( __freebsd__ )
	void * sp;
	void *operator new( size_t size ) {
	    return ::operator new( size );
	} // Pthread_cleanup::operator new
#else
	void *operator new( size_t, void *storage ) {
	    return storage;
	} // Pthread_cleanup::operator new
#endif
    }; // PthreadCleanup

    typedef uStack<PthreadCleanup> Cleanup;
#endif
    void unwindStack();    

    // exception handling

    uEHM::uResumptionHandlers *handlerStackTop, *handlerStackVisualTop;
    uBaseEvent *resumedObj;				// the object that is currently being handled during resumption
    const std::type_info *topResumedType;		// the top of the currently handled resumption stack (unchanged during stack unwind through EH)
    uEHM::uDeliverEStack *DEStack;			// manage exception enable/disable
    std::unexpected_handler unexpectedRtn;		// per coroutine handling unexpected action
    bool unexpected;					// indicate if unexpected error occurs

    __cxxabiv1::__cxa_eh_globals ehGlobals;
    friend __cxxabiv1::__cxa_eh_globals *__cxxabiv1::__cxa_get_globals_fast() throw();
    friend __cxxabiv1::__cxa_eh_globals *__cxxabiv1::__cxa_get_globals() throw();

    // profiling

    mutable uProfileTaskSampler *profileTaskSamplerInstance; // pointer to related profiling object

    void createCoroutine();

    void setState( State s ) {
	state = s;
    } // uBaseCoroutine::setState

    void contextSw();					// switch between a task and the kernel
    void contextSw2();					// switch between two coroutine contexts

    void corStarter() {					// remembers who started a coroutine
	starter_ = last;
    } // uBaseCoroutine::corStarter
    void corFinish() __attribute__(( noreturn ));

    uBaseCoroutine( uBaseCoroutine & );			// no copy
    uBaseCoroutine &operator=( uBaseCoroutine & );	// no assignment
  protected:
    // Duplicate "main" (see uMachContext) to get better error message about
    // missing "main" member for coroutine.
    virtual void main() = 0;				// starting routine for coroutine
    // Only allow direct access, i.e., preclude indirect access C.resume()/C.suspend()
    void resume();
    void suspend();
  public:
    uBaseCoroutine();

    uBaseCoroutine( unsigned int stackSize ) : uMachContext( stackSize ) {
	createCoroutine();
    } // uBaseCoroutine::uBaseCoroutine

    uBaseCoroutine( void *storage, unsigned int storageSize ) : uMachContext( storage, storageSize ) {
	createCoroutine();
    } // uBaseCoroutine::uBaseCoroutine

    const char *setName( const char *name );
    const char *getName() const;

    State getState() const {
	return notHalted ? state : Halt;
    } // uBaseCoroutine::getState

    uBaseCoroutine &starter() const {			// starter coroutine => did first resume
	return *starter_;
    } // uBaseCoroutine::starter

    uBaseCoroutine &resumer() const {			// last resumer coroutine
	return *last;
    } // uBaseCoroutine::resumer

    // cancellation

    void cancel() { cancelled_ = true; }
    bool cancelled() { return cancelled_; }
    bool cancelInProgress() { return cancelInProgress_; }

    // exception handling

    _Event Failure;					// general failure
    _Event UnhandledException;
  private:
    void handleUnhandled( const UnhandledException *ex );
    void handleUnhandled( const uBaseEvent *ex = NULL );
  public:
    // These members should be private but cannot be because they are
    // referenced from user code.

    void setCancelState( CancellationState state );
    CancellationState getCancelState() { return cancelState; }
    void setCancelType( CancellationType type );
    CancellationType getCancelType() { return cancelType; }

    template<CancellationState newState> class Cancel {	// RAII helper to enable/disable cancellation
	CancellationState prev;
      public:
	Cancel() {
	    uBaseCoroutine &coroutine = uThisCoroutine();
	    prev = coroutine.getCancelState();
	    coroutine.setCancelState( newState );
	} // Cancel::Cancel

	~Cancel() {
	    uThisCoroutine().setCancelState( prev );
	} // Cancel::~Cancel
    }; // uBaseCoroutine::Cancel

    uEHM::AsyncEMsgBuffer asyncEBuf;			// list of pending nonlocal exceptions
}; // uBaseCoroutine


_Event uBaseCoroutine::Failure : public uKernelFailure {
  protected:
    Failure( const char *const msg = "" );
  public:
}; // uBaseCoroutine::Failure


_Event uBaseCoroutine::UnhandledException : public uBaseCoroutine::Failure {
    friend class uBaseCoroutine;			// access: all

    const uBaseEvent *cause;				// initial exception
    uBaseCoroutine &origFailedCor;
    char origFailedCorName[uEHMMaxName];
    bool multiple;					// multiple exceptions ?
    mutable bool cleanup;				// => delete "cause"

    UnhandledException( const uBaseEvent *cause, const char *const msg = "" );
    UnhandledException( const UnhandledException *cause );
    UnhandledException( const uBaseCoroutine::UnhandledException &ex );
  public:
    virtual ~UnhandledException();
    const uBaseCoroutine &origSource() const;
    const char *origName() const;
    virtual void defaultTerminate() const;
    void triggerCause();
}; // uBaseCoroutine::UnhandledException


_Event uBaseCoroutine::UnwindStack {
    friend class uBaseCoroutine;
    friend class uMachContext;				// access: exec_dtor
    friend _Task Pthread;				// access: exec_dtor

    bool exec_dtor;

    UnwindStack( bool = false );
  public:
    ~UnwindStack();			
}; // uBaseCoroutine::UnwindStack


//######################### Real-Time (cont) #########################


class uBaseScheduleFriend {
  protected:
    virtual ~uBaseScheduleFriend() {}
    uBaseTask &getInheritTask( uBaseTask &task ) const;
    int getActivePriority( uBaseTask &task ) const;
    int getActivePriorityValue( uBaseTask &task ) const;
    int setActivePriority( uBaseTask &task1, int priority );
    int setActivePriority( uBaseTask &task1, uBaseTask &task2 );
    int getBasePriority( uBaseTask &task ) const;
    int setBasePriority( uBaseTask &task, int priority );
    int getActiveQueueValue( uBaseTask &task ) const;
    int setActiveQueue( uBaseTask &task1, int priority );
    int getBaseQueue( uBaseTask &task ) const;
    int setBaseQueue( uBaseTask &task, int priority );
    bool isEntryBlocked( uBaseTask &task ) const;
    bool checkHookConditions( uBaseTask &task1, uBaseTask &task2 ) const;
}; // uBaseScheduleFriend


template<class Node> class uBaseSchedule : protected uBaseScheduleFriend {
  public:
    virtual bool empty() const = 0;
    virtual void add( Node *node ) = 0;
    virtual Node *drop() = 0;
    virtual void remove( uBaseTaskDL *node ) = 0;
    virtual bool checkPriority( Node &owner, Node &calling ) = 0;
    virtual void resetPriority( Node &owner, Node &calling ) = 0;
    virtual void addInitialize( uBaseTaskSeq &taskList ) = 0;
    virtual void removeInitialize( uBaseTaskSeq &taskList ) = 0;
    virtual void rescheduleTask( uBaseTaskDL *taskNode, uBaseTaskSeq &taskList ) = 0;
}; // uBaseSchedule


class uBasePrioritySeq : public uBaseScheduleFriend {
    friend class uSerial;
  protected:
    uBaseTaskSeq list;
    bool executeHooks;
  public:
    uBasePrioritySeq();
    virtual bool empty() const;
    virtual uBaseTaskDL *head() const;
    virtual int add( uBaseTaskDL *node, uBaseTask *owner );
    virtual uBaseTaskDL *drop();
    virtual void remove( uBaseTaskDL *node );
    virtual void onAcquire( uBaseTask &owner );
    virtual void onRelease( uBaseTask &oldOwner );
    int reposition( uBaseTask &task, uSerial &sserial );
}; // uBasePrioritySeq


class uBasePriorityQueue : public uBasePrioritySeq {
    uQueue<uBaseTaskDL> list;
  public:
    virtual bool empty() const;
    virtual uBaseTaskDL *head() const;
    virtual int add( uBaseTaskDL *node, uBaseTask *owner ); 
    virtual uBaseTaskDL *drop();
    virtual void remove( uBaseTaskDL *node ); 
    virtual void onAcquire( uBaseTask &owner );
    virtual void onRelease( uBaseTask &oldOwner );
}; // uBasePriorityQueue


class uRepositionEntry {
    uBaseTask &blocked, &calling;
    uSerial &bSerial, &cSerial;
  public:
    uRepositionEntry( uBaseTask &blocked, uBaseTask &calling );
    int uReposition( bool relCallingLock );
}; // uRepositionEntry


#ifdef KNOT
extern int g_ready_queue, g_mutex_queue, g_pthread_lock, g_io_lock;
#endif // KNOT


class uDefaultScheduler : public uBaseSchedule<uBaseTaskDL> {
    uBaseTaskSeq list;					// list of tasks awaiting execution
  public:
    bool empty() const { return list.empty(); }
    void add( uBaseTaskDL *taskNode ) {
#ifdef KNOT
	g_ready_queue += 1;
#endif // KNOT
	list.addTail( taskNode );
    }
    uBaseTaskDL *drop() {
#ifdef KNOT
	g_ready_queue -= 1;
#endif // KNOT
	return list.dropHead();
    }
    void remove( uBaseTaskDL *node ) {
#ifdef KNOT
	g_ready_queue -= 1;
#endif // KNOT
	list.remove( node );
    }
    bool checkPriority( uBaseTaskDL &owner, uBaseTaskDL &calling ) { return false; }
    void resetPriority( uBaseTaskDL &owner, uBaseTaskDL &calling ) {}
    void addInitialize( uBaseTaskSeq &taskList ) {};
    void removeInitialize( uBaseTaskSeq &taskList ) {};
    void rescheduleTask( uBaseTaskDL *taskNode, uBaseTaskSeq &taskList ) {};
}; // uDefaultScheduler


//######################### Translator Generated Definitions #########################


class uBasePIQ {
  protected:
    virtual ~uBasePIQ() {}
  public:
    virtual int getHighestPriority() = 0;
}; // uBasePIQ


#include <uPIHeap.h>


enum uAction { uNo, uYes };				// used to indicate the most derived constructor/destructor of an object


class uCoroutineConstructor {				// placed in the constructor of a coroutine
  public:
    uCoroutineConstructor( uAction f, uSerial &serial, uBaseCoroutine &coroutine, const char *name );
} __attribute__(( unused )); // uCoroutineConstructor


class uCoroutineDestructor {				// placed in the destructor of a coroutine
    uAction f;
    uBaseCoroutine &coroutine;
  public:
    uCoroutineDestructor( uAction f, uBaseCoroutine &coroutine );
    ~uCoroutineDestructor();
}; // uCoroutineDestructor


class uCoroutineMain {					// placed in the main member of a coroutine
  private:
    uBaseCoroutine &coroutine;
  public:
    uCoroutineMain( uBaseCoroutine &coroutine );
    ~uCoroutineMain();					// needed by the debugger
}; // uCoroutineMain


class uSerial {
    friend class uCondition;				// access: acceptSignalled, leave2
    friend class uSerialConstructor;			// access: mr, prevSerial, leave
    friend class uSerialDestructor;			// access: prevSerial, acceptSignalled, lastAcceptor, mutexOwner, leave, ~uSerialDestructor, enterDestructor
    friend class uTaskConstructor;			// access: acceptSignalled
    friend class uSerialMember;				// access: lastAcceptor, alive, enter, leave
    friend _Task uBootTask;				// access: acceptSignalled
    friend class uMachContext;				// access: leave2
    friend class uBasePrioritySeq;			// access: mutexOwnerlock TEMPORARY
    friend class uRepositionEntry;			// access: lock, entryList TEMPORARY
    friend class uBaseScheduleFriend;			// access: checkHookConditions
    friend class uTimeoutHndlr;				// access: enterTimeout

    // profiling

    friend class uProfileTaskSampler;			// access: profileTaskSamplerInstance

    uSpinLock lock;					// provide mutual exclusion while examining serial state
    uBaseTask *mutexOwner;				// active thread in the mutex object
    uBitSet< __U_MAXENTRYBITS__ > mask;			// entry mask of accepted mutex members and timeout
    unsigned int *mutexMaskPosn;			// location to place mask position in accept statement
    uBasePrioritySeq &entryList;			// tasks waiting to enter mutex object
#if 1							// normally use stack; use queue for experiments
    uStack<uBaseTaskDL> acceptSignalled;		// tasks suspended within the mutex object
#else
    uQueue<uBaseTaskDL> acceptSignalled;		// tasks suspended within the mutex object
#endif
    uBaseTask *constructorTask;				// identity of task creating mutex object
    uBaseTask *destructorTask;				// identity of task calling mutex object's destructor
    uSerial *prevSerial;				// task's previous serial (see uSerialMember, recursive entry during constructor)
    unsigned int mr;					// mutex recursion counter for multiple serial-object entry
    enum uDestructorState { NoDestructor, DestrCalled, DestrScheduled }; // identify the state of the destructor
    uDestructorState destructorStatus;			// has the destructor been called ? 
    bool alive;						// serial destroyed ?
    bool acceptMask;					// entry mask set by uAcceptReturn or uAcceptWait
    bool acceptLocked;					// flag indicating if mutex lock has been acquired for the accept statement

    // real-time

    uEventNode uTimeoutEvent;				// event node for event list
    uEventList *events;					// event list when event added

    // exception handling

    uBaseTask *lastAcceptor;				// acceptor of current entry for communication between acceptor and caller

    // profiling

    mutable uProfileTaskSampler *profileSerialSamplerInstance; // pointer to related profiling object

    void resetDestructorStatus();			// allow destructor to be called
    void enter( unsigned int &mr, uBasePrioritySeq &ml, int mp );
    void enterDestructor( unsigned int &mr, uBasePrioritySeq &ml, int mp );
    void enterTimeout();
    void leave( unsigned int mr );
    void leave2();
    void removeTimeout();
    bool checkHookConditions( uBaseTask *task );	// check conditions for executing hooks

    uSerial( uSerial & );				// no copy
    uSerial &operator=( uSerial & );			// no assignment

    void acceptStart( unsigned int &mutexMaskPosn );
    void acceptTry();

    bool acceptTestMask() {
	return mask.isAllClr();
    } // uSerial::acceptTestMask

    void acceptPause();
    void acceptPause( uDuration duration );
    void acceptPause( uTime time );
    void acceptEnd();

    void acceptElse() {
	if ( acceptLocked ) {
	    mask.clrAll();
	    lock.release();
	} // if
    } // uSerial::acceptElse
  public:
    // These members should be private but cannot be because they are
    // referenced from user code.

    uSerial( uBasePrioritySeq &entryList );
    ~uSerial();

    // calls generated by translator in application code
    bool acceptTry( uBasePrioritySeq &ml, int mp );
    bool acceptTry2( uBasePrioritySeq &ml, int mp );

    void acceptSetMask() {
	// The lock acquired at the start of the accept statement cannot be
	// released here, otherwise, it is necessary to recheck the mutex queues
	// before exit. As a consequence, all destructors between here and
	// ~uSerialMember (which executes leave) are executed with the mutex lock
	// closed, preventing tasks from queuing on this mutex object.
	acceptMask = true;
    } // uSerial::acceptSetMask

    bool executeU() {
	acceptPause();
	return true;
    } // uSerial::executeU

    bool executeC() {
	if ( acceptTestMask() ) {
	    acceptElse();
	    return false;
	} // if
	return executeU();
    } // uSerial::executeC

    bool executeU( bool else_ ) {
	if ( else_ ) {
	    acceptElse();
	    return false;
	} // if
	return executeU();
    } // uSerial::executeU

    bool executeC( bool else_ ) {
	if ( else_ ) {
	    acceptElse();
	    return false;
	} // if
	return executeC();
    } // uSerial::executeC

    bool executeU( bool timeout, uTime time ) {
	if ( timeout ) {
	    acceptTry();
	    acceptPause( time );
	    return true;
	} // if
	return executeU();
    } // uSerial::executeU

    bool executeU( bool timeout, uDuration duration );

    bool executeC( bool timeout, uTime time ) {
	if ( timeout ) {
	    acceptTry();
	    acceptPause( time );
	    return true;
	} // if
	return executeC();
    } // uSerial::executeC

    bool executeC( bool timeout, uDuration duration );

    bool executeU( bool timeout, uTime time, bool else_ ) {
	if ( else_ ) {
	    acceptElse();
	    return false;
	} // if
	if ( timeout ) {
	    return executeU( timeout, time );
	} // if
	return executeU();
    } // uSerial::executeU

    bool executeU( bool timeout, uDuration duration, bool else_ );

    bool executeC( bool timeout, uTime time, bool else_ ) {
	if ( else_ ) {
	    acceptElse();
	    return false;
	} // if
	if ( timeout ) {
	    return executeC( timeout, time );
	} // if
	return executeC();
    } // uSerial::executeC

    bool executeC( bool timeout, uDuration duration, bool else_ );

    class uProtectAcceptStmt {
	uSerial &serial;
      public:
	unsigned int mutexMaskPosn;			// bit position (0-127) in the entry mask for active mutex member

	uProtectAcceptStmt( uSerial &serial ) : serial( serial ) {
	    serial.acceptStart( mutexMaskPosn );
	} // uSerial::uProtectAcceptStmt::uProtectAcceptStmt

	uProtectAcceptStmt( uSerial &serial, bool ) : serial( serial ) {
	    serial.removeTimeout();
	    serial.acceptStart( mutexMaskPosn );
	} // uSerial::uProtectAcceptStmt::uProtectAcceptStmt

	~uProtectAcceptStmt() {
	    serial.acceptEnd();
	} // uSerial::uProtectAcceptStmt::~uProtectAcceptStmt
    }; // uSerial::uProtectAcceptStmt

    // exception handling

    _Event Failure;					// raised when entry encounters an exceptional event
    _Event EntryFailure;				// derived from Failure
    _Event RendezvousFailure;				// derived from Failure as accepted statement may be hidden inside any mutex entry
}; // uSerial


class uTimeoutHndlr : public uSignalHandler {		// real-time
    friend class uSerial;				// access: uWakeupHndlr
    
    uSerial &serial;

    uTimeoutHndlr( uBaseTask &task, uSerial &serial );
    uTimeoutHndlr( uSerial &serial );
    void uHandler();
}; // uTimeoutHndlr


_Event uSerial::Failure : public uKernelFailure {	// general event for failure in a mutex member routine
    const uSerial *const serial;			// identify the mutex object of _Cormonitor or _Task
  protected:
    Failure( const uSerial *const serial, const char *const msg = "" );
    Failure( const char *const msg = "" );
  public:
    const uSerial *serialId() const;
}; // uSerial::Failure


_Event uSerial::EntryFailure : public uSerial::Failure {
  public:
    EntryFailure( const uSerial *const serial, const char *const msg = "" );
    EntryFailure( const char *const msg = "" );
    virtual ~EntryFailure();
    virtual void defaultTerminate() const;
}; // uSerial::EntryFailure


_Event uSerial::RendezvousFailure : public uSerial::Failure {
    const uBaseCoroutine *const caller_;
  public:
    RendezvousFailure( const uSerial *const serial, const char *const msg = "" );
    virtual ~RendezvousFailure();
    const uBaseCoroutine *caller() const;
    virtual void defaultTerminate() const;
}; // uSerial::RendezvousFailure


class uTaskConstructor {				// placed in the constructor of a task
    uAction f;
    uSerial &serial;
    uBaseTask &task;
  public:
    uTaskConstructor( uAction f, uSerial &serial, uBaseTask &task, uBasePIQ &piq, const char *n, bool profile );
    ~uTaskConstructor();
} __attribute__(( unused )); // uTaskConstructor


class uTaskDestructor {					// placed in the destructor of a task
    friend class uTaskConstructor;			// access: cleanup

    uAction f;
    uBaseTask &task;

    static void cleanup( uBaseTask &task );
  public:
    uTaskDestructor( uAction f, uBaseTask &task ) : f( f ), task( task ) {
    } // uTaskDestructor::uTaskDestructor

    ~uTaskDestructor();
}; // uTaskDestructor


class uTaskMain {					// placed in the main member of a task
    uBaseTask &task;
  public:
    uTaskMain( uBaseTask &task );
    ~uTaskMain();
}; // uTaskMain


class uSerialConstructor {				// placed in the constructor of a mutex class
    uAction f;
    uSerial &serial;
  public:
    uSerialConstructor( uAction f, uSerial &serial );
    uSerialConstructor( uAction f, uSerial &serial, const char *n );
    ~uSerialConstructor();
}; // uSerialConstructor


class uSerialDestructor {				// placed in the destructor of a mutex class
    uAction f;
    uSerial *prevSerial;				// task's previous serial
    unsigned int mr;					// mutex recursion counter for multiple serial-object entry
    unsigned int nlevel;				// nesting level counter for accessed serial-objects
  public:
    uSerialDestructor( uAction f, uSerial &serial, uBasePrioritySeq &ml, int mp );
    ~uSerialDestructor();
}; // uSerialDestructor


class uSerialMember {					// placed in the mutex member of a mutex class
    uSerial *prevSerial;				// task's previous serial
    unsigned int mr;					// mutex recursion counter for multiple serial-object entry
    unsigned int nlevel;				// nesting level counter for accessed serial-objects

    // exception handling

    friend class uSerial;				// access: caller, acceptor
    friend class uCondition;				// access: caller

    uBaseTask *acceptor;				// acceptor of the entry invocation, NULL => no acceptor
    bool acceptorSuspended;				// true only when the acceptor, if there is one, remains blocked
    bool noUserOverride;				// true when acceptor has not been called

    void finalize( uBaseTask &task );
  public:
    uSerialMember( uSerial &serial, uBasePrioritySeq &ml, int mp );
    ~uSerialMember();
    uBaseTask *uAcceptor() { uBaseTask *temp = acceptor; acceptor = NULL; return temp; }
}; // uSerialMember


//######################### uBaseTask (cont) #########################


class uBaseTask : public uBaseCoroutine {
    friend class uMachContext;				// access: currCoroutine, profileActive, setState, main
    friend class uBaseCoroutine;			// access: currCoroutine, profileActive, setState
    friend uBaseCoroutine &uThisCoroutine();		// access: currCoroutine
    friend class uOwnerLock;				// access: entryRef, profileActive, wake
    friend class uCondLock;				// access: entryRef, ownerLock, profileActive, wake
    friend class uSpinLock;                             // access: profileActive
    friend class UPP::uSemaphore;			// access: entryRef, wake
    friend class uSerial;				// access: everything
    friend class uSerialConstructor;			// access: profileActive, setSerial
    friend class uSerialDestructor;			// access: mutexRef, profileActive, mutexRecursion, setState
    friend class uCondition;				// access: currCoroutine, mutexRef, info, profileActive
    friend class uCoroutineConstructor;			// access: profileActive
    friend class uCoroutineDestructor;			// access: profileActive
    friend class uTaskConstructor;			// access: currCluster, mutexRef, profileActive, setSerial
    friend class uTaskDestructor;			// access: currCluster, profileActive
    friend class uTaskMain;				// access: recursion, profileActive 
    friend _Coroutine UPP::uProcessorKernel;		// access: currCoroutine, setState, wake
    friend _Task uProcessorTask;			// access: currCluster, uBaseTask
    friend class uCluster;				// access: currCluster, readyRef, clusterRef, bound
    friend _Task uBootTask;				// access: wake
    friend class uCoroutineMain;			// access: profileActive
    friend class uHeapManager;				// access: profileActive
    friend class uKernelModule;				// access: currCoroutine, inheritTask
    friend void *malloc( size_t size ) __THROW;		// access: profileActive
    friend void *memalign( size_t alignment, size_t size ) __THROW; // access: profileActive
    friend class uEventList;				// access: profileActive

    // exception handling

    friend void uEHM::terminateHandler();		// access: terminateRtn
    friend std::terminate_handler std::set_terminate( std::terminate_handler func ) throw(); // access: terminateRtn

    // debugging

    friend class uLocalDebuggerHandler;			// access: taskDebugMask, processBP
    friend _Task uLocalDebugger;			// access: bound, taskDebugMask, debugPCandSRR
    friend class UPP::uSigHandlerModule;		// access: debugPCandSRR

    // profiling

    friend _Task uProfiler;				// access: currCoroutine
    friend void __cyg_profile_func_enter( void *pcCurrentFunction, void *pcCallingFunction );
    friend void __cyg_profile_func_exit( void *pcCurrentFunction, void *pcCallingFunction );
    friend class uExecutionMonitor;			// access: profileActive
    friend _Task uMain;					// access: profileActive
  public:
    enum State { Start, Ready, Running, Blocked, Terminate };
  private:
    void createTask( uCluster &cluster );
    uBaseTask( uCluster &cluster, uProcessor &processor ); // only used by uProcessorTask
    void setState( State state );
    void wake();

    uBaseTask( uBaseTask & );				// no copy
    uBaseTask &operator=( uBaseTask & );		// no assignment

    // debugging

    char taskDebugMask[8];				// 64 bit breakpoint mask for task (used only by debugger)
    void *debugPCandSRR;				// PC of break point; address of return message for IPC SRR
    bool processBP;					// true if task is in the middle of processing a breakpoint

    State state;					// current state of task
    unsigned int recursion;				// allow recursive entry of main member
    unsigned int mutexRecursion;			// number of recursive calls while holding mutex
    uCluster *currCluster;				// cluster task is executing on
    uBaseCoroutine *currCoroutine;			// coroutine being executed by tasks thread
    long int info;					// condition information stored with blocked task

    uBaseTaskDL clusterRef;				// double link field: list of tasks on cluster
    uBaseTaskDL readyRef;				// double link field: ready queue
    uBaseTaskDL entryRef;				// double link field: general entry deque (all waiting tasks)
    uBaseTaskDL mutexRef;				// double link field: mutex member, suspend stack, condition variable
    uProcessor &bound;					// processor to which this task is bound, if applicable
    uBasePrioritySeq *calledEntryMem;			// pointer to called mutex queue
    uOwnerLock *ownerLock;				// pointer to owner lock used for signalling conditions

    // profiling

    bool profileActive;					// indicates if this context is supposed to be profiled
    void profileActivate( uBaseTask &task );

    // exception handling

    uSerialMember *acceptedCall;                       // pointer to the last mutex entry accepted by this thread
    std::terminate_handler terminateRtn;		// per task handling termination action
  protected:
    // real-time

    friend class uSerialMember;				// access: setSerial, currCoroutine, profileActive, acceptedCall
    friend class uBaseScheduleFriend;			// access: entryRef, getInheritTask, setActivePriority, setBasePriority, setActiveQueue, setBaseQueue, uIsEntryBlocked
    friend class uWakeupHndlr;                          // access: wake
    friend class uBasePrioritySeq;			// access: entryRef, mutexRef, calledEntryMem
    friend class uRepositionEntry;			// access: entryList TEMPORARY

    // Duplicate "main" (see uMachContext) to get better error message about
    // missing "main" member for task.
    virtual void main() = 0;				// starting routine for task

    int priority;
    int activePriority;
    uBaseTask *inheritTask;
    int queueIndex;
    int activeQueueIndex;
    uSerial *currSerial;				// current serial task is using (not original serial)

    unsigned int currSerialLevel;			// counter for checking non-nested entry/exit from multiple accessed mutex objects

    uBaseTask &getInheritTask() {
	return *inheritTask;
    } // uBaseTask::getInheritTask

    int setActivePriority( int priority ) {
	int temp = activePriority;
	activePriority = priority;
	return temp;
    } // uBaseTask::setActivePriority

    int setActivePriority( uBaseTask &task ) {
	int temp = activePriority;
	inheritTask = &task;
	activePriority = inheritTask->getActivePriority();
	return temp;
    } // uBaseTask::setActivePriority

    int setBasePriority( int priority ) {
	int temp = priority;
	uBaseTask::priority = priority;
	return temp;
    } // uBaseTask::setBasePriority

    int setActiveQueue( int q ) {
	int temp = activeQueueIndex;
	// uInheritTask = &t;  is this needed or should this just be called from setActivePriority ??
	activeQueueIndex = q;
	return temp;
    } // uBaseTask::setActiveQueue

    int setBaseQueue( int q ) {
	int temp = queueIndex;
	queueIndex = q;
	return temp;
    } // uBaseTask::setBaseQueue

    uSerial &setSerial( uSerial &serial ) {
	uSerial *temp = currSerial;
	currSerial = &serial;
	return *temp;
    } // uBaseTask::setSerial
  public:
    uBaseTask() : clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( uThisCluster() );
    } // uBaseTask::uBaseTask

    uBaseTask( unsigned int stackSize ) : uBaseCoroutine ( stackSize ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( uThisCluster() );
    } // uBaseTask::uBaseTask

    uBaseTask( void *storage, unsigned int storageSize ) : uBaseCoroutine ( storage, storageSize ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( uThisCluster() );
    } // uBaseTask::uBaseTask

    uBaseTask( uCluster &cluster );

    uBaseTask( uCluster &cluster, unsigned int stackSize ) : uBaseCoroutine( stackSize ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( cluster );
    } // uBaseTask::uBaseTask

    uBaseTask( uCluster &cluster, void *storage, unsigned int storageSize ) : uBaseCoroutine( storage, storageSize ), clusterRef( *this ), readyRef( *this ), entryRef( *this ), mutexRef( *this ), bound( *(uProcessor *)0 ) {
	createTask( cluster );
    } // uBaseTask::uBaseTask

    ~uBaseTask() {
    } // uBaseTask::~uBaseTask

    void yield();
    void yield( unsigned int times );

    uCluster &migrate( uCluster &cluster );

    uCluster &getCluster() const {
	return *currCluster;
    } // uBaseTask::getCluster

    uBaseCoroutine &getCoroutine() const {
	return *currCoroutine;
    } // uBaseTask::getCoroutine

    State getState() const {
	return state;
    } // uBaseTask::getState

    int getActivePriority() const {
	// special case for base of active priority stack
	return this == inheritTask ? priority : activePriority;
    } // uBaseTask::getActivePriority

    int getActivePriorityValue() const {		// TEMPORARY: replace previous member?
	 return activePriority;
    } // uBaseTask::getActivePriorityValue

    int getBasePriority() const {
	return priority;
    } // uBaseTask::getBasePriority

    int getActiveQueueValue() const {			// TEMPORARY: rename
	return activeQueueIndex;
    } // uBaseTask::getActiveQueueValue

    int getBaseQueue() const {
	return queueIndex;
    } // uBaseTask::getBaseQueue

    uSerial &getSerial() const {
	return *currSerial;
    } // uBaseTask::getSerial

    // profiling

    void profileActivate();
    void profileInactivate();
    void printCallStack() const;

    // These members should be private but cannot be because they are
    // referenced from user code.

    uBasePIQ *uPIQ;					// TEMPORARY
    void *pthreadData;					// pointer to pthread specific data

    void uYieldNoPoll();
    void uYieldYield( unsigned int times );		// inserted by translator for -yield
    void uYieldInvoluntary();                           // pre-allocates metric memory before yielding

    void uSleep( uDuration duration );
    void uSleep( uTime time );
}; // uBaseTask


class uWakeupHndlr : public uSignalHandler {		// real-time
    friend class uBaseTask;				// access: uWakeupHndlr

    uWakeupHndlr( uBaseTask &task ) {
	This = &task;
    } // uWakeupHndlr::uWakeupHndlr

    void uHandler() {
	This->wake();
    } // uWakeupHndlr::uHandler
}; // uWakeupHndlr


inline uBaseCoroutine &uThisCoroutine() {
    return *uThisTask().currCoroutine;
} // uThisCoroutine


#if defined( __U_ERRNO_FUNC__ )
// ___errno on solaris, __errno_location on linux, __oserror on irix, __error on freebsd
extern "C" inline int *__U_ERRNO_FUNC__ __THROW;
extern "C" inline int *__U_ERRNO_FUNC__ __THROW {
    return &uThisCoroutine().uErrno;
} // __U_ERRNO_FUNC__
#endif // __U_ERRNO_FUNC__


//######################### uEHM (cont) #########################


inline bool uEHM::pollCheck() {
    uBaseCoroutine &coroutine = uThisCoroutine();
    return ! coroutine.asyncEBuf.empty() || coroutine.cancelled();
} // uEHM::pollCheck 


//######################### Cancellation #########################


class uEnableCancel : public uBaseCoroutine::Cancel<uBaseCoroutine::CancelEnabled> {
  public:
    uEnableCancel() {
	uEHM::poll();
    } // uEnableCancel::uEnableCancel
}; // uEnableCancel

class uDisableCancel : public uBaseCoroutine::Cancel<uBaseCoroutine::CancelDisabled> {
}; // uDisableCancel


//######################### uCondition #########################


class uCondition {
    uQueue<uBaseTaskDL> condQueue;			// queue of blocked tasks
    uSerial *owner;					// mutex object owning condition, only set in wait

    uCondition( uCondition & );				// no copy
    uCondition &operator=( uCondition & );		// no assignment
  public:
    uCondition();
    ~uCondition();

    void wait();					// wait on condition
    void wait( long int info );				// wait on condition with information
    void signal();					// signal condition
    void signalBlock();					// signal condition
    bool empty() const;					// test for tasks on condition
    long int front() const;				// return task information

    // exception handling

    _Event WaitingFailure;				// condition queue deleted before restarted from waiting

    // These members should be private but cannot be because they are
    // referenced from user code.
}; // uCondition


_Event uCondition::WaitingFailure : public uSerial::Failure {
    friend class uCondition;

    const uCondition &cond;
    WaitingFailure( const uCondition &cond, const char *const msg = "" );
  public:
    virtual ~WaitingFailure();
    const uCondition &conditionId() const;
    virtual void defaultTerminate() const;
}; // uCondition::WaitingFailure 


//######################### uNBIO #########################


namespace UPP {
    class uNBIO {					// monitor (private mutex member)
	friend class ::uCluster;			// access: closeFD
	friend _Coroutine uProcessorKernel;		// access: okToSelect, IOPoller
	friend class uSelectTimeoutHndlr;		// access: NBIOnode
	friend class ::uKernelBoot;			// access: uNBIO

	struct NBIOnode : public uSeqable {
	    uSemaphore pending;				// wait for I/O completion
	    uBaseTask *pendingTask;			// name of waiting task in case nominated to IOPoller
	    int nfds;					// return value
	    enum { singleFd, multipleFds } fdType;
	    bool timedout;				// has timeout
	    union {
		struct {				// used if waiting for only one fd
		    uIOClosure *closure;
		    int *uRWE;
		} sfd;
		struct {				// used if waiting for multiple fds
		    fd_mask uNFDs;
		    fd_set *uRFDs;
		    fd_set *uWFDs;
		    fd_set *uEFDs;
		} mfd;
	    } smfd;
	}; // NBIOnode

	fd_set uRFDs, uWFDs, uEFDs;			// master copy of all waiting I/O for all tasks
	fd_set lrfds, lwfds, lefds;			// temporary copies for select call
	fd_set tfds;					// temporary mask to construct result mask

	int maxFD;					// highest FD used in select mask
	bool okToSelect;				// uniprocessor flag indicating blocking select
	int descriptors;				// declared here so uniprocessor kernel can check if I/O occurred
	uBaseTask *IOPoller;				// pointer to current IO poller task, or 0
	uSequence<NBIOnode> pendingIO;			// list of waiting tasks

	void setMaxFD();
	void checkIO( uDuration delay );
	bool checkPoller( NBIOnode &node );

	void waitOrPoll( NBIOnode &node, uEventNode *uTimeoutEvent = NULL ) {
	    switch ( initSfd( node, uTimeoutEvent ) ) {
	      case false:				// not poller task ?
		node.pending.P();
		if ( ! node.listed() ) break;		// not poller task ?
		// FALL THROUGH
	      case true:
		while ( pollIO( node ) ) uThisTask().uYieldNoPoll(); // busy wait
	    } // if
	} // uNBIO::waitOrPoll
    
	void waitOrPoll( fd_mask nfds, NBIOnode &node, uEventNode *uTimeoutEvent = NULL ) {
	    switch ( initMfds( nfds, node, uTimeoutEvent ) ) {
	      case false:				// not poller task ?
		node.pending.P();
		if ( ! node.listed() ) break;		// not poller task ?
		// FALL THROUGH
	      case true:
		while ( pollIO( node ) ) uThisTask().uYieldNoPoll(); // busy wait
	    } // if
	} // uNBIO::waitOrPoll

	_Mutex void closeFD( int fd );			// called by uCluster
	_Mutex bool pollIO( NBIOnode &node );
	_Mutex bool initSfd( NBIOnode &node, uEventNode *uTimeoutEvent = NULL );
	_Mutex bool initMfds( fd_mask nfds, NBIOnode &node, uEventNode *uTimeoutEvent = NULL );

	uNBIO();

	int select( uIOClosure &closure, int &rwe, timeval *timeout = NULL );
	int select( fd_mask nfds, fd_set *rfds, fd_set *wfds, fd_set *efds, timeval *timeout = NULL );
      public:
    }; // uNBIO


    class uSelectTimeoutHndlr : public uSignalHandler {	// real-time
	friend class uNBIO;
  
	uNBIO::NBIOnode &node;

	uSelectTimeoutHndlr( uBaseTask &task, uNBIO::NBIOnode &node ) : node( node ) {
	    This = &task;
	} // uSelectTimeoutHndlr::uSelectTimeoutHndlr

	void uHandler() {
	    node.timedout = true;
	} // uSelectTimeoutHndlr::uHandle
    }; // uSelectTimeoutHndlr
} // UPP


//######################### uProcessorKernel #########################


namespace UPP {
    _Coroutine uProcessorKernel {
	friend class ::uKernelBoot;			// access: new, uProcessorKernel, ~uProcessorKernel
	friend class ::uSerial;				// access: schedule, kernelClock
	friend class ::uSerialDestructor;		// access: schedule
	friend class ::uOwnerLock;			// access: schedule
	friend class ::uCondLock;			// access: schedule, kernelClock
	friend class uSemaphore;			// access: schedule
	friend class ::uBaseTask;			// access: schedule, kernelClock
	friend _Task ::uProcessorTask;			// access: terminated, kernelClock
	friend class ::uProcessor;			// access: uProcessorKernel
	friend class uNBIO;				// access: kernelClock

	// real-time

	friend class ::uEventList;			// access: schedule
	friend class ::uEventListPop;			// access: kernelClock
	friend class ::uSysEventList;			// access: setTimer
#if defined( __U_ONETIMER__ ) && defined( __U_MULTI__ )
	friend class ::uProcEventList;			// access: kernelClock
#endif // __U_ONETIMER__ && defined( __U_MULTI__ )

	unsigned int kind;				// specific kind of schedule operation
	uSpinLock *prevLock;				// comunication
	uBaseTask *nextTask;				// task to be wakened

	void taskIsBlocking();
	static void schedule();
	static void schedule( uSpinLock *lock );
	static void schedule( uBaseTask *task );
	static void schedule( uSpinLock *lock, uBaseTask *task );
	void scheduleInternal();
	void scheduleInternal( uSpinLock *lock );
	void scheduleInternal( uBaseTask *task );
	void scheduleInternal( uSpinLock *lock, uBaseTask *task );
	void onBehalfOfUser();
	void setTimer( uDuration time );
	void setTimer( uTime time );
#ifndef __U_MULTI__
	void nextProcessor( uProcessorDL *&currProc, uProcessorDL *cycleStart );
#endif // __U_MULTI__
	void main();

	uProcessorKernel();
	~uProcessorKernel();

	void *operator new( size_t, void *storage ) {
	    return storage;
	} // uProcessorKernel::operator new

	void *operator new( size_t size ) {
	    return ::operator new( size );
	} // uProcessorKernel::operator new

	uClock kernelClock;
      public:
    }; // uProcessorKernel
} // UPP


//######################### uProcessor (cont) #########################


class uProcessor {
    friend class uKernelBoot;				// access: new, uProcessor, events, contextEvent, contextSwitchHandler, setContextSwitchEvent
    friend class uKernelModule;				// access: events
    friend class uCluster;				// access: pid, idleRef, external, processorRef, setContextSwitchEvent
    friend _Task uBootTask;				// access: currTask
    friend _Coroutine UPP::uProcessorKernel;		// access: events, currTask, currCluster, procTask, external, globalRef, setContextSwitchEvent
    friend _Task uProcessorTask;			// access: pid, processorClock, preemption, currCluster, setContextSwitchEvent
    friend class UPP::uNBIO;				// access: setContextSwitchEvent
    friend class uEventList;				// access: events, contextSwitchHandler
#if defined( __U_ONETIMER__ ) && defined( __U_MULTI__ )
    friend class uProcEventList;			// access: events
#endif // __U_ONETIMER__ && defined( __U_MULTI__ )
    friend class uEventNode;                            // access: events
    friend class uEventListPop;                         // access: contextSwitchHandler
    friend void exit( int retcode ) __THROW;		// access: uProcExit
    friend __U_START_KT_TYPE__;				// access: everything
#if defined( __i386__ ) || defined( __ia64__ ) && ! defined( __old_perfmon__ )
    friend class HWCounters;				// access: uPerfctrContext (i386) or uPerfmon_fd (ia64)
#endif

    // debugging

    friend _Task uLocalDebugger;			// access: debugIgnore
    friend _Task uLocalDebuggerReader;			// access: debugIgnore

    bool debugIgnore;					// ignore processor migration

    // profiling
    
    friend class uProfileProcessorSampler;		// access: profileProcessorSamplerInstance

#if ! defined( __U_MULTI__ ) || defined( __U_ONETIMER__ )
    static
#endif // ! __U_MULTI__ || __U_ONETIMER__
    uEventList *events;					// list of events for all processors (ONETIMER or !MULTI) or this processor (MULTI & !ONETIMER)
#if defined( __U_ONETIMER__ ) && defined( __U_MULTI__ )
    uProcEventList *procEvents;				// list of events specific to this processor
#endif // __U_ONETIMER__ && defined( __U_MULTI__ )
#if ! defined( __U_MULTI__ )
    static						// shared info on uniprocessor
#endif // ! __U_MULTI__
    uEventNode *contextEvent;				// context-switch node for event list
#if ! defined( __U_MULTI__ )
    static						// shared info on uniprocessor
#endif // ! __U_MULTI__
    uCxtSwtchHndlr *contextSwitchHandler;		// special time slice handler

#ifdef __U_MULTI__
    UPP::uProcessorKernel processorKer;			// need a uProcessorKernel
#endif // __U_MULTI__

#if defined( __i386__ ) && defined( __U_MULTI__ ) && ! defined( __U_PTHREAD__ )
    unsigned int ldtValue;
    
#   define U_MAX_LDT 8192
    static uBitSet< U_MAX_LDT > LDTFreeSet;
    static uOwnerLock LDTFreeSetLock;
    static bool LDTFreeSetInit;
    static int allocLDT();
    static void freeLDT( int oldLDT );
#endif // __i386__ &&__U_MULTI__ && ! __U_PTHREAD__

    // profiling

#ifndef __U_MULTI__
    static
#else
    mutable
#endif // ! __U_MULTI__
    uProfileProcessorSampler *profileProcessorSamplerInstance; // pointer to related profiling object

    uProcessor( uProcessor & );				// no copy
    uProcessor &operator=( uProcessor &);		// no assignment

    void *operator new( size_t, void *storage ) {
	return storage;
    } // uProcessor::operator new
  protected:
    uClock *processorClock;				// clock bound to processor

    uPid_t pid;

    unsigned int preemption;
    unsigned int spin;

    uProcessorTask *procTask;				// handle processor specific requests
    uBaseTaskSeq external;				// ready queue for processor task

    uBaseTask *currTask;				// task executing on processor
    uCluster *currCluster;				// cluster processor currently associated with

    bool detached;					// processor detached ?
    bool terminated;                                    // processor being deleted ?

    uProcessorDL idleRef;				// double link field: list of idle processors
    uProcessorDL processorRef;				// double link field: list of processors on a cluster
    uProcessorDL globalRef;				// double link field: list of all processors

    void createProcessor( uCluster &cluster, bool detached, int ms, int spin );
    void fork( uProcessor *processor );
    void setContextSwitchEvent( int msecs );		// set the real-time timer
    void setContextSwitchEvent( uDuration duration );	// set the real-time timer
#ifdef __U_MULTI__
#if defined( __irix__ ) || defined( __freebsd__ )
    pid_t procCreate( uProcessor *processor );
#endif
#if defined( __irix__ ) ||  ( defined( __linux__ ) && ! defined( __U_PTHREAD__ ) ) || defined( __freebsd__ )
    void procWait( pid_t pid );
#endif
    void procExit( int retcode ) __attribute__(( noreturn ));
#endif // __U_MULTI__

    uProcessor( uCluster &cluster, double );		// used solely during kernel boot
  public:
    void *operator new( size_t size ) {			// must be public for normal dynamic "new"
	return ::operator new( size );
    } // uProcessor::operator new

    uProcessor( unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    uProcessor( bool detached, unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    uProcessor( uCluster &cluster, unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    uProcessor( uCluster &cluster, bool detached, unsigned int ms = uDefaultPreemption(), unsigned int spin = uDefaultSpin() );
    ~uProcessor();

    uClock &getClock() const;
    uPid_t getPid() const;
    uCluster &setCluster( uCluster &cluster );
    uCluster &getCluster() const;
    uBaseTask &getTask() const;
    bool getDetach() const;
    unsigned int setPreemption( unsigned int ms );
    unsigned int getPreemption() const;
    unsigned int setSpin( unsigned int spin );
    unsigned int getSpin() const;
    bool idle() const;
} __attribute__(( unused )); // uProcessor


//######################### uSerial (cont) #########################


inline bool uSerial::executeU( bool timeout, uDuration duration ) {
    return executeU( timeout, activeProcessorKernel->kernelClock.getTime() + duration );
} // uSerial::executeU

inline bool uSerial::executeC( bool timeout, uDuration duration ) {
    return executeC( timeout, activeProcessorKernel->kernelClock.getTime() + duration );
} // uSerial::executeC

inline bool uSerial::executeU( bool timeout, uDuration duration, bool else_ ) {
    return executeU( timeout, activeProcessorKernel->kernelClock.getTime() + duration, else_ );
} // uSerial::executeU

inline bool uSerial::executeC( bool timeout, uDuration duration, bool else_ ) {
    return executeC( timeout, activeProcessorKernel->kernelClock.getTime() + duration, else_ );
} // uSerial::executeC


//######################### uCluster (cont) #########################


#if defined( __solaris__ ) || defined( __irix__ )
#include <sys/select.h>					// fd_set
#else
#include <sys/types.h>					// fd_set
#endif


class uCluster {
    // The cluster cannot be a monitor because the processor kernel is only a
    // coroutine. If the kernel called a mutex routine in the cluster and could
    // not get in, the previous task executed by the kernel is put on the mutex
    // entry queue and the kernel is restarted. When the kernel restarts, it
    // now enters the critical section when it should be scheduling a new
    // task. Therefore explicit locks must be used for these queues.

    friend class uBaseTask;				// access: makeTaskReady, taskAdd, taskRemove
    friend class UPP::uNBIO;				// access: makeProcessorIdle, makeProcessorActive
    friend class uKernelBoot;				// access: new, NBIO, taskAdd, taskRemove
    friend _Coroutine UPP::uProcessorKernel;		// access: NBIO, readyTaskTryRemove, readyTasksEmpty, tasksOnCluster, makeProcessorActive, processorPause
    friend _Task uProcessorTask;			// access: processorAdd, processorRemove
    friend class uTaskConstructor;			// access: taskAdd
    friend class uTaskDestructor;			// access: taskRemove
    friend class uProcessor;				// access: processorAdd, processorRemove
    friend class uRealTimeBaseTask;			// access: taskReschedule
    friend class uPeriodicBaseTask;			// access: taskReschedule
    friend class uSporadicBaseTask;			// access: taskReschedule
    friend class uFileAccess;				// access: closeFD
    friend class uSocket;				// access: closeFD
    friend class uSocketAccept;				// access: closeFD
    friend class uSocketClient;				// access: select
    friend class uFileIO;				// access: select
    friend class uSocketIO;				// access: select

    // debugging

    friend _Task uLocalDebugger;			// access: debugIgnore
    friend _Task uLocalDebuggerReader;			// access: debugIgnore

    bool debugIgnore;					// ignore cluster migration

    // profiling

    friend class uProfileClusterSampler;		// access: profileClusterSamplerInstance

    // real-time

    friend class uEventList;				// access: wakeProcessor
    friend class uProcWakeupHndlr;			// access: wakeProcessor

    uClusterDL globalRef;				// double link field: list of all clusters

    uCluster( uCluster & );				// no copy
    uCluster &operator=( uCluster & );			// no assignment

    void *operator new( size_t, void *storage ) {
	return storage;
    } // uCluster::operator new
  protected:
    const char *name;					// textual name for cluster, default value
    uSpinLock readyIdleTaskLock;			// protect readyTasks, idleProcessors and tasksOnCluster
    uBaseSchedule<uBaseTaskDL> *readyTasks;		// list of tasks awaiting execution by processors on this cluster
    bool defaultReadyTasks;				// indicates if the cluster allocated the ready task
    uProcessorSeq idleProcessors;			// list of idle processors associated with this cluster
    uBaseTaskSeq tasksOnCluster;			// list of tasks on this cluster
    uSpinLock processorsOnClusterLock;
    uProcessorSeq processorsOnCluster;			// list of processors associated with this cluster
    unsigned int stackSize;

    // Make a pointer to allow static declaration for uniprocessor.
#ifndef __U_MULTI__
    static						// shared info on uniprocessor
#endif // ! __U_MULTI__
    UPP::uNBIO *NBIO;					// non-blocking I/O facilities

    // profiling

    mutable uProfileClusterSampler *profileClusterSamplerInstance; // pointer to related profiling object

    static void wakeProcessor( uPid_t pid );
    void processorPause();
    void makeProcessorIdle( uProcessor &processor );
    void makeProcessorActive( uProcessor &processor );
    void makeProcessorActive();

    bool readyTasksEmpty() {
	return readyTasks->empty();
    } // uCluster::readyTasksEmpty

    void makeTaskReady( uBaseTask &readyTask );
    void readyTaskRemove( uBaseTaskDL *task );
    uBaseTask &readyTaskTryRemove();
    void taskAdd( uBaseTask &task );
    void taskRemove( uBaseTask &task );
    void taskReschedule( uBaseTask &task );
    virtual void processorAdd( uProcessor &processor );
    void processorRemove( uProcessor &processor );
    void closeFD( int fd );
    void createCluster( unsigned int stackSize, const char *name );

    int select( uIOClosure &closure, int rwe, timeval *timeout = NULL ) {
	return NBIO->select( closure, rwe, timeout );
    } // uCluster::select
  public:
    void *operator new( size_t size ) {			// must be public for normal dynamic "new"
	return ::operator new( size );
    } // uCluster::operator new

    uCluster( unsigned int stackSize = uDefaultStackSize(), const char *name = "*unnamed*" );
    uCluster( const char *name );
    uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, unsigned int stackSize = uDefaultStackSize(), const char *name = "*unnamed*" );
    uCluster( uBaseSchedule<uBaseTaskDL> &ReadyQueue, const char *name = "*unnamed*" );
    virtual ~uCluster();

    const char *setName( const char *name ) {
	const char *prev = name;
	uCluster::name = name;
	return prev;
    } // uCluster::setName

    const char *getName() const {
	return
#ifdef __U_DEBUG__
	    ( name == NULL || name == (const char *)-1 ) ? "*unknown*" : // storage might be scrubbed
#endif // __U_DEBUG__
	    name;
    } // uCluster::getName

    unsigned int setStackSize( unsigned int stackSize_ ) {
	unsigned int prev = stackSize;
	stackSize = stackSize_;
	return prev;
    } // uCluster::setStackSize

    unsigned int getStackSize() const {
	return stackSize;
    } // uCluster::getStackSize

    void taskResetPriority( uBaseTask &owner, uBaseTask &calling );
    void taskSetPriority( uBaseTask &owner, uBaseTask &calling );

    static const int readSelect;
    static const int writeSelect;
    static const int exceptSelect;

    int select( int fd, int rwe, timeval *timeout = NULL );

    int select( int nfds, fd_set *rfd, fd_set *wfd, fd_set *efd, timeval *timeout = NULL ) {
	return NBIO->select( nfds, rfd, wfd, efd, timeout );
    } // uCluster::select

    const uBaseTaskSeq &getTasksOnCluster() {
	return tasksOnCluster;
    } // uCluster::getTasksOnCluster

    const uProcessorSeq &getProcessorsOnCluster() {
	return processorsOnCluster;
    } // uCluster::getProcessorsOnCluster
}; // uCluster


//######################### uBaseCoroutine (cont) #########################


inline uBaseCoroutine::uBaseCoroutine() : uMachContext( uThisCluster().getStackSize() ) {
    createCoroutine();
} // uBaseCoroutine::uBaseCoroutine


//######################### uPthreadable #########################


extern "C" {						// not all prototypes in pthread.h
#if defined( __linux__ )
    void _pthread_cleanup_push( _pthread_cleanup_buffer *, void (*) (void *), void * ) __THROW;
    void _pthread_cleanup_pop( _pthread_cleanup_buffer *, int ) __THROW;  
#endif
    int pthread_tryjoin_np( pthread_t, void **) __THROW;
    int pthread_getattr_np(pthread_t, pthread_attr_t*) __THROW;
    int pthread_attr_setstack( pthread_attr_t *attr, void *stackaddr, size_t stackSize ) __THROW;
    int pthread_attr_getstack( const pthread_attr_t *attr, void **stackaddr, size_t *stackSize ) __THROW;
} // extern "C"


// uC++ tasks derived from the following have "almost" pthread capabilities

_Task uPthreadable {					// abstract class (inheritance only)
    friend void uMachContext::invokeTask( uBaseTask & ); // access: stop_unwinding
    friend int pthread_create( pthread_t *new_thread_id, const pthread_attr_t *attr, void * (*start_func)( void * ), void *arg ) __THROW;
    friend int pthread_attr_init( pthread_attr_t *attr ) __THROW;
    friend int pthread_attr_destroy( pthread_attr_t *attr ) __THROW;
    friend int pthread_attr_setscope( pthread_attr_t *attr, int contentionscope ) __THROW;
    friend int pthread_attr_getscope( const pthread_attr_t *attr, int *contentionscope ) __THROW;
    friend int pthread_attr_setdetachstate( pthread_attr_t *attr, int detachstate ) __THROW;
    friend int pthread_attr_getdetachstate( const pthread_attr_t *attr, int *detachstate ) __THROW;
    friend int pthread_attr_setstacksize( pthread_attr_t *attr, size_t stacksize ) __THROW;
    friend int pthread_attr_getstacksize( const pthread_attr_t *attr, size_t *stacksize ) __THROW;
    friend int pthread_attr_setstackaddr( pthread_attr_t *attr, void *stackaddr ) __THROW;
    friend int pthread_attr_getstackaddr( const pthread_attr_t *attr, void **stackaddr ) __THROW;
    friend int pthread_attr_setstack( pthread_attr_t *attr, void *stackaddr, size_t stacksize ) __THROW;
    friend int pthread_attr_getstack( const pthread_attr_t *attr, void **stackaddr, size_t *stacksize ) __THROW;
    friend int pthread_getattr_np( pthread_t threadID, pthread_attr_t *attr ) __THROW;
    friend int pthread_attr_setschedpolicy( pthread_attr_t *attr, int policy ) __THROW;
    friend int pthread_attr_getschedpolicy( const pthread_attr_t *attr, int *policy ) __THROW;
    friend int pthread_attr_setinheritsched( pthread_attr_t *attr, int inheritsched ) __THROW;
    friend int pthread_attr_getinheritsched( const pthread_attr_t *attr, int *inheritsched ) __THROW;
    friend int pthread_attr_setschedparam( pthread_attr_t *attr, const struct sched_param *param ) __THROW;
    friend int pthread_attr_getschedparam( const pthread_attr_t *attr, struct sched_param *param ) __THROW;
    friend void pthread_exit( void *status );		// access: joinval
    friend int pthread_join( pthread_t, void **);	// access: attr
    friend int pthread_tryjoin_np( pthread_t, void **) __THROW;
    friend int pthread_detach( pthread_t ) __THROW;
#if defined( __solaris__ )
    friend void __pthread_cleanup_push( void (*)(void *), void *, caddr_t, _cleanup_t *);
    friend void __pthread_cleanup_pop( int, _cleanup_t *);
#elif defined( __linux__ )
    friend void _pthread_cleanup_push( _pthread_cleanup_buffer *, void (*) (void *), void *) __THROW;
    friend void _pthread_cleanup_pop ( _pthread_cleanup_buffer *, int) __THROW;
#elif defined( __freebsd__ )
    friend void pthread_cleanup_push( void (*) (void *), void * );
    friend void pthread_cleanup_pop( int ex );
// #else don't need anything for irix
#endif
    struct Pthread_attr_t {				// thread attributes
	int contentionscope;
	int detachstate;
	size_t stacksize;
	void *stackaddr;
	int policy;
	int inheritsched;
	struct sched_param param;
    } attr;

    static const Pthread_attr_t u_pthread_attr_defaults;

#if defined(__irix__)	
    static uLock unwind_lock;				// unwinding is not thread safe by default on IRIX
#endif
    uBaseCoroutine::Cleanup cleanup_handlers;
    pthread_t pthreadId_;				// used to allow joining with uPthreadables
    _Unwind_Exception uexc;
    bool stop_unwinding;				// indicates that forced unwinding should stop

    uPthreadable( uPthreadable & );			// no copy
    uPthreadable &operator=( uPthreadable & );		// no assignment
    void createPthreadable( const pthread_attr_t *attr_ = NULL );

    static Pthread_attr_t *&get( const pthread_attr_t *attr ) {
	return *((Pthread_attr_t **)attr);
    } // uPthreadable::get
  protected:
    void *joinval;					// pthreads return value
    pthread_attr_t pthread_attr;			// pthread attributes

    uPthreadable( const pthread_attr_t *attr ) :
	    uBaseTask( attr != NULL ? get( attr )->stackaddr : u_pthread_attr_defaults.stackaddr,
		       attr != NULL ? get( attr )->stacksize : u_pthread_attr_defaults.stacksize ) {
	createPthreadable( attr );
    } // uPthreadable::uPthreadable

    uPthreadable() {					// same constructors as for uBaseTask
	createPthreadable();
    } // uPthreadable::uPthreadable

    uPthreadable( unsigned int stackSize ) : uBaseTask( stackSize ) {
	createPthreadable();
    } // uPthreadable::uPthreadable

    uPthreadable( void *storage, unsigned int storageSize ) : uBaseTask( storage, storageSize ) {
	createPthreadable();
    } // uPthreadable::uPthreadable

    uPthreadable( uCluster &cluster ) : uBaseTask( cluster ) {
	createPthreadable();
    } // uPthreadable::uPthreadable

    uPthreadable( uCluster &cluster, unsigned int stackSize ) : uBaseTask( cluster, stackSize ) {
	createPthreadable();
    } // uPthreadable::uPthreadable

    uPthreadable( uCluster &cluster, void *storage, unsigned int storageSize ) : uBaseTask( cluster, storage, storageSize ) {
	createPthreadable();
    } // uPthreadable::uPthreadable

    virtual void main() = 0;				// remove BaseTask from error message
  public:
    ~uPthreadable();
    _Nomutex pthread_t pthreadId() { return pthreadId_; } // returns a pthread id for a uC++ task

    // exception handling

    _Event Failure;					// general failure
    _Event CreationFailure;				// pthread creation failure
  private:
    // The following routines should only have to be called by the original task owner.

    _Mutex void *join() {				// may only be accepted after the task which inherits
	return joinval;					// this terminates and turns into a monitor
    } // uPthreadable::join

    static void restart_unwinding( _Unwind_Reason_Code urc, _Unwind_Exception *e );
    static _Unwind_Reason_Code unwinder_cleaner( int version, _Unwind_Action ,_Unwind_Exception_Class, _Unwind_Exception *, _Unwind_Context *, void * );
    void do_unwind();
    void cleanup_pop( int ex );
#if ! defined( __irix__ )	
    void cleanup_push( void (*routine)(void *), void *args, void *stackaddress );	
    uBaseCoroutine::PthreadCleanup *cleanupStackTop();
#else
  public:
    _Nomutex uBaseCoroutine::Cleanup &cleanupStackTop();
#endif // ! __irix__
}; // uPthreadable


_Event uPthreadable::Failure : public uKernelFailure {
  protected:
    Failure();
}; // uPthreadable::Failure


_Event uPthreadable::CreationFailure : public uPthreadable::Failure {
}; // uPthreadable::CreationFailure


//######################### uMain #########################


_Task uMain : public uPthreadable {
    friend _Task Pthread;				// access: cleanup_handlers

    int argc;
    char **argv;

    // A reference to a variable that holds the return code that the uMain task
    // returns to the OS.

    int &uRetCode;

    // Main routine for the first user task, declared here, defined by user.

    void main();
  public:
    uMain( uCluster &cluster, int size, int argc, char *argv[], int &retcode );
    ~uMain();
}; // uMain


//######################### Kernel Boot #########################


class uKernelBoot {
    static int count;

    static void startup();
    static void finishup();
  public:
    uKernelBoot() {
	count += 1;
	if ( count == 1 ) {
	    startup();
	} // if
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uKernelBoot &)0x%p.uKernelBoot\n", this );
#endif // __U_DEBUG_H__
    } // uKernelBoot::uKernelBoot

    ~uKernelBoot() {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uKernelBoot &)0x%p.~uKernelBoot\n", this );
#endif // __U_DEBUG_H__
	if ( count == 1 ) {
	    finishup();
	} // if
	count -= 1;
    } // uKernelBoot::uKernelBoot
}; // uKernelBoot


class uInitProcessorsBoot {
    static int count;

    static void startup();
    static void finishup();
  public:
    uInitProcessorsBoot() {
	count += 1;
	if ( count == 1 ) {
	    startup();
	} // if
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uInitProcessorsBoot &)0x%p.uInitProcessorsBoot\n", this );
#endif // __U_DEBUG_H__
    } // uInitProcessorsBoot::uInitProcessorsBoot

    ~uInitProcessorsBoot() {
#ifdef __U_DEBUG_H__
	uDebugPrt( "(uInitProcessorsBoot &)0x%p.~uInitProcessorsBoot\n", this );
#endif // __U_DEBUG_H__
	if ( count == 1 ) {
	    finishup();
	} // if
	count -= 1;
    } // uInitProcessorsBoot::~uInitProcessorsBoot
}; // uInitProcessorsBoot


#include <uBaseSelector.h>				// select statement


// debugging

#ifdef __U_DEBUG__
#include <uLocalDebugger.h>
#endif // __U_DEBUG__


// Create an instance in each translation unit, but only the first instance to
// execute performs the system bootstrap. Do not include instances in the
// kernel modules.

#ifndef __U_KERNEL__
#include <ios>
static std::ios_base::Init __ioinit;			// ensure streams are initialized before startup
static uKernelBoot uBootKernel;

#if __U_LOCALDEBUGGER_H__
static uLocalDebuggerBoot uBootLocalDebugger;
#endif // __U_LOCALDEBUGGER_H__

#if defined( __U_PROFILE__ )
#ifndef __U_PROFILEABLE_ONLY__
#include <uProfilerBoot.h>
static uProfilerBoot uBootProfiler;

#endif // __U_PROFILEABLE_ONLY__
#endif // __U_PROFILE__

static uInitProcessorsBoot uBootProcessorsInit;

#endif // __U_KERNEL__


#pragma __U_USER_CODE__


#endif // __U_CPLUSPLUS_H__


// Local Variables: //
// compile-command: "gmake install" //
// End: //
